#!/usr/local/bin/ruby
# Directory Syncronizer (source priority)
#   by Shin-ichiro HARA (sinara@blade.nagaokaut.ac.jp)
#
# Date: 2003.02.28
# Version: 2.00.00

require "dir-compare"
require "ftools"

#################### Rm_rf #####################

class Rm_rf < DirWalker
  def file(s)
    File.unlink(s)
  end

  def dir_out(s)
    Dir.unlink(s)
  end
end

def rm_rf(paths)
  rm = Rm_rf.new
  dc = DirCompare.new(rm)
  Dir.glob(paths).each do |path|
    dc.run(path)
  end
end

#################### Mv_o #####################
class Mv_o < DirPairWalker
  def initialize(force = false)
    @force = force
    @file_dir = false
  end

  def file_file(s, t)
    File.mv(s, t)
  end

  def file_dir(s, t)
    @file_dir = true
  end

  def file_dir_out(s, t)
    @file_dir = false
    Dir.unlink(t)
    File.mv(s, t)
  end

  def file_non(s, t)
    dir = File.dirname(t)
    File.mkpath(dir) unless File.directory?(dir)
    File.mv(s, t)
  end

  def dir_file(s, t)
    File.unlink(t)
    File.mv(s, t)
    throw :prune
  end

  def dir_dir(s, t)
  end

  def dir_dir_out(s, t)
    Dir.rmdir(s)
  end

  def dir_non(s, t)
    dir = File.dirname(t)
    File.mkpath(dir) unless File.directory?(dir)
    File.mv(s, t)
    throw :prune
  end

  def non_file(s, t)
    if @force || @file_dir
      File.unlink(t)
    end
  end

  def non_dir(s, t)
  end

  def non_dir_out(s, t)
    if @force
      Dir.unlink(t)
    end
  end

  def non_non(s, t)
  end
end

def mv_o(target, source, force = false)
  visitor = Mv_o.new(force)
  dc = DirCompare.new(visitor)
  dc.compare(target, source)
end

########## Utility ##########
module FileOperation
  def cp(from, to)
    mkdir(File.dirname(to.to_s))
    File.cp(from.to_s, to.to_s)
    stat = from.stat
    File.chmod(stat.mode, to.to_s)
    File.utime(stat.atime, stat.mtime, to.to_s)
  end
  
  def mkdir(dir)
    File.mkpath(dir.to_s)
  end

  def rm_r(path)
    rm_rf(path)
  end
  
  def mv(from, to)
    mv_o(from.to_s, to.to_s)
  end

  def rpath(targetfile, basedir)
    targetfile, basedir = targetfile.to_s, basedir.to_s
    target = File.expand_path(targetfile).split(File::SEPARATOR)
    base = File.expand_path(basedir).split(File::SEPARATOR)
    depth = 0
    while base[depth] == target[depth]
      depth += 1
    end
    File.join([".."] * (base.size - depth) + target[depth .. -1])
  end
end

########## DirySyncSourcePrior ##########
class DirSyncSourcePrior < DirPairRunner
  include FileOperation

  attr_reader :warns

  def initialize(sdir, tdir, trash, show = false, verbose = true)
    @sdir = sdir
    @tdir = tdir
    @trash = File.join(trash, Time.now.strftime("%Y%m%d"))
    @show = show
    @verbose = verbose
    @warns = []
  end

  def compare(sfd, tfd)
    unless filter(sfd.path) || filter(tfd.path)
      super
    end
  end

  def msg(*msgs)
    puts msgs.join(" ") if @verbose
  end

  def filter(s)
    s =~ /^\.[^.\/]/ || s =~ /~$/ #/
  end  

  def move_to_trash(file)
    s = File.expand_path(file)
    t = File.expand_path(@tdir)
    newfile = File.join(@trash, rpath(s, t))
    msg("MOVE: #{rpath(file, @tdir)} #{rpath(newfile, @tdir)}")
    mv(file, newfile) unless @show
  end

  def file_lt_file(s, t)
    @warns << "OLDER: #{s} < #{t}"
  end

  def file_gt_file(s, t)
    msg(rpath(s, @sdir), "=>", rpath(t, @tdir))

    move_to_trash(t)
    cp(s, t) unless @show
  end

  def file_eq_file(s, t)
    #nop
  end

  def file_dir(s, t)
    @warns << "FILE-DIR: #{s}, #{t} ... SKIP."
    throw :prune
  end

  def file_non(s, t)
    msg(rpath(s, @sdir), "=>", rpath(t, @tdir))
    cp(s, t) unless @show
  end

  def dir_file(s, t)
    @warns << "DIR-FILE: #{s}, #{t} ... SKIP."
    throw :prune
  end

  def dir_dir(s, t)
    #nop
  end

  def dir_non(s, t)
    msg("MKDIR #{t}")
    mkdir(t) unless @show
  end

  def non_file(s, t)
    move_to_trash(t)
  end

  def non_dir(s, t)
    move_to_trash(t)
    throw :prune
  end

  def non_non(s, t)
    #nop
  end
end

if $0 == __FILE__
  require "getopts"
  getopts("hcsWV")
  sdir, tdir, trash = ARGV.shift, ARGV.shift, ARGV.shift
  sdir.sub!(/#{File::SEPARATOR}?$/o, '')
  tdir.sub!(/#{File::SEPARATOR}?$/o, '')
  trash ||= "trash"
  if trash !~ /^\// #/
    dir = File.dirname(File.expand_path(tdir))
    trash = File.join(dir, trash)
  end
  
  if $OPT_h || !tdir
    puts "COPY source-dir to target-dir, by source priority"
    puts "Usage:"
    puts "  #{File.basename($0)} [options] source_dir target_dir"
    puts "Options:"
    puts "        -s:  show process only"
    puts "        -c:  directory check"
    puts "        -W:  not show warning"
    puts "        -V:  terse mode"
    exit
  end
  
  if $OPT_c
    visitor = DirPairViewer.new
  else
    raise "#{sdir} must be a directory." unless File.directory?(sdir)
    raise "#{tdir} must be a directory." if File.file?(tdir)
    visitor = DirSyncSourcePrior.new(sdir, tdir, trash, $OPT_s, !$OPT_V)
  end
  
  DirCompare.new(visitor).compare(sdir, tdir)
  
  if !$OPT_c && !$OPT_W && !visitor.warns.empty?
    $stderr.puts "\nWARNING!! ***************************"
    visitor.warns.each do |msg|
      $stderr.puts msg
    end
  end
end
