=begin
=TestRunner.rb
A test runner UI for GTK.
=end

#<standard_header>
#
# Copyright (C) 2000-2001 Nathaniel Talbott
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#</standard_header>

require 'gtk'
require 'Lapidary/UI/TestRunnerMediator'
require 'Lapidary/UI/TestRunnerUtilities'

module Lapidary
	module UI
		module GTK
			class TestRunner
				extend TestRunnerUtilities
				def TestRunner.run(suite)
					new(suite).start
					
				end
				def initialize(suite)
					@suite = suite
				end
				def start
					setupMediator
					setupUI
					attachToMediator
					startUI
				end
				def setupMediator
					@mediator = TestRunnerMediator.new(@suite)
					suiteName = @suite.to_s
					if ( @suite.kind_of?(Module) )
						suiteName = @suite.name
					end
					suiteNameEntry.set_text(suiteName)
				end
				def setupUI
					mainWindow.signal_connect("destroy", nil) { stop }
					mainWindow.show_all
					faultList.signal_connect("select-child", nil) {
						| list, item, data |
						showFault(item.fault)
					}
					faultList.signal_connect("unselect-child", nil) {
						clearFault
					}
					@red = false
				end
				def attachToMediator
					runButton.signal_connect("clicked", nil) {
						@mediator.runSuite
					}
					@mediator.tuner.addListener(TestRunnerMediator::RESET, method(:resetUI))
					@mediator.tuner.addListener(TestRunnerMediator::FAULT_ADDED, method(:addFault))
					@mediator.tuner.addListener(TestRunnerMediator::RESULT_CHANGED, method(:resultChanged))
					@mediator.tuner.addListener(TestRunnerMediator::STATUS_CHANGED, method(:updateStatus))
				end
				def startUI
					timer = Gtk::timeout_add(0) {
						Gtk::timeout_remove(timer)
						@mediator.runSuite
					}
					Gtk.main
				end
				def stop
					Gtk.main_quit
				end
				def resetUI(count)
					testProgressBar.set_style(greenStyle)
					testProgressBar.configure(0, 0, count)
					@red = false

					runCountLabel.set_text("0")
					assertionCountLabel.set_text("0")
					failureCountLabel.set_text("0")
					errorCountLabel.set_text("0")

					faultList.remove_items(faultList.children)
				end
				def addFault(fault)
					if ( ! @red )
						testProgressBar.set_style(redStyle)
						@red = true
					end
					item = FaultListItem.new(fault)
					item.show
					faultList.append_items([item])
				end
				def showFault(fault)
					rawShowFault(fault.longDisplay)
				end
				def rawShowFault(string)
					faultDetailLabel.set_text(string)
					outerDetailSubPanel.queue_resize
				end
				def clearFault
					rawShowFault("")
				end
				def resultChanged(result)
					testProgressBar.set_value(testProgressBar.get_value + 1)

					runCountLabel.set_text(result.runs.to_s)
					assertionCountLabel.set_text(result.assertions.to_s)
					failureCountLabel.set_text(result.failures.to_s)
					errorCountLabel.set_text(result.errors.to_s)
				end
				def updateStatus(status)
					statusEntry.set_text(status.message)
				end

				def mainWindow
					lazyInitialize(:mainWindow) {
						@mainWindow = Gtk::Window.new(Gtk::WINDOW_TOPLEVEL)
						@mainWindow.set_title("R2 TestRunner")
						@mainWindow.set_usize(800, 600)
						@mainWindow.set_uposition(20, 20)
						@mainWindow.set_policy(true, true, false)
						@mainWindow.add(mainPanel)
					}
				end
				def mainPanel
					lazyInitialize(:mainPanel) {
						@mainPanel = Gtk::VBox.new(false, 0)
						@mainPanel.pack_start(suitePanel, false, false, 0)
						@mainPanel.pack_start(progressPanel, false, false, 0)
						@mainPanel.pack_start(infoPanel, false, false, 0)
						@mainPanel.pack_start(listPanel, false, false, 0)
						@mainPanel.pack_start(detailPanel, true, true, 0)
						@mainPanel.pack_start(statusPanel, false, false, 0)
					}
				end
				def suitePanel
					lazyInitialize(:suitePanel) {
						@suitePanel = Gtk::HBox.new(false, 10)
						@suitePanel.border_width(10)
						@suitePanel.pack_start(Gtk::Label.new("Suite:"), false, false, 0)
						@suitePanel.pack_start(suiteNameEntry, true, true, 0)
						@suitePanel.pack_start(runButton, false, false, 0)
					}
				end
				def suiteNameEntry
					lazyInitialize(:suiteNameEntry) {
						@suiteNameEntry = Gtk::Entry.new
						@suiteNameEntry.set_editable(false)
					}
				end
				def runButton
					lazyInitialize(:runButton) {
						@runButton = Gtk::Button.new("Run")
					}
				end
				def progressPanel
					lazyInitialize(:progressPanel) {
						@progressPanel = Gtk::HBox.new(false, 10)
						@progressPanel.border_width(10)
						@progressPanel.pack_start(testProgressBar, true, true, 0)
					}
				end
				def testProgressBar
					lazyInitialize(:testProgressBar) {
						@testProgressBar = EnhancedProgressBar.new
						@testProgressBar.set_usize(@testProgressBar.allocation.width, 50)
						@testProgressBar.set_style(greenStyle)
					}
				end
				def greenStyle
					lazyInitialize(:greenStyle) {
						@greenStyle = Gtk::Style.new
						@greenStyle.set_bg(Gtk::STATE_PRELIGHT, 0x0000, 0xFFFF, 0x0000)
					}
				end
				def redStyle
					lazyInitialize(:redStyle) {
						@redStyle = Gtk::Style.new
						@redStyle.set_bg(Gtk::STATE_PRELIGHT, 0xFFFF, 0x0000, 0x0000)
					}
				end
				def infoPanel
					lazyInitialize(:infoPanel) {
						@infoPanel = Gtk::HBox.new(false, 0)
						@infoPanel.border_width(10)
						@infoPanel.pack_start(Gtk::Label.new("Runs:"), false, false, 0)
						@infoPanel.pack_start(runCountLabel, true, false, 0)
						@infoPanel.pack_start(Gtk::Label.new("Assertions:"), false, false, 0)
						@infoPanel.pack_start(assertionCountLabel, true, false, 0)
						@infoPanel.pack_start(Gtk::Label.new("Failures:"), false, false, 0)
						@infoPanel.pack_start(failureCountLabel, true, false, 0)
						@infoPanel.pack_start(Gtk::Label.new("Errors:"), false, false, 0)
						@infoPanel.pack_start(errorCountLabel, true, false, 0)
					}
				end
				def runCountLabel
					lazyInitialize(:runCountLabel) {
						@runCountLabel = Gtk::Label.new("0")
						@runCountLabel.set_justify(Gtk::JUSTIFY_LEFT)
					}
				end
				def assertionCountLabel
					lazyInitialize(:assertionCountLabel) {
						@assertionCountLabel = Gtk::Label.new("0")
						@assertionCountLabel.set_justify(Gtk::JUSTIFY_LEFT)
					}
				end
				def failureCountLabel
					lazyInitialize(:failureCountLabel) {
						@failureCountLabel = Gtk::Label.new("0")
						@failureCountLabel.set_justify(Gtk::JUSTIFY_LEFT)
					}
				end
				def errorCountLabel
					lazyInitialize(:errorCountLabel) {
						@errorCountLabel = Gtk::Label.new("0")
						@errorCountLabel.set_justify(Gtk::JUSTIFY_LEFT)
					}
				end
				def listPanel
					lazyInitialize(:listPanel) {
						@listPanel = Gtk::HBox.new
						@listPanel.border_width(10)
						@listPanel.pack_start(listScrolledWindow, true, true, 0)
					}
				end
				def listScrolledWindow
					lazyInitialize(:listScrolledWindow) {
						@listScrolledWindow = Gtk::ScrolledWindow.new
						@listScrolledWindow.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC)
						@listScrolledWindow.set_usize(@listScrolledWindow.allocation.width, 150)
						@listScrolledWindow.add_with_viewport(faultList)
					}
				end
				def faultList
					lazyInitialize(:faultList) {
						@faultList = Gtk::List.new
					}
				end
				def detailPanel
					lazyInitialize(:detailPanel) {
						@detailPanel = Gtk::HBox.new
						@detailPanel.border_width(10)
						@detailPanel.pack_start(detailScrolledWindow, true, true, 0)
					}
				end
				def detailScrolledWindow
					lazyInitialize(:detailScrolledWindow) {
						@detailScrolledWindow = Gtk::ScrolledWindow.new
						@detailScrolledWindow.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC)
						@detailScrolledWindow.set_usize(400, @detailScrolledWindow.allocation.height)
						@detailScrolledWindow.add_with_viewport(outerDetailSubPanel)
					}
				end
				def outerDetailSubPanel
					lazyInitialize(:outerDetailSubPanel) {
						@outerDetailSubPanel = Gtk::VBox.new
						@outerDetailSubPanel.pack_start(innerDetailSubPanel, false, false, 0)
					}
				end
				def innerDetailSubPanel
					lazyInitialize(:innerDetailSubPanel) {
						@innerDetailSubPanel = Gtk::HBox.new
						@innerDetailSubPanel.pack_start(faultDetailLabel, false, false, 0)
					}
				end
				def faultDetailLabel
					lazyInitialize(:faultDetailLabel) {
						@faultDetailLabel = EnhancedLabel.new("")
						style = Gtk::Style.new
						font = Gdk::Font.font_load("-*-Courier New-medium-r-normal--*-120-*-*-*-*-*-*")
						style.set_font(font)
						@faultDetailLabel.set_style(style)
						@faultDetailLabel.set_justify(Gtk::JUSTIFY_LEFT)
						@faultDetailLabel.set_line_wrap(false)
					}
				end
				def statusPanel
					lazyInitialize(:statusPanel) {
						@statusPanel = Gtk::HBox.new
						@statusPanel.border_width(10)
						@statusPanel.pack_start(statusEntry, true, true, 0)
					}
				end
				def statusEntry
					lazyInitialize(:statusEntry) {
						@statusEntry = Gtk::Entry.new
						@statusEntry.set_editable(false)
					}
				end

				def lazyInitialize(symbol)
					if ( instance_eval("@" + symbol.to_s).nil? )
						yield
					end
					return instance_eval("@" + symbol.to_s)
				end
			end

			class EnhancedProgressBar < Gtk::ProgressBar
				def set_style(style)
					super
					hide
					show
				end
			end

			class EnhancedLabel < Gtk::Label
				def set_text(text)
					super(text.gsub(/\n\t/, "\n" + (" " * 4)))
				end
			end

			class FaultListItem < Gtk::ListItem
				attr_reader(:fault)
				def initialize(fault)
					super(fault.shortDisplay)
					@fault = fault
				end
			end
		end
	end
end

if __FILE__ == $0
	Lapidary::UI::GTK::TestRunner.startCommandLineTest
end
