# timerholder.rb -- 
#
# Copyright (c) 2000,2002 Masatoshi SEKI
#
# timerholder.rb is copyrighted free software by Masatoshi SEKI.
# You can redistribute it and/or modify it under the same terms as Ruby.

require 'monitor'
require 'md5'

class TimerHolder
  include MonitorMixin

  class InvalidIndexError < RuntimeError; end

  class Entry
    def initialize(value, sec = nil)
      @expires = sec || Time.at(1)
      @value = value
    end
    attr_accessor :value
    attr_accessor :expires
    
    def expired?
      @expires < Time.now
    end
  end

  def initialize(timeout=600, max_timeout=nil, min_keeper_sleep=5)
    super()
    @pool = {}
    @timeout = timeout
    @max_timeout = max_timeout || timeout
    @min_keeper_sleep = min_keeper_sleep
    @keeper = keeper
  end

  def add(data, extend=nil)
    key = nil
    entry = Entry.new(data, expires(extend))
    synchronize do 
      while true
	key = create_id(entry)
	next if @pool.include? key
	@pool[key] = entry
	break
      end
    end
    @keeper.wakeup
    key
  end

  def fetch(key, dv=nil, extend=nil)
    synchronize do 
      entry = @pool[key]
      return dv if (entry.nil? || entry.expired?)
      entry.expires = expires(extend)
      return entry.value
    end
  end

  def store(key, value, extend=nil)
    synchronize do 
      entry = @pool[key]
      raise InvalidIndexError, key.to_s unless entry && !entry.expired?
      entry.expires = expires(extend)
      entry.value = value
    end
  end

  def include?(key)
    synchronize do 
      entry = @pool[key]
      return false unless entry && !entry.expired?
      true
    end
  end

  def peek(key)
  end

  private
  def expires(extend=nil)
    extend = @max_timeout if extend && extend > @max_timeout
    Time.now + (extend || @timeout)
  end

  def keeper
    keeper_sleep = (@timeout>@min_keeper_sleep) ? @timeout : @min_keeper_sleep
    Thread.new do
      loop do
	synchronize do
	  keys = @pool.keys 
	  keys.each do |key|
	    entry =  @pool[key] 
	    @pool.delete(key) if entry && entry.expired?
	  end
	end
	Thread.stop if @pool.size == 0
	sleep(keeper_sleep)
      end
    end
  end

  def create_id(entry)
    md5 = MD5.new(Time.now.to_s)
    md5.update($$.to_s)
    md5.update(entry.__id__.to_s)
    md5.hexdigest[0,16]
  end
end
