require "dpklib/class"
require "dpklib/dpkstd"
require "dpklib/error"

module Dpklib
  class << self
    def install_singleton_format_and_parse(format_class, format = nil, &block)
      Dpklib.install_singleton_instance(format_class, format, &block)
      format = format_class.instance

      defmethod = class << format_class
                    method(:define_method)
                  end
      # defmethod.call(:format, format.method(:format)) doesn't work.
      # Causes "method `parse' overridden (TypeError)".
      defmethod.call(:format) { |obj|
        format.format(obj)
      }
      defmethod.call(:parse) { |str|
        format.parse(str)
      }
      format
    end
  end #/<< self

  class Format
    FormatUnimplementedError = Class.new(NotImplementedError)
    ParseUnimplementedError = Class.new(NotImplementedError)
    
    FormatError = Dpklib.new_error_class(StandardError) { |msg|
      "Unable to format: #{msg}"
    }
    ParseError = Dpklib.new_error_class(StandardError) { |msg|
      "Unable to parse: #{msg}"
    }
    
    def format(object)
      raise FormatUnimplementedError
    end

    def parse(string)
      raise ParseUnimplementedError
    end
  end #/Format

  class StringFormat < Format
    def format(object)
      object.to_s
    end
    
    def parse(string)
      string
    end

    Dpklib.install_singleton_format_and_parse(self)
  end #/StringFormat

  class HashedFormat < Format
    attr_accessor :object_formatted_hash
    def initialize(object_formatted_hash)
      super()
      @object_formatted_hash = object_formatted_hash
    end

    def format(object)
      if object_formatted_hash.include?(object)
        object_formatted_hash[object]
      else
        format_unknown_object(object)
      end
    end

    def format_unknown_object(obj)
      "UNKOWN(#{obj})"
    end
  end #/HashedFormat

  class EnumFormat < Format
    attr_accessor :first_index, :formatted_array
    def initialize(first_index, *formatted_array)
      @first_index = first_index
      @formatted_array = formatted_array
    end

    def format(object)
      formatted = nil
      case object
      when Fixnum
        formatted = formatted_array[object - first_index]
      end
      formatted || format_unknown_object(object)
    end

    def each_formatted_with_index(&block)
      index = first_index
      for formatted in formatted_array
        yield(formatted, index)
        index += 1
      end
      nil
    end

    def format_unknown_object(obj)
      "UNKOWN(#{obj})"
    end
  end #/EnumFormat

  class DecimalFormat < Format
    def format(object)
      object.to_s
    end

    def parse(string)
      begin
        Dpklib.decimal_integer(string)
      rescue ArgumentError
        raise(ParseError, $!.to_s)
      end
    end

    Dpklib.install_singleton_format_and_parse(self)
    def self.decimal_array(array)
      array.collect { |item|
        parse(item)
      }
    end
  end #/DecimalFormat
  
  class BitArrayFormat < Format
    def format(bools)
      str = ""
      for bool in bools
        str << (bool ? "1" : "0")
      end
      str
    end

    def parse(string)
      array = []
      string.each_byte do |char|
        case char
        when ?0
          array << false
        when ?1
          array << true
        else
          raise ParseError, "#{char.chr} is not suitable for bit."
        end
      end
      array
    end

    Dpklib.install_singleton_format_and_parse(self)
  end #/BitArrayFormat
end #/Dpklib
