#include "ruby.h"
#include "io.h"
#include "gd.h"
#include "gdfontg.h"		/* giant */
#include "gdfontl.h"		/* large */
#include "gdfontmb.h"		/* medium bold */
#include "gdfonts.h"		/* small */
#include "gdfontt.h"		/* tiny */

static VALUE mGD, cImage, cFont, cPolygon;

static void
free_img(iptr)
    gdImagePtr iptr;
{
    if (iptr) {
	gdImageDestroy(iptr);
    }
}

static VALUE
img_s_new(class, w, h)
    VALUE class, w, h;
{
    VALUE img;
    gdImagePtr iptr;

    iptr = gdImageCreate(NUM2INT(w), NUM2INT(h));
    return Data_Wrap_Struct(class,0,free_img,iptr);
}

static VALUE
img_from_gif(class, f)
    VALUE class, f;
{
    VALUE img;
    OpenFile *fptr;
    gdImagePtr iptr;

    Check_Type(f, T_FILE); 
    GetOpenFile(f, fptr);
    io_readable(fptr);

    iptr = gdImageCreateFromGif(fptr->f);
    return Data_Wrap_Struct(class,0,free_img,iptr);
}

static VALUE
img_from_gd(class, f)
    VALUE class, f;
{
    VALUE img;
    OpenFile *fptr;
    gdImagePtr iptr;

    Check_Type(f, T_FILE); 
    GetOpenFile(f, fptr);
    io_readable(fptr);

    iptr = gdImageCreateFromGd(fptr->f);
    return Data_Wrap_Struct(class,0,free_img,iptr);
}

static VALUE
img_from_xbm(class, f)
    VALUE class, f;
{
    VALUE img;
    OpenFile *fptr;
    gdImagePtr iptr;

    Check_Type(f, T_FILE); 
    GetOpenFile(f, fptr);
    io_readable(fptr);

    iptr = gdImageCreateFromXbm(fptr->f);
    return Data_Wrap_Struct(class,0,free_img,iptr);
}

static VALUE
img_destroy(img)
    struct RData *img;
{
    if (img->data) {
	gdImageDestroy((gdImagePtr)img->data);
	img->data = 0;
    }
    return Qnil;
}

static VALUE
img_color_allocate(img, r, g, b)
    VALUE img, r, g, b;
{
    gdImagePtr im;
    int c;

    Data_Get_Struct(img, gdImage, im);
    c = gdImageColorAllocate(im, NUM2INT(r), NUM2INT(g), NUM2INT(b));

    return INT2FIX(c);
}

static VALUE
img_color_deallocate(img, color)
    VALUE img, color;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageColorDeallocate(im, NUM2INT(color));

    return img;
}

static VALUE
img_color_closest(img, r, g, b)
    VALUE img, r, g, b;
{
    gdImagePtr im;
    int c;

    Data_Get_Struct(img, gdImage, im);
    c = gdImageColorClosest(im, NUM2INT(r), NUM2INT(g), NUM2INT(b));

    return INT2FIX(c);
}

static VALUE
img_color_exact(img, r, g, b)
    VALUE img, r, g, b;
{
    gdImagePtr im;
    int c;

    Data_Get_Struct(img, gdImage, im);
    c = gdImageColorExact(im, NUM2INT(r), NUM2INT(g), NUM2INT(b));

    return INT2FIX(c);
}

static VALUE
img_colors_total(img)
    VALUE img;
{
    gdImagePtr im;
    int c;

    Data_Get_Struct(img, gdImage, im);
    c = gdImageColorsTotal(im);

    return INT2FIX(c);
}

static VALUE
img_get_pixel(img, x, y)
    VALUE img, x, y;
{
    gdImagePtr im;
    int c;

    Data_Get_Struct(img, gdImage, im);
    c = gdImageGetPixel(im, NUM2INT(x), NUM2INT(y));

    return INT2FIX(c);
}

static VALUE
img_set_pixel(img, x, y, c)
    VALUE img, x, y, c;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageSetPixel(im, NUM2INT(x), NUM2INT(y), NUM2INT(c));

    return img;
}

static VALUE
img_rgb(img, idx)
    VALUE img, idx;
{
    gdImagePtr im;
    VALUE ary = ary_new2(3);
    int i, c;

    Data_Get_Struct(img, gdImage, im);
    i = NUM2INT(idx);

    c = gdImageRed(im, i);
    ary_push(ary, INT2FIX(c));
    c = gdImageGreen(im, i);
    ary_push(ary, INT2FIX(c));
    c = gdImageBlue(im, i);
    ary_push(ary, INT2FIX(c));

    return ary;
}

static VALUE
img_transparent(img, idx)
    VALUE img, idx;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);

    gdImageColorTransparent(im, NUM2INT(idx));

    return img;
}

static void
image_req(img)
    VALUE img;
{
    if (!obj_is_kind_of(img, cImage)) {
	TypeError("GD::Image required");
    }
}

static VALUE
img_set_blush(img, brush)
    VALUE img, brush;
{
    gdImagePtr im, br;

    Data_Get_Struct(img, gdImage, im);
    image_req(brush);
    Data_Get_Struct(brush, gdImage, br);
    gdImageSetBrush(im, br);

    return img;
}

static VALUE
img_set_style(argc, argv, img)
    int argc;
    VALUE *argv;
    VALUE img;
{
    gdImagePtr im, st;
    int *style;
    int i;

    Data_Get_Struct(img, gdImage, im);
    style = ALLOCA_N(int, argc);
    for (i=0; i<argc; i++) {
	style[i] = NUM2INT(argv[i]);
    }

    gdImageSetStyle(im, style, argc);

    return img;
}

static VALUE
img_set_tile(img, tile)
    VALUE img, tile;
{
    gdImagePtr im, ti;

    Data_Get_Struct(img, gdImage, im);
    image_req(tile);
    Data_Get_Struct(tile, gdImage, ti);
    gdImageSetTile(im, ti);

    return img;
}

static VALUE
img_line(img, x1, y1, x2, y2, c)
    VALUE img, x1, y1, x2, y2, c;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageLine(im,NUM2INT(x1),NUM2INT(y1),NUM2INT(x2),NUM2INT(y2),NUM2INT(c));

    return img;
}

static VALUE
img_dashed_line(img, x1, y1, x2, y2, c)
    VALUE img, x1, y1, x2, y2, c;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageDashedLine(im,NUM2INT(x1),NUM2INT(y1),NUM2INT(x2),NUM2INT(y2),NUM2INT(c));

    return img;
}


static VALUE
img_rectangle(img, x1, y1, x2, y2, c)
    VALUE img, x1, y1, x2, y2, c;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageRectangle(im,NUM2INT(x1),NUM2INT(y1),NUM2INT(x2),NUM2INT(y2),NUM2INT(c));

    return img;
}


static VALUE
img_filled_rectangle(img, x1, y1, x2, y2, c)
    VALUE img, x1, y1, x2, y2, c;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageFilledRectangle(im,NUM2INT(x1),NUM2INT(y1),NUM2INT(x2),NUM2INT(y2),NUM2INT(c));

    return img;
}

static void
poly_req(ply)
    VALUE ply;
{
    if (!obj_is_kind_of(ply, cPolygon) || TYPE(ply) != T_ARRAY) {
	TypeError("GD::Polygon required");
    }
}

static VALUE
img_polygon(img, ply, c)
    VALUE img;
    struct RArray *ply;
    VALUE c;
{
    gdImagePtr im;
    gdPointPtr pnt;
    int i, len;

    Data_Get_Struct(img, gdImage, im);

    poly_req(ply);
    len = ply->len/2;
    pnt = ALLOCA_N(gdPoint, len);
    for (i=0; i<len; i++) {
	pnt[i].x = NUM2INT(ply->ptr[i*2]);
	pnt[i].y = NUM2INT(ply->ptr[i*2+1]);
    }

    gdImagePolygon(im, pnt, len, NUM2INT(c));

    return img;
}

static VALUE
img_filled_polygon(img, ply, c)
    VALUE img;
    struct RArray *ply;
    VALUE c;
{
    gdImagePtr im;
    gdPointPtr pnt;
    int i, len;

    Data_Get_Struct(img, gdImage, im);

    poly_req(ply);
    len = ply->len/2;
    pnt = ALLOCA_N(gdPoint, len);
    for (i=0; i<len; i++) {
	pnt[i].x = NUM2INT(ply->ptr[i*2]);
	pnt[i].y = NUM2INT(ply->ptr[i*2+1]);
    }

    gdImageFilledPolygon(im, pnt, len, NUM2INT(c));

    return img;
}

static VALUE
img_arc(img, cx, cy, w, h, s, e, c)
    VALUE img, cx, cy, w, h, s, e, c;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageArc(im,NUM2INT(cx),NUM2INT(cy),NUM2INT(w),NUM2INT(h),NUM2INT(s),NUM2INT(e),NUM2INT(c));

    return img;
}

static VALUE
img_fill(img, x, y, c)
    VALUE img, x, y, c;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageFill(im,NUM2INT(x),NUM2INT(y),NUM2INT(c));

    return img;
}

static VALUE
img_fill_to_border(img, x, y, b, c)
    VALUE img, x, y, b, c;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageFillToBorder(im,NUM2INT(x),NUM2INT(y),NUM2INT(b),NUM2INT(c));

    return img;
}

static VALUE
img_copy(img, img2, dx, dy, sx, sy, w, h)
    VALUE img, img2, dx, dy, sx, sy, w, h;
{
    gdImagePtr im, im2;

    Data_Get_Struct(img, gdImage, im);
    image_req(img2);
    Data_Get_Struct(img2, gdImage, im2);

    gdImageCopy(im,im2,NUM2INT(dx),NUM2INT(dy),NUM2INT(sx),NUM2INT(sy),NUM2INT(w),NUM2INT(h));

    return img;
}

static VALUE
img_copy_resized(img, img2, dx, dy, sx, sy, dw, dh, sw, sh)
    VALUE img, img2, dx, dy, sx, sy, dw, dh, sw, sh;
{
    gdImagePtr im, im2;

    Data_Get_Struct(img, gdImage, im);
    image_req(img2);
    Data_Get_Struct(img2, gdImage, im2);

    gdImageCopyResized(im,im2,NUM2INT(dx),NUM2INT(dy),NUM2INT(sx),NUM2INT(sy),NUM2INT(dw),NUM2INT(dh),NUM2INT(sw),NUM2INT(sh));

    return img;
}

static void
font_req(fnt)
    VALUE fnt;
{
    if (!obj_is_kind_of(fnt, cFont)) {
	TypeError("GD::Font required");
    }
}

static VALUE
img_string(img, fnt, x, y, str, c)
    VALUE img, fnt, x, y, str, c;
{
    gdImagePtr im;
    gdFontPtr  f;

    Check_Type(str, T_STRING);
    Data_Get_Struct(img, gdImage, im);
    font_req(fnt);
    Data_Get_Struct(fnt, gdFont, f);

    gdImageString(im,f,NUM2INT(x),NUM2INT(y),RSTRING(str)->ptr,NUM2INT(c));

    return img;
}

static VALUE
img_string_up(img, fnt, x, y, str, c)
    VALUE img, fnt, x, y, str, c;
{
    gdImagePtr im;
    gdFontPtr  f;

    Check_Type(str, T_STRING);
    Data_Get_Struct(img, gdImage, im);
    font_req(fnt);
    Data_Get_Struct(fnt, gdFont, f);

    gdImageStringUp(im,f,NUM2INT(x),NUM2INT(y),RSTRING(str)->ptr,NUM2INT(c));

    return img;
}

static VALUE
img_char(img, fnt, x, y, ch, c)
    VALUE img, fnt, x, y, ch, c;
{
    gdImagePtr im;
    gdFontPtr  f;
    int ci;

    Data_Get_Struct(img, gdImage, im);
    font_req(fnt);
    Data_Get_Struct(fnt, gdFont, f);

    if (TYPE(ch) == T_STRING) {
	if (RSTRING(ch)->len != 1) {
	    ArgError("string must be 1 byte(%d bytes)", RSTRING(ch)->len);
	}
	ci = RSTRING(ch)->ptr[0];
    }
    else {
	ci = NUM2INT(ch);
    }
    gdImageChar(im,f,NUM2INT(x),NUM2INT(y),c,NUM2INT(c));

    return img;
}

static VALUE
img_char_up(img, fnt, x, y, ch, c)
    VALUE img, fnt, x, y, ch, c;
{
    gdImagePtr im;
    gdFontPtr  f;
    int ci;

    Data_Get_Struct(img, gdImage, im);
    font_req(fnt);
    Data_Get_Struct(fnt, gdFont, f);

    if (TYPE(ch) == T_STRING) {
	if (RSTRING(ch)->len != 1) {
	    ArgError("string must be 1 byte(%d bytes)", RSTRING(ch)->len);
	}
	ci = RSTRING(ch)->ptr[0];
    }
    else {
	ci = NUM2INT(ch);
    }
    gdImageCharUp(im,f,NUM2INT(x),NUM2INT(y),ci,NUM2INT(c));

    return img;
}

static VALUE
img_get_interlace(img)
    VALUE img;
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    if (gdImageGetInterlaced(im)) {
	return TRUE;
    }
    return FALSE;
}

static VALUE
img_set_interlace(img, val)
{
    gdImagePtr im;

    Data_Get_Struct(img, gdImage, im);
    gdImageInterlace(im, RTEST(val));

    return img;
}

static VALUE
img_bounds(img)
    VALUE img;
{
    gdImagePtr im;
    VALUE ary = ary_new2(2);
    int i;

    Data_Get_Struct(img, gdImage, im);
    i = gdImageSX(im);
    ary_push(ary, INT2FIX(i));
    i = gdImageSY(im);
    ary_push(ary, INT2FIX(i));

    return ary;
}

static VALUE
img_width(img)
    VALUE img;
{
    gdImagePtr im;
    int i;

    Data_Get_Struct(img, gdImage, im);
    i = gdImageSX(im);
    return INT2FIX(i);
}

static VALUE
img_height(img)
    VALUE img;
{
    gdImagePtr im;
    int i;

    Data_Get_Struct(img, gdImage, im);
    i = gdImageSY(im);
    return INT2FIX(i);
}

static VALUE
img_gif(img, out)
    VALUE img, out;
{
    gdImagePtr im;
    OpenFile *fptr;
    FILE *f;

    Data_Get_Struct(img, gdImage, im);
    Check_Type(out, T_FILE); 
    GetOpenFile(out, fptr);
    io_writable(fptr);
    f = (fptr->f2) ? fptr->f2 : fptr->f;

    gdImageGif(im, f);

    return img;
}

static VALUE
img_gd(img, out)
    VALUE img, out;
{
    gdImagePtr im;
    OpenFile *fptr;
    FILE *f;

    Data_Get_Struct(img, gdImage, im);
    Check_Type(out, T_FILE); 
    GetOpenFile(out, fptr);
    io_writable(fptr);
    f = (fptr->f2) ? fptr->f2 : fptr->f;

    gdImageGd(im, f);

    return img;
}

static VALUE
ply_new(class)
    VALUE class;
{
    VALUE self = ary_new();

    RBASIC(self)->class = class;
    return self;
}

static VALUE
ply_add_pt(ply, x, y)
    VALUE ply, x, y;
{
    /* type check */
    NUM2INT(x);
    NUM2INT(y);

    ary_push(ply, x);
    ary_push(ply, y);
    return ply;
}

static VALUE
ply_get_pt(ply, idx)
    VALUE ply, idx;
{
    int i = NUM2INT(idx);

    if (RARRAY(ply)->len < idx) return Qnil;
    i *= 2;

    return assoc_new(ary_entry(ply, i), ary_entry(ply, i+1));
}

static VALUE
ply_set_pt(ply, idx, x, y)
    VALUE ply, idx, x, y;
{
    int i = NUM2INT(idx)*2;

    /* type check */
    NUM2INT(x);
    NUM2INT(y);

    ary_store(ply, i,   x);
    ary_store(ply, i+1, y);

    return ply;
}

static VALUE
ply_delete_pt(ply, idx)
    VALUE ply, idx;
{
    int i = NUM2INT(idx)*2;

    ary_delete_at(ply, INT2FIX(i));
    ary_delete_at(ply, INT2FIX(i+1));

    return ply;
}

static VALUE
ply_length(ply)
    VALUE ply;
{
    return INT2FIX(RARRAY(ply)->len);
}

static VALUE
ply_vertices(ply)
    struct RArray *ply;
{
    int i;
    VALUE ary = ary_new2(ply->len/2);

    for (i = 0; i<ply->len; i+=2) {
	ary_push(ary, assoc_new(ply->ptr[i], ply->ptr[i]));
    }
    return ary;
}

static VALUE
ply_bounds(ply)
    struct RArray *ply;
{
    int i, l, t, r, b;
    int nx, ny;

    if (ply->len == 0) {
	l = t = r = b = 0;
    }
    else {
	l = r = NUM2INT(ply->ptr[0]);
	t = b = NUM2INT(ply->ptr[1]);
    }
    for (i = 2; i<ply->len; i+=2) {
	nx = NUM2INT(ply->ptr[i]);
	if (nx < l) l = nx;
	if (nx > r) r = nx;
	ny = NUM2INT(ply->ptr[i+1]);
	if (ny < t) t = ny;
	if (ny > b) b = ny;
    }
    return ary_new3(4, INT2FIX(l), INT2FIX(t), INT2FIX(r), INT2FIX(b));
}

static VALUE
ply_offset(ply, vx, vy)
    struct RArray *ply;
    VALUE vx, vy;
{
    int i, x, y, c;

    x = NUM2INT(vx);
    y = NUM2INT(vy);

    for (i = 0; i<ply->len; i+=2) {
	c = NUM2INT(ply->ptr[i]) + x;
	ply->ptr[i] = INT2FIX(c);
	c = NUM2INT(ply->ptr[i+1]) + y;
	ply->ptr[i+1] = INT2FIX(c);
    }

    return (VALUE)ply;
}

static VALUE
ply_map(argc, argv, ply)
    int argc;
    VALUE *argv;
    struct RArray *ply;
{
    VALUE sl, st, sr, sb, dl, dt, dr, db;
    int sx, sy, dx, dy;
    double xmag, ymag;
    int i, c;

    i = rb_scan_args(argc,argv,"44",&sl,&st,&sr,&sb, &dl,&dt,&dr,&db);

    if (i == 4) {
	int i, l, t, r, b;
	int nx, ny;

	if (ply->len == 0) {
	    l = t = r = b = 0;
	}
	else {
	    l = r = NUM2INT(ply->ptr[0]);
	    t = b = NUM2INT(ply->ptr[1]);
	}
	for (i = 2; i<ply->len; i+=2) {
	    nx = NUM2INT(ply->ptr[i]);
	    if (nx < l) l = nx;
	    if (nx > r) r = nx;
	    ny = NUM2INT(ply->ptr[i+1]);
	    if (ny < t) t = ny;
	    if (ny > b) b = ny;
	}
	sx = l;
	sy = t;
	dx = NUM2INT(sl);
	dy = NUM2INT(st);
	xmag = (double)(NUM2INT(sr) - NUM2INT(sl))/(double)(r - l);
	ymag = (double)(NUM2INT(sb) - NUM2INT(st))/(double)(b - t);
    }
    else if (i == 8) {
	sx = NUM2INT(sl);
	sy = NUM2INT(st);
	dx = NUM2INT(dl);
	dy = NUM2INT(dt);
	xmag = (double)(NUM2INT(dr) - NUM2INT(dl))/
	    (double)(NUM2INT(sr) - NUM2INT(sl));
	ymag = (double)(NUM2INT(db) - NUM2INT(dt))/
	    (double)(NUM2INT(sb) - NUM2INT(st));
    }
    else {
	ArgError("wrong # of arguments (%d for 4 or 8)", argc);
    }

    for (i = 0; i<ply->len; i+=2) {
	c = NUM2INT(ply->ptr[i]);
	c = (c-sx)*xmag+dx;
	ply->ptr[i] = INT2FIX(c);

	c = NUM2INT(ply->ptr[i+1]);
	c = (c-sy)*ymag+dy;
	ply->ptr[i+1] = INT2FIX(c);
    }

    return (VALUE)ply;
}

static VALUE
fnt_create(fnt)
    gdFontPtr fnt;
{
    return Data_Wrap_Struct(cFont, 0, 0, fnt);
}

static VALUE
fnt_new(name)
    char *name;
{
    if (strcmp(name, "Giant") == 0) {
	return fnt_create(gdFontGiant);
    }
    if (strcmp(name, "Large") == 0) {
	return fnt_create(gdFontLarge);
    }
    if (strcmp(name, "Medium") == 0) {
	return fnt_create(gdFontMediumBold);
    }
    if (strcmp(name, "Small") == 0) {
	return fnt_create(gdFontSmall);
    }
    if (strcmp(name, "Tiny") == 0) {
	return fnt_create(gdFontTiny);
    }
    ArgError("undefined font name `%s'", name);
}

static VALUE
fnt_s_new(obj, name)
    VALUE obj;
    struct RString *name;
{
    Check_Type(name, T_STRING);
    return fnt_new(name->ptr);
}

static VALUE
fnt_nchars(fnt)
    VALUE fnt;
{
    gdFontPtr fp;

    Data_Get_Struct(fnt,gdFont,fp);
    return INT2FIX(fp->nchars);
}

static VALUE
fnt_offset(fnt)
    VALUE fnt;
{
    gdFontPtr fp;

    Data_Get_Struct(fnt,gdFont,fp);
    return INT2FIX(fp->offset);
}

static VALUE
fnt_width(fnt)
    VALUE fnt;
{
    gdFontPtr fp;

    Data_Get_Struct(fnt,gdFont,fp);
    return INT2FIX(fp->w);
}

static VALUE
fnt_height(fnt)
    VALUE fnt;
{
    gdFontPtr fp;

    Data_Get_Struct(fnt,gdFont,fp);
    return INT2FIX(fp->h);
}

void
Init_GD()
{
    mGD = rb_define_module("GD");
    cImage = rb_define_class_under(mGD, "Image", cObject);
    rb_define_singleton_method(cImage, "new", img_s_new, 2);
    rb_define_singleton_method(cImage, "newFromGif", img_from_gif, 1);
    rb_define_singleton_method(cImage, "newFromXbm", img_from_xbm, 1);
    rb_define_singleton_method(cImage, "newFromGd",  img_from_gd, 1);
    rb_define_method(cImage, "destroy", img_destroy, 0);

    rb_define_method(cImage, "colorAllocate", img_color_allocate, 3);
    rb_define_method(cImage, "colorDeallocate", img_color_deallocate, 1);
    rb_define_method(cImage, "colorClosest", img_color_closest, 3);
    rb_define_method(cImage, "colorExact", img_color_exact, 3);
    rb_define_method(cImage, "colorsTotal", img_colors_total, 0);
    rb_define_method(cImage, "getPixel", img_get_pixel, 2);
    rb_define_method(cImage, "rgb", img_rgb, 1);
    rb_define_method(cImage, "transparent", img_transparent, 1);

    rb_define_method(cImage, "setBrush", img_set_blush, 1);
    rb_define_const(mGD, "Brushed", INT2FIX(gdBrushed));
    rb_define_method(cImage, "setStyle", img_set_style, -1);
    rb_define_const(mGD, "Styled", INT2FIX(gdStyled));
    rb_define_const(mGD, "StyledBrushed", INT2FIX(gdStyledBrushed));
    rb_define_method(cImage, "setTile", img_set_tile, 1);
    rb_define_const(mGD, "Tiled", INT2FIX(gdTiled));
    rb_define_const(mGD, "Transparent", INT2FIX(gdTransparent));

    rb_define_method(cImage, "setPixel", img_set_pixel, 3);
    rb_define_method(cImage, "line", img_line, 5);
    rb_define_method(cImage, "dashedLine", img_dashed_line, 5);
    rb_define_method(cImage, "rectangle", img_rectangle, 5);
    rb_define_method(cImage, "filledRectangle", img_filled_rectangle, 5);
    rb_define_method(cImage, "polygon", img_polygon, 2);
    rb_define_method(cImage, "filledPolygon", img_filled_polygon, 2);
    rb_define_method(cImage, "arc", img_arc, 7);
    rb_define_method(cImage, "fill", img_fill, 3);
    rb_define_method(cImage, "fillToBorder", img_fill_to_border, 4);
    rb_define_method(cImage, "line", img_line, 5);

    rb_define_method(cImage, "copy", img_copy, 7);
    rb_define_method(cImage, "copyResized", img_copy_resized, 9);

    rb_define_method(cImage, "string", img_string, 5);
    rb_define_method(cImage, "stringUp", img_string_up, 5);
    rb_define_method(cImage, "char", img_char, 5);
    rb_define_method(cImage, "charUp", img_char_up, 5);

    rb_define_method(cImage, "interlace",  img_get_interlace, 0);
    rb_define_method(cImage, "interlace=", img_set_interlace, 1);

    rb_define_method(cImage, "bounds", img_bounds, 0);
    rb_define_method(cImage, "width", img_width, 0);
    rb_define_method(cImage, "height", img_height, 0);

    rb_define_method(cImage, "gif", img_gif, 1);
    rb_define_method(cImage, "gd", img_gd, 1);

    cPolygon = rb_define_class_under(mGD, "Polygon", cObject);
    rb_define_singleton_method(cPolygon, "new", ply_new, 0);

    rb_define_method(cPolygon, "addPt", ply_add_pt, 2);
    rb_define_method(cPolygon, "getPt", ply_get_pt, 1);
    rb_define_method(cPolygon, "setPt", ply_set_pt, 3);
    rb_define_method(cPolygon, "deletePt", ply_delete_pt, 1);
    rb_define_method(cPolygon, "length", ply_length, 0);
    rb_define_method(cPolygon, "vertices", ply_vertices, 0);
    rb_define_method(cPolygon, "bounds", ply_bounds, 0);
    rb_define_method(cPolygon, "offset", ply_offset, 2);
    rb_define_method(cPolygon, "map", ply_map, -1);

    cFont = rb_define_class_under(mGD, "Font", cObject);
    rb_define_singleton_method(cFont, "new", fnt_s_new, 1);

    rb_define_const(cFont, "GiantFont", fnt_new("Giant"));
    rb_define_const(cFont, "SmallFont", fnt_new("Small"));
    rb_define_const(cFont, "LargeFont", fnt_new("Large"));
    rb_define_const(cFont, "MediumFont", fnt_new("Medium"));
    rb_define_const(cFont, "MediumBoldFont", fnt_new("Medium"));
    rb_define_const(cFont, "TinyFont", fnt_new("Tiny"));

    rb_define_method(cFont, "nchars", fnt_nchars, 0);
    rb_define_method(cFont, "offset", fnt_nchars, 0);
    rb_define_method(cFont, "width", fnt_width, 0);
    rb_define_method(cFont, "height", fnt_width, 0);
}
