// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_TESTER_H
#define POOMA_UTILITIES_TESTER_H

//-----------------------------------------------------------------------------
// Classes:
// Tester
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
// Tester: A simple class with some utility methods used to write test
// programs.  It includes an Inform class used to print messages and
// results, a method to update an 'OK' boolean variable about whether the
// test is working correctly, and the ability to search for a few common
// command-line arguments to control how the test program should function.
// In POOMA test codes, Tester object should be created at the beginning,
// after initializing POOMA.  It can be destroyed after finalizing POOMA.
// 
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Utilities/Inform.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
//
// Full Description:
//
// Tester is used to help make it easier to write simple test programs.
// It provides:
//   - A built-in Inform stream
//   - A boolean flag with an "OK"/"Not OK" current setting
//   - Methods to set or update this OK flag
//   - Methods to print out messages to the stream
//   - Exception handlers for use in catch() blocks
//   - The ability to parse some simple command-line options to control
//     a test code's behavior.
//
// You use Tester in this way:
//
// 1. Initialize Pooma as normal.
// 2. Create a Tester instance, giving it the argc, argv from main:
//        Pooma::Tester tester(argc, argv);
// 3. For each test you want to do, you can print out messages to tester,
//    then call the "check" method with a boolean argument:
//        tester << "This is the first test." << endl;
//        tester.check(testval == true);
//    If the argument to "check" is false, it will clear the flag.  Once
//    cleared, it will stay cleared even if further "check" calls are done
//    with an argument of true (so once one test fails, the whole test code
//    will fail).
// 4. When you're done, you can print out a result message, like this:
//        int retval = tester.results("Test code description string");
//    This will print out the message:
//        PASSED ..... message
//    or
//        FAILED ..... message
//    depending on the current value of the OK flag.  It return an exit
//    code value, 1 if the test FAILED, 0 if the test PASSED, that you can
//    use to return from main.  For example:
//
//        int retval = tester.results("My current test.");
//        Pooma::finalize();
//        return retval;
//
// The optional command-line arguments that Tester understands are: (these
// will NOT be stripped from the argc, argv given to Tester in its constructor)
//
//   -v       : turn on verbose output from test program
//   -p <str> : change prefix of test program messages to <str>
//   -q       : do not print out anything at all, just have program
//              return 0 or 1
//
//-----------------------------------------------------------------------------


namespace Pooma {

class Tester
{
public:
  //============================================================
  // Constructors
  //============================================================

  // Create a default Tester object.

  Tester();

  // Create a Tester object which parses argc and argv.

  Tester(int argc, char **argv);


  //============================================================
  // Destructors
  //============================================================

  ~Tester();


  //============================================================
  // Testing accessors
  //============================================================

  // Return the Inform stream used to print out test messages

  inline Inform &out()
    {
      return inform_m;
    }

  // Return the current state of the status flag

  inline bool ok() const
    {
      return ok_m;
    }

  // Return the proper main() return value.  This should be used
  // at the end of a test program to return the proper error code.

  inline int returnValue() const
    {
      return (ok() ? 0 : 1);
    }


  //============================================================
  // Testing operations
  //============================================================

  // Take the provided boolean value, and if it is false, set our
  // ok status to false.  If it is true, do not change the ok status.
  // Return the value of the boolean we are checking.

  inline bool check(bool val)
    {
      ok_m = (ok_m && val);
      if (!ok_m && abort_m) 
        {
          PInsist(0,"Check failed!");
        }
      return val;
    }

  // Take the provided boolean value (2nd arg), and if it is false, set our
  // ok status to false.  If it is true, do not change the ok status.
  // Return the value of the boolean we are checking.  This version 
  // of check will also print out a message of the form:
  //   "Checking <string>: check = <val>, updated status = <status>"

  inline bool check(const char *str, bool val)
    {
      check(val);
      if (str != 0)
	out() << "Checking " << str;
      else
	out() << "Checking";
      out() << ": check = " << val << ", updated status = " << ok_m;
      out() << std::endl;
      return val;
    }

  template<class T>    
  bool check(const char *str, const T &val, const T &correct)
    {
      bool res = check(val == correct);
      if (str != 0)
	out() << "Checking " << str;
      else
	out() << "Checking";
      out() << ": val = " << val << ", correct = " << correct
            << ", updated status = " << ok_m;
      out() << std::endl;
      return res;
    }

  template<class T>    
  bool check(const char *str, const T &val, const T &correct, 
    const T &tol)
    {
      bool res = check(abs(val - correct) < tol);
      if (str != 0)
	out() << "Checking " << str;
      else
	out() << "Checking";
      out() << ": val = " << val << ", correct = " << correct
            << ", updated status = " << ok_m;
      out() << std::endl;
      return res;
    }

  // Just set the OK flag to the given value

  inline void set(bool val)
    {
      ok_m = val;
    }

  // Setters for the quiet_m member and output prefix (needed at least for now
  // on Windows with CodeWarrior, where command-line arguments don't exist.

  inline void setQuiet(bool quiet) { 
    quiet_m = quiet;
    // If we're not verbose, turn off the inform stream.
    if (!verbose_m || quiet_m) {
      out().setOutputLevel(Inform::off);
    } else {
      out().setOutputLevel(Inform::on);
    }
  }
  inline void setVerbose(bool verbose) { 
    verbose_m = verbose;
    // If we're not verbose, turn off the inform stream.
    if (!verbose_m || quiet_m) {
      out().setOutputLevel(Inform::off);
    } else {
      out().setOutputLevel(Inform::on);
    }
  }
  
  inline bool verbose() { return verbose_m; }
  
  inline void setPrefix(char *prefix) { out().setPrefix(prefix); }
  

  // Print out a message to cout about the current status.  If a
  // string is given, print that message out on the same line.
  // This will either print PASSED or FAILED as the first word.
  // This will also return the current main() return code setting.

  int results(const char *msg = 0) const;

  // Exception handlers for use in catch() blocks.  The handlers print
  // notifications of exceptions.

  void exceptionHandler(const char *msg = 0);
  void exceptionHandler(const Assertion &asrt);

private:
  //============================================================
  // Private data members.
  //============================================================

  // The status of the test

  bool ok_m;

  // Should we be quiet about printing everything out?

  bool quiet_m;

  // An Inform stream used to print out messages.

  Inform inform_m;

  // Turn on/off the Inform output stream inform_m:

  bool verbose_m;

  // If this is set, failure will cause a PInsist to be called.
  
  bool abort_m;
  
  //============================================================
  // Private methods.
  //============================================================

  // Parse the arguments and set up our testing state.

  void parse(int argc, char **argv);
};

} // namespace POOMA

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_UTILITIES_TESTER_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Tester.h,v $   $Author: jac $
// $Revision: 1.10 $   $Date: 2000/03/28 01:10:57 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
