// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// SendKernel
// ReceiveKernel
// SendReceive
//-----------------------------------------------------------------------------

#ifndef POOMA_CHEETAH_SEND_RECEIVE_H
#define POOMA_CHEETAH_SEND_RECEIVE_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// SendKernel and ReceiveKernel are special iterates that interact with cheetah
// to send and receive data that gets used in expressions.
//
// SendReceive is a wrapper class that contains a send() and receive()
// function that encapsulate generating the necessary tag and launching the
// SendKernel and ReceiveKernel iterates.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Evaluator/InlineEvaluator.h"
#include "Tulip/Messaging.h"
#include "Utilities/PAssert.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

#if POOMA_CHEETAH

//-----------------------------------------------------------------------------
//
// SendIterate<View>
//
// A SendIterate requests a read lock on a piece of data.  When that read lock
// is granted, we call a cheetah matching handler to send the data to the
// appropriate context.  We construct the SendIterate with a tag that is used
// to match the appropriate ReceiveIterate on the remote context.
//
//-----------------------------------------------------------------------------

template<class View>
class SendIterate
  : public Pooma::Iterate_t
{
public:
  SendIterate(const View &view, int toContext, int tag)
    : Pooma::Iterate_t(Pooma::scheduler()),
      view_m(view),
      toContext_m(toContext),
      tag_m(tag)
 {
    PAssert(toContext >= 0);
   
    hintAffinity(engineFunctor(view_m,
			       DataObjectRequest<BlockAffinity>()));

#if POOMA_REORDER_ITERATES
    // Priority interface was added to r2 version of serial async so that
    // message iterates would run before any other iterates.
    priority(-1);
#endif

    DataObjectRequest<WriteRequest> writeReq(*this);
    DataObjectRequest<ReadRequest> readReq(writeReq);
    engineFunctor(view_m, readReq);
  }

  virtual ~SendIterate()
  {
    DataObjectRequest<WriteRelease> writeReq;
    DataObjectRequest<ReadRelease> readReq(writeReq);
    engineFunctor(view_m, readReq);
  }

  virtual void run()
  {
    Pooma::remoteEngineHandler()->send(toContext_m, tag_m, view_m);
  }

private:

  // Context we're sending the data to.

  int toContext_m;

  // A tag used to match the sent data with the right receive.

  int tag_m;

  // The data we're sending (typically a view of an array).

  View view_m;
};

//-----------------------------------------------------------------------------
//
// ReceiveIterate:
//
// ReceiveIterate requests a write lock on a piece of data.  When that lock
// is granted, we register the data with the cheetah matching handler which
// will fill the block when a message arrives.  The write lock is released
// by the matching handler.
//
//-----------------------------------------------------------------------------

template<class View, class IncomingView>
class ReceiveIterate
  : public Pooma::Iterate_t
{
public:

  typedef ReceiveIterate<View, IncomingView> This_t;

  ReceiveIterate(const View &view, int fromContext, int tag)
    : view_m(view),
      fromContext_m(fromContext),
      tag_m(tag),
      Pooma::Iterate_t(Pooma::scheduler())
  {
    PAssert(fromContext >= 0);
    
    hintAffinity(engineFunctor(view,
			       DataObjectRequest<BlockAffinity>()));

#if POOMA_REORDER_ITERATES
    // Priority interface was added to r2 version of serial async so that
    // message iterates would run before any other iterates.
    priority(-1);
#endif

    DataObjectRequest<WriteRequest> writeReq(*this);
    engineFunctor(view, writeReq);

    Pooma::addIncomingMessage();
  }

  virtual ~ReceiveIterate()
  {
  }

  // If we're using cheetah, but don't support out-of-order execution, then
  // the run method of this iterate must block until the message has been
  // received.  Unlike typical iterates, the work implied by this iterate
  // isn't actually performed in the run method.  The run method merely
  // registers a method that gets handled by cheetah when the appropriate
  // message arrives.

#if !POOMA_REORDER_ITERATES

  bool ready_m;

  static void handle(This_t *me, IncomingView &viewMessage)
  {
    apply(me->view_m, viewMessage);
    me->ready_m = true;
  }

  virtual void run()
  {
    ready_m = false;
    Pooma::remoteEngineHandler()->request(fromContext_m, tag_m,
					  handle, this);

    while (!ready_m)
    {
      Pooma::poll();
    }
  }

#else

  virtual void run()
  {
    Pooma::remoteEngineHandler()->request(fromContext_m, tag_m,
					  apply, view_m);
  }

#endif

private:

  static void apply(const View &viewLocal, IncomingView &viewMessage)
  {
    // For now, we just copy the message into the brick accepting the data.

    KernelEvaluator<InlineKernelTag>::evaluate(viewLocal, OpAssign(),
					       viewMessage);
    
    // Release the received block:
    DataObjectRequest<WriteRelease> writeReq;
    engineFunctor(viewLocal, writeReq);
    
    Pooma::gotIncomingMessage();
  }

  // Context we're sending the data to.

  int fromContext_m;

  // A tag used to match the sent data with the right send.

  int tag_m;

  // The place to put the data we're receiving (typically a view of the
  // engine).;

  View view_m;
};

//-----------------------------------------------------------------------------
//
// SendReceive
//
// SendReceive contains two static functions, send(view, context) and
// receive(view, context).  These functions encapsulate generating matching
// tags for the send and receive and launching the iterates to perform the
// send and receive.
//
//-----------------------------------------------------------------------------

struct SendReceive
{
  template<class View>
  static
  void send(const View &view, int toContext)
  {
    int tag = Pooma::sendTag(toContext);
    Pooma::scheduler().handOff(new SendIterate<View>(view, toContext, tag));
  }
};

template<class IncomingView>
struct Receive
{
  template<class View>
  static
  void receive(const View &view, int fromContext)
  {
    PAssert(fromContext >= 0);
    int tag = Pooma::receiveTag(fromContext);
    Pooma::scheduler().handOff(new ReceiveIterate<View,
			       IncomingView>(view,
					     fromContext, tag));
  }
};


#else // not POOMA_CHEETAH

//-----------------------------------------------------------------------------
//
// SendReceive
//
// The non-cheetah versions of send and receive are empty and should never
// actually be used, since a remote view should only happen when the data
// lives on another context.
//
//-----------------------------------------------------------------------------

struct SendReceive
{
  template<class View>
  static
  void send(const View &view, int toContext)
  {
    PAssert(false);
  }
};

template<class IncomingView>
struct Receive
{
  template<class View>
  static
  void receive(const View &view, int fromContext)
  {
    PAssert(false);
  }
};

#endif // not POOMA_CHEETAH

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_CHEETAH_SEND_RECEIVE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: SendReceive.h,v $   $Author: sa_smith $
// $Revision: 1.8 $   $Date: 2000/06/20 20:46:30 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
