// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// fftwTest1: basic test of FFTW wrapper.
//-----------------------------------------------------------------------------

// Bring in Pooma array machinery.

#include "Pooma/Arrays.h"
#include "Utilities/Tester.h"
#include "Transform/WrapFFTW.h"

static bool OK = true;

#if POOMA_FFTW

template<int D2>
inline
void test1(const Array<2, complex<double>, BrickView<D2> > &a,
	   Pooma::Tester &tester)
{
  Interval<2> dom = a.domain();

  int length  = dom[0].size();
  int lot     = dom[1].size();
  int istride = a.engine().strides()[0];
  int idist   = a.engine().strides()[1];

  double pi = 3.1415926535897932;

  a = sin(2 * pi * (1 + iota(dom).comp(1) ) * iota(dom).comp(0) / length);

  FftwCC fft(1);  // forward fft

  fft.initialize(Loc<2>(length, lot), Loc<2>(istride, idist));

  fft.apply(&a(0,0));

  fft.finalize();

  complex<double> result = double(length) / complex<double>(0.0, 2.0);

  Array<2, complex<double>, Brick> compare(length, lot);

  compare = complex<double>(0.0, 0.0);

  Pooma::blockAndEvaluate();

  int col;
  for (col = 0; col < lot; ++col)
  {
    compare(col + 1, col) = result;
    compare(length - col - 1, col) = -result;
  }

  double diff = sum(abs(a - compare));

  tester.out() << "difference:" << diff << std::endl;

  tester.check("checking result of transform of sin",
	       diff < 0.00001);

  tester.out() << a << std::endl;
}

#endif // POOMA_FFTW

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

#if POOMA_FFTW

  Interval<1> I1(20);
  Interval<1> Is(10,14);
  Interval<3> I3(I1, I1, I1);

  Array<3, complex<double>, Brick> a(I3);

  test1(a(1, I1, Is), tester);

#else // POOMA_FFTW

  tester.out() << "Warning: FFTW is not enabled, so this" << std::endl
	       << "test is empty.  To enable FFTW," << std::endl
	       << "run configure with --fftw." << std::endl;

#endif // POOMA_FFTW
 
  tester.out() << "-------------------------------------" << std::endl;
  int ret = tester.results("fftwTest1");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: fftwTest1.cpp,v $   $Author: jac $
// $Revision: 1.4 $   $Date: 2000/06/19 22:18:43 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
