// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ----------------------------------------------------------------------------
// Various tests of the symmetrize<>() template function on Tensors
// ----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

//-----------------------------------------------------------------------------
// Simplistic named functions that return a Full Tensor with specific symmetry.
// Use these for checking correctness of symmetrize<>() template functions.
//-----------------------------------------------------------------------------

template<int D, class T, class EngineTag>
Tensor<D,T,Full>
makeSymmetric(Tensor<D,T,EngineTag> &x)
{
  Tensor<D,T,Full> y(0.0);
  for (int i = 0; i < D; i++) {
    y(i,i) = x(i,i);
    for (int j = i + 1; j < D; j++) {
      y(i,j) = (x(i,j) + x(j,i))*0.5;
      y(j,i) = y(i,j);
    }
  }
  return y;
}

template<int D, class T, class EngineTag>
Tensor<D,T,Full>
makeAntisymmetric(Tensor<D,T,EngineTag> &x)
{
  Tensor<D,T,Full> y(0.0);
  for (int i = 1; i < D; i++) {
    for (int j = 0; j < i; j++) {
      y(i,j) = (x(i,j) - x(j,i))*0.5;
      y(j,i) = -y(i,j);
    }
  }
  return y;
}

template<int D, class T, class EngineTag>
Tensor<D,T,Full>
makeDiagonal(Tensor<D,T,EngineTag> &x)
{
  Tensor<D,T,Full> y(0.0);
  for (int i = 0; i < D; i++) {
    y(i,i) = x(i,i);
  }
  return y;
}

template<int D, class T, class EngineTag>
Tensor<D,T,Full>
makeFull(Tensor<D,T,EngineTag> &x)
{
  Tensor<D,T,Full> y(0.0);
  for (int i = 0; i < D; i++) {
    for (int j = 0; j < D; j++) {
      y(i,j) = x(i,j);
    }
  }
  return y;
}


//-----------------------------------------------------------------------------
// testSymmetrize function; main program calls this for various D values:
//-----------------------------------------------------------------------------

template<int D>
void testSymmetrize(Pooma::Tester &tester)
{

  tester.out() << std::endl << "========= " << D << "D =========" << std::endl;

  // Create Full, Antisymmetric, Symmetric, and Diagonal Tensors as inputs:
  Tensor<D,double,Full> tf;
  double value = 1.0;
  for (int i = 0; i < D; i++) {
    for (int j = 0; j < D; j++) {
      tf(i,j) = value;
      value++;
    }
  }
  tester.out() << "tf: " << tf << std::endl;
  Tensor<D,double,Antisymmetric> ta;
  value = 1.0;
  for (int i = 0; i < TensorStorageSize<D,Antisymmetric>::Size; i++) {
    ta(i) = i + 1.0;
  }
  tester.out() << "ta: " << ta << std::endl;
  Tensor<D,double,Symmetric> ts;
  for (int i = 0; i < TensorStorageSize<D,Symmetric>::Size; i++) {
    ts(i) = i + 1.0;
  }
  tester.out() << "ts: " << ts << std::endl;
  Tensor<D,double,Diagonal> td;
  for (int i = 0; i < TensorStorageSize<D,Diagonal>::Size; i++) {
    td(i) = i + 1.0;
  }
  tester.out() << "td: " << td << std::endl;

  //---------------------------------------------------------------------------
  // Make Fields of these types, to test forwarding of symmetrize<>() function:
  // Create the physical Domains:
  const int nVerts = 6;
  const int nCells = nVerts - 1;
  int nCellsTot = 1;
  Interval<D> vertexDomain;
  for (int d = 0; d < D; d++) {
    vertexDomain[d] = Interval<1>(nVerts);
    nCellsTot *= nCells;
  }

  // Create the (uniform, logically rectilinear) mesh.
  Vector<D> origin(0.0), spacings(0.2);
  typedef UniformRectilinearMesh<D> Mesh_t;
  Mesh_t mesh(vertexDomain, origin, spacings);

  // Create geometry object:
  typedef DiscreteGeometry<Cell, Mesh_t> Geometry_t;
  Geometry_t geom(mesh);

  // Create the Fields:

  // Full, Antisymmetric, Symmetric, Diagonal Tensor Fields:
  Field<Geometry_t,Tensor<D,double,Full> > tff(geom);
  Field<Geometry_t,Tensor<D,double,Symmetric> > tfs(geom);
  Field<Geometry_t,Tensor<D,double,Antisymmetric> > tfa(geom);
  Field<Geometry_t,Tensor<D,double,Diagonal> > tfd(geom);

  // Assign to the single-Tensor values:
  tff = tf;
  tfs = ts;
  tfa = ta;
  tfd = td;

#ifndef __MWERKS__ // This whole module is too much for CW5 to compile
  // --------------------------------------------------------------------------
  // Symmetrize from Full tensor to {Antisymmetric, Symmetric, Diagonal}:

  // To Antisymmetric:
  if (!tester.check("symmetrize<Antisymmetric>(tf): ", 
                    (symmetrize<Antisymmetric>(tf) == 
                     makeAntisymmetric(tf)))) {
    tester.out() << "symmetrize<Antisymmetric>(tf) = " 
                 << symmetrize<Antisymmetric>(tf) 
                 << " != makeAntisymmetric(tf) = " 
                 << makeAntisymmetric(tf) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Antisymmetric>(tff)): ", 
                    (sum(symmetrize<Antisymmetric>(tff)) == 
                     nCellsTot*makeAntisymmetric(tf)))) {
    tester.out() << "sum(symmetrize<Antisymmetric>(tff)) = " 
                 << sum(symmetrize<Antisymmetric>(tff)) 
                 << " != nCellsTot*makeAntisymmetric(tf) = "
                 << nCellsTot*makeAntisymmetric(tf) << std::endl;
  }


  // To Symmetric:
  if (!tester.check("symmetrize<Symmetric>(tf): ", 
                    (symmetrize<Symmetric>(tf) == 
                     makeSymmetric(tf)))) {
    tester.out() << "symmetrize<Symmetric>(tf) = " 
                 << symmetrize<Symmetric>(tf) 
                 << " != makeSymmetric(tf) = " 
                 << makeSymmetric(tf) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Symmetric>(tff)): ", 
                    (sum(symmetrize<Symmetric>(tff)) == 
                     nCellsTot*makeSymmetric(tf)))) {
    tester.out() << "sum(symmetrize<Symmetric>(tff)) = " 
                 << sum(symmetrize<Symmetric>(tff)) 
                 << " != nCellsTot*makeSymmetric(tf) = "
                 << nCellsTot*makeSymmetric(tf) << std::endl;
  }

  // To Diagonal::
  if (!tester.check("symmetrize<Diagonal>(tf): ", 
                    (symmetrize<Diagonal>(tf) == 
                     makeDiagonal(tf)))) {
    tester.out() << "symmetrize<Diagonal>(tf) = " 
                 << symmetrize<Diagonal>(tf) 
                 << " != makeDiagonal(tf) = " 
                 << makeDiagonal(tf) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Diagonal>(tff)): ", 
                    (sum(symmetrize<Diagonal>(tff)) == 
                     nCellsTot*makeDiagonal(tf)))) {
    tester.out() << "sum(symmetrize<Diagonal>(tff)) = " 
                 << sum(symmetrize<Diagonal>(tff)) 
                 << " != nCellsTot*makeDiagonal(tf) = "
                 << nCellsTot*makeDiagonal(tf) << std::endl;
  }

  // --------------------------------------------------------------------------
  // Symmetrize from Antisymmetric tensor to {Full, Symmetric, Diagonal}:

  // To Full:
  if (!tester.check("symmetrize<Full>(ta): ", 
                    (symmetrize<Full>(ta) == 
                     makeFull(ta)))) {
    tester.out() << "symmetrize<Full>(ta) = " 
                 << symmetrize<Full>(ta) 
                 << " != makeFull(ta) = " 
                 << makeFull(ta) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Full>(tfa)): ", 
                    (sum(symmetrize<Full>(tfa)) == 
                     nCellsTot*makeFull(ta)))) {
    tester.out() << "sum(symmetrize<Full>(tfa)) = " 
                 << sum(symmetrize<Full>(tfa)) 
                 << " != nCellsTot*makeFull(ta) = "
                 << nCellsTot*makeFull(ta) << std::endl;
  }

  // To Symmetric:
  if (!tester.check("symmetrize<Symmetric>(ta): ", 
                    (symmetrize<Symmetric>(ta) == 
                     makeSymmetric(ta)))) {
    tester.out() << "symmetrize<Symmetric>(ta) = " 
                 << symmetrize<Symmetric>(ta) 
                 << " != makeSymmetric(ta) = " 
                 << makeSymmetric(ta) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Symmetric>(tfa)): ", 
                    (sum(symmetrize<Symmetric>(tfa)) == 
                     nCellsTot*makeSymmetric(ta)))) {
    tester.out() << "sum(symmetrize<Symmetric>(tfa)) = " 
                 << sum(symmetrize<Symmetric>(tfa)) 
                 << " != nCellsTot*makeSymmetric(ta) = "
                 << nCellsTot*makeSymmetric(ta) << std::endl;
  }

  // To Diagonal::
  if (!tester.check("symmetrize<Diagonal>(ta): ", 
                    (symmetrize<Diagonal>(ta) == 
                     makeDiagonal(ta)))) {
    tester.out() << "symmetrize<Diagonal>(ta) = " 
                 << symmetrize<Diagonal>(ta) 
                 << " != makeDiagonal(ta) = " 
                 << makeDiagonal(ta) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Diagonal>(tfa)): ", 
                    (sum(symmetrize<Diagonal>(tfa)) == 
                     nCellsTot*makeDiagonal(ta)))) {
    tester.out() << "sum(symmetrize<Diagonal>(tfa)) = " 
                 << sum(symmetrize<Diagonal>(tfa)) 
                 << " != nCellsTot*makeDiagonal(ta) = "
                 << nCellsTot*makeDiagonal(ta) << std::endl;
  }
#endif // __MWERKS__ 

  // --------------------------------------------------------------------------
  // Symmetrize from Symmetric tensor to {Full, Antisymmetric, Diagonal}:

  // To Full:
  if (!tester.check("symmetrize<Full>(ts): ", 
                    (symmetrize<Full>(ts) == 
                     makeFull(ts)))) {
    tester.out() << "symmetrize<Full>(ts) = " 
                 << symmetrize<Full>(ts) 
                 << " != makeFull(ts) = " 
                 << makeFull(ts) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Full>(tfs)): ", 
                    (sum(symmetrize<Full>(tfs)) == 
                     nCellsTot*makeFull(ts)))) {
    tester.out() << "sum(symmetrize<Full>(tfs)) = " 
                 << sum(symmetrize<Full>(tfs)) 
                 << " != nCellsTot*makeFull(ts) = "
                 << nCellsTot*makeFull(ts) << std::endl;
  }

  // To Antisymmetric:
  if (!tester.check("symmetrize<Antisymmetric>(ts): ", 
                    (symmetrize<Antisymmetric>(ts) == 
                     makeAntisymmetric(ts)))) {
    tester.out() << "symmetrize<Antisymmetric>(ts) = " 
                 << symmetrize<Antisymmetric>(ts) 
                 << " != makeAntisymmetric(ts) = " 
                 << makeAntisymmetric(ts) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Antisymmetric>(tfs)): ", 
                    (sum(symmetrize<Antisymmetric>(tfs)) == 
                     nCellsTot*makeAntisymmetric(ts)))) {
    tester.out() << "sum(symmetrize<Antisymmetric>(tfs)) = " 
                 << sum(symmetrize<Antisymmetric>(tfs)) 
                 << " != nCellsTot*makeAntisymmetric(ts) = "
                 << nCellsTot*makeAntisymmetric(ts) << std::endl;
  }

  // To Diagonal::
  if (!tester.check("symmetrize<Diagonal>(ts): ", 
                    (symmetrize<Diagonal>(ts) == 
                     makeDiagonal(ts)))) {
    tester.out() << "symmetrize<Diagonal>(ts) = " 
                 << symmetrize<Diagonal>(ts) 
                 << " != makeDiagonal(ts) = " 
                 << makeDiagonal(ts) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Diagonal>(tfs)): ", 
                    (sum(symmetrize<Diagonal>(tfs)) == 
                     nCellsTot*makeDiagonal(ts)))) {
    tester.out() << "sum(symmetrize<Diagonal>(tfs)) = " 
                 << sum(symmetrize<Diagonal>(tfs)) 
                 << " != nCellsTot*makeDiagonal(ts) = "
                 << nCellsTot*makeDiagonal(ts) << std::endl;
  }

  // --------------------------------------------------------------------------
  // Symmetrize from Diagonal tensor to {Full, Antisymmetric, Symmetric}:

  // To Full:
  if (!tester.check("symmetrize<Full>(td): ", 
                    (symmetrize<Full>(td) == 
                     makeFull(td)))) {
    tester.out() << "symmetrize<Full>(td) = " 
                 << symmetrize<Full>(td) 
                 << " != makeFull(td) = " 
                 << makeFull(td) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Full>(tfd)): ", 
                    (sum(symmetrize<Full>(tfd)) == 
                     nCellsTot*makeFull(td)))) {
    tester.out() << "sum(symmetrize<Full>(tfd)) = " 
                 << sum(symmetrize<Full>(tfd)) 
                 << " != nCellsTot*makeFull(td) = "
                 << nCellsTot*makeFull(td) << std::endl;
  }

  // To Antisymmetric:
  if (!tester.check("symmetrize<Antisymmetric>(td): ", 
                    (symmetrize<Antisymmetric>(td) == 
                     makeAntisymmetric(td)))) {
    tester.out() << "symmetrize<Antisymmetric>(td) = " 
                 << symmetrize<Antisymmetric>(td) 
                 << " != makeAntisymmetric(td) = " 
                 << makeAntisymmetric(td) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Antisymmetric>(tfd)): ", 
                    (sum(symmetrize<Antisymmetric>(tfd)) == 
                     nCellsTot*makeAntisymmetric(td)))) {
    tester.out() << "sum(symmetrize<Antisymmetric>(tfd)) = " 
                 << sum(symmetrize<Antisymmetric>(tfd)) 
                 << " != nCellsTot*makeAntisymmetric(td) = "
                 << nCellsTot*makeAntisymmetric(td) << std::endl;
  }

  // To Symmetric:
  if (!tester.check("symmetrize<Symmetric>(td): ", 
                    (symmetrize<Symmetric>(td) == 
                     makeSymmetric(td)))) {
    tester.out() << "symmetrize<Symmetric>(td) = " 
                 << symmetrize<Symmetric>(td) 
                 << " != makeSymmetric(td) = " 
                 << makeSymmetric(td) << std::endl;
  }
  if (!tester.check("sum(symmetrize<Symmetric>(tfd)): ", 
                    (sum(symmetrize<Symmetric>(tfd)) == 
                     nCellsTot*makeSymmetric(td)))) {
    tester.out() << "sum(symmetrize<Symmetric>(tfd)) = " 
                 << sum(symmetrize<Symmetric>(tfd)) 
                 << " != nCellsTot*makeSymmetric(td) = "
                 << nCellsTot*makeSymmetric(td) << std::endl;
  }

}

//-----------------------------------------------------------------------------
// Main program:
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  testSymmetrize<3>(tester);
  testSymmetrize<2>(tester);
  testSymmetrize<1>(tester);

  int ret = tester.results("TestSymmetrize");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TestSymmetrize.cpp,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/03/07 13:18:18 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
