// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// This program was prepared by the Regents of the University of California
// at Los Alamos National Laboratory (the University) under Contract No. 
// W-7405-ENG-36 with the U.S. Department of Energy (DOE). The University has 
// certain rights in the program pursuant to the contract and the program 
// should not be copied or distributed outside your organization. All rights
// in the program are reserved by the DOE and the University. Neither the U.S.
// Government nor the University makes any warranty, express or implied, or
// assumes any liability or responsibility for the use of this software
//-----------------------------------------------------------------------------
// Class:
// IterateScheduler<SerialAsync>
// Iterate<SerialAsync>
// DataObject<SerialAsync>
//-----------------------------------------------------------------------------

#include <iostream>

#ifndef _SerialAsync_h_
#define _SerialAsync_h_
/*
LIBRARY:
        SerialAsync

CLASSES: IterateScheduler

CLASSES: DataObject

CLASSES: Iterate

OVERVIEW 
        SerialAsync IterateScheduler is a policy template to create a
        dependence graphs and executes the graph respecting the
        dependencies without using threads. There is no parallelism,
        but Iterates may be executed out-of-order with respect to the
        program text.

-----------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
// Smarts classes for times when you want no threads but you do want
// dataflow evaluation.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include <list>
#include "Threads/IterateSchedulers/IterateScheduler.h"
#include "Threads/IterateSchedulers/Runnable.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//inline ostream& lock(ostream& s) { return s; }
//inline ostream& unlock(ostream& s) { return s; }

namespace Smarts {

#define MYID 0
#define MAX_CPUS 1
//
// Tag class for specializing IterateScheduler, Iterate and DataObject.
//
struct SerialAsync 
{
  enum Action { Read, Write};
};


//-----------------------------------------------------------------------------

struct SystemContext
{
  void addNCpus(int) {}
  void wait() {}
  void concurrency(int){}
  int concurrency() {return 1;}
  void mustRunOn() {}

  // We have a separate message queue because they are
  // higher priority.
  static std::list<RunnablePtr_t> workQueueMessages_m;
  static std::list<RunnablePtr_t> workQueue_m;

  ///////////////////////////
  // This function lets you check if there are iterates that are
  // ready to run.
  inline static
  bool workReady()
  {
    return !(workQueue_m.empty() && workQueueMessages_m.empty());
  }

  ///////////////////////////
  // Run an iterate if one is ready.
  inline static
  void runSomething()
  {
    if (!workQueueMessages_m.empty())
    {
      // Get the top iterate.
      // Delete it from the queue.
      // Run the iterate.
      // Delete the iterate.  This could put more iterates in the queue.

      RunnablePtr_t p = workQueueMessages_m.front();
      workQueueMessages_m.pop_front();
      p->execute();
      delete p;
    }
    else
    {
      if (!workQueue_m.empty())
      {
	RunnablePtr_t p = workQueue_m.front();
	workQueue_m.pop_front();
	p->execute();
	delete p;
      }
    }
  }

};

inline void addRunnable(RunnablePtr_t rn)
{
  SystemContext::workQueue_m.push_front(rn);
}

inline void add(RunnablePtr_t rn)
{
  if (rn->priority() == -1)
  {
    SystemContext::workQueueMessages_m.push_front(rn);
  }
  else
  {
    SystemContext::workQueue_m.push_front(rn);
  }
}

inline  void concurrency(int){}
inline  int concurrency() {return 1;}
inline  void wait() {}
inline  void mustRunOn(){}

/*------------------------------------------------------------------------
CLASS
	IterateScheduler_Serial_Async

	Implements a asynchronous scheduler for a data driven execution.
	Specializes a IterateScheduler.

KEYWORDS
	Data-parallelism, Native-interface, IterateScheduler.

DESCRIPTION

        The SerialAsync IterateScheduler, Iterate and DataObject
	implement a SMARTS scheduler that does dataflow without threads.
	What that means is that when you hand iterates to the
	IterateScheduler it stores them up until you call
	IterateScheduler::blockingEvaluate(), at which point it evaluates
	iterates until the queue is empty.
-----------------------------------------------------------------------------*/

template<>
class IterateScheduler<SerialAsync>
{
  friend class DataObject<SerialAsync>;
  friend class Iterate<SerialAsync>;
public:

  typedef DataObject<SerialAsync> DataObject_t;
  typedef Iterate<SerialAsync> Iterate_t;

  ///////////////////////////
  // Constructor
  //
  IterateScheduler() {}

  ///////////////////////////
  // Destructor
  //
  ~IterateScheduler() {}
  void setConcurrency(int) {}

  //---------------------------------------------------------------------------
  // Mutators.
  //---------------------------------------------------------------------------

  ///////////////////////////
  // Tells the scheduler that the parser thread is starting a new
  // data-parallel statement.  Any Iterate that is handed off to the
  // scheduler between beginGeneration() and endGeneration() belongs
  // to the same data-paralllel statement and therefore has the same
  // generation number.
  //
  inline void beginGeneration() { }

  ///////////////////////////
  // Tells the scheduler that no more Iterates will be handed off for
  // the data parallel statement that was begun with a
  // beginGeneration().
  //  
  inline void endGeneration() {}

  ///////////////////////////
  // The parser thread calls this method to evaluate the generated
  // graph until all the nodes in the dependence graph has been
  // executed by the scheduler.  That is to say, the scheduler
  // executes all the Iterates that has been handed off to it by the
  // parser thread.
  //
  inline
  void blockingEvaluate();

  ///////////////////////////
  // The parser thread calls this method to ask the scheduler to run
  // the given Iterate when the dependence on that Iterate has been
  // satisfied.
  //
  inline void handOff(Iterate<SerialAsync>* it);

  inline
  void releaseIterates() { }

protected:
private:

  typedef std::list<Iterate_t*> Container_t;
  typedef Container_t::iterator Iterator_t;

};

//-----------------------------------------------------------------------------

/*------------------------------------------------------------------------
CLASS
	Iterate_SerialAsync

	Iterate<SerialAsync> is used to implement the SerialAsync
	scheduling policy.

KEYWORDS
	Data_Parallelism, Native_Interface, IterateScheduler, Data_Flow.

DESCRIPTION
        An Iterate is a non-blocking unit of concurrency that is used
	to describe a chunk of work. It inherits from the Runnable
	class and as all subclasses of Runnable, the user specializes
	the run() method to specify the operation.
	Iterate<SerialAsync> is a further specialization of the
	Iterate class to use the SerialAsync Scheduling algorithm to
	generate the data dependency graph for a data-driven
	execution.  */

template<>
class Iterate<SerialAsync> : public Runnable
{
  friend class IterateScheduler<SerialAsync>;
  friend class DataObject<SerialAsync>;

public:

  typedef DataObject<SerialAsync> DataObject_t;
  typedef IterateScheduler<SerialAsync> IterateScheduler_t;


  ///////////////////////////
  // The Constructor for this class takes the IterateScheduler and a
  // CPU affinity.  CPU affinity has a default value of -1 which means
  // it may run on any CPU available.
  //
  inline Iterate(IterateScheduler<SerialAsync> & s, int affinity=-1);

  ///////////////////////////
  // The dtor is virtual because the subclasses will need to add to it.
  //
  virtual ~Iterate() {}

  ///////////////////////////
  // The run method does the core work of the Iterate.
  // It is supplied by the subclass.
  //
  virtual void run() = 0;

  ///////////////////////////
  // Stub in all the affinities, because there is no such thing
  // in serial.
  //
  inline int affinity() const {return 0;}
  ///////////////////////////
  // Stub in all the affinities, because there is no such thing
  // in serial.
  //
  inline int hintAffinity() const {return 0;}
  ///////////////////////////
  // Stub in all the affinities, because there is no such thing
  // in serial.
  //
  inline void affinity(int) {}
  ///////////////////////////
  // Stub in all the affinities, because there is no such thing
  // in serial.
  //
  inline void hintAffinity(int) {}

  ///////////////////////////
  // Notify is used to indicate to the Iterate that one of the data
  // objects it had requested has been granted. To do this, we dec a
  // dependence counter which, if equal to 0, the Iterate is ready for
  // execution.
  //
  inline void notify();

  ///////////////////////////
  // How many notifications remain?
  //
  inline
  int notifications() const { return notifications_m; }

  inline void addNotification()
  {
    notifications_m++;
  }

protected:

  // What scheduler are we working with?
  IterateScheduler<SerialAsync> &scheduler_m;

  // How many notifications should we receive before we can run?
  int notifications_m;

private:
  // Set notifications dynamically and automatically every time a
  // request is made by the iterate
  void incr_notifications() { notifications_m++;}

};


//-----------------------------------------------------------------------------

/*------------------------------------------------------------------------
CLASS
	DataObject_SerialAsync

	Implements a asynchronous scheduler for a data driven execution.
KEYWORDS
	Data-parallelism, Native-interface, IterateScheduler.

DESCRIPTION 
        The DataObject Class is used introduce a type to represent
	a resources (normally) blocks of data) that Iterates contend
	for atomic access. Iterates make request for either a read or
	write to the DataObjects. DataObjects may grant the request if
	the object is currently available. Otherwise, the request is
	enqueue in a queue private to the data object until the
	DataObject is release by another Iterate. A set of read
	requests may be granted all at once if there are no
	intervening write request to that DataObject.
	DataObject<SerialAsync> is a specialization of DataObject for
	the policy template SerialAsync. 
*/

template<>
class DataObject<SerialAsync>
{
  friend class IterateScheduler<SerialAsync>;
  friend class Iterate<SerialAsync>;
public:

  typedef IterateScheduler<SerialAsync> IterateScheduler_t;
  typedef Iterate<SerialAsync> Iterate_t;

  // There are two ways data can be used: to read or to write.
  // Don't change this to give more than two states:
  // things inside depend on that.

  ///////////////////////////
  // Construct the data object with an empty set of requests
  // and the given affinity.
  //
  inline DataObject(int affinity=-1);
    
  ///////////////////////////
  // for compatibility with other SMARTS schedulers, accept
  // Scheduler arguments (unused)
  //
  inline
  DataObject(int affinity, IterateScheduler<SerialAsync>&);

  ///////////////////////////
  // Stub out affinity because there is no affinity in serial.
  //
  inline int affinity() const { return 0; }

  ///////////////////////////
  // Stub out affinity because there is no affinity in serial.
  //
  inline void affinity(int) {}

  ///////////////////////////
  // An iterate makes a request for a certain action in a certain
  // generation.
  //
  inline
  void request(Iterate<SerialAsync>&, SerialAsync::Action);

  ///////////////////////////
  // An iterate finishes and tells the DataObject it no longer needs
  // it.  If this is the last release for the current set of
  // requests, have the IterateScheduler release some more.
  //
  inline void release(SerialAsync::Action);

protected:
private:

  // If release needs to let more iterates go, it calls this.
  inline void releaseIterates();

  // The type for a request.
  class Request
  {
  public:
    Request() {}
    Request(Iterate<SerialAsync>& it, SerialAsync::Action act) 
      : iterate_m(&it), act_m(act) {}

    Iterate<SerialAsync>& iterate() const { return *iterate_m; }
    SerialAsync::Action act() const { return act_m; }
  private:
    Iterate<SerialAsync>* iterate_m;
    SerialAsync::Action act_m;
  };

  // The type of the queue and iterator.
  typedef std::list<Request> Container_t;
  typedef Container_t::iterator Iterator_t;

  // The list of requests from various iterates.
  // They're granted in FIFO order.
  Container_t queue_m;

  // Pointer to the last request that has been granted.
  Iterator_t released_m;

  // The number of outstanding notifications.
  int notifications_m;
};

//////////////////////////////////////////////////////////////////////
//
// Inline implementation of the functions for
// IterateScheduler<SerialAsync>
//
//////////////////////////////////////////////////////////////////////

//
// IterateScheduler<SerialAsync>::handOff(Iterate<SerialAsync>*)
// No action needs to be taken here.  Iterates will make their
// own way into the execution queue.
//

inline void
IterateScheduler<SerialAsync>::handOff(Iterate<SerialAsync>* it)
{
  it->notify();
}

//////////////////////////////////////////////////////////////////////
//
// Inline implementation of the functions for Iterate<SerialAsync>
//
//////////////////////////////////////////////////////////////////////

//
// Iterate<SerialAsync>::Iterate
// Construct with the scheduler and the number of notifications.
// Ignore the affinity.
//

inline
Iterate<SerialAsync>::Iterate(IterateScheduler<SerialAsync>& s, int)
: scheduler_m(s), notifications_m(1)
{
}

//
// Iterate<SerialAsync>::notify
// Notify the iterate that a DataObject is ready.
// Decrement the counter, and if it is zero, alert the scheduler.
//

inline void
Iterate<SerialAsync>::notify()
{
  if ( --notifications_m == 0 )
  {
    add(this);
  }
}

//////////////////////////////////////////////////////////////////////
//
// Inline implementation of the functions for DataObject<SerialAsync>
//
//////////////////////////////////////////////////////////////////////

//
// DataObject::DataObject()
// Initialize:
//   released_m to the end of the queue (which should) also be the
//   beginning.  notifications_m to zero, since nothing has been
//   released yet.
//

inline
DataObject<SerialAsync>::DataObject(int)
: released_m(queue_m.end()), notifications_m(0)
{
}

//
// void DataObject::release(Action)
// An iterate has finished and is telling the DataObject that
// it is no longer needed.
//

inline void
DataObject<SerialAsync>::release(SerialAsync::Action)
{
  if ( --notifications_m == 0 )
    releaseIterates();
}



//-----------------------------------------------------------------------------
// 
// void IterateScheduler<SerialAsync>::blockingEvaluate
// Evaluate all the iterates in the queue.
//
//-----------------------------------------------------------------------------
inline
void
IterateScheduler<SerialAsync>::blockingEvaluate()
{
  // Loop as long as there is anything in the queue.
  while (SystemContext::workReady())
  {
    SystemContext::runSomething();
  }
}

//-----------------------------------------------------------------------------
//
// void DataObject::releaseIterates(SerialAsync::Action)
// When the last released iterate dies, we need to
// look at the beginning of the queue and tell more iterates
// that they can access this data.
//
//-----------------------------------------------------------------------------
inline
void
DataObject<SerialAsync>::releaseIterates()
{
  // Get rid of the reservations that have finished.
  queue_m.erase( queue_m.begin() , released_m );

  // Next we'll see if we can release some new ones.
  // Get the begin and end iterators.
  released_m = queue_m.begin();
  DataObject<SerialAsync>::Iterator_t end;
  end = queue_m.end();

  // If there are any in the queue, we'll release something.
  if ( released_m != end )
    {
      // Release the first one whatever it is.
      released_m->iterate().notify();
      ++notifications_m;

      // Record what action that one will take.
      SerialAsync::Action act = released_m->act();

      // Look at the next iterate.
      ++released_m;

      // If the first one was a read, release more.
      if ( act == SerialAsync::Read )

        // As long as we aren't at the end and we have more reads... 
        while ((released_m != end) && 
               (released_m->act()==SerialAsync::Read))
          {
            // Release it...
            released_m->iterate().notify();
            ++notifications_m;

            // And go on to the next.
            ++released_m;
          }
    }
}


//
// void DataObject::request(Iterate&, action)
// An iterate makes a reservation with this DataObject for a given
// action in a given generation.  The request may be granted
// immediately.
//
inline
void
DataObject<SerialAsync>::request(Iterate<SerialAsync>& it, 
                                 SerialAsync::Action act) 

{
  // The request can be granted immediately if:
  // The queue is currently empty, or
  // The request is a read and everything in the queue is a read.

  // Set notifications dynamically and automatically 
  //     every time a request is made by the iterate 
  it.incr_notifications();

  bool allReleased = (queue_m.end() == released_m);
  bool releasable =  queue_m.empty() ||
      ((act == SerialAsync::Read) &&
       (queue_m.begin()->act() == SerialAsync::Read) && 
       allReleased);

  // Push the request on the stack.
  queue_m.push_back( Request(it, act) );

  // If it's releasable, release it and record the release.
  if (releasable)
    {
      it.notify();
      ++notifications_m;
    }

  // Otherwise, if this is the first nonreleasable iterate,
  // make released_m point to the last element of the queue.
  else if (allReleased)
    {
      --released_m;
    }
}


//----------------------------------------------------------------------


//
// End of Smarts namespace.
//
}

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_PACKAGE_CLASS_H

/***************************************************************************
 * $RCSfile: SerialAsync.h,v $   $Author: sa_smith $
 * $Revision: 1.9 $   $Date: 2000/06/08 22:16:50 $
 ***************************************************************************/
