// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_PARTICLES_ATTRIBUTE_H
#define POOMA_PARTICLES_ATTRIBUTE_H

//-----------------------------------------------------------------------------
// Classes:
//   Attribute
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
// Attribute is a non-templated base class used to provide an interface
// to DynamicArray objects used as attributes in Particle classes.
// The dynamic operations such as create, destroy, copy, etc., are actually
// performed via requests to a layout object that each DynamicArray will
// use, but this class defines a small set of virtual/nonvirtual functions to:
//   print the DynamicArray contents to a stream (for debugging)
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Pooma/Configuration.h"
#include <iosfwd>

//-----------------------------------------------------------------------------
// Forward References
//-----------------------------------------------------------------------------

template <class T> class IndirectionList;
template <int Dim> class Interval;

//-----------------------------------------------------------------------------
//
// Full Description of Attribute:
//
// Attribute is an abstract base class used to implement an external
// polymorphism interface to DynamicArray objects.  The AttributeWrapper
// subclass is templated on the type of Array (really, DynamicArray) that
// the user wants to provide an abstract interface to.  This is used to
// let users create heterogenous collections of DynamicArray's, and to
// perform common tasks on all of them such as print.
//
//-----------------------------------------------------------------------------


///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

class Attribute
{
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================

  //============================================================
  // Constructors
  //============================================================

  // Default constructor.

  Attribute()
    {
    }

  // Copy constructor.

  Attribute(const Attribute &)
    {
    }


  //============================================================
  // Destructor
  //============================================================

  // Attribute needs a virtual destructor, since we will be deleting
  // Attribute's from a base class pointer.

  virtual ~Attribute()
    {
    }


  //============================================================
  // Public virtual interface
  //============================================================

  // Print the contents of the Array to the given stream.

  virtual void print(std::ostream &) const = 0;

  /* Omit this until we have serialize/deserialize functions for DynamicArray.

  // serialize/deserialize the Array using the given stream.

  virtual int serialize(std::ostream &) const = 0;
  virtual int serialize(std::fstream &) const = 0;
  virtual int deserialize(std::ostream &) = 0;
  virtual int deserialize(std::fstream &) = 0;

  */

#if POOMA_CHEETAH

  // packSize, pack and unpack function interface for particle swapping

  virtual int packSize(int) const = 0;
  virtual int pack(int, const IndirectionList<int> &, char *) const = 0;
  virtual int unpack(int, const Interval<1> &, char *) = 0;

#endif // POOMA_CHEETAH

};


//-----------------------------------------------------------------------------
//
// A specialization of the Inform traits used to say that Attribute has
// a print method.
//
//-----------------------------------------------------------------------------

inline
std::ostream&
operator<<(std::ostream& o, const Attribute& attrib)
{
  attrib.print(o);
  return o;
}


// } // namespace POOMA

//////////////////////////////////////////////////////////////////////

#endif // POOMA_PARTICLES_ATTRIBUTE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Attribute.h,v $   $Author: julianc $
// $Revision: 1.11 $   $Date: 2000/06/22 06:37:54 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
