// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
//   Div
// Global Function Templates:
//   div
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_DIFFOPS_DIV_H
#define POOMA_NEWFIELD_DIFFOPS_DIV_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// Classes:
//
// Div	:  Divergence operator (functor) on discrete Fields.
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// 
// Global Function Templates:
// 
// div()   :  Wrapper function around FieldStencil<Div>::operator() . The Div
//            functors actually used are partial specializations of the generic
//            Div that come from Div.URM.h or Div.RM.h, for example.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "NewField/Field.h"
#include "NewField/FieldInitializers.h"
#include "NewField/DiffOps/FieldStencil.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
// 
// Classes:
//
// Div:
// 
// Div is a functor class serving as the "Functor" template parameter for
// FieldStencil<Functor,Expression>, which implements a discrete 
// divergence operator.
// Partial specializations implement various combinations of input and output
// centerings, for specific coordinate systems, and different finite-differenc
// orders, are defined in other headers like Div.[URM,RM].h .
// 
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// 
// Global Function Templates:
// 
// div() : Divergence. Takes a ConstField of Vectors (or Tensors) on a 
//         discrete geometry with one centering and returns a ConstField of
//         scalars (or Vectors) on a geometry that's the same except
//         (possibly) for the centering. All the work happens in the embedded
//         Div functor partial specialization, in its operator() methods.
//-----------------------------------------------------------------------------


// ----------------------------------------------------------------------------
// General Div template
// ----------------------------------------------------------------------------

template<class T2, class Geometry>
class DivCellToVert;

template<class T2, class Geometry>
class DivVertToCell;

template<class T2, class Geometry, class OC>
class DivSameToSame;


// ----------------------------------------------------------------------------
// 
// Global Function Templates:
//
// ----------------------------------------------------------------------------

// Divergence.

template<class Geometry, class T, class EngineTag>
typename
FieldStencilSimple<DivSameToSame<T, Geometry, Cell>,
  Field<Geometry, T, EngineTag> >::Type_t
divCellToCell(const Field<Geometry, T, EngineTag> &f)
{
  typedef DivSameToSame<T, Geometry, Cell> Div_t;
  typedef FieldStencilSimple<Div_t, Field<Geometry, T, EngineTag> > Ret_t;
  return Ret_t::make(Div_t(f.fieldEngine()), f);
}

template<class Geometry, class T, class EngineTag>
typename
FieldStencilSimple<DivVertToCell<T, Geometry>,
  Field<Geometry, T, EngineTag> >::Type_t
divVertToCell(const Field<Geometry, T, EngineTag> &f)
{
  typedef DivVertToCell<T, Geometry> Div_t;
  typedef FieldStencilSimple<Div_t, Field<Geometry, T, EngineTag> > Ret_t;
  return Ret_t::make(Div_t(f.fieldEngine()), f);
}

template<class Geometry, class T, class EngineTag>
typename
FieldStencilSimple<DivCellToVert<T, Geometry>,
  Field<Geometry, T, EngineTag> >::Type_t
divCellToVert(const Field<Geometry, T, EngineTag> &f)
{
  typedef DivCellToVert<T, Geometry> Div_t;
  typedef FieldStencilSimple<Div_t, Field<Geometry, T, EngineTag> > Ret_t;
  return Ret_t::make(Div_t(f.fieldEngine()), f);
}

template<class Geometry, class T, class EngineTag>
typename
FieldStencilSimple<DivSameToSame<T, Geometry, Vert>,
  Field<Geometry, T, EngineTag> >::Type_t
divVertToVert(const Field<Geometry, T, EngineTag> &f)
{
  typedef DivSameToSame<T, Geometry, Vert> Div_t;
  typedef FieldStencilSimple<Div_t, Field<Geometry, T, EngineTag> > Ret_t;
  return Ret_t::make(Div_t(f.fieldEngine()), f);
}

#endif     // POOMA_FIELD_DIFFOPS_DIV_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Div.h,v $   $Author: sa_smith $
// $Revision: 1.2 $   $Date: 2000/07/26 22:27:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
