// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   MeshView<RectilinearMesh>
//-----------------------------------------------------------------------------

#ifndef POOMA_MESHES_MESHVIEW_RM_H
#define POOMA_MESHES_MESHVIEW_RM_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// MeshView<RectilinearMesh> : A view of a RectilinearMesh.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Shrink.h"
#include "Meshes/RectilinearMesh.h"

#include <iosfwd>

//-----------------------------------------------------------------------------
// Full Description:
// 
// MeshView<Mesh> is a lightweight view class.
//-----------------------------------------------------------------------------

template<class Mesh>
class MeshView;


//-----------------------------------------------------------------------------
// Full Description:
// 
// MeshView<RectilinearMesh> is a lightweight view of a 
// RectilinearMesh. See Meshes/RectilinearMesh.h for a discussion
// of that class.
//-----------------------------------------------------------------------------

template<int Dim, class CoordinateSystem, class T>
class MeshView<RectilinearMesh<Dim, CoordinateSystem, T> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef MeshView<RectilinearMesh<Dim, CoordinateSystem, T> > This_t;

  // The viewed class.

  typedef RectilinearMesh<Dim, CoordinateSystem, T> Viewed_t;
    
  // The range of the coordinate axes.
  
  typedef typename Viewed_t::AxisType_t AxisType_t;
  
  // The coordinate system.
  
  typedef typename Viewed_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef Interval<Dim> Domain_t;
  
  // The number of indices required to select a point in this mesh.
  
  enum { dimensions = Viewed_t::dimensions };
  
  // The dimensionality of the coordinate system.
  
  enum { coordinateDimensions = CoordinateSystem_t::dimensions };
  
  // The type used to represent a point in the mesh.
  
  typedef typename Viewed_t::PointType_t PointType_t;
  
  // The type of our GuardLayers object.
  
  typedef typename Viewed_t::GuardLayers_t GuardLayers_t;
  
  // The type of our BC container object.
  
  typedef typename Viewed_t::BCTypes_t BCTypes_t;
  
  // Types of internal arrays used to store volumes, positions, spacings, 
  // and surface normals. We will need to intercept these and view them
  // using our domain.

  typedef typename Viewed_t::CellVolumesArray_t ViewedCellVolumesArray_t;
  typedef typename Viewed_t::PositionsArray_t ViewedPositionsArray_t;
  typedef typename Viewed_t::SpacingsArray_t ViewedSpacingsArray_t;
  typedef typename Viewed_t::SurfaceNormalsArray_t ViewedSurfaceNormalsArray_t;
  
  typedef typename View1<ViewedCellVolumesArray_t, Domain_t>::Type_t 
    CellVolumesArray_t;
  typedef typename View1<ViewedPositionsArray_t, Domain_t>::Type_t 
    PositionsArray_t;
  typedef typename View1<ViewedSpacingsArray_t, Domain_t>::Type_t 
    SpacingsArray_t;
  typedef typename View1<ViewedSurfaceNormalsArray_t, Domain_t>::Type_t 
    SurfaceNormalsArray_t;

  //---------------------------------------------------------------------------
  // Constructors. 
  
  // We only build this puppy from an existing Rectilinear
  // mesh or mesh-view and an Interval<Dim>.

  MeshView(const Viewed_t &viewedMesh, const Domain_t &dom)
  : viewedMesh_m(viewedMesh), 
    domain_m(dom), 
    cellDomain_m(growRight(dom, -1))
  { }  

  MeshView(const This_t &viewedMesh, const Domain_t &dom)
  : viewedMesh_m(viewedMesh.viewedMesh_m), 
    domain_m(dom + viewedMesh.domain_m.firsts()), 
    cellDomain_m(growRight(domain_m, -1))
  { }  
    
  // Copy constructor.
  
  MeshView(const This_t &model)
  : viewedMesh_m(model.viewedMesh_m), 
    domain_m(model.domain_m),
    cellDomain_m(model.cellDomain_m)
  { }

  //---------------------------------------------------------------------------
  // Copy assignment operator (disabled). This is OK since we won't be pre-
  // instantiating this.
  
  This_t &operator=(const This_t &);

  //---------------------------------------------------------------------------
  // Empty destructor is fine.
  
  ~MeshView() { }

  //---------------------------------------------------------------------------
  // General accessors.

  // The coordinate system.
  
  const CoordinateSystem_t &coordinateSystem() const 
  {
    return viewedMesh_m.coordinateSystem(); 
  }

  // Our guard layers object.
  
  const GuardLayers_t guardLayers() const
  {
    return GuardLayers_t(0); 
  }
  
  // Whether or not we're initialized.
  
  bool initialized() const 
  { 
    return viewedMesh_m.initialized(); 
  }

  // The origin.
    
  PointType_t origin() const
  {
    return viewedMesh_m.vertexPositions().read(domain_m.firsts()); 
  }

  //---------------------------------------------------------------------------
  // Domain functions.
  
  // The vertex domain, as the mesh was constructed with.

  Domain_t physicalDomain() const
  {
    return domain_m - domain_m.firsts(); 
  }

  // Function that returns a domain adjusted to give the indices of the cells.

  Domain_t physicalCellDomain() const
  {
    return cellDomain_m - cellDomain_m.firsts(); 
  }

  // The total vertex domain. Equal to the physical domain since there are
  // no guard layers.

  Domain_t totalDomain() const
  {
    return physicalDomain(); 
  }

  // The total cell domain. Equal to the physical cell domain since there are
  // no guard layers.

  Domain_t totalCellDomain() const
  {
    return physicalCellDomain(); 
  }

  //---------------------------------------------------------------------------
  // Volume functions.

  // Returns the total volume of the mesh (physical zones only). This means
  // we turn our domain into a cell-domain (by shrinking on the right).
  
  T totalVolumeOfCells() const
  {
    return viewedMesh_m.totalVolumeOfCells(cellDomain_m); 
  }

  // Uses input domain to subset member Array of cell volumes; returns 
  // total subset volume. The domain passed in is a zero-based cell
  // domain. we must adjust this to the original coordinates by adding
  // the offset.

  template <class DomainType>
  T totalVolumeOfCells(const DomainType &domain) const
  {
    return viewedMesh_m.totalVolumeOfCells(domain + domain_m.firsts()); 
  }

  // Return entire array of cell volumes. We

  CellVolumesArray_t cellVolumes() const
  {
    return viewedMesh_m.cellVolumes()(cellDomain_m); 
  }
          
  //---------------------------------------------------------------------------
  // Vertex functions.

  // Nearest vertex index, adjusted for the new zero-based domain.

  Loc<Dim> nearestVertex(const PointType_t &point) const
  {
    return viewedMesh_m.nearestVertex(point) - domain_m.firsts(); 
  } 

  // Nearest vertex index with all vertex coordinates below point,
  // adjusted for the new zero-based domain.

  Loc<Dim> vertexBelow(const PointType_t &point) const
  {
    return viewedMesh_m.vertexBelow(point) - domain_m.firsts(); 
  }

  // Loc for cell in cell-ctrd Field containing the point (x,y,z).

  Loc<Dim> cellContaining(const PointType_t &point) const
  {
    return viewedMesh_m.cellContaining(point) - domain_m.firsts(); 
  }

  // Return coordinates (position vectors) of indexed vertex subsets.

  PositionsArray_t vertexPositions() const
  {
    return viewedMesh_m.vertexPositions()(domain_m); 
  }

  // Typical vertex-vertex grid spacings for indexed cell subsets, returned as
  // Vectors of spacings, with one component per coordinate dimension

  SpacingsArray_t vertexDeltas() const
  {
    return viewedMesh_m.vertexDeltas()(cellDomain_m); 
  }

  // --------------------------------------------------------------------------
  // The surface normals (unit vectors), like the volumes, are constant
  // everywhere in this type of mesh, for cartesian coordinates. For
  // curvilinear coordinates, they would vary across the mesh. These functions
  // return elements from the compute-engine-based Array of surface
  // normals. Each element of the Array is an array of 2*Dim Vectors, one 
  // for each face along each dimension of the mesh. The ordering of the faces 
  // is the usual thing (fortran ordering, 0 means low face of 1st dim, 1 means
  // high face of 1st dim, 2 means low face of 2nd dim, 3 means high face of
  // 2nd dim, and so on.

  SurfaceNormalsArray_t cellSurfaceNormals() const
  {
    return viewedMesh_m.cellSurfaceNormals()(cellDomain_m); 
  }

  // --------------------------------------------------------------------------
  // Print a MeshView<RectilinearMesh> on an output stream.
  
  template <class Ostream>
  void print(Ostream &ostr) const
  {
    ostr << "MeshView<RectilinearMesh>" 
      << "\n--------------------------------\n";
    ostr << "Dimension: " << Dim << '\n';
    ostr << "CoordinateSystem: " << coordinateSystem() << '\n';
    ostr << "Viewed domain: " << domain_m << '\n';
    ostr << "Physical domain: " << physicalDomain() << '\n';
    ostr << "Total domain: " << totalDomain() << '\n';
    ostr << "GuardLayers: " << guardLayers() << '\n';
    ostr << "Origin: " << origin() << '\n';
    ostr << "Vertex positions: " << '\n' << vertexPositions() << '\n';
    ostr << "Cell volumes: " << '\n' << cellVolumes();
  }
  
private:

  // Our data.
  
  Viewed_t viewedMesh_m;
  Domain_t domain_m, cellDomain_m;
};

//-----------------------------------------------------------------------------
//
// ostream inserter for MeshView<RectilinearMesh>
//
//-----------------------------------------------------------------------------

template<int Dim, class CoordinateSystem, class T>
std::ostream &operator<<(std::ostream &ostr, 
  const MeshView<RectilinearMesh<Dim, CoordinateSystem, T> > &mesh)
{
  mesh.print(ostr);
  return ostr;
}

#endif // POOMA_MESHES_MESHVIEW_RM_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: MeshView.RM.h,v $   $Author: julianc $
// $Revision: 1.4 $   $Date: 2000/07/25 00:27:24 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

