// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// include files
//-----------------------------------------------------------------------------

#include "Utilities/ElementProperties.h"

#include <string>
#include <iostream> // For debugging.
extern "C"{
#include <string.h>
}

//-----------------------------------------------------------------------------
// Generic serializer functions
//-----------------------------------------------------------------------------
// Generic size computation
//-----------------------------------------------------------------------------
template<class T>
int serialSizeof(const T&){
  return sizeof(T);
}

//-----------------------------------------------------------------------------
// Essential serializers for generic types to/from specific streams
//-----------------------------------------------------------------------------
template<class T>
int streamOut(std::iostream& s, const T& t){
  s.write((const char*)&t ,sizeof(T));
  return sizeof(T);
}

//-----------------------------------------------------------------------------
template<class T>
int streamIn(T& t, std::iostream& s){
  s.read((char*)&t, sizeof(T));
  return sizeof(T);
}

//-----------------------------------------------------------------------------
template<class T>
int streamOut(char*& s, const T& t){
  // Memcpy implementation
  memcpy((void*)s,(const void*)&t, sizeof(T));
  // Placement new implementation
  //	new (s) T(t);
  s+=sizeof(T);
  return sizeof(T);
}

//-----------------------------------------------------------------------------
template<class T>
int streamIn(T& t, char*& s){
  // Memcpy implementation
  memcpy((void*)&t, (const void*)s, sizeof(T));
  // Placement new implementation
  //	t= *(T*)s;
  s+=sizeof(T);
  return sizeof(T);
}

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
template<class T>
int streamNOut(std::iostream& s, const T* t, int num){
  int nBytes= num*sizeof(T);
  s.write((const char*)t ,nBytes);
  return nBytes;
}

//-----------------------------------------------------------------------------
template<class T>
int streamNIn(T* t, std::iostream& s, int num){
  int nBytes= num*sizeof(T);
  s.read((char*)t, nBytes);
  return nBytes;
}

//-----------------------------------------------------------------------------
template<class T>
int streamNOut(char*& s, const T* t, int num){
  int nBytes= num*sizeof(T);
  // Memcpy implementation
  memcpy((void*)s,(const void*)t, nBytes);
  s+=nBytes;
  return nBytes;
}

//-----------------------------------------------------------------------------
template<class T>
int streamNIn(T* t, char*& s, int num){
  int nBytes= num*sizeof(T);
  // Memcpy implementation
  memcpy((void*)t, (const void*)s, nBytes);
  s+= nBytes;
  return nBytes;
}

//-----------------------------------------------------------------------------
// Essential generic serializers use the streamIn() streamOut() functions.
//-----------------------------------------------------------------------------
template <class Stream, class T>
int serialize(Stream& s, const T& t){
  // Assert this is a concrete type.
  PAssert(ElementProperties<T>::concrete);
  return streamOut(s,t);
}
//
template <class Stream, class T>
int deserialize(T& t, Stream& s){
  // Assert this is a concrete type.
  PAssert(ElementProperties<T>::concrete);
  return streamIn(t,s);
}
//-----------------------------------------------------------------------------
// Generic serializers with index to stream position first.
//-----------------------------------------------------------------------------
template<class Stream, class T>
int serialize(Stream& s, const T& t, int index){
  seekOut(s,index);
  return serialize(s,t);
}

//-----------------------------------------------------------------------------
template<class Stream, class T>
int deserialize(T& t, Stream& s, int index){
  seekIn(s,index);
  return deserialize(t,s);
}


//-----------------------------------------------------------------------------
// Serializers for simple arrays
//-----------------------------------------------------------------------------

template<class Stream, class T>
int serializeN(Stream& s, const T* t, int num){
  int nBytes=0;
  ElementProperties<T> eprops;
  // Check for concrete type and optimize if so.
  if(eprops.concrete){
    nBytes= streamNOut(s, t , num);
  }
  else{
    for(int i=0;i<num;i++){
      nBytes+= serialize(s,t[i]);
    }
  }
  // std::cout<<"Serialized generic array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}

//-----------------------------------------------------------------------------
template<class Stream, class T>
int deserializeN(T* t, Stream& s, int num){
  int nBytes=0;
  ElementProperties<T> eprops;
  // Check for concrete type and optimize if so.
  if(eprops.concrete){
    nBytes= streamNIn(t, s, num);
  }
  else{
    for(int i=0;i<num;i++){
      nBytes+= deserialize(t[i],s);
    }
  }
  return nBytes;
}

//-----------------------------------------------------------------------------
// Serializers for simple arrays with seek to index first.
//-----------------------------------------------------------------------------
template<class Stream, class T>
int serializeN(Stream& s, const T* t, int num, int index){
  seekOut(s,index);
  return serializeN(s,t,num);
}

//-----------------------------------------------------------------------------
template<class Stream, class T>
int deserializeN(T* t, Stream& s, int num, int index){
  seekIn(s,index);
  return deserializeN(t,s,num);
}

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Specializations for C++ standard strings
//-----------------------------------------------------------------------------
template <class Stream>
int serialize(Stream& s, const std::string& t){
  return streamStringOut(s,t);
}
//
template <class Stream>
int deserialize(std::string& t, Stream& s){
  return streamStringIn(t,s);
}

//-----------------------------------------------------------------------------
/*
//-----------------------------------------------------------------------------
// Serializers for simple arrays optimized for known fixed-length
// types.
//-----------------------------------------------------------------------------
// For int:
//-----------------------------------------------------------------------------
template <class Stream>
int serializeN(Stream& s, const int* t, int num){
  int nBytes= streamNOut(s, t , num);
  // std::cout<<"Serialized int array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}

//-----------------------------------------------------------------------------
template <class Stream>
int deserializeN(int* t, Stream& s, int num){
  int nBytes= streamNIn(t, s, num);
  // std::cout<<"Deserialized int array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}

//-----------------------------------------------------------------------------
// For long:
//-----------------------------------------------------------------------------
template <class Stream>
int serializeN(Stream& s, const long* t, int num){
  int nBytes= streamNOut(s, t , num);
  // std::cout<<"Serialized long array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}

//-----------------------------------------------------------------------------
template <class Stream>
int deserializeN(long* t, Stream& s, int num){
  int nBytes= streamNIn(t, s, num);
  // std::cout<<"Deserialized long array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}

//-----------------------------------------------------------------------------
// For float:
//-----------------------------------------------------------------------------
template <class Stream>
int serializeN(Stream& s, const float* t, int num){
  int nBytes= streamNOut(s, t , num);
  // std::cout<<"Serialized float array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}

//-----------------------------------------------------------------------------
template <class Stream>
int deserializeN(float* t, Stream& s, int num){
  int nBytes= streamNIn(t, s, num);
  // std::cout<<"Deserialized float array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}

//-----------------------------------------------------------------------------
// For double:
//-----------------------------------------------------------------------------
template <class Stream>
int serializeN(Stream& s, const double* t, int num){
  int nBytes= streamNOut(s, t , num);
  // std::cout<<"Serialized double array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}

//-----------------------------------------------------------------------------
template <class Stream>
int deserializeN(double* t, Stream& s, int num){
  int nBytes= streamNIn(t, s, num);
  // std::cout<<"Deserialized int array "<<nBytes<<" bytes."<<std::endl;
  return nBytes;
}
*/
//-----------------------------------------------------------------------------

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Serializers.cpp,v $   $Author: julianc $
// $Revision: 1.10 $   $Date: 2000/07/24 16:03:39 $
// ----------------------------------------------------------------------
// ACL:rcsinfo





