// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// RNGSimple
//-----------------------------------------------------------------------------

#ifndef POOMA_FUNCTIONS_RNG_SIMPLE_H
#define POOMA_FUNCTIONS_RNG_SIMPLE_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
//
// class RNGSimple
//
// Simple class that implements random number generator a la
// Numerical Recipes, in the range [0...1].
//-----------------------------------------------------------------------------

class RNGSimple
{
public:

  // return type
  typedef double Type_t;

public:

  // default constructor

  RNGSimple(int adv = 0)
    : currentRand_m(randShift + 1)
  {
    advance(adv);
  }

  // copy constructor

  RNGSimple(const RNGSimple& rng)
    : currentRand_m(rng.currentRand_m)
  { }

  // destructor

  ~RNGSimple()
  { }

  //   advance indicates number of times to advance random number source

  inline void advance(int advance = 1)
  {
    int iadv;
    for (iadv = 0; iadv < advance; iadv++)
    {
      currentRand_m = (currentRand_m * randMultiplier + randShift) % randModulus;
    }
  }

  // set seed to user-specified value, plus shift to ensure it is large

  inline void seed(long seed)
  {
    currentRand_m = (seed + randShift) % randModulus;
  }

  inline long seed() const
  {
    return currentRand_m;
  }

  // return the next pseudo-random number (from 0 ... 1)

  inline Type_t value() const
  {
    return ( Type_t(currentRand_m) / Type_t(randModulus) );
  }

  // return the period of the RNG

  static Type_t period()
  {
    return Type_t(randModulus);
  }

private:

  mutable long currentRand_m;

  enum { randModulus    = 714025 };
  enum { randMultiplier = 1366 };
  enum { randShift      = 150889 };
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_FUNCTIONS_RNG_SIMPLE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RNGSimple.h,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/03/07 13:17:19 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
