// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ----------------------------------------------------------------------------
// Scalar advection example, illustrating use of Field, Mesh,
// DiscreteGeometry, and the canned second-order divergence operator
// div<Cell>()
// ----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Clock.h"
#include "Pooma/Lux.h"
#include "BAOptions.h"
#include "BAInitialConditions.h"

// ----------------------------------------------------------------------------
// The Real Main Program:

template <int Dim>
void BAmain(int argc, char *argv[], Inform &pout)
{
  // The BAOptions object sets the default option values and parses argv for
  // options that override the defaults.
  BAOptions<Dim> opts(argc, argv);
  opts.print(pout);

  // PrintArray class, to format output as desired:
  PrintArray pa(3, 10, 3, 6, true, 1);

  // Create the physical domains:
  Interval<Dim> vertexDomain, cellDomain;
  int d;
  for (d = 0; d < Dim; d++) {
    vertexDomain[d] = Interval<1>(opts.nVerts[d]);
    cellDomain[d] = Interval<1>(opts.nCells[d]);
  }

  // Create the (uniform, logically rectilinear) mesh.
  Vector<Dim> origin(0.0);
  Vector<Dim> spacings;
  for (d = 0; d < Dim; d++) { spacings(d) = opts.dx[d]; }
  typedef UniformRectilinearMesh<Dim> Mesh_t;
  Mesh_t mesh(vertexDomain, origin, spacings);

  // Create two geometry objects - one allowing 1 guard layer to account for
  // stencil width and another with no guard layers to support temporaries:
  typedef DiscreteGeometry<Cell, UniformRectilinearMesh<Dim> > Geometry_t;
  Geometry_t geom(mesh, GuardLayers<Dim>(1));
  Geometry_t geomNG(mesh);

  // Create the layouts
  Loc<Dim> patches(8);
  for (d = 0; d < Dim; d++) { patches[d] = Loc<1>(opts.nPatches[d]); }
  GridLayout<Dim> layoutc(cellDomain, patches, 
                          GuardLayers<Dim>(1), GuardLayers<Dim>(1), ReplicatedTag());
  GridLayout<Dim> layoutcNG(cellDomain, patches, ReplicatedTag());
  typedef MultiPatch<GridTag, CompressibleBrick> MP_t;

  // Create the Fields:

  // The flow Field u(x,t), a duplicate (stored at the previous
  // timestep for staggered leapfrog), and a useful temporary:
  Field<Geometry_t, double, MP_t> 
    u(geom, layoutc), uPrev(geomNG, layoutcNG), uTemp(geomNG, layoutcNG);

  // Initialize Fields to zero everywhere, even global guard layers:
  u.all() = 0.0;

  // Set up periodic boundary conditions on all mesh faces:
  u.addBoundaryConditions(AllPeriodicFaceBC());

  // Load initial condition u(x,0):
  if (opts.initialCondition == 1) {
    loadSphere<Dim>(u, opts);
  } else {
    loadOtherShape(u, uTemp, opts);
  }

  // Propagation velocity:
  Vector<Dim> v(0.2);
  for (d = 0; d < Dim; d++) v(d) = opts.v[d];

  const double dt = opts.dt;  // Timestep

  // Establish optional Lux runtime visualization connection:
  Connection<Lux> *lux;
  if (opts.doLuxOut) {
    lux = new Connection<Lux>("ScalarAdvection");
    lux->connect("u", u);
    lux->ready();
  }

  // Print out information at t = 0:
  if (opts.purge) {
    u = where((fabs(u) < opts.epsilon), 0.0); // Purge initial conditions
  }
  Pooma::blockAndEvaluate();
  if (opts.doLuxOut) {
    lux->update();
    lux->interact();
  }
  pout << "t = " << double(0.0);
  if (opts.doCompressedFractionOut) pout << " ; compressedFraction(u) = "
                                         << compressedFraction(u);
  if (opts.doSumOut) pout << " ; sum(u) = " << sum(u);
  pout << std::endl;
  if (opts.doTextOut) { pa.print(pout, u); }

  // Prime the leapfrog by setting the field at the previous timestep using the
  // initial conditions:
  uPrev = u;

  // Do a preliminary timestep using forward Euler, using the canned POOMA
  // div() function:
  u -= div<Cell>(v * dt * u);

  // Start timer to time main loop; do blockAndEvaluate first so results are
  // meaningful:
  Pooma::blockAndEvaluate();
  double startTime = Pooma::Clock::value();

  // Now use staggered leapfrog (second-order) for the remaining timesteps.
  // The spatial derivative is just the second-order finite difference in the
  // canned POOMA stencil-based divergence operator div():
  for (int timestep = 2; timestep <= opts.lastTimeStep; timestep++) {

    uTemp = u;
    u = uPrev - 2.0 * div<Cell>(v * dt * u);

    if (opts.purge) {
      if ((timestep % opts.purgeIncrement) == 0) {
        u = where((fabs(u) < opts.epsilon), 0.0);
      }
    }

    if ((timestep % opts.outputIncrement) == 0) {
      Pooma::blockAndEvaluate();
      pout << "t = " << timestep*dt << std::endl;
      if (opts.doCompressedFractionOut) pout << " ; compressedFraction(u) = " 
                                             << compressedFraction(u);
      if (opts.doSumOut) pout << " ; sum(u) = " << sum(u);
      pout << std::endl;
      if (opts.doTextOut) { pa.print(pout, u); }
      if (opts.doLuxOut) {
        lux->update();
        lux->interact();
      }
    }

    uPrev = uTemp;
  }

  // Compute the wallclock time for the loop; first do blockAndEvaluate() to
  // make sure the calculation has actually been done first:
  Pooma::blockAndEvaluate();
  double wallTime = Pooma::Clock::value() - startTime;

  pout << "Done. Wallclock seconds = " << wallTime << std::endl;
  Pooma::finalize();
}

// ----------------------------------------------------------------------------
// The Main Program:

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv); // Initialize the library
  Inform pout(NULL, 0);         // Output, via process 0 only

  // Select dimensionalities to compile (from ONED, TWOD, THREED):
//define ONED
#define TWOD
#define THREED
  bool compiledDims[3] = {false, false, false};
#ifdef ONED
  compiledDims[0] = true;
#endif
#ifdef TWOD
  compiledDims[1] = true;
#endif
#ifdef THREED
  compiledDims[2] = true;
#endif

  // Set the dimensionality from mandatory "-dim <Dim>" argument:
  int Dim = 0;
  using std::string;
  string arg(argv[1]);
  if (arg == "-dim") {
    if (2 == argc) {
      pout << "First argument must always be: -dim <Dim>" << std::endl;
      exit(1);
    }
    bool hasarg = true;
    // Make sure the argument is a number:
    char firstchar = argv[2][0];
    if (firstchar < '0' || firstchar > '9') {
      if ((firstchar != '-' && firstchar != '+') || argv[2][1] == 0 ||
          (argv[2][1] < '0' || argv[2][1] > '9'))
        hasarg = false;
    }
    if (!hasarg) {
      pout << "Must supply the <Dim> argument in -dim <Dim>" << std::endl;
      exit(1);
    }
    // Get the value and assign it:
    Dim = atoi(argv[2]);
    PInsist(Dim<4, "Will only run for Dim<=3.");
    PInsist(compiledDims[Dim-1], "Not compiled for requested Dim value.");
  } else {
    pout << "First argument must always be: -dim <Dim>" << std::endl;
    exit(1);
  }

  // Now run the simulation for the desired input dimensionality:
  switch (Dim) {
#ifdef ONED
  case 1:
    BAmain<1>(argc, argv, pout);
    break;
#endif
#ifdef TWOD
  case 2:
    BAmain<2>(argc, argv, pout);
    break;
#endif
#ifdef THREED
  case 3:
    BAmain<3>(argc, argv, pout);
    break;
#endif
  default:
    PInsist(Dim<4, "Will only run for Dim=1, 2, or 3.");
    PInsist(Dim>0, "Will only run for Dim=1, 2, or 3.");
    break;
  }
return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BAdvection.cpp,v $   $Author: julianc $
// $Revision: 1.3 $   $Date: 2000/07/24 16:07:25 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
