// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef BA_INITIAL_CONDITIONS_H
#define BA_INITIAL_CONDITIONS_H

// Load initial condition u(x,0), a symmetric pulse centered around nCells/4
// and decaying to zero away from nCells/4 all directions, with a height of
// 1.0, with a half-width equal to opts.pulseHalfWidthCells (defaults to
// nCells/8):
template<int Dim, class FieldType>
void loadSphere(FieldType &u, BAOptions<Dim> &opts)
{
  Vector<Dim> spacings = u.geometry().mesh().meshSpacing();
  double pulseHalfWidth = spacings(0)*opts.pulseHalfWidthCells;
  Loc<Dim> pulseCenter;
  for (int d = 0; d < Dim; d++) { pulseCenter[d] = Loc<1>(opts.nCells[d]/2); }
  Pooma::blockAndEvaluate(); // Needed pre-scalar-indexing read
  Vector<Dim> u0 = u.x(pulseCenter);
  u = 1.0 * exp(-dot(u.x() - u0, u.x() - u0) / (2.0 * pulseHalfWidth));
}

// Load initial condition u(x,0), an ellipsoid with 4 "fins" near the rear,
// with various geometrical parameters controlled by opts.pulseHalfWidthCells,
// opts.aspectRatio, opts.finHalfWidthCells, opts.finLength,
// opts.finOffsetCellsX, opts.finOffsetCellsYZ. Ellipsoid centered around
// nCells/2 and decaying to zero away from nCells/2 all directions, with pulse
// "height" of 1.0 and a half-width equal to opts.pulseHalfWidthCells in the
// YZ directions and opts.pulseHalfWidthCells*aspectRatio in the X direction.
// This requires partial specializations for 1D, 2D, 3D:

// 3D
template<class FieldType>
void loadOtherShape(FieldType &u, FieldType &uTemp, BAOptions<3> &opts)
{
  const int Dim = 3;
  Vector<Dim> spacings = u.geometry().mesh().meshSpacing();
  double pulseHalfWidthYZ = spacings(0)*opts.pulseHalfWidthCells;
  double pulseHalfWidthX = opts.aspectRatio*pulseHalfWidthYZ;
  double finHalfWidth = spacings(0)*opts.finHalfWidthCells/4.0;
  Loc<Dim> pulseCenter;
  for (int d = 0; d < Dim; d++) { pulseCenter[d] = Loc<1>(opts.nCells[d]/2); }
  Pooma::blockAndEvaluate(); // Needed pre-scalar-indexing read
  Vector<Dim> u0 = u.x(pulseCenter);
  u = 1.0 * exp(-(pow((u.x().comp(0) - u0(0)),2)/(2.0*pulseHalfWidthX)
                  + pow((u.x().comp(1) - u0(1)),2)/(2.0*pulseHalfWidthYZ)
                  + pow((u.x().comp(2) - u0(2)),2)/(2.0*pulseHalfWidthYZ)));
  // --------------------------------------------------------------------------
  // Fins:

  // High Z:
  Loc<Dim> finLoc3(pulseCenter);
  finLoc3[0] -= opts.finOffsetCellsX;
  finLoc3[2] += opts.finOffsetCellsYZ;
  Loc<Dim> finLoc1(finLoc3);
  finLoc1[2] += opts.finLengthCells;
  Loc<Dim> finLoc2(finLoc1);
  finLoc2[0] -= opts.finLengthCells;
  Vector<Dim> pt1 = u.x(finLoc1);
  Vector<Dim> pt2 = u.x(finLoc2);
  Vector<Dim> pt3 = u.x(finLoc3);
  Vector<Dim> pt2a = pt2;
  pt2a(2) -= spacings(2)/4.0;
  pt2a(0) -= spacings(0)/4.0;
  double slope = (pt3(2) - pt2(2))/(pt3(0) - pt2(0));
  uTemp = exp(-(pow((uTemp.xComp(1) - pt1(1)),2)/(2.0*finHalfWidth)));
  uTemp = where((uTemp.xComp(0) >= pt1(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt1(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(0) <= pt2(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt2(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(2) <= pt3(2)),
                uTemp*
                exp(-(pow((uTemp.xComp(2) - pt3(2)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(2) >= pt2(2)),
                uTemp*
                exp(-(pow((uTemp.xComp(2) - pt2(2)),2)/(2.0*finHalfWidth))));
  uTemp = where(((uTemp.xComp(2) - slope*uTemp.xComp(0)) <= 
                 (pt2a(2) - slope*pt2a(0))),
                uTemp*
                exp(-(pow(((uTemp.xComp(2) - slope*uTemp.xComp(0)) -
                          (pt2a(2) - slope*pt2a(0))),2)/(2.0*finHalfWidth))));
  u = where((uTemp >= u), uTemp, u);

  // Low Z:
  finLoc3 = pulseCenter;
  finLoc3[0] -= opts.finOffsetCellsX;
  finLoc3[2] -= opts.finOffsetCellsYZ;
  finLoc1 = finLoc3;
  finLoc1[2] -= opts.finLengthCells;
  finLoc2 = finLoc1;
  finLoc2[0] -= opts.finLengthCells;
  pt1 = u.x(finLoc1);
  pt2 = u.x(finLoc2);
  pt3 = u.x(finLoc3);
  pt2a = pt2;
  pt2a(2) += spacings(2)/4.0;
  pt2a(0) -= spacings(0)/4.0;
  slope = (pt3(2) - pt2(2))/(pt3(0) - pt2(0));
  uTemp = 0.0;
  uTemp = exp(-(pow((uTemp.xComp(1) - pt1(1)),2)/(2.0*finHalfWidth)));
  uTemp = where((uTemp.xComp(0) >= pt1(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt1(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(0) <= pt2(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt2(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(2) >= pt3(2)),
                uTemp*
                exp(-(pow((uTemp.xComp(2) - pt3(2)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(2) <= pt2(2)),
                uTemp*
                exp(-(pow((uTemp.xComp(2) - pt2(2)),2)/(2.0*finHalfWidth))));
  uTemp = where(((uTemp.xComp(2) - slope*uTemp.xComp(0)) >= 
                 (pt2a(2) - slope*pt2a(0))),
                uTemp*
                exp(-(pow(((uTemp.xComp(2) - slope*uTemp.xComp(0)) -
                          (pt2a(2) - slope*pt2a(0))),2)/(2.0*finHalfWidth))));
  u = where((uTemp >= u), uTemp, u);

  // High Y:
  finLoc3 = pulseCenter;
  finLoc3[0] -= opts.finOffsetCellsX;
  finLoc3[1] += opts.finOffsetCellsYZ;
  finLoc1 = finLoc3;
  finLoc1[1] += opts.finLengthCells;
  finLoc2 = finLoc1;
  finLoc2[0] -= opts.finLengthCells;
  pt1 = u.x(finLoc1);
  pt2 = u.x(finLoc2);
  pt3 = u.x(finLoc3);
  pt2a = pt2;
  pt2a(1) -= spacings(1)/4.0;
  pt2a(0) -= spacings(0)/4.0;
  slope = (pt3(1) - pt2(1))/(pt3(0) - pt2(0));
  uTemp = 0.0;
  uTemp = exp(-(pow((uTemp.xComp(2) - pt1(2)),2)/(2.0*finHalfWidth)));
  uTemp = where((uTemp.xComp(0) >= pt1(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt1(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(0) <= pt2(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt2(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(1) <= pt3(1)),
                uTemp*
                exp(-(pow((uTemp.xComp(1) - pt3(1)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(1) >= pt2(1)),
                uTemp*
                exp(-(pow((uTemp.xComp(1) - pt2(1)),2)/(2.0*finHalfWidth))));
  uTemp = where(((uTemp.xComp(1) - slope*uTemp.xComp(0)) <= 
                 (pt2a(1) - slope*pt2a(0))),
                uTemp*
                exp(-(pow(((uTemp.xComp(1) - slope*uTemp.xComp(0)) -
                          (pt2a(1) - slope*pt2a(0))),2)/(2.0*finHalfWidth))));
  u = where((uTemp >= u), uTemp, u);

  // Low Y:
  finLoc3 = pulseCenter;
  finLoc3[0] -= opts.finOffsetCellsX;
  finLoc3[1] -= opts.finOffsetCellsYZ;
  finLoc1 = finLoc3;
  finLoc1[1] -= opts.finLengthCells;
  finLoc2 = finLoc1;
  finLoc2[0] -= opts.finLengthCells;
  pt1 = u.x(finLoc1);
  pt2 = u.x(finLoc2);
  pt3 = u.x(finLoc3);
  pt2a = pt2;
  pt2a(1) += spacings(1)/4.0;
  pt2a(0) -= spacings(0)/4.0;
  slope = (pt3(1) - pt2(1))/(pt3(0) - pt2(0));
  uTemp = 0.0;
  uTemp = exp(-(pow((uTemp.xComp(2) - pt1(2)),2)/(2.0*finHalfWidth)));
  uTemp = where((uTemp.xComp(0) >= pt1(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt1(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(0) <= pt2(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt2(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(1) >= pt3(1)),
                uTemp*
                exp(-(pow((uTemp.xComp(1) - pt3(1)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(1) <= pt2(1)),
                uTemp*
                exp(-(pow((uTemp.xComp(1) - pt2(1)),2)/(2.0*finHalfWidth))));
  uTemp = where(((uTemp.xComp(1) - slope*uTemp.xComp(0)) >= 
                 (pt2a(1) - slope*pt2a(0))),
                uTemp*
                exp(-(pow(((uTemp.xComp(1) - slope*uTemp.xComp(0)) -
                          (pt2a(1) - slope*pt2a(0))),2)/(2.0*finHalfWidth))));
  u = where((uTemp >= u), uTemp, u);

  // Try smoothing things:
  Loc<Dim> avgWidth(2), avgOffset(1);
  Interval<Dim> interior = shrinkRight(u.physicalDomain(), avgWidth) +
    avgOffset;
  int iterations = opts.smoothing;
  double c3 = 0.5/6.0;
  for (int i = 0; i < iterations; i++) {
    u(interior) = 0.5*u(interior) + 
      c3*u(interior[0] + 1, interior[1]    , interior[2]    ) +
      c3*u(interior[0] - 1, interior[1]    , interior[2]    ) +
      c3*u(interior[0]    , interior[1] + 1, interior[2]    ) +
      c3*u(interior[0]    , interior[1] - 1, interior[2]    ) +
      c3*u(interior[0]    , interior[1]    , interior[2] + 1) +
      c3*u(interior[0]    , interior[1]    , interior[2] - 1);
  }
}


// 2D
template<class FieldType>
void loadOtherShape(FieldType &u, FieldType &uTemp, BAOptions<2> &opts)
{
  const int Dim = 2;
  Vector<Dim> spacings = u.geometry().mesh().meshSpacing();
  double pulseHalfWidthYZ = spacings(0)*opts.pulseHalfWidthCells;
  double pulseHalfWidthX = opts.aspectRatio*pulseHalfWidthYZ;
  double finHalfWidth = spacings(0)*opts.finHalfWidthCells/4.0;
  Loc<Dim> pulseCenter;
  for (int d = 0; d < Dim; d++) { pulseCenter[d] = Loc<1>(opts.nCells[d]/2); }
  Pooma::blockAndEvaluate(); // Needed pre-scalar-indexing read
  Vector<Dim> u0 = u.x(pulseCenter);
  u = 1.0 * exp(-(pow((u.x().comp(0) - u0(0)),2)/(2.0*pulseHalfWidthX)
                  + pow((u.x().comp(1) - u0(1)),2)/(2.0*pulseHalfWidthYZ)));
  // --------------------------------------------------------------------------
  // Fins:

  // High Y:
  Loc<Dim> finLoc3(pulseCenter);
  finLoc3[0] -= opts.finOffsetCellsX;
  finLoc3[1] += opts.finOffsetCellsYZ;
  Loc<Dim> finLoc1(finLoc3);
  finLoc1[1] += opts.finLengthCells;
  Loc<Dim> finLoc2(finLoc1);
  finLoc2[0] -= opts.finLengthCells;
  Vector<Dim> pt1 = u.x(finLoc1);
  Vector<Dim> pt2 = u.x(finLoc2);
  Vector<Dim> pt3 = u.x(finLoc3);
  Vector<Dim> pt2a = pt2;
  pt2a(1) -= spacings(1)/4.0;
  pt2a(0) -= spacings(0)/4.0;
  double slope = (pt3(1) - pt2(1))/(pt3(0) - pt2(0));
  uTemp = 1.0;
  uTemp = where((uTemp.xComp(0) >= pt1(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt1(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(0) <= pt2(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt2(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(1) <= pt3(1)),
                uTemp*
                exp(-(pow((uTemp.xComp(1) - pt3(1)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(1) >= pt2(1)),
                uTemp*
                exp(-(pow((uTemp.xComp(1) - pt2(1)),2)/(2.0*finHalfWidth))));
  uTemp = where(((uTemp.xComp(1) - slope*uTemp.xComp(0)) <= 
                 (pt2a(1) - slope*pt2a(0))),
                uTemp*
                exp(-(pow(((uTemp.xComp(1) - slope*uTemp.xComp(0)) -
                          (pt2a(1) - slope*pt2a(0))),2)/(2.0*finHalfWidth))));
  u = where((uTemp >= u), uTemp, u);

  // Low Y:
  finLoc3 = pulseCenter;
  finLoc3[0] -= opts.finOffsetCellsX;
  finLoc3[1] -= opts.finOffsetCellsYZ;
  finLoc1 = finLoc3;
  finLoc1[1] -= opts.finLengthCells;
  finLoc2 = finLoc1;
  finLoc2[0] -= opts.finLengthCells;
  pt1 = u.x(finLoc1);
  pt2 = u.x(finLoc2);
  pt3 = u.x(finLoc3);
  pt2a = pt2;
  pt2a(1) += spacings(1)/4.0;
  pt2a(0) -= spacings(0)/4.0;
  slope = (pt3(1) - pt2(1))/(pt3(0) - pt2(0));
  uTemp = 1.0;
  uTemp = where((uTemp.xComp(0) >= pt1(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt1(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(0) <= pt2(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt2(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(1) >= pt3(1)),
                uTemp*
                exp(-(pow((uTemp.xComp(1) - pt3(1)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(1) <= pt2(1)),
                uTemp*
                exp(-(pow((uTemp.xComp(1) - pt2(1)),2)/(2.0*finHalfWidth))));
  uTemp = where(((uTemp.xComp(1) - slope*uTemp.xComp(0)) >= 
                 (pt2a(1) - slope*pt2a(0))),
                uTemp*
                exp(-(pow(((uTemp.xComp(1) - slope*uTemp.xComp(0)) -
                          (pt2a(1) - slope*pt2a(0))),2)/(2.0*finHalfWidth))));
  u = where((uTemp >= u), uTemp, u);

  // Try smoothing things:
  Loc<Dim> avgWidth(2), avgOffset(1);
  Interval<Dim> interior = shrinkRight(u.physicalDomain(), avgWidth) +
    avgOffset;
  int iterations = opts.smoothing;
  double c2 = 0.5/4.0;
  for (int i = 0; i < iterations; i++) {
    u(interior) = 0.5*u(interior) + 
      c2*u(interior[0] + 1, interior[1]    ) + 
      c2*u(interior[0] - 1, interior[1]    ) +
      c2*u(interior[0]    , interior[1] + 1) + 
      c2*u(interior[0]    , interior[1] - 1);
  }
}


// 1D
template<class FieldType>
void loadOtherShape(FieldType &u, FieldType &uTemp, BAOptions<1> &opts)
{
  const int Dim = 1;
  Vector<Dim> spacings = u.geometry().mesh().meshSpacing();
  double pulseHalfWidthYZ = spacings(0)*opts.pulseHalfWidthCells;
  double pulseHalfWidthX = opts.aspectRatio*pulseHalfWidthYZ;
  double finHalfWidth = spacings(0)*opts.finHalfWidthCells/4.0;
  Loc<Dim> pulseCenter;
  for (int d = 0; d < Dim; d++) { pulseCenter[d] = Loc<1>(opts.nCells[d]/2); }
  Pooma::blockAndEvaluate(); // Needed pre-scalar-indexing read
  Vector<Dim> u0 = u.x(pulseCenter);
  u = 1.0 * exp(-(pow((u.x().comp(0) - u0(0)),2)/(2.0*pulseHalfWidthX)));

  // --------------------------------------------------------------------------
  // Fin:

  Loc<Dim> finLoc3(pulseCenter);
  finLoc3[0] -= opts.finOffsetCellsX;
  Loc<Dim> finLoc1(finLoc3);
  Loc<Dim> finLoc2(finLoc1);
  finLoc2[0] -= opts.finLengthCells;
  Vector<Dim> pt1 = u.x(finLoc1);
  Vector<Dim> pt2 = u.x(finLoc2);
  Vector<Dim> pt3 = u.x(finLoc3);
  Vector<Dim> pt2a = pt2;
  pt2a(0) -= spacings(0)/4.0;
  uTemp = 1.0;
  uTemp = where((uTemp.xComp(0) >= pt1(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt1(0)),2)/(2.0*finHalfWidth))));
  uTemp = where((uTemp.xComp(0) <= pt2(0)),
                uTemp*
                exp(-(pow((uTemp.xComp(0) - pt2(0)),2)/(2.0*finHalfWidth))));
  u = where((uTemp >= u), uTemp, u);

  // Try smoothing things:
  Loc<Dim> avgWidth(2), avgOffset(1);
  Interval<Dim> interior = shrinkRight(u.physicalDomain(), avgWidth) +
    avgOffset;
  int iterations = opts.smoothing;
  double c1 = 0.5/2.0;
  for (int i = 0; i < iterations; i++) {
    u(interior) = 0.5*u(interior) + 
      c1*u(interior[0] + 1) + 
      c1*u(interior[0] - 1);
  }
}

#endif // BA_INITIAL_CONDITIONS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BAInitialConditions.h,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/03/07 13:17:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
