// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   Field
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_FIELD_H
#define POOMA_FIELD_FIELD_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// Field: An Array-like container that adds the notions of geometry and
//        the management of automatic boundary conditions. Works closely
//        with its read-only cousin, ConstField.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Forward declarations
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag> class Field;

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Array/Array.h"
#include "Field/ConstField.h"
#include "Field/CreateLeaf.h"
#include "Field/Reductions.h"
#include "Engine/EnginePatch.h"
#include "Engine/CompressedFraction.h"

// We want to use the Field-specific stuff.

#undef MakeReturn
#define MakeReturn MakeFieldReturn
#include "Field/FieldOperators.h"
#include "Field/PoomaFieldOperators.h"
#include "Field/VectorFieldOperators.h"
#undef MakeReturn

//-----------------------------------------------------------------------------
// Prototypes for the assign function used to assign an expression to a Field.
//
// Prototypes defined here:
//   Field = Array
//   Field = ConstField
//   Field = Field
//   Array = ConstField
//   Array = Field
//   Field = scalar
///
// If you wish to have Field work with other types of objects on the right-
// hand side (for example, other classes that derive from Field), define
// extra assign() functions that take the following arguments:
//
//   assign(Field<Geom,T,EngineTag>, yourclass, Operator)
//
// where "yourclass" is the class that you would like to work on the
// right-hand side in an expression with a Field on the left-hand side.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag,
  int OtherDim, class OtherT, class OtherEngineTag, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs,
       const Array<OtherDim, OtherT, OtherEngineTag> &rhs,
       const Op &op);
       
template<class Geom, class T, class EngineTag,
  class OtherGeom, class OtherT, class OtherEngineTag, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs,
       const ConstField<OtherGeom, OtherT, OtherEngineTag> &rhs,
       const Op &op);
  
template<class Geom, class T, class EngineTag,
  class OtherGeom, class OtherT, class OtherEngineTag, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs,
       const Field<OtherGeom, OtherT, OtherEngineTag> &rhs,
       const Op &op);
       
template<int Dim, class T, class EngineTag,
  class OtherGeom, class OtherT, class OtherEngineTag, class Op>
const Array<Dim, T, EngineTag> &
assign(const Array<Dim, T, EngineTag> &lhs,
       const ConstField<OtherGeom, OtherT, OtherEngineTag> &rhs,
       const Op &op);
       
template<int Dim, class T, class EngineTag,
  class OtherGeom, class OtherT, class OtherEngineTag, class Op>
const Array<Dim, T, EngineTag> &
assign(const Array<Dim, T, EngineTag> &lhs,
       const Field<OtherGeom, OtherT, OtherEngineTag> &rhs,
       const Op &op);

template<class Geom, class T, class EngineTag, class T1, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs, const T1 &rhs, const Op &op);

//-----------------------------------------------------------------------------
// Traits classes
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// View specializations for Field.
//-----------------------------------------------------------------------------

// Create a Array view.

template<class Subject, class Dom> struct ArrayView;

template<class Geometry, class T, class EngineTag, class Sub1>
struct ArrayView<Field<Geometry, T, EngineTag>, Sub1>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Engine_t Engine_t;
  typedef Sub1 SDomain_t;

  // Deduce the template parameters for the output type.
  
  typedef typename NewEngine<Engine_t, SDomain_t>::Type_t NewEngine_t;
  static const int newDim = NewEngine_t::dimensions;
  typedef typename NewEngine_t::Element_t NewT_t;
  typedef typename NewEngine_t::Tag_t NewEngineTag_t;
  
  // The output type.
  
  typedef Array<newDim, NewT_t, NewEngineTag_t> Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s)
    {
      typedef NewEngineEngine<Engine_t, SDomain_t> NewEE_t;
      typedef NewEngineDomain<Engine_t, SDomain_t> NewED_t;

#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field::arrayRead view bounds error.");
#endif

      return Type_t(
        NewEE_t::apply(f.engine(), s),
        NewED_t::apply(f.engine(), s),
        EngineConstructTag());
    }
};

// These are all the views that are obtained by the syntax a(b).

// Single-valued version. Handles scalars and Locs.

template<class Subject, class Domain, bool SV>
struct View1Implementation;

// Single-valued version. Handles scalars and Locs.

template<class Geometry, class T, class EngineTag, class Domain>
struct View1Implementation<Field<Geometry, T, EngineTag>, Domain, true>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  // The return type is pretty simple here.
  
  typedef typename Subject_t::ElementRef_t Type_t;
  
#if defined(__MWERKS__)

  inline static 
  Type_t make(const Subject_t &f, const Domain &s)
    {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine()(s);
    }

#else // not metrowerks

  template<class S1, class Combine>
  inline static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const Combine &)
    {
      Domain s(Combine::make(f, s1));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine()(s);
    }

  template<class S1, class S2, class Combine>
  inline static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2,
	      const Combine &)
    {
      Domain s(Combine::make(f, s1, s2));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine()(s);
    }

  template<class S1, class S2, class S3,
    class Combine>
  inline static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2, const S3 &s3,
	      const Combine &)
    {
      Domain s(Combine::make(f, s1, s2, s3));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine()(s);
    }

#endif // not metrowerks
};

// Non-single-valued implementation. Works for general domains
// including Nodes and INodes.

template<class Geometry, class T, class EngineTag, class Domain>
struct View1Implementation<Field<Geometry, T, EngineTag>, Domain, false>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Engine_t Engine_t;
  typedef typename NewEngine<Engine_t, Domain>::Type_t NewEngine_t;
  typedef typename NewEngine_t::Element_t NewT_t;
  typedef typename NewEngine_t::Tag_t NewEngineTag_t;
  typedef typename NewGeometry<Geometry, Domain>::Type_t NewGeometry_t;
  
  // The output type.
  
  typedef Field<NewGeometry_t, NewT_t, NewEngineTag_t> Type_t;

  // A function that creates the view.
  
  typedef NewEngineEngine<Engine_t, Domain> NewEE_t;
  typedef NewEngineDomain<Engine_t, Domain> NewED_t;
  typedef NewGeometryGeometry<Geometry, Domain> NewGG_t;
  typedef NewGeometryDomain<Geometry, Domain> NewGD_t;

#if defined(__MWERKS__)

  inline static
  Type_t make(const Subject_t &f, const Domain &s)
    {

#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif

      return Type_t(
        NewGeometry_t(NewGG_t::apply(f.geometry(), s), 
          NewGD_t::apply(f.geometry(), s)),
        NewEngine_t(NewEE_t::apply(f.engine(), s),
          NewED_t::apply(f.engine(), s)),
        f.bconds());
    }


#else // not metrowerks

  template<class S1, class Combine>
  static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const Combine &)
  {
    Domain s(Combine::make(f, s1));
#if POOMA_BOUNDS_CHECK
    PInsist(contains(f.totalDomain(), s),
	    "Field view bounds error.");
#endif

      return Type_t(
        NewGeometry_t(NewGG_t::apply(f.geometry(), s), 
          NewGD_t::apply(f.geometry(), s)),
        NewEngine_t(NewEE_t::apply(f.engine(), s),
          NewED_t::apply(f.engine(), s)),
        f.bconds());
  }

  template<class S1, class S2, class Combine>
  static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const S2 &s2, const Combine &)
  {
    Domain s(Combine::make(f, s1, s2));
#if POOMA_BOUNDS_CHECK
    PInsist(contains(f.totalDomain(), s),
	    "Field view bounds error.");
#endif

      return Type_t(
        NewGeometry_t(NewGG_t::apply(f.geometry(), s), 
          NewGD_t::apply(f.geometry(), s)),
        NewEngine_t(NewEE_t::apply(f.engine(), s),
          NewED_t::apply(f.engine(), s)),
        f.bconds());
  }

  template<class S1, class S2, class S3,
    class Combine>
  static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2, const S3 &s3,
	      const Combine &)
  {
    Domain s(Combine::make(f, s1, s2, s3));
#if POOMA_BOUNDS_CHECK
    PInsist(contains(f.totalDomain(), s),
	    "Field view bounds error.");
#endif

      return Type_t(
        NewGeometry_t(NewGG_t::apply(f.geometry(), s), 
          NewGD_t::apply(f.geometry(), s)),
        NewEngine_t(NewEE_t::apply(f.engine(), s),
          NewED_t::apply(f.engine(), s)),
        f.bconds());
  }

#endif
};

// General version.

template<class Geometry, class T, class EngineTag, class Sub1>
struct View1<Field<Geometry, T, EngineTag>, Sub1>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  // At some point, we need to fix NewDomain1; until then, use
  // the temporary version from Array.h.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef TemporaryNewDomain1<Domain_t, Sub1> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The optimized domain combiner.
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return type.
  
  typedef typename Dispatch_t::Type_t Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1)
    {
#if defined(__MWERKS__)
      SDomain_t s(Combine_t::make(f, s1));
      return Dispatch_t::make(f, s);
#else
      return Dispatch_t::make(f, s1, Combine_t());
#endif
    }
};

template<class Geometry, class T, class EngineTag>
struct View1<Field<Geometry, T, EngineTag>, int>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  typedef typename Subject_t::ElementRef_t Type_t;

  inline static
  Type_t make(const Subject_t &f, int s1)
  {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<1>(s1)),
        "Field view bounds error.");
#endif
    return f.engine()(s1);
  }
};

// These for views that are obtained by the syntax a(b, c).

template<class Geometry, class T, class EngineTag, 
  class Sub1, class Sub2>
struct View2<Field<Geometry, T, EngineTag>, Sub1, Sub2>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef NewDomain2<Sub1, Sub2> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The optimized domain combiner.
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return type.
  
  typedef typename Dispatch_t::Type_t Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1, const Sub2 &s2)
    {
#if defined(__MWERKS__)
      SDomain_t s(Combine_t::make(f, s1, s2));
      return Dispatch_t::make(f, s);
#else
      return Dispatch_t::make(f, s1, s2, Combine_t());
#endif
    }
};

template<class Geometry, class T, class EngineTag>
struct View2<Field<Geometry, T, EngineTag>, int, int>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  typedef typename Subject_t::ElementRef_t Type_t;

  inline static
  Type_t make(const Subject_t &f, int s1, int s2)
  {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<2>(s1, s2)),
        "Field view bounds error.");
#endif
    return f.engine()(s1, s2);
  }
};

// These ones are for views that are obtained by the syntax a(b, c, d).

template<class Geometry, class T, class EngineTag, 
  class Sub1, class Sub2, class Sub3>
struct View3<Field<Geometry, T, EngineTag>, Sub1, Sub2, Sub3>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef NewDomain3<Sub1, Sub2, Sub3> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The optimized domain combiner.
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return type.
  
  typedef typename Dispatch_t::Type_t Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1, const Sub2 &s2, 
    const Sub3 &s3)
    {
#if defined(__MWERKS__)
      SDomain_t s(Combine_t::make(f, s1, s2, s3));
      return Dispatch_t::make(f, s);
#else
      return Dispatch_t::make(f, s1, s2, s3, Combine_t());
#endif
    }
};

template<class Geometry, class T, class EngineTag>
struct View3<Field<Geometry, T, EngineTag>, int, int, int>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<Geometry, T, EngineTag> Subject_t;

  typedef typename Subject_t::ElementRef_t Type_t;

  inline static
  Type_t make(const Subject_t &f, int s1, int s2, int s3)
  {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<3>(s1, s2, s3)),
        "Field view bounds error.");
#endif
    return f.engine()(s1, s2, s3);
  }
};

//-----------------------------------------------------------------------------
// Patch specialization for Field.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
struct Patch<Field<Geom, T, EngineTag> >
{
  typedef Field<Geom, T, EngineTag> Subject_t;
  typedef typename Subject_t::Engine_t OldEngine_t;

  typedef typename EngineFunctor<OldEngine_t, EnginePatch>::Type_t Engine_t;

  // Should a patch view of a field be a field or an array?
  // or should there even be such a thing?

  typedef Array<Subject_t::dimensions, T, typename Engine_t::Tag_t> Type_t;

  inline static
  Type_t make(const Subject_t &subject, int i)
  {
    return Type_t(engineFunctor(subject.engine(), EnginePatch(i)));
  }
};

//-----------------------------------------------------------------------------
// ComponentView specialization for Field.
//-----------------------------------------------------------------------------

// This is for views of the form f.comp(loc).

// the int version gets instantiated when f.comp(i) is resolved, but isn't used
// it's provided to avoid errors

template<class Geom, class T, class EngineTag>
struct ComponentView<int, Field<Geom, T, EngineTag> >
{
  typedef int Type_t;
};

template<class Components, class Geom, class T, class EngineTag>
struct ComponentView<Components, Field<Geom, T, EngineTag> >
{
  // Convenience typedef for the thing we're taking a component view of.
  
  typedef Field<Geom, T, EngineTag> Subject_t;

  // Deduce the template parameters for the output type.
  
  typedef typename Subject_t::Element_t Element_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef typename ComponentAccess<Element_t, Components>::Element_t NewT_t;
  typedef CompFwd<Engine_t, Components> NewEngineTag_t;
  typedef Engine<Geom::dimensions, NewT_t, NewEngineTag_t> NewEngine_t;
  typedef NewComponentGeometry<Components, Geom> NewCompGeom_t;
  typedef typename NewCompGeom_t::Type_t NewGeometry_t;
  
  // The output type.
  
  typedef Field<NewGeometry_t, NewT_t, NewEngineTag_t> Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Components &loc)
    {
      return Type_t(NewCompGeom_t::make(f.geometry(), loc),
        NewEngine_t(f.engine(), loc), 
        f.bconds());
    }
};

//-----------------------------------------------------------------------------
// Full Description:
// 
// Field resembles Array in that it is a read-write version inheriting from
// ConstField. It adds a few extra capabilities to the Array concept:
//
//   o Geometry - in addition to being able to return values, it can also
//       return the position in physical coordinate space corresponding to the
//       value.
//   o Automatic boundary conditions - Field can automatically update
//        parts of its domain using boundary condition objects stored in
//        a list. Before viewing, these boundary condition objects can be
//        queried as to whether the domain they manage needs updating and
//        then told to update, in necessary.
//
// Field has three template parameters:
//   o Geom - the geomtry object used to associate values with physical 
//       positions;
//   o T - the type of element stored in the field;
//   o EngineTag - the tag that selects the Engine used to turn indices int
//       values.
// Default template arguments are provided for T and EngineTag.
//
// Exported typedefs:
//
//   Base_t - this class' ConstField parent class.
//   CoordinateSystem_t - the coordinate system for this field.
//   Domain_t - the type of physical/total domain for this field.
//   Element_t - the type of elements managed by this field.
//   ElementRef_t - a reference or proxy to allow changing the elements
//     managed by this field.
//   Engine_t - the type of Engine running this field.
//   EngineTag_t - the tag used to select Engine_t.
//   Geometry_t - the type of this field's geometry.
//   Layout_t - the type of the engine's layout.
//   PointType_t - the type used to represent a point corresponding to a
//     field value.
//   PositionArray_t - the type of array returned by x().
//   This_t - this class.
//
// Exported enumerations and constants:
//
//   dimensions - the dimensionality of the field.
//   coordinateDimensions - the dimension of the coordinate system.
//
// Constructors:
//
//   Field() - default constructor. Creates an uninitialized Field.
//     Use the initialize(...) functions below to complete initialization.
//   Field(Geometry) - set the geometry and use its domain to initialize
//     the Engine. 
//   Field(Geometry, BCondInitFunctor])
//   Field(Geometry, BCondList)
//      - Same as Geometry only, but initialize the BCondList either
//      with an initlization functor or another BCondLlist (which
//      will be retargeted.
//   Field(Geometry, Layout) - Like above, but initialize the engine
//      with a Layout object (and check for consistency).
//   Field(Geometry, Layout, BCondInitFunctor)
//   Field(Geometry, Layout, BCondList)
//      - Like Field(Geometry, Layout), but also initialize the
//      BCondList.
//   Field(Field[, DontCopyBoundaryConditions]) - copy constructor.
//      Makes a shallow copy (i.e. copies the engine, not the data).
//      If DontCopyBoundaryConditions tag object is supplied, then
//      BCs are not copied.
//   Field(Geometry_t, Engine_t[, BCondList]) - constructor used to
//     construct views.
//
// Initialize functions:
//
//   initialize(Geometry)
//   inititlize(Geometry, BCondInitFunctor)
//   initialize(Geometry, BCondList)
//   initialize(Geometry, Layout)
//   initialize(Geometry, Layout, BCondInitFunctor)
//   initialize(Geometry, Layout, BCondList)
//   initialize(Field)
//   initialize(Field, DontCopyBoundaryConditions)
//   initialize(Geometry, Engine, BCondList)
//   initialize(Geometry, Engine)
//     - See constructors with same signatures.
//
// Indexing functions:
//
//   read(int1[,...,int7]) - returns the field value given the indices 
//     (read only).
//   operator()(int1[,...,int7]) - returns the field value given the indices
//     (read/write).
//   x(int1[,...,int7]) - returns the position value given the indices.
//
// View creation functions:
//
//   read([dom1,...,dom3]) - returns a view of the field in the specified
//     domain (read only). The zero argument version reads the physical
//     domain.
//   operator()([dom1,...,dom3]) - returns a view of the field in the 
//     specified domain (read/write). The zero argument version reads the 
//     physical domain.
//   x(dom1[,...,dom3]) - returns a view of the position array in the 
//     specified domain.
//
// Accessors:
//
//   bconds() - returns the boundary condition list.
//   engine() - returns this field's engine.
//   geometry() - returns this field's geometry.
//   physicalDomain() - returns this field's physical domain.
//   totalDomain() - returns this field's total domain.
//   x() - all of the positions in the physical domain.
//
// BC handling:
//
//   addBoundaryCondition(BCCategory) - adds a boundary condition from 
//    the appropriate category.
//   addBoundaryConditions(BCFunctor) - adds a set of boundary conditions 
//    using the specified functor.
//-----------------------------------------------------------------------------

template<class Geom, 
         class T = POOMA_DEFAULT_ELEMENT_TYPE,
         class EngineTag = POOMA_DEFAULT_ENGINE_TYPE>
class Field : public ConstField<Geom, T, EngineTag>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  // Base_t is a convenience typedef referring to this class' parent.

  typedef ConstField<Geom, T, EngineTag> Base_t;

  // This_t is a convenience typedef referring to this class.

  typedef Field<Geom, T, EngineTag> This_t;

  // The number of indices required to select a point in this ConstField.
  
  enum { dimensions = Geom::dimensions };
  
  // The dimensionality of the coordinate system.
  
  enum { coordinateDimensions = Geom::coordinateDimensions };

  // This field's coordinate system.
  
  typedef typename Geom::CoordinateSystem_t CoordinateSystem_t;
  
  // EngineTag_t is a typedef that specifies the type of engine.
  // Engine_t is a typedef that maps the engine-tag to the Engine type.
  
  typedef EngineTag EngineTag_t;
  typedef Engine<dimensions, T, EngineTag_t> Engine_t;
  
  // Layout_t is the Engine's layout.
  
  typedef typename Engine_t::Layout_t Layout_t;

  // Geometry_t is this field's geometry.
  
  typedef Geom Geometry_t;

  // Element_t is the type of elements managed by this field's engine. 
  // ElementRef_t is the writable version.
  
  typedef typename Engine_t::Element_t Element_t;
  typedef typename Engine_t::ElementRef_t ElementRef_t;

  // The types of the total/physical domains. 

  typedef typename Geometry_t::Domain_t Domain_t; 
  
  // The type of array returned by x().
  
  typedef typename Geometry_t::PositionArray_t PositionArray_t;
  
  // The type of points returned by the geometry.
  
  typedef typename Geometry_t::PointType_t PointType_t;

  //---------------------------------------------------------------------------
  // Default constructor. Creates default geometry and engine objects.
  
  Field() { }  
  
  //---------------------------------------------------------------------------
  // Geometry constructor. We use the specified Geometry object to supply the
  // Engine's domain. This is appropriate for single patch engines. Also
  // supply a version that allows the boundary conditions to be initialized.
  
  Field(const Geometry_t &geom)
  : Base_t(geom)
  { } 

  template<class BCondInitFunctor>
  Field(const Geometry_t &geom, const BCondInitFunctor &bcInit)
  : Base_t(geom)
  {
    bcInit(*this);
  } 

  //---------------------------------------------------------------------------
  // Initialize a field using a geometry and an already existing boundary
  // condition list. The subject of these boundary conditions is retargeted.
  
  Field(const Geometry_t &geom, const BCondList &bc)
  : Base_t(geom, bc)
  {
    retargetBoundaryConditions(); 
  } 

  //---------------------------------------------------------------------------
  // Geometry/layout constructors. We use the specified Geometry object to
  // initialize our geometry and the Layout to initialize the engine. Clearly,
  // these must be synchronized. This is appropriate for multi-patch engines.
  // Versions are also supplied to initialize the boundary conditions.
  
  Field(const Geometry_t &geom, const Layout_t &layout)
  : Base_t(geom, layout)
  { 
    // Base checks for Geometry/Layout consistency.
  } 

  template <class BCondInitFunctor>
  Field(const Geometry_t &geom, const Layout_t &layout,
        const BCondInitFunctor &bcInit)
  : Base_t(geom, layout)
  {
    // Base checks for Geometry/Layout consistency.

    bcInit(*this);
  } 

  Field(const Geometry_t &geom, const Layout_t &layout,
        const BCondList &bc)
  : Base_t(geom, layout, bc)
  {
    // Base checks for Geometry/Layout consistency.

    retargetBoundaryConditions(); 
  } 

  //---------------------------------------------------------------------------
  // Copy constructor. Just pass the model on to the base classes.
  // Note that as with Array, this is a shallow copy.
  // As a result, this does not retarget boundary conditions.
  
  Field(const This_t &model)
  : Base_t(model)
  { }  

  //---------------------------------------------------------------------------
  // Copy constructor, except do not copy the boundary conditions. 
  // Just pass the model on to the base classes.
  
  Field(const This_t &model, const DontCopyBoundaryConditions &tag)
  : Base_t(model, tag)
  { }  

  //---------------------------------------------------------------------------
  // View creation constructor. This is a shallow copy. In particular, the
  // boundary conditions point to the original field.
  
  Field(const Geometry_t &g, const Engine_t &e, const BCondList &bc)
  : Base_t(g, e, bc)
  { }

  Field(const Geometry_t &g, const Engine_t &e)
  : Base_t(g, e)
  { }

  //---------------------------------------------------------------------------
  // Simple initialize functions.
  
  void initialize(const Geometry_t &g)
  {
    Base_t::initialize(g);
  }
  
  template <class BCondInitFunctor>
  void initialize(const Geometry_t &g, const BCondInitFunctor &bcInit)
  {
    Base_t::initialize(g);
    bcInit(*this);
  }

  void initialize(const Geometry_t &g, const BCondList &bc)
  { 
    Base_t::initialize(g,bc);
    retargetBoundaryConditions(); 
  }

  void initialize(const Geometry_t &g, const Layout_t &layout)
  {
    Base_t::initialize(g, layout);
  }
      
  void initialize(const Geometry_t &g, const Layout_t &layout, 
                  const BCondList &bc)
  { 
    // Base checks for consistency.
    
    Base_t::initialize(g,layout,bc);
    retargetBoundaryConditions(); 
  }

  template <class BCondInitFunctor>
  void initialize(const Geometry_t &g, const Layout_t &layout, 
                  const BCondInitFunctor &bcinit)
  { 
    // Base checks for consistency.
    
    Base_t::initialize(g,layout);
    bcinit(*this); 
  }

  void initialize(const This_t &model)
  { 
    Base_t::initialize(model);
  }

  void initialize(const This_t &model, const DontCopyBoundaryConditions &tag)
  { 
    Base_t::initialize(model, tag);
  }

  void initialize(const Geometry_t &g, const Engine_t &e, const BCondList &bc)
  { 
    Base_t::initialize(g, e, bc);
  }

  void initialize(const Geometry_t &g, const Engine_t &e)
  { 
    Base_t::initialize(g, e);
  }

  //---------------------------------------------------------------------------
  // Functions for adding and running boundary conditions.
  
  template<class Category>
  void addBoundaryCondition(const Category &bc)
  {
    bconds().addBoundaryCondition(*this, bc);
  }

  template<class BCondInitFunctor>
  void addBoundaryConditions(const BCondInitFunctor &bcs)
  {
    bcs(*this);
  }
  
  void applyBoundaryConditions()
  {
    bconds().setDirty();
    notifyPreRead();
  }
  
  //---------------------------------------------------------------------------
  // Instruct the field to make its own copy of its data.
  
  void makeOwnCopy();

  //---------------------------------------------------------------------------
  // View-creation operations. These operator() functions take zero or more
  // sub-domains, which combine to form a domain with dimensionality identical
  // to the rank of the field. They return a Field with no geometry and
  // an engine capable of viewing the specified domain. Views based on up 
  // to 3 subdomains are supported. The zero argument version returns a view of
  // the physical domain.
  //
  // Theory of operation:
  //
  // Is the view of a Field an Array or a Field? If it were an Array, we 
  // wouldn't be able to update the boundary conditions. Therefore, it makes 
  // sense that views, along with all the other field-related entities that 
  // can find themselves at a leaf in a PETE expression tree, be Fields of 
  // some sort. What sort of Field is it? This is a little tricky, because 
  // what is possible depends on the sort of view. For example, if a Field f 
  // has a rectilinear mesh we take the view f(I) where I is an 
  // Interval<1> and it is fully contained in the physical domain (PD) of f, 
  // one can imagine that the view can be as full-featured a field as any 
  // other, except for the fact that the view's total domain (TD) would be 
  // identical to its PD. In particular, one could conceivably take a view 
  // of the field's geometry/mesh and, importantly, retain connectivity 
  // information. However, if I is a more complicated domain like a Range<>
  // or indirection list, there is no plausible prescription for deducing 
  // connectivity. This means we lose the notions of a mesh and centering. 
  // Therefore, a view of a Field cannot be used in a differential operator. 
  // Can we retain the ability for the Field to return positions? This is
  // less clear since we clearly know which points correspond to which
  // Field values. We could just take a view of the position arrays over the
  // same domain we're using to view the field values. However, in really
  // isn't clear how useful this is, so we opt for simplicity by making the
  // a view of a Field a Field to ensure boundary conditions are run, but
  // making the Geometry of the resulting Field a dumb NoGeometry<>.

  inline typename View1<This_t, Domain_t>::Type_t 
  operator()() const
  {
    typedef View1<This_t, Domain_t> Ret_t;
    return Ret_t::make(*this, physicalDomain());
  }

  inline typename View1<This_t, Domain_t>::Type_t 
  all() const
  {
    typedef View1<This_t, Domain_t> Ret_t;
    return Ret_t::make(*this, totalDomain());
  }

  template<class Sub1> 
  inline typename View1<This_t, Sub1>::Type_t 
  operator()(const Sub1 &s1) const
  {
    typedef View1<This_t, Sub1> Ret_t;
    return Ret_t::make(*this, s1);
  }

  template<class Sub1, class Sub2> 
  inline typename View2<This_t, Sub1, Sub2>::Type_t 
  operator()(const Sub1 &s1, const Sub2 &s2) const
  {
    typedef View2<This_t, Sub1, Sub2> Ret_t;
    return Ret_t::make(*this, s1, s2);
  }

  template<class Sub1, class Sub2, class Sub3> 
  inline typename View3<This_t, Sub1, Sub2, Sub3>::Type_t 
  operator()(const Sub1 &s1, const Sub2 &s2, const Sub3 &s3) const
  {
    typedef View3<This_t, Sub1, Sub2, Sub3> Ret_t;
    return Ret_t::make(*this, s1, s2, s3);
  }
  
  //---------------------------------------------------------------------------
  // View-creation operations yielding Arrays.
   
  typename ArrayView<This_t, Domain_t>::Type_t
  inline array() const
  {
    typedef ArrayView<This_t, Domain_t> Ret_t;
    return Ret_t::make(*this, physicalDomain());
  };
   
  typename ArrayView<This_t, Domain_t>::Type_t
  inline arrayAll() const
  {
    typedef ArrayView<This_t, Domain_t> Ret_t;
    return Ret_t::make(*this, totalDomain());
  };

  //---------------------------------------------------------------------------
  // Component-forwarding functions. These work quite similar to the ones from
  // Array except we produce a Field with the same Geometry.

  inline typename ComponentView<Loc<1>, This_t>::Type_t
  comp(int i1) const
  {
    return ComponentView<Loc<1>, This_t>::make(*this, Loc<1>(i1));
  }

  inline typename ComponentView<Loc<2>, This_t>::Type_t
  comp(int i1, int i2) const
  {
    return ComponentView<Loc<2>, This_t>::make(*this, Loc<2>(i1, i2));
  }

  template<class Components>
  inline typename ComponentView<Components, This_t>::Type_t
  comp(const Components &loc) const
  {
    return ComponentView<Components, This_t>::make(*this, loc);
  }

  //---------------------------------------------------------------------------
  // Patch accessor function returns the i'th patch.
  //---------------------------------------------------------------------------

  inline typename Patch<This_t>::Type_t
  patchLocal(EnginePatch::PatchID_t i) const
  {
    return Patch<This_t>::make(*this,i);
  }

  //---------------------------------------------------------------------------
  // Copy assignment operators. We pack this assignment expression into a
  // PETE binary expression tree node and then use this to construct an
  // array with an expression engine. We then pass this on to an evaluator,
  // which handles the computation. The first two versions handle assigning
  // Fields to Fields and the fourth one handles assigning
  // scalars.
  //
  // Note: PARTIAL ORDERING is required for the scalar assignment operator 
  // to be distinguishable from the others.

  This_t &operator=(const Field<Geom, T, EngineTag> &rhs)
  {
    assign(*this, rhs, OpAssign());
    return *this;
  }

  const This_t &operator=(const Field<Geom, T, EngineTag> &rhs) const
  {
    return assign(*this, rhs, OpAssign());
  }

  template<class T1>
  const This_t &operator=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpAssign());
  }

  //---------------------------------------------------------------------------
  // Op-assignment operators. 
  //
  // Note: PARTIAL ORDERING is required for the scalar assignment operator 
  // to be distinguishable from the others.

  // Addition.

  template<class T1>
  const This_t &operator+=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpAddAssign());
  }

  // Subtraction.

  template<class T1>
  const This_t &operator-=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpSubtractAssign());
  }

  // Multiplication.

  template<class T1>
  const This_t &operator*=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpMultiplyAssign());
  }

  // Division.

  template<class T1>
  const This_t &operator/=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpDivideAssign());
  }

  // Modulus.

  template<class T1>
  const This_t &operator%=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpModAssign());
  }

  // Bitwise-Or.

  template<class T1>
  const This_t &operator|=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpBitwiseOrAssign());
  }

  // Bitwise-And.

  template<class T1>
  const This_t &operator&=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpBitwiseAndAssign());
  }

  // Bitwise-Xor.

  template<class T1>
  const This_t &operator^=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpBitwiseXorAssign());
  }

  // Left shift.

  template<class T1>
  const This_t &operator<<=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpLeftShiftAssign());
  }

  // Right shift.

  template<class T1>
  const This_t &operator>>=(const T1 &rhs) const
  {
    return assign(*this, rhs, OpRightShiftAssign());
  }

private:
 
  //---------------------------------------------------------------------------
  // Retarget our boundary conditions (i.e., make this Field their subject).
  
  void retargetBoundaryConditions();

};

//-----------------------------------------------------------------------------
// This specialization of LeafFunctor is used to get the domain type or the
// (zero-based) domain itself from a Field. Used only by Expression-Engine.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
struct LeafFunctor<Field<Geom, T, EngineTag>, DomainFunctorTag>
{
  typedef typename Field<Geom, T, EngineTag>::Domain_t Type_t;
  static Type_t apply(const Field<Geom, T, EngineTag> &f, 
    const DomainFunctorTag &)
    {
      return f.physicalDomain();
    }
};

//-----------------------------------------------------------------------------
// This specialization of LeafFunctor is used to apply a view (subsetting) 
// operation to a Field. The domain will always be zero-based since this
// is used only by Expression-Engine. This is why we add the firsts() to
// the domain.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag, class Domain>
struct LeafFunctor<Field<Geom, T, EngineTag>, ViewFunctorTag<Domain> >
{
  typedef typename View1<Field<Geom, T, EngineTag>, Domain>::Type_t Type_t;
  inline static Type_t apply(const Field<Geom, T, EngineTag> &f, 
    const ViewFunctorTag<Domain> &t) 
    {
      typedef View1<Field<Geom, T, EngineTag>, Domain> Ret_t;
      return Ret_t::make(f, t.domain_m);
    }
};

//-----------------------------------------------------------------------------
// This version of LeafFunctor is used by Expression-Engines to used to 
// evaluate a Field using indices. 
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag, int Dim>
struct LeafFunctor<Field<Geom, T, EngineTag>, EvalLeaf<Dim> >
{
  typedef typename Field<Geom, T, EngineTag>::Element_t Type_t;
  inline static
  Type_t apply(const Field<Geom, T, EngineTag> &f, const EvalLeaf<Dim> &t) 
  {
    return t.eval(f.engine());
  }
};

//-----------------------------------------------------------------------------
// EngineView functor acting on ConstField.
// This version is probably wrong in general, because you want to take a view
// of the geometry as well as the field.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class E, class Tag>
struct LeafFunctor<Field<Geometry, T, E>, EngineView<Tag> >
{
  typedef Field<Geometry, T, E> Subject_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef LeafFunctor<Engine_t, EngineView<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t NewEngine_t;

  typedef typename NewEngine_t::Tag_t NewTag_t;
  typedef Field<Geometry, T, NewTag_t> Type_t;

  inline static
  Type_t apply(const Subject_t &field, 
	       const EngineView<Tag> &tag)
  {
    return Type_t(field.geometry(),
		  LeafFunctor_t::apply(field.engine(), tag));
  }
};

//-----------------------------------------------------------------------------
// EngineApply functor acting on Field.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class E, class Tag>
struct LeafFunctor<Field<Geometry, T, E>, EngineApply<Tag> >
{
  typedef Field<Geometry, T, E> Subject_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef LeafFunctor<Engine_t, EngineApply<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t Type_t;

  inline static
  Type_t apply(const Subject_t &field, 
	       const EngineApply<Tag> &tag)
  {
    return LeafFunctor_t::apply(field.engine(), tag);
  }
};

//----------------------------------------------------------------------
// Apply the ConformTag to the leaves of the tree.
// Check to see if a given Field conforms.
//----------------------------------------------------------------------

template<class Geom, class T, class EngineTag, int Dim>
struct LeafFunctor<Field<Geom, T, EngineTag>, ConformTag<Dim> >
{
  typedef bool Type_t;
  static Type_t apply1(const Interval<Dim> &d, 
    const ConformTag<Dim> &ct)
    {
      return conforms(d, ct);
    }
  template<int Dim2>
  static Type_t apply1(const Interval<Dim2> &d, 
    const ConformTag<Dim> &ct)
    {
      return false;
    }
  static Type_t apply(const Field<Geom, T, EngineTag> &f,
    const ConformTag<Dim> &ct)
    {
      return apply1(f.physicalDomain(), ct);
    }
};

//----------------------------------------------------------------------
// For Fields, we use DataObjectApply to pass the request on to array
// (or return the default, if the field has no data block).
//----------------------------------------------------------------------

template<class RequestType, class Geom, class T, class EngineTag>
struct LeafFunctor<Field<Geom, T, EngineTag>,
  DataObjectRequest<RequestType> >
{
  typedef typename Field<Geom, T, EngineTag>::Engine_t Engine_t;
  enum { dataObject = Engine_t::dataObject };
  typedef typename DataObjectRequest<RequestType>::Type_t Type_t;
  inline static
  Type_t apply(const Field<Geom, T, EngineTag> &f,
	       const DataObjectRequest<RequestType> &functor)
  {
    return DataObjectApply<dataObject>::apply(f.engine(), functor);
  }
};

//----------------------------------------------------------------------
// Do what needs to be done prior to reading. For Fields, this means
// telling the boundary conditions where we're going to read.
//----------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
struct LeafFunctor<Field<Geom, T, EngineTag>, NotifyPreReadTag>
{
  typedef bool Type_t;
  static Type_t apply(const Field<Geom, T, EngineTag> &f,
    const NotifyPreReadTag &)
    {
      f.notifyPreRead();
      return true;
    }
};

//-----------------------------------------------------------------------------
// Handle general engine functor requests.
//-----------------------------------------------------------------------------

template<class Geom, class T, class E, class Tag>
struct LeafFunctor<Field<Geom, T, E>, EngineFunctorTag<Tag> >
{
  typedef typename Field<Geom,T,E>::Engine_t Engine_t;
  typedef typename EngineFunctor<Engine_t,Tag>::Type_t Type_t;
  inline static
  Type_t apply(const Field<Geom, T, E> &field,
	       const EngineFunctorTag<Tag> &tag)
  {
    return EngineFunctor<Engine_t,Tag>::apply(field.engine(), tag.tag());
  }
};

//---------------------------------------------------------------------------
// A specialization of EngineFunctor for field.
//---------------------------------------------------------------------------

template<class Geom, class T, class E, class Tag>
struct EngineFunctor<Field<Geom, T, E>, Tag>
{
  typedef typename Field<Geom,T,E>::Engine_t Engine_t;
  typedef typename EngineFunctor<Engine_t, Tag>::Type_t Type_t;

  inline static 
  Type_t apply(const Field<Geom, T, E> &field,
	       const Tag &tag)
  {
    return engineFunctor(field.engine(), tag);
  }
};

//-----------------------------------------------------------------------------
// Traits class for expressions containing field.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
struct ExpressionTraits<Field<Geom, T, EngineTag> >
{
  typedef ExpressionIsField Type_t;
};


//-----------------------------------------------------------------------------
//
// double compressedFraction()
//
// Compute the fraction of the total domain that is currently compressed.
//
//-----------------------------------------------------------------------------

template<class Geometry, class T, class LTag>
inline double
compressedFraction(Field<Geometry, T, MultiPatch<LTag,CompressibleBrick> > &x)
{
  return compressedFraction(x.engine());
}

//-----------------------------------------------------------------------------
//
// void compress()
//
// (Try to) compress all the patches of a Field or Array using a
// MultiPatchEngine with compressible patches
//
//-----------------------------------------------------------------------------

template<class Geometry, class T, class LTag>
inline void
compress(Field<Geometry, T, MultiPatch<LTag,CompressibleBrick> > &x)
{
  compress(x.engine());
}

//-----------------------------------------------------------------------------
//
// void uncompress()
//
// Manually uncompress all the patches of a Field or Array using a
// MultiPatchEngine with compressible patches 
//
//-----------------------------------------------------------------------------

template<class Geometry, class T, class LTag>
inline void
uncompress(Field<Geometry, T, MultiPatch<LTag,CompressibleBrick> > &x)
{
  uncompress(x.engine());
}

#include "Field/Field.cpp"

#endif // POOMA_FIELD_FIELD_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Field.h,v $   $Author: swhaney $
// $Revision: 1.67 $   $Date: 2000/07/20 15:39:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
