// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// DynamicEngine test code
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Domain/Loc.h"
#include "Domain/Interval.h"
#include "Domain/Range.h"
#include "Engine/DynamicEngine.h"

typedef Engine<1,double,Dynamic> Array_t;

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc,argv);
#if POOMA_EXCEPTIONS
  try {
#endif
    tester.out() << "\nTesting DynamicEngine." << std::endl;

    Interval<1> I(10);
    Array_t A(I);

    for (int i = 0; i < 10; i++)
      A(Loc<1>(i)) = 2.0 + i - i*i;
    
    for (int i = 0; i < 10; i++)
      tester.out() <<  A(Loc<1>(i)) << " ";
    tester.out() << std::endl;

    Interval<1> J(2,5);
    NewEngine<Array_t,Interval<1> >::Type_t B(A,J); 
    
    for (int i = 0; i < 4; i++)
      {
        tester.check(B.read(i) == A.read(i + 2));
        tester.out() <<  B.read(Loc<1>(i)) << " ";
      }
    tester.out() << std::endl;

    Range<1> K(1,9,2);
    NewEngine<Array_t, Range<1> >::Type_t C(A,K); 
    
    tester.check(C.stride() == 2);
    
    for (int i = 0; i < 5; i++)
      {
        tester.check(
          C(i) == A(i * 2 + 1) );
        tester.out() << C.read(i) << " ";
      }
    tester.out() << std::endl;

    A(3) = -444;
    tester.check(A(3) == -444);
    tester.check(B(1) == -444);
    tester.check(C(1) == -444);
    
    Range<1> KV(0,4,2);
    typedef NewEngine<Array_t, Range<1> >::Type_t View1_t;
    
    NewEngine<View1_t,Range<1> >::Type_t CV(C,KV); // A(1), A(5), A(9)
                                                   // C(0), C(2), C(4)
                                                   //       B(3)
    tester.check(CV(0) == A(1));
    tester.check(CV(1) == B(3));

    Interval<1> IV(0,4);
    NewEngine<View1_t,Range<1> >::Type_t CV2(C,IV); 

    tester.check(CV2(0) == A(1));
    tester.check(CV2(1) == A(3));

    Array_t AC = A;

    AC(Loc<1>(2)) = -999;
    
    tester.check(AC(2) == -999);
    tester.check(A(2)  == -999);
    tester.check(AC.read(2) == A.read(2));
    tester.check(B(0) == -999);
    tester.out() << "AC(2) = " << AC(Loc<1>(2)) << std::endl;
    tester.out() << "A(2) = " << A(Loc<1>(2)) << std::endl;

    tester.check(A.isShared());
    tester.check(AC.isShared());
    
    AC.makeOwnCopy();
    tester.check(A.isShared());
    tester.check(!AC.isShared());
    
    double save = AC.read(7);
    A(7) = -111;
    
    tester.check(A.read(7) == -111);
    tester.check(AC.read(7)  == save);
    tester.check(C(3) == -111);
    tester.out() << "AC(2) = " << AC(7) << std::endl;
    tester.out() << "A(2) = " << A(7) << std::endl;

    Array_t E(I);
    for (int i = 0; i < 10; i++)
      E(i) = i;

    tester.out() << "E: ";
    for (int i = 0; i < 10; i++)
      tester.out() << E.read(i) << " ";
    tester.out() << std::endl;

    Array_t F;
    tester.check(F.domain().size() == 0);
    
    F = E;
    
    tester.check(F.isShared());
    tester.check(E.isShared());
    
    tester.out() << "F == E" << std::endl;
    tester.out() << "F: ";
    for (int i = 0; i < 10; i++)
      {
        tester.out() << F(i) << " ";
        tester.check(F(i) == E(i));
      }
    tester.out() << std::endl;

    Array_t G(I);
    for (int i = 0; i < 10; i++)
      G(i) = i*i;

    tester.out() << "G: ";
    for (int i = 0; i < 10; i++)
      tester.out() << G(i) << " ";
    tester.out() << std::endl;
    
    tester.check(!G.isShared());
    
    E = G;

    tester.check(E.isShared());
    tester.check(G.isShared());
    tester.check(!F.isShared());
    
    tester.out() << "E = G;" << std::endl;
    tester.out() << "E: ";
    for (int i = 0; i < 10; i++)
      {
        tester.check(E(i) == G(i));
        tester.out() << E(i) << " ";
      }
    tester.out() << std::endl;

    AC(Loc<1>(2)) = -222;
    tester.check(AC(2) == -222);
    tester.check(A(2)  == -999);
    tester.check(B(0)  == -999);
    tester.out() << "AC(2) = " << AC(Loc<1>(2)) << std::endl;
    tester.out() << "A(2) = " << A(Loc<1>(2)) << std::endl;

    A.makeOwnCopy();
    
    tester.check(!A.isShared());
    tester.check(B.dataBlock().isShared());
    tester.check(C.dataBlock().isShared());
    
    B(1) = -888;
    tester.check(A(3) == -444);
    tester.check(B(1) == -888);
    tester.check(C(1) == -888);
    
    B(3) = -555;
    tester.check(C(2) == -555);
    tester.check(B(3) == -555);
    tester.check(A(5) != -555);
    tester.check(CV(1) == -555);

#if POOMA_EXCEPTIONS
  }
  catch(const char *err) 
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
  catch(const Pooma::Assertion &err)
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
#endif    
  int ret = tester.results("dynamic_test1");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: dynamic_test1.cpp,v $   $Author: jac $
// $Revision: 1.5 $   $Date: 2000/07/19 23:03:34 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
