// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// NotifyEngineWrite
//
// function
// notifyEngineWrite()
//-----------------------------------------------------------------------------

#ifndef POOMA_ENGINE_NOTIFYENGINEWRITE_H
#define POOMA_ENGINE_NOTIFYENGINEWRITE_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// NotifyEngineWrite is a general wrapper class the is used to tell an engine
// that we're going to write to it.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Utilities/WrappedInt.h"

//-----------------------------------------------------------------------------
//
// Full Description:
//
// Multipatch engines will want to fill their guard cells after being written
// to.  This class allows us to notify the engines that need to be notified,
// and do nothing to other engines.
//
// You must specialize NotifyEngineWrite for multiPatch engines.
//-----------------------------------------------------------------------------

template<class Engine>
struct NotifyEngineWrite
{
  NotifyEngineWrite(){}
 ~NotifyEngineWrite(){}

 inline static void
  notify(const Engine &)
  {
    // Engines that are multipatch must specialize this functor.
    CTAssert(!(Engine::multiPatch));
  }
};

//-----------------------------------------------------------------------------
// This helper function simplifies use of the NotifyEngineWrite functor.
//-----------------------------------------------------------------------------

template<class Engine>
inline
void notifyEngineWrite(const Engine &e)
{
  NotifyEngineWrite<Engine>::notify(e);
}

//-----------------------------------------------------------------------------
// This function lets us skip the notification at compile time.  (If we're
// actually reading from the engine, for example.)
//-----------------------------------------------------------------------------

template<class Engine>
inline
void notifyEngineWrite(const Engine &, const WrappedInt<false> &)
{
}

template<class Engine>
inline
void notifyEngineWrite(const Engine &e, const WrappedInt<true> &)
{
  NotifyEngineWrite<Engine>::notify(e);
}

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_ENGINE_NOTIFYENGINEWRITE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: NotifyEngineWrite.h,v $   $Author: jac $
// $Revision: 1.6 $   $Date: 2000/05/10 05:07:37 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
