// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// EngineFunctor<Engine,Tag>
// EngineFunctorDefault<Engine,Tag>
// EngineFunctorScalar<T,Tag>
// EngineView<Tag>
// EngineApply<Tag>
//
// function:
// engineFunctor(eng,tag)
//-----------------------------------------------------------------------------

#ifndef POOMA_ENGINE_ENGINEFUNCTOR_H
#define POOMA_ENGINE_ENGINEFUNCTOR_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// EngineFunctor provides a common interface to a variety of engine queries,
// like "are you compressed" or "are you shifted with respect to another
// engine".  By providing a common interface, we minimize the number of changes
// you need to make to add a new capability that makes non-standard queries of
// engines.
//
// This approach replaces the previous message() function which made queries
// of engines.  The current version doesn't require new member functions to be
// added to engines to support new capabilities.  Also this approach allows for
// simple default cases, since you use partial specialization on EngineFunctor.
//
// WARNING: If you use a default action, you should probably have some
// verification mechanism to ensure that the engine doesn't need to have a
// special action defined.  For example, the DataObject default action checks
// the dataObject enum in the engine to make sure it doesn't have a data
// object.
//
// Default actions for a given functor are specified using
// EngineFunctorDefault.  We define EngineFunctor for Expression Engines and
// general tags, so you cannot specify the action of a specific functor
// for general engines.  (The compiler would not be able to choose between
// EngineFunctor<ExpressionEngine,T> and EngineFunctor<T,YourFunctor>)
// The generic version of EngineFunctor<> calls DefaultEngineFunctor<>, so
// you may define EngineFunctorDefault for a general engine and tag.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/PETE.h"


//-----------------------------------------------------------------------------
//
// Full Description:
//
// EngineFunctor<Eng,Tag> - defines the action corresponding to Tag on the
//   engine of type Eng.  Requires the action return type and apply function:
//
//   typedef return_type   Type_t;     // return type of the action
//   static Type_t apply(const Eng &engine, const Tag &tag);
//   
// Tag - policy tag for the action.  Requires 1 typedef:
//
//   typedef combiner_type Combine_t;  // PETE combiner for the return
//
// EngineFunctorScalar<T,Tag> - computes a value for a scalar leaf. Requires:
//
//   typedef return_type   Type_t;     // return type of the action
//   static Type_t apply(const Scalar<T> &scalar, const Tag &tag);
//
// EngineFunctorDefault<Eng,Tag> - allows you to define generic actions
//   for a specific tag.
//-----------------------------------------------------------------------------

template<class Eng, class Tag>
struct EngineFunctorDefault
{ };

//-----------------------------------------------------------------------------
// The most generic version of EngineFunctor falls through to
// EngineFunctorDefault.
//
// (We define
// template<class Tag> EngineFunctor<ExpressionEngine, Tag>
// so you cannot define
// template<class Engine> EngineFunctor<Engine, SpecificTag>
// without ambiguities.
//
// By using EngineFunctorDefault, you can define general operations
// for a given tag that will be caught by all non-expression engines.
//-----------------------------------------------------------------------------

template<class Eng, class Tag>
struct EngineFunctor
{
  typedef typename EngineFunctorDefault<Eng,Tag>::Type_t Type_t;
  inline static
  Type_t apply(const Eng &e, const Tag &t)
  {
    return EngineFunctorDefault<Eng,Tag>::apply(e,t);
  }
};

//-----------------------------------------------------------------------------
//
// engineFunctor(engine, tag)
//
// This helper function is shorthand for:
//
// EngineFunctor<Engine, Tag>::apply(engine, tag)
//
//-----------------------------------------------------------------------------

template<class Eng, class Tag>
inline typename EngineFunctor<Eng,Tag>::Type_t
engineFunctor(const Eng &e, const Tag &tag)
{
  return EngineFunctor<Eng,Tag>::apply(e,tag);
}

//-----------------------------------------------------------------------------
//
// EngineFunctorScalar<T, Tag>
//
// Users must specialize this struct for all tags.
// The specialization needs to contain Type_t and an apply method:
//
// template<class T>
// struct EngineFunctorScalar<T, MyTag >
// {
//   typedef ... Type_t;
//
//   inline static
//   Type_t apply(const T &s, const MyTag &t)
//   {
//     ...
//   }
// };
//
//-----------------------------------------------------------------------------

template<class T, class Tag>
struct EngineFunctorScalar
{ };

//-----------------------------------------------------------------------------
// Overview: 
//
// EngineView<Tag> and EngineApply<Tag> are replacements for EngineFunctor.
// EngineFunctor applied to an expression uses forEach, which means there are
// two levels of indirection at the leaves.  EngineView and EngineTag are
// forEach functors, reducing the number of levels of indirection.
//
//-----------------------------------------------------------------------------

template<class Tag>
struct EngineView;

//-----------------------------------------------------------------------------
// LeafFunctor specializations for EngineView.
//
// Applying EngineView to a general node is an error.
// Applying EngineView to a scalar just returns the scalar.
//-----------------------------------------------------------------------------

template<class Node, class Tag>
struct LeafFunctor<Node, EngineView<Tag> >
{
};

template<class T, class Tag>
struct LeafFunctor<Scalar<T>, EngineView<Tag> >
{
  typedef Scalar<T> Type_t;
  static inline
  Type_t apply(const Scalar<T> &s, const EngineView<Tag> &)
  {
    return s;
  }
};

//-----------------------------------------------------------------------------
// For a given type of engine view, you must either specialize LeafFunctor
// for all engines or provide a specialization of DefaultEngineView.
//
// This level of indirection is necessary to avoid the ambiguity that would
// result from attempting to provide the specializations:
//
// LeafFunctor<ExpressionEngine, EngineView<GeneralTag>>
// LeafFunctor<GeneralEngine, EngineView<SpecificTag>>
//-----------------------------------------------------------------------------

template<class Engine, class Tag>
struct DefaultEngineView;

template<int Dim, class T, class E, class Tag>
struct LeafFunctor<Engine<Dim, T, E>, EngineView<Tag> >
{
  typedef Engine<Dim, T, E> Subject_t;
  typedef DefaultEngineView<Subject_t, Tag> EngineView_t;
  typedef typename EngineView_t::Type_t Type_t;

  static inline
  Type_t apply(const Subject_t &engine,
	       const EngineView<Tag> &tag)
  {
    return EngineView_t::apply(engine, tag);
  }
};


template<class Tag>
struct EngineApply;

//-----------------------------------------------------------------------------
// LeafFunctor specializations for EngineApply.
//
// Applying EngineView to a general node is an error.
// There is no scalar specialization since the default varies from tag to tag.
//-----------------------------------------------------------------------------

template<class Node, class Tag>
struct LeafFunctor<Node, EngineApply<Tag> >
{
};

//-----------------------------------------------------------------------------
// For a given type of engine view, you must either specialize LeafFunctor
// for all engines or provide a specialization of DefaultEngineView.
//
// This level of indirection is necessary to avoid the ambiguity that would
// result from attempting to provide the specializations:
//
// LeafFunctor<ExpressionEngine, EngineView<GeneralTag>>
// LeafFunctor<GeneralEngine, EngineView<SpecificTag>>
//-----------------------------------------------------------------------------

template<class Engine, class Tag>
struct DefaultEngineApply;

template<int Dim, class T, class E, class Tag>
struct LeafFunctor<Engine<Dim, T, E>, EngineApply<Tag> >
{
  typedef Engine<Dim, T, E> Subject_t;
  typedef DefaultEngineApply<Subject_t, Tag> EngineApply_t;
  typedef typename EngineApply_t::Type_t Type_t;

  static inline
  Type_t apply(const Subject_t &engine,
	       const EngineApply<Tag> &tag)
  {
    return EngineApply_t::apply(engine, tag);
  }
};


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_ENGINE_ENGINEFUNCTOR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: EngineFunctor.h,v $   $Author: sa_smith $
// $Revision: 1.7 $   $Date: 2000/07/04 03:05:45 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
