// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// DataObjectApply
// DataObjectRequest<RequestTag>
// LeafFunctor<DataObjectRequest,Leaf>
// GetDataObject
// DataObjectRequest<GetDataObject>
// BlockAffinity
// DataObjectRequest<BlockAffinity>
//-----------------------------------------------------------------------------

#ifndef POOMA_ENGINE_DATAOBJECT_H
#define POOMA_ENGINE_DATAOBJECT_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
// Traits and tags necessary for dealing with the Smarts data object
// inside some engines.
//
// Engine_t::dataObject is true if the engine has
// a smarts data object, false otherwise.  Anything that requires information
// about smarts data objects should query this trait first.
//
// The tag needs to be of type
// DataObjectTag<RequestType>, which satisfy the interface of array message
// tags and ForEach LeafFunctor tags.  In this file we define two request types,
// GetDataObject and BlockAffinity to return the data object and the affinity:
//
// Pooma::DataObject_t obj = forEach(array,DataObjectRequest<GetDataObject>())
// int affinity = forEach(array,DataObjectRequest<BlockAffinity>())
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Threads/PoomaSmarts.h"
#include "Engine/EngineFunctor.h"
#include "PETE/PETE.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// DataObjectApply is used to wrap the two cases of whether an object has a
// smarts data object or not.
//-----------------------------------------------------------------------------

template<bool Block>
struct DataObjectApply
{ };

template<>
struct DataObjectApply<false>
{
  // engine has no data object, so return default value
  template<class Engine,class Functor>
  inline static
  typename Functor::Type_t
  apply(const Engine&, const Functor& functor)
  {
    return functor.defaultValue();
  }
};

template<>
struct DataObjectApply<true>
{
  // engine has a data object, so pass request via message
  template<class Engine, class Functor>

  inline static
  typename Functor::Type_t
  apply(const Engine& engine, const Functor& functor)
  {
    return functor(engine.dataObject());
  }
};

//-----------------------------------------------------------------------------
//  DataObjectRequest<RequestType>
//
//  This class has two functions.
//  1) It's a message functor that can be handed to arrays.  Arrays that have
//  data objects will support the function:
//  array.dataBlockRequest(datablocktag);
//  To implement the message function, the engine should call operator() on the
//  DataObjectRequest with a pointer to the data object.
//  2) It's a LeafFunctor tag class that allows us to apply smarts data object
//  to expressions.
//
//  In cases where the engine contains an expression, the request will be
//  passed on to engines in the expression that have data objects.  For example,
//  if we use DataObjectRequest to request a lock on a stencil engine, it uses
//  forEach to request the same lock on all the engines in the expression
//  contained inside the stencil.
//
//  To implement a version of DataObjectRequest<Request>, the following
//  interface is required:
//
//  typedef ... Type_t;
//  typedef ... Combine_t;
//  Type_t operator()(Pooma::DataObject_t*) const;
//  Type_t defaultValue() const;
//
//  Type_t is the return type of the functor.  Combine_t is a ForEach combine
//  tag that will be used if the engine contains an expression.  operator()
//  lets you compute the return value given a pointer to the actual data
//  object. defaultValue() returns the value that should be returned when
//  there is no data object.
//-----------------------------------------------------------------------------

template<class RequestType>
class DataObjectRequest
{};

template<class Eng, class RequestType>
struct EngineFunctorDefault<Eng, DataObjectRequest<RequestType> >
{
  enum { hasDataObject = Eng::hasDataObject };

  typedef typename DataObjectRequest<RequestType>::Type_t Type_t;

  static inline
  Type_t apply(const Eng &e, const DataObjectRequest<RequestType> &request)
  {
    return DataObjectApply<hasDataObject>::apply(e, request);
  }
};

//-----------------------------------------------------------------------------
// LeafFunctors for DataObjectRequest.
// For scalars, we return the default value provided by the functor.
//-----------------------------------------------------------------------------

template<class RequestType,class T>
struct EngineFunctorScalar<T, DataObjectRequest<RequestType> >
{
  typedef typename DataObjectRequest<RequestType>::Type_t Type_t;
  inline static
  Type_t apply(const T &, const DataObjectRequest<RequestType> &tag)
  {
    return tag.defaultValue();
  }
};

//-----------------------------------------------------------------------------
// DataObjectRequest<BlockAffinity>
// Used to get the affinity for an array.
//-----------------------------------------------------------------------------

struct BlockAffinity { };


//-----------------------------------------------------------------------------
//
// CLASS NAME
//   AffinityCombine
//
// DESCRIPTION
//   This trivial combiner returns the left-most object in an expression.
//
//-----------------------------------------------------------------------------

struct AffinityCombine
{
  AffinityCombine() { }
  AffinityCombine(const AffinityCombine &) { }
};

template<class Op>
struct Combine2<int, int, Op, AffinityCombine>
{
  typedef int Type_t;
  inline static
  Type_t combine(int a,int b,Op ,AffinityCombine )
  {
    return a;
  }
};

template<>
class DataObjectRequest<BlockAffinity>
{
public:
  // This functor returns an affinity
  typedef int Type_t;

  // Affinities combine to return the left-most affinity.  It might make
  // more sense to perform a more intelligent combination.
  // (Currently the affinity for an iterate comes from the LHS. If the LHS
  // has multiple parts, then we're only interested in the leftmost array.)

  typedef AffinityCombine Combine_t;

  DataObjectRequest() { }

  // Just return the DataObject pointer if there is one, otherwise
  // return a null pointer.
  inline Type_t operator()(Pooma::DataObject_t* obj) const
  {
    return obj->affinity();
  }

  inline Type_t defaultValue() const
  {
    return (-1);
  }
};

#endif // POOMA_ENGINE_DATAOBJECT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DataObject.h,v $   $Author: swhaney $
// $Revision: 1.29 $   $Date: 2000/03/07 13:16:50 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
