// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// DynamicArray test 4: Create/Destroy ops on arrays that are views
// on the same data.
//-----------------------------------------------------------------------------

// include files

#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Pooma/DynamicArrays.h"
#include "Pooma/GMPArrays.h"


int main(int argc, char *argv[])
{
  int i;

  // Initialize POOMA and output stream, using Tester class

  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);
  tester.out() << argv[0] << ": DynamicArray view dynamic ops." << std::endl;
  tester.out() << "-------------------------------------------" << std::endl;

  // Create simple single-patch dynamic arrays.

  tester.out() << "Creating DynamicArray objects ..." << std::endl;
  Interval<1> D(6);
  DynamicArray<int,Dynamic> b2(D);
  b2 = 4;
  DynamicArray<int,Dynamic> b3(b2);
  Pooma::blockAndEvaluate();
  tester.out() << "Created DynamicArray b2 = " << b2 << std::endl;
  tester.out() << "Created DynamicArray b3 = " << b3 << std::endl;
  for (i=D.first(); i <= D.last(); ++i)
    {
      tester.check(b2.read(i) == b3.read(i));
      tester.check(b3.read(i) == 4);
    }
  tester.check("b2 sum matches b3", sum(b2) == sum(b3));

  // Create multi-patch dynamic arrays.

  tester.out() << "Creating MP DynamicArray objects ...";
  tester.out() << std::endl;
  Loc<1> blocks(3);
  GridPartition<1> gpar(blocks);
  LocalMapper<1> cmap(gpar);  
  DynamicLayout dynlayout(D,gpar,cmap);
  DynamicArray< int, MultiPatch<DynamicTag,Dynamic> > c2(dynlayout);
  c2 = 7;
  DynamicArray< int, MultiPatch<DynamicTag,Dynamic> > c3(c2);
  Pooma::blockAndEvaluate();
  tester.out() << "Created MP DynamicArray c2 = " << c2 << std::endl;
  tester.out() << "Created MP DynamicArray c3 = " << c3 << std::endl;
  for (i=D.first(); i <= D.last(); ++i)
    {
      tester.check(c2.read(i) == c3.read(i));
      tester.check(c3.read(i) == 7);
    }
  tester.check("c2 sum matches c3", sum(c2) == sum(c3));

  // Change the array value and see that they are both affected.

  tester.out() << "Changing Array's to be equal to 2 ..." << std::endl;
  b3 = 2;
  c3 = 2;
  Pooma::blockAndEvaluate();
  tester.out() << "New b2 = " << b2 << std::endl;
  tester.out() << "New b3 = " << b3 << std::endl;
  tester.out() << "New c2 = " << c2 << std::endl;
  tester.out() << "New c3 = " << c3 << std::endl;
  for (i=D.first(); i <= D.last(); ++i)
    {
      tester.check(b2.read(i) == b3.read(i));
      tester.check(b2.read(i) == 2);
      tester.check(c2.read(i) == c3.read(i));
      tester.check(c2.read(i) == 2);
    }
  tester.check("b2 sum matches b3", sum(b2) == sum(b3));
  tester.check("c2 sum matches c3", sum(c2) == sum(c3));

  // Create new elements in c2; this should change c3 as well.

  tester.out() << "Creating 3 new elements in c2, set to 3 ..." << std::endl;
  c2.create(3);
  c2.sync();
  c2(i++) = 3;
  c2(i++) = 3;
  c2(i++) = 3;
  tester.out() << "New c2 = " << c2 << std::endl;
  tester.out() << "New c3 = " << c3 << std::endl;
  for (i=D.first(); i <= (D.last() + 3); ++i)
    {
      tester.check(c2.read(i) == c3.read(i));
      tester.check(c2.read(i) == (i <= D.last() ? 2 : 3));
      c2(i) = i - D.first();
    }

  // Create a view of the first three odd elements, and then do
  // a dynamic op.

  tester.out() << "Creating view of the 0, 2, 4th elements ..." << std::endl;
  Array< 1, int, DynamicView> v2(b2(Range<1>(0,4,2)));
  Array< 1, int, MultiPatchView<DynamicTag,Dynamic,1> > v3(c2(Range<1>(0,4,2)));
  tester.out() << "Current b2 = " << b2 << std::endl;
  tester.out() << "Current b3 = " << b3 << std::endl;
  tester.out() << "Current v2 = " << v2 << std::endl;
  tester.out() << "Current c2 = " << c2 << std::endl;
  tester.out() << "Current c3 = " << c3 << std::endl;
  tester.out() << "Current v3 = " << v3 << std::endl;
  tester.out() << "Deleting first four elements of original array c2 ...";
  tester.out() << std::endl;
  c2.destroy(Interval<1>(4), ShiftUp());
  c2.sync();
  tester.out() << "New c2 = " << c2 << std::endl;
  tester.out() << "New c3 = " << c3 << std::endl;
  tester.out() << "New v3 = " << v3 << std::endl;
  for (i=D.first(); i <= (D.last() - 1); ++i)
    {
      int offi = i - D.first();
      tester.check(c2.read(i) == c3.read(i));
      tester.check(c2.read(i) == offi + 4);
      if (offi < 3)
        tester.check(v3.read(offi) == c2.read(2*offi + D.first()));
    }

  // Return resulting error code and exit; Tester will shut down POOMA.

  tester.out() << "-------------------------------------------" << std::endl;
  int retval = tester.results("DynamicArray view dynamic ops");
  Pooma::finalize();
  return retval;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: dynamic_array_test4.cpp,v $   $Author: jac $
// $Revision: 1.8 $   $Date: 2000/07/19 23:17:36 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
