// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_LEFT_DOMAIN_H
#define POOMA_DOMAIN_LEFT_DOMAIN_H

//-----------------------------------------------------------------------------
// Class:
// LeftDomain<int>
// DomainTraits<LeftDomain<int> >
//-----------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// LeftDomain is one of the domain wildcards, which are used when constructing
// other domains using specific combination rules.  LeftDomain means to use
// the starting endpoint of the domain of a second 'reference' domain, with
// a new user-provided right endpoint, when constructing a new
// domain.  It is also used when constructing new domains with no other
// arguments to mean that the domain should not be initialized, which can
// save considerable time in some circumstances.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/DomainTraits.h"
#include "Domain/Loc.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description of LeftDomain:
//
// LeftDomain is a special domain class which is used as a 'wildcard'.
// Wildcards are useful when constructing new domains based on some other
// 'reference' domain, which is done when doing things like making a new
// view of an Array.  Wildcard domains use the reference domain to determine
// what the 'final' domain should be.  LeftDomain refers to 'use the left
// endpoint of the reference domain, with newly provided right endpoint,
// as the new domain values'.
//
// LeftDomain can be used as one of the arguments to the 'combineSlice' or
// 'fillSlice' routines in the NewDomain combiners, in which case the user-
// supplied reference domain is used with the 'setWildcardDomain' method
// of the domain being filled to get the final domain settings.
//
// Wildcard domains in general can also be used in the constructors for
// regular domain objects.  If they are given, they indicate that those
// dimensions should not be initialized, which can be helpful to avoid
// extra unneeded work when the domain will be filled with new values very
// soon.
//
//-----------------------------------------------------------------------------

template<int Dim>
class LeftDomain
{
public:
  //
  // Typedefs and enumerations
  //

  typedef LeftDomain<Dim> Domain_t;
  typedef LeftDomain<1>   OneDomain_t;
  typedef int             Element_t;
  enum { dimensions = Dim };

  //
  // Constructors.
  //

  // default constructor
  LeftDomain() : endpoints_m(Pooma::NoInit()) { CTAssert(Dim > 0); }

  // copy constructor
  LeftDomain(const LeftDomain<Dim> &d) : endpoints_m(d.endpoints_m) {
    CTAssert(Dim > 0);
  }

  // templated constructors for 1 ... 7 arguments, used to fill up the
  // values for the endpoints
  template<class T1>
  explicit LeftDomain(const T1 &a)
    : endpoints_m(a) { CTAssert(Dim > 0); }

  template<class T1, class T2>
  LeftDomain(const T1 &a, const T2 &b)
    : endpoints_m(a,b) { CTAssert(Dim > 0); }

  template<class T1, class T2, class T3>
  LeftDomain(const T1 &a, const T2 &b, const T3 &c)
    : endpoints_m(a,b,c) { CTAssert(Dim > 0); }

  template<class T1, class T2, class T3, class T4>
  LeftDomain(const T1 &a, const T2 &b, const T3 &c, const T4 &d)
    : endpoints_m(a,b,c,d) { CTAssert(Dim > 0); }

  template<class T1, class T2, class T3, class T4, class T5>
  LeftDomain(const T1 &a, const T2 &b, const T3 &c, const T4 &d, const T5 &e)
    : endpoints_m(a,b,c,d,e) { CTAssert(Dim > 0); }

  template<class T1, class T2, class T3, class T4, class T5,
           class T6>
  LeftDomain(const T1 &a, const T2 &b, const T3 &c, const T4 &d, const T5 &e,
      const T6 &f)
    : endpoints_m(a,b,c,d,e,f) { CTAssert(Dim > 0); }

  template<class T1, class T2, class T3, class T4, class T5,
           class T6, class T7>
  LeftDomain(const T1 &a, const T2 &b, const T3 &c, const T4 &d, const T5 &e,
      const T6 &f, const T7 &g)
    : endpoints_m(a,b,c,d,e,f,g) { CTAssert(Dim > 0); }

  //
  // Destructor.  For this class there is nothing to do.
  //

  ~LeftDomain() { }

  //
  // Domain-like methods
  //

  // Get the Nth element of our domain.  This only returns a copy; there
  // is no way to modify a LeftDomain after it has been constructed, except
  // through the setDomain method and the operator=.
  OneDomain_t operator[](int n) const { return OneDomain_t(endpoints_m[n]); }

  // setDomain: change our LeftDomain to be the newly provided one.
  void setDomain(const LeftDomain<Dim> &d) { endpoints_m = d.endpoints_m; }

  // given another reference domain, return the proper values for first,
  // length, and stride
  template<class T>
  typename DomainTraits<T>::Element_t first(const T &u) const {
    // for a LeftDomain, first is taken from the given argument's first
    return u.first();
  }
  int first(int u) const { return u; }

  template<class T>
  typename DomainTraits<T>::Element_t length(const T &u) const {
    // for a LeftDomain, length is determined by our right endpoint and
    // the given argument's first.  We do this by making a temporary of type
    // T with the proper endpoints and calling length on that.  If the values
    // for the endpoints are inconsistent with the domain type T, it will
    // be an error.
    CTAssert(Dim == 1);
    T dom(u.first(), endpoints_m[0].first(), u.stride());
    return dom.length();
  }
  int length(int u) const {
    CTAssert(Dim == 1);
    Interval<1> dom(u, endpoints_m[0].first());
    return dom.length();
  }

  template<class T>
  typename DomainTraits<T>::Element_t stride(const T &u) const {
    // for a LeftDomain, the stride is the same as the given domain's stride
    return u.stride();
  }
  int stride(int) const { return 1; }


  //
  // operator=
  //

  LeftDomain<Dim> &operator=(const LeftDomain<Dim> &d) {
    endpoints_m = d.endpoints_m;
    return *this;
  }

protected:

private:
  // our list of right endpoints
  Loc<Dim> endpoints_m;
};


//-----------------------------------------------------------------------------
//
// Full Description:
//
// DomainTraits<LeftDomain<Dim>> provides traits information about LeftDomain,
// which is one of the domain wildcards.  It has a quite stripped-down
// selection of traits, the basic ones needed to allow wildcards to be used
// in the construction of regular and strided domains.  This includes the
// dimension and the type of the wildcard, and an enum indicating that it is
// a wildcard.  Also, getDomain returns a 1D element of the N-dimensional
// list of wildcards.
//
//-----------------------------------------------------------------------------

template<int Dim>
struct DomainTraits< LeftDomain<Dim> >
{
  // necessary typedefs: the domain type when this is copied
  typedef LeftDomain<Dim> Domain_t;
  typedef LeftDomain<1>   OneDomain_t;
  typedef LeftDomain<1>   PointDomain_t;
  typedef LeftDomain<Dim> AskDomain_t;

  // necessary enums
  enum { domain          = 1 };
  enum { dimensions      = Dim };
  enum { sliceDimensions = Dim };
  enum { wildcard        = 1 };
  enum { singleValued    = 0 };

  // get the Nth element of the domain, and return a OneDomain_t
  // object with it (here, as a copy).  Since LeftDomain does not store any
  // data, we can just return a new copy.
  static OneDomain_t getDomain(const Domain_t &d, int n) {
    return d[n];
  }

  // convert from the Nth element of the domain to a single point, if
  // possible, and return a PointDomain_t.  Here, we just return a new copy
  // of PointDomain_t, since this object does not have any data.
  static PointDomain_t getPointDomain(const Domain_t &d, int n) {
    return d[n];
  }
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_DOMAIN_LEFT_DOMAIN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: LeftDomain.h,v $   $Author: jac $
// $Revision: 1.9 $   $Date: 2000/06/27 01:58:28 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
