// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Tests ZeroFaceBC
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"

#include <iostream>

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  // Useful ingredients for constructing and initializing Fields below:

  // Vertex domain:
  int nVerts = 4;
  Interval<1> I(nVerts), J(nVerts);
  Interval<2> domain(I,J);

  // Construct 2D mesh, zero-origin, unit-spacing:
  typedef UniformRectilinearMesh<2> Mesh_t;
  Mesh_t mesh(domain);

  // Construct 2D Cell-centered Geometry, with 1 guard layer:
  GuardLayers<2> gl(1);
  typedef DiscreteGeometry<Cell, Mesh_t> Geometry_t;
  Geometry_t geometry(mesh, gl);

  // --------------------------------------------------------------------------
  // Set a ZeroFaceBC on one face of a scalar Field:

  // Create scalar Field (Field<2,double,....>):
  Field<Geometry_t, double> a(geometry);

  // Assign it to the X position values
  //won't compile  a() = a.x().comp(0)();
  a(a.geometry().totalDomain()) = 9.0;
  std::cout << "~~~~~a~~~~~~~" << std::endl 
            << a.all() << std::endl << std::endl;

  // ZeroFaceBC
  ZeroFaceBC zf(1);
  
  BCondItem *bcScalar = zf.create(a);  
  
  Range<1> r(0,nVerts - 2), s(-1,nVerts - 1);
  Range<2> rs(r, s), ss(s, s);
  
  bcScalar->notifyPreRead(rs);
  std::cout << a.all() << std::endl;
  
  bcScalar->notifyPreRead(ss);
  std::cout << a.all() << std::endl;
  
  
  // --------------------------------------------------------------------------
  // Set ZeroFaceBC on all faces of a vector Field:

  Field<Geometry_t, Vector<3> > b(geometry);

  // Assign it to the position values, and a constant value in 3rd component:
  // won't compile:
//   (b.comp(0))() = 1.3;
//  b.comp(1)() = b.x().comp(1)();
//   b.comp(0) = 1.3;
//   b.comp(1) = 2.3;
//   b.comp(2) = 3.3;

  Vector<3> v3(9.1, 9.2, 9.3);
  b(b.geometry().totalDomain()) = v3;
  std::cout << "~~~~~~~~~~~~b~~~~~~~~~~" << std::endl << b.all()
	    << std::endl << std::endl;

  // Use a BcondList, for convenience:
  BCondList vbc;

  // Make ZeroFaceBC BCs, and add to the BCondList

  // Low X face:
  vbc.addBoundaryCondition(b, ZeroFaceBC(0, true));

  // High X face:
  vbc.addBoundaryCondition(b, ZeroFaceBC(1, true));

  // Low Y face:
  vbc.addBoundaryCondition(b, ZeroFaceBC(2, true));

  // High Y face:
  vbc.addBoundaryCondition(b, ZeroFaceBC(3, true));
  
  vbc.notifyPreRead(rs);
  std::cout << b.all() << std::endl;
  
  vbc.notifyPreRead(ss);
  std::cout << b.all() << std::endl;

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TestZeroFaceBC.cpp,v $   $Author: swhaney $
// $Revision: 1.11 $   $Date: 2000/03/07 13:16:15 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
