// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Contents:
//   Doof2d Example
//-----------------------------------------------------------------------------

// include files

// Bring in Pooma Array machinery.
#include "Pooma/Arrays.h"

// Define stencil object.
#include "stencil9.h"

#include <iostream>

//-----------------------------------------------------------------------------
// Main program.
//-----------------------------------------------------------------------------

int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);

  Inform out;

  // create arrays

  Array<2, double, MultiPatch<UniformTag, Remote<Brick> > > a, b;

  // set problem size

  int n = 2000;
  out << "Problem Size: " << n << std::endl;

  int i, niters = 20;

  // create array domain and resize arrays

  Interval<1> I(1,n), J(1,n);
  Interval<2> domain(I,J);

  // Create a 2 x 2 layout

  UniformGridPartition<2> partition(Loc<2>(2, 2), GuardLayers<2>(1),
				    GuardLayers<2>(1));

  UniformGridLayout<2> layout(domain, partition, DistributedTag());

  a.initialize(layout);
  b.initialize(layout);

  // get domains and constant for diffusion stencil

  const double fact = 1.0/9.0;

  // reset array element values

  a = b = 0.0;
  Pooma::blockAndEvaluate();
  a(niters,niters) = 1000.0;

  // Run 9pt doof2d without coefficients using expression

  out << "Diffusion using expression ..." << std::endl;
  out << "iter = 0, a_mid = " << a(niters,niters) << std::endl;
  for (i=1; i<=niters; ++i)
  {
    b(I,J) = fact * (a(I+1,J+1) + a(I+1,J  ) + a(I+1,J-1) +
                     a(I  ,J+1) + a(I  ,J  ) + a(I  ,J-1) +
                     a(I-1,J+1) + a(I-1,J  ) + a(I-1,J-1));
    a = b;
    Pooma::blockAndEvaluate();
    out << "iter = " << i << ", a_mid = " << a(niters,niters)
              << std::endl;
  }

  // reset array element values

  a = b = 0.0;
  Pooma::blockAndEvaluate();
  a(niters,niters) = 1000.0;

  // Run 9pt doof2d without coefficients using stencil

  out << "\nDiffusion using stencil ..." << std::endl;
  out << "iter = 0, a_mid = " << a(niters,niters) << std::endl;
  Stencil<DoofNinePt> stencil9;
  for (i=1; i<=niters; ++i)
  {
    b(I,J) = stencil9(a);
    a = b;
    Pooma::blockAndEvaluate();
    out << "iter = " << i << ", a_mid = " << a(niters,niters)
              << std::endl;
  }

  out << "All Done!" << std::endl;
  
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Doof2d.cpp,v $   $Author: sa_smith $
// $Revision: 1.16 $   $Date: 2000/08/18 03:28:52 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
