// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// ParticleBC template definitions for ReflectBC specialization.
//-----------------------------------------------------------------------------

// include files

#include "Domain/Interval.h"
#include "Array/Array.h"
#include "Evaluator/PatchFunction.h"
#include "Utilities/PAssert.h"
#include "Tiny/Vector.h"

//-----------------------------------------------------------------------------
// ReflectBCFunc functor
// Apply the reflecting BC in a patchwise fashion.
//-----------------------------------------------------------------------------

template <class T>
struct ReflectBCFunc
{
  // Constructor
  ReflectBCFunc(const T& min, const T& max)
    : min_m(min), max_m(max) { }

  // apply method implements BC
  template <class ArrayPatch1, class ArrayPatch2>
  void apply(const ArrayPatch1& obj, const ArrayPatch2& sub, int)
  {
    PAssert(sub.domain() == obj.domain());
    int from = sub.domain()[0].first();
    int to = sub.domain()[0].last();
    for (int i = from; i <= to; ++i)
      {
        // check lower boundary
        if (sub(i) < min_m)
          obj(i) = 2 * min_m - sub(i);

        // check upper boundary
        else if (sub(i) > max_m)
          obj(i) = 2 * max_m - sub(i);
      }
  }

  // bounds for BC
  T min_m, max_m;
};

//-----------------------------------------------------------------------------
// ReflectBCFunc functor specialized for Vector
// Apply the reflecting BC in a patchwise fashion.
// Each component is treated separately.
//-----------------------------------------------------------------------------

template <int Dim, class T, class E>
struct ReflectBCFunc< Vector<Dim,T,E> >
{
  // Typedefs
  typedef Vector<Dim,T,E> Type_t;

  // Constructor
  ReflectBCFunc(const Type_t& min, const Type_t& max)
    : min_m(min), max_m(max) { }

  // apply method implements BC
  template <class ArrayPatch1, class ArrayPatch2>
  void apply(const ArrayPatch1& obj, const ArrayPatch2& sub, int)
  {
    PAssert(sub.domain() == obj.domain());
    int from = sub.domain()[0].first();
    int to = sub.domain()[0].last();
    for (int i = from; i <= to; ++i)
      {
        for (int d = 0; d < Dim; ++d)
          {
            // check lower boundary
            if (sub(i)(d) < min_m(d))
              obj(i)(d) = 2 * min_m(d) - sub(i)(d);

            // check upper boundary
            else if (sub(i)(d) > max_m(d))
              obj(i)(d) = 2 * max_m(d) - sub(i)(d);
          }
      }
  }

  // bounds for BC
  Type_t min_m, max_m;
};

//-----------------------------------------------------------------------------
// void applyBoundaryCondition()
// Apply the reflecting boundary condition to the subject.
//-----------------------------------------------------------------------------

template <class Subject, class Object, class T>
void
ParticleBC< Subject, Object, ReflectBC<T> >::
applyBoundaryCondition(int pid)
{
  // get limits of reflecting range
  T min = bc_m.min(), max = bc_m.max();

  // loop over local patches and apply BC using functor
  ReflectBCFunc<T> bcfun(min,max);
  if (pid < 0)
    {
      PatchFunction< ReflectBCFunc<T>,
	             PatchParticle2<true,false> > patchfun(bcfun);
      patchfun.block(object_m,subject_m);
    }
  else
    {
      bcfun.apply(object_m.patchLocal(pid)(), subject_m.patchLocal(pid)(), pid);
    }
}


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ReflectBC.cpp,v $   $Author: sa_smith $
// $Revision: 1.13 $   $Date: 2000/05/04 22:38:04 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
