// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Field test 10: Field::makeOwnCopy().
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

//-----------------------------------------------------------------------------
// Globals
//-----------------------------------------------------------------------------

const int NX = 5, NY = 5;
Interval<1> I(NX), J(NY);
  

//-----------------------------------------------------------------------------
// Test function
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
void doTest1(Pooma::Tester &tester, Field<Geom, T, EngineTag> &f)
{
  Field<Geom, T, EngineTag> g(f);
  
  f = 2;
  tester.check("1: Before makeOwnCopy()", all(f.all() == g.all()), true);
  g.makeOwnCopy();

  Pooma::blockAndEvaluate();

  f(0,0) = 3; f(-1,-1) = 3.0;  
  tester.check("1: After makeOwnCopy()", g(0,0), 2.0);
  g.applyBoundaryConditions();
  tester.check("1: After makeOwnCopy() + BC", f(-1,-1), 3.0);
}

template<class Geom, class T, class EngineTag>
void doTest2(Pooma::Tester &tester, Field<Geom, T, EngineTag> &f)
{
  Field<Geom, T, EngineTag> g;
  g.initialize(f);

  f = 2;
  tester.check("2: Before makeOwnCopy()", all(f.all() == g.all()), true);
  g.makeOwnCopy();

  Pooma::blockAndEvaluate();

  f(0,0) = 3; f(-1,-1) = 3.0;  
  tester.check("2: After makeOwnCopy()", g(0,0), 2.0);
  g.applyBoundaryConditions();
  tester.check("2: After makeOwnCopy() + BC", f(-1,-1), 3.0);
}

template<class Geom, class T, class EngineTag>
void doTest3(Pooma::Tester &tester, Field<Geom, T, EngineTag> &f)
{
  Field<Geom, T, EngineTag> g;
  g.initialize(f.geometry(), f.engine(), f.bconds());

  f = 2;
  tester.check("3: Before makeOwnCopy()", all(f.all() == g.all()), true);
  g.makeOwnCopy();

  Pooma::blockAndEvaluate();

  f(0,0) = 3; f(-1,-1) = 3.0;  
  tester.check("3: After makeOwnCopy()", g(0,0), 2.0);
  g.applyBoundaryConditions();
  tester.check("3: After makeOwnCopy() + BC", f(-1,-1), 3.0);
}

//-----------------------------------------------------------------------------
// Main program
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  // Create the mesh.
  
  Vector<2,double> origin;
  Vector<2,double> spacings;
  for (int d = 0; d < 2; d++) 
    {
      origin(d) = d;
      spacings(d) = d + 1;
    }
  UniformRectilinearMesh<2> mesh(I, J, origin, spacings);
  
  // Create the geometry.
  
  typedef DiscreteGeometry<Cell, UniformRectilinearMesh<2> > Geometry_t;
  
  Geometry_t geom(mesh, GuardLayers<2>(1));
  
  // Make a Brick-Engine-based field.
  
  Field<Geometry_t> f(geom, AllZeroFaceBC());

  doTest1(tester, f);
  doTest2(tester, f);
  doTest3(tester, f);
    
  int ret = tester.results("field_test10");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: field_test10.cpp,v $   $Author: swhaney $
// $Revision: 1.3 $   $Date: 2000/03/07 13:17:15 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
