/*
    Copyright (C) 1999 by  XCIN TEAM

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For any question or suggestion, please mail to xcin mailing-list:
    xcin@linux.org.tw, or the maintainer Tung-Han Hsieh: thhsieh@linux.org.tw
*/      


#include <string.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include "xcintool.h"
#include "module.h"
#include "bims.h"
#include "bimsphone.h"



/*----------------------------------------------------------------------------

	phone_init()

----------------------------------------------------------------------------*/

static int
phone_init(void *conf, char *objname, core_config_t *xc)
{
    phone_conf_t *cf = (phone_conf_t *)conf;
    char *s, *cmd[2], value[50], truefn[256];
    FILE *fp;

    cmd[0] = objname;
    cmd[1] = "INP_CNAME";                               /* inp_names */
    if (get_resource(cmd, value, 50, 2))
        cf->inp_cname = strdup(value);
    else
	cf->inp_cname = "`";
    cf->inp_ename = objname;

    cmd[1] = "SETKEY";                                  /* setkey */
    if (get_resource(cmd, value, 50, 2))
        cf->setkey = (char)atoi(value);
    else {
        perr(XCINMSG_WARNING, "%s: %s: value not found.\n", objname, cmd);
        return False;
    }

    cmd[1] = "TSI_FNAME";
    if (get_resource(cmd, value, 50, 2) &&
	(fp = open_data(value, "r", xc->default_dir, xc->user_dir, "tab",
		xc->locale.lc_ctype, truefn, 256, XCINMSG_WARNING))) {
	fclose(fp);
	cf->tsi_fname = strdup(truefn);
    }
    else
	return False;

    cmd[1] = "YIN_FNAME";
    if (get_resource(cmd, value, 50, 2) &&
	(fp = open_data(value, "r", xc->default_dir, xc->user_dir, "tab",
		xc->locale.lc_ctype, truefn, 256, XCINMSG_WARNING))) {
	fclose(fp);
	cf->yin_fname = strdup(truefn);
    }
    else
	return False;

    cmd[1] = "KEYMAP";
    if (get_resource(cmd, value, 50, 2)) {
	int mapnum;

	mapnum = atoi(value);
	if (mapnum >= 0 && mapnum < N_KEYMAPS)
	    cf->keymap = (char)mapnum;
    }

    if (bimsInit(cf->tsi_fname, cf->yin_fname) == 0)
        return True;
    else
	return False;
}


/*----------------------------------------------------------------------------

	Tool Functions.

----------------------------------------------------------------------------*/

#define N_MAX_KEYCODE  4

static int
big5_mbs_wcs(wch_t *wcs, char *mbs, int wcs_len)
{
    int len;
    char *s;

    for (s=mbs, len=0; *s && len<wcs_len-1; s+=2, len++) {
	wcs[len].s[0] = *s;
	wcs[len].s[1] = *(s+1);
    }
    wcs[len].wch = (wchar_t)0;

    return len;
}

static void
commit_string(inpinfo_t *inpinfo, int n_chars)
{
    static char *str=NULL;

    if (str)
	free(str);
    str = bimsFetchText(inpinfo->icid, n_chars);
    DebugLog("%s\n", str);
    inpinfo->cch = str;
}

static void
editing_status(inpinfo_t *inpinfo, phone_iccf_t *iccf)
{
    char *str;
    int len;

    inpinfo->edit_pos = bimsQueryPos(inpinfo->icid);

    str = bimsQueryZuYinString(inpinfo->icid);
    inpinfo->keystroke_len =
        big5_mbs_wcs(inpinfo->s_keystroke, str, N_MAX_KEYCODE+1);
    free(str);

    str = bimsQueryInternalText(inpinfo->icid);
    len = strlen(str) / 2;
    if (len >= iccf->lcch_size) {
	iccf->lcch_size = len+1;
	inpinfo->lcch = realloc(inpinfo->lcch, (len+1)*sizeof(wch_t));
    }
    inpinfo->n_lcch = big5_mbs_wcs(inpinfo->lcch, str, len+1);
    free(str);

    inpinfo->cch_publish.wch = (wchar_t)0;
    if (! inpinfo->keystroke_len && len) {
	inpinfo->cch_publish.wch = inpinfo->lcch[inpinfo->n_lcch-1].wch;
	iccf->mode |= PHONE_MODE_COMPOSEDOK;
    }
    else
	iccf->mode &= ~(PHONE_MODE_COMPOSEDOK);
}


/*----------------------------------------------------------------------------

	phone_xim_init() & phone_xim_end()

----------------------------------------------------------------------------*/

static void
check_winsize(inpinfo_t *inpinfo, phone_iccf_t *iccf)
{
    int size;

    size = inpinfo->xcin_wlen / 2 - 1;
    if (iccf->lcch_size != size) {
        bimsSetMaxLen(inpinfo->icid, size-1);
	iccf->lcch_max_len = size-1;
    }
    if (iccf->lcch_size <= size) {
	if (inpinfo->lcch)
	    free(inpinfo->lcch);
	iccf->lcch_size = size+1;
        inpinfo->lcch = calloc(iccf->lcch_size, sizeof(wch_t));
    }
}

static int keymaplist[N_KEYMAPS] = {
	BC_KEYMAP_ZO, BC_KEYMAP_ETEN, BC_KEYMAP_HSU};

static int
phone_xim_init(void *conf, inpinfo_t *inpinfo)
{
    phone_conf_t *cf = (phone_conf_t *)conf;
    phone_iccf_t *iccf;
    int i;

    inpinfo->iccf = calloc(1, sizeof(phone_iccf_t));
    iccf = (phone_iccf_t *)inpinfo->iccf;

    inpinfo->inp_cname = cf->inp_cname;
    inpinfo->inp_ename = cf->inp_ename;
    inpinfo->area3_len = N_MAX_KEYCODE * 2 + 2;
    inpinfo->guimode = GUIMOD_LISTCHAR | GUIMOD_SELKEYSPOT;
    inpinfo->keystroke_len = 0;
    inpinfo->s_keystroke = calloc(N_MAX_KEYCODE+1, sizeof(wch_t));
    inpinfo->n_mcch = 0;
    inpinfo->mcch[0].wch = (wchar_t)0;
    inpinfo->n_lcch = 0;
    inpinfo->edit_pos = 0;
    inpinfo->cch_publish.wch = (wchar_t)0;
    check_winsize(inpinfo, iccf);

    inpinfo->n_selkey = 10;
    for (i=0; i<9; i++) {
	inpinfo->s_selkey[i].wch  = (wchar_t)0;
	inpinfo->s_selkey[i].s[0] = i + '1';
    }
    inpinfo->s_selkey[i].wch  = (wchar_t)0;
    inpinfo->s_selkey[i].s[0] = '0';
    bimsSetKeyMap(inpinfo->icid, keymaplist[cf->keymap]);
}

static unsigned int
phone_xim_end(void *conf, inpinfo_t *inpinfo)
{
    phone_conf_t *cf = (phone_conf_t *)conf;
    phone_iccf_t *iccf = (phone_iccf_t *)inpinfo->iccf;
    unsigned int ret;

    if (inpinfo->n_lcch) {
	commit_string(inpinfo, inpinfo->n_lcch);
	editing_status(inpinfo, iccf);
	ret = IMKEY_COMMIT;
    }
    else
	ret = IMKEY_ABSORB;

    free(inpinfo->iccf);
    free(inpinfo->s_keystroke);
    free(inpinfo->lcch);
    inpinfo->iccf = NULL;
    inpinfo->s_keystroke = NULL;
    inpinfo->lcch = NULL;
    bimsFreeBC(inpinfo->icid);

    return ret;
}


/*----------------------------------------------------------------------------

	phone_keystroke() & phone_show_keystroke()

----------------------------------------------------------------------------*/

static int
determine_selection(inpinfo_t *inpinfo, phone_iccf_t *iccf, 
			KeySym key, int num, int *base, char *selection)
{
    int i;

    if (key) {
        switch(key) {
        case XK_Escape:
            return(11);
        case XK_less:
            if (*base - 10 >= 0)
                *base -= 10;
            break;
	case XK_space:
        case XK_greater:
            if (*base + 10 <= num)
                *base += 10;
            break;
        case XK_0:
            return(10);
        case XK_1:
            return(1);
        case XK_2:
            return(2);
        case XK_3:
            return(3);
        case XK_4:
            return(4);
        case XK_5:
            return(5);
        case XK_6:
            return(6);
        case XK_7:
            return(7);
        case XK_8:
            return(8);
        case XK_9:
            return(9);
        default:
            break;
        }
    }

    if (*base - 10 >= 0) {
        if (*base + 10 <= num)
	    inpinfo->mcch_pgstate = MCCH_MIDDLE;
	else
            inpinfo->mcch_pgstate = MCCH_END;
    }
    else if (*base + 10 <= num)
	inpinfo->mcch_pgstate = MCCH_BEGIN;
    else 
	inpinfo->mcch_pgstate = MCCH_ONEPG;

    for (i=0; i<10 && i + *base < num; i++) {
	inpinfo->mcch[i].wch = (wchar_t)0;
        strncpy(inpinfo->mcch[i].s, selection+(*base+i)*2, 2);
    }
    inpinfo->mcch[i].wch = (wchar_t)0;
    inpinfo->n_mcch = i;
    iccf->mode &= ~(PHONE_MODE_COMPOSEDOK);

    return(0);
}


static unsigned int
phone_keystroke(void *conf, inpinfo_t *inpinfo, keyinfo_t *keyinfo)
{
    phone_conf_t *cf   = (phone_conf_t *)conf;
    phone_iccf_t *iccf = (phone_iccf_t *)inpinfo->iccf;
    KeySym keysym = keyinfo->keysym;
    int state, rval, n_commit, num_selection, base;
    char *selection;

    check_winsize(inpinfo, iccf);
    if (keysym == XK_Up)
	bimsToggleSelection(inpinfo->icid);
    else if (keysym == XK_Return) {
	if (inpinfo->n_lcch) {
	    commit_string(inpinfo, inpinfo->n_lcch);
	    editing_status(inpinfo, iccf);
	    return IMKEY_COMMIT;
	}
	else
	    return IMKEY_IGNORE;
    }
    else if (keysym == XK_space) {
	if ((iccf->mode & PHONE_MODE_COMPOSEDOK)) {
	    commit_string(inpinfo, inpinfo->n_lcch);
	    editing_status(inpinfo, iccf);
	    return IMKEY_COMMIT;
	}
	else if (! inpinfo->n_lcch && ! inpinfo->keystroke_len)
	    return IMKEY_IGNORE;
    }
    else if (keysym == XK_BackSpace && 
	     ! inpinfo->n_lcch && ! inpinfo->keystroke_len)
	return IMKEY_IGNORE;

    state = bimsQueryState(inpinfo->icid);
    switch(state) {
    case BC_STATE_EDITING:
	inpinfo->n_mcch = 0;
	inpinfo->mcch[0].wch = (wchar_t)0;
        rval = bimsFeedKey(inpinfo->icid, keysym);
        switch (rval) {
        case BC_VAL_COMMIT:
	    n_commit = (inpinfo->n_lcch >= iccf->lcch_max_len) ? 
			inpinfo->n_lcch - iccf->lcch_max_len + 1 : 1;
	    commit_string(inpinfo, n_commit);
	case BC_VAL_ABSORB:
	    editing_status(inpinfo, iccf);
	    if (rval == BC_VAL_COMMIT)
	        return IMKEY_COMMIT;
	    else
	        return IMKEY_ABSORB;
        default:
	    return IMKEY_IGNORE;
        }
        break;

    case BC_STATE_SELECTION:
        selection = bimsQuerySelectionText(inpinfo->icid);
        num_selection = bimsQuerySelectionNumber(inpinfo->icid);
        base = bimsQuerySelectionBase(inpinfo->icid);
        rval = determine_selection(inpinfo, iccf, keysym,
			num_selection, &base, selection);
        bimsSetSelectionBase(inpinfo->icid, base);
        if (! rval)
            return IMKEY_ABSORB;
        else {
	    if (rval < 11)
                bimsPindownByNumber(inpinfo->icid, rval+base);
	    inpinfo->n_mcch = 0;
	    inpinfo->mcch[0].wch = (wchar_t)0;
            bimsToggleEditing(inpinfo->icid);
	    editing_status(inpinfo, iccf);
            return IMKEY_ABSORB;
        }
        break;

    default:
        break;
    }
    return IMKEY_IGNORE;
}

static int 
phone_show_keystroke(void *conf, simdinfo_t *simdinfo)
{
    phone_conf_t *cf = (phone_conf_t *)conf;
    static wch_t keystroke_list[5];
    char *str;
    unsigned int code;
    struct ZhiInfo zhi_info;

    if (simdinfo->cch_publish.wch) {
	code = ((unsigned int)(simdinfo->cch_publish.s[0])) << 8;
	code |= (unsigned int)(simdinfo->cch_publish.s[1]);
	bzero(&zhi_info, sizeof(struct ZhiInfo));
	zhi_info.code = (ZhiCode)code;

        if (tabeZhiInfoLookupYin(&zhi_info) == 0) {
    	    simdinfo->s_keystroke = keystroke_list;
	    str = (char *)tabeYinToZuYinSymbolSequence(zhi_info.yin[0]);
	    big5_mbs_wcs(keystroke_list, str, 5);
	    free(str);
	    return True;
	}
    }
    simdinfo->s_keystroke = NULL;
    return False;
}


/*----------------------------------------------------------------------------

        Definition of phonetic input method module (templet).

----------------------------------------------------------------------------*/

static char *phone_valid_objname[] = { "bimsphone", "phone", NULL };

module_t module_ptr = {
    "phone",                                    /* name */
    MODULE_VERSION,                             /* version */
    "phonetic input method module.",            /* comments */
    phone_valid_objname,                        /* valid_objname */
    MOD_CINPUT,                                 /* module_type */
    sizeof(phone_conf_t),                       /* conf_size */
    phone_init,                                 /* init */
    phone_xim_init,                             /* xim_init */
    phone_xim_end,                              /* xim_end */
    NULL,					/* switch_in */
    NULL,					/* switch_out */
    phone_keystroke,                            /* keystroke */
    phone_show_keystroke,                       /* show_keystroke */
};

