/***************************************************************************
                             rmcast.c
                             -------------------
    begin                : May 2001
    copyright            : (C) 2001 by Jorge Allyson Azevedo
                                       Milena Scanferla
                                       Magnos Martinello
                                       Daniel Sadoc
    email                : {allyson,milena,magnos,sadoc}@land.ufrj.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/******************************************************************************************************
 *																									  *
 * Reliable Multicast Protocol (rmcast.c)															  *
 *																									  *
 * Top-level routines used by the reliable multicast library.										  *
 *																									  *
 * use tab spacing = 4																				  *
 *																									  *
 ******************************************************************************************************/

#ifndef RMCAST_C
#define RMCAST_C

#include <math.h>
#include <unistd.h> 
#include <signal.h>
#include <pthread.h>  
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <netdb.h>

#include "rmcast.h"
#include "rminternals.h"
#include "rmtcp.h"

#ifdef SOLARIS

int siginterrupt(int sig, int flag) {
               struct sigaction act;
               (void)sigaction(sig, NULL, &act);
               if (flag)
                       act.sa_flags &= ~SA_RESTART;
               else
                       act.sa_flags |= SA_RESTART;
               return
               sigaction(sig, &act, NULL);
       }
       
#endif       


pthread_t rcv_thread, signal_handler_thread;


extern int		    h_errno;
extern EVENT_LIST   *event_list;
extern CACHE	    *cache;
extern USER_INFO    local_user_info;
extern sigset_t     alrmset;
extern FILE         *logfile;
extern GLOBAL_OPTIONS rmcast_options;


static int r_qid = -1; /* message queue identifier */

static int initialized = 0; /* flag that specifies if the user is connected to
                               the multicast group.

                               It may also be read as: did the user try to
                               get the current state?*/

#ifdef SOLARIS
pthread_mutex_t  change_local_user_sn;
#else
pthread_mutex_t  change_local_user_sn = PTHREAD_MUTEX_INITIALIZER;
#endif

void RM_readConfigFile(char *filename)
{
    FILE *fpin = NULL;
    int  i, optionnumber;
    char option[255], value[255], trash[4];
    char line[255];

    fpin = fopen(filename,"r");
    


    if (fpin != NULL)
    {
        fprintf(stderr,"RELIABLE MULTICAST LIBRARY\n\n");
        fprintf(stderr,"---- Reading configuration file ----\n");
        
        fgets(line, 255, fpin);        
       
        while (!feof(fpin))
        {

            i = sscanf(line,"%[^=#] %[=#] %s",option, trash, value);
            if (i != 3 || trash[0]=='#')
            {
                // fprintf(stderr,"skiping line (%d)\n",i);
                
            }
            else
            {
                fprintf(stderr,"%s=%s\n",option,value);
                
                if (strcmp(option,"DEST_IP")==0) {
                    optionnumber = DEST_IP;
                } else if (strcmp(option,"DEST_PORT")==0) {
                    optionnumber = DEST_PORT;
                } else if (strcmp(option,"TTL")==0) {
                    optionnumber = TTL;
                } else if (strcmp(option,"MICROSLEEP")==0) {
                    optionnumber = MICROSLEEP;
                } else if (strcmp(option,"AVG_DELAY")==0) {
                    optionnumber = TIMER_LOWER;
                } else if (strcmp(option,"TIMER_LOWER")==0) {
                    optionnumber = TIMER_LOWER;
                } else if (strcmp(option,"TIMER_UPPER")==0) {
                    optionnumber = TIMER_UPPER;
                } else if (strcmp(option,"TIMER_DISTRIBUTION")==0) {
                    optionnumber = TIMER_LOWER;
                } else if (strcmp(option,"MAX_NAK")==0) {
                    optionnumber = MAX_NAK;                    
                } else if (strcmp(option,"MAX_MEMBER_CACHE_SIZE")==0) {
                    optionnumber = MAX_NAK;                    
                } else if (strcmp(option,"VERSION")==0) {
                    optionnumber = VERSION;
                } else if (strcmp(option,"TRANSMISSION_MODE")==0) {
                    optionnumber = TRANSMISSION_MODE;
                } else if (strcmp(option,"LOG_FILE")==0) {
                    optionnumber = LOG_FILE;
                } else if (strcmp(option,"NEW_USER_SUPPORT")==0) {
                    optionnumber = NEW_USER_SUPPORT;
                } else if (strcmp(option,"STATISTICS")==0) {
                    optionnumber = STATISTICS;
                } else if (strcmp(option,"REFRESH_TIMER")==0) {
                    optionnumber = REFRESH_TIMER;
                } else if (strcmp(option,"LOSS_PROB")==0) {
                    optionnumber = LOSS_PROB;
                } else {
                
                    fprintf(stderr,"Invalid option in input file: %s",line);
                    continue;
                }
                
                if (optionnumber!=LOG_FILE && optionnumber!=DEST_IP)
                {
                    RM_setOption(optionnumber,(void*)atoi(value));
                }
                else 
                {
/*
 *                     if (optionnumber==TIMER_LOWER || optionnumber==TIMER_UPPER)
 *                     {
 *                         RM_setOption(optionnumber,(void*)atof(value));
 *                     }
 *                     else
 */
                        RM_setOption(optionnumber,(void*)value);
                }
                    
            }
            
            fgets(line, 255, fpin);                    
        }
    }

    fprintf(stderr,"----- Configuration file OK.-----\n\n");    
    if (fpin!=NULL)
        fclose(fpin);
   
}


/**************************************************************************************
 *
 * void RM_initialize()
 *
 * Initializes internal structures (cache, event list and message queue) and
 * local user information (ip, pid and sn).
 *
 * Parameters: no parameteres. 
 *
 **************************************************************************************/

#ifdef SOLARIS
int gethostname(char *name, int namelen);	
#endif

void RM_initialize()
{
	struct hostent *localhost_ent;
	int i;
	char localhost_ip[MAX_IP_STRING_SIZE], localhost_name[31];
    
/*
 * #ifdef  SOLARIS
 * 
 *     sigset_t     alrmset;
 *     
 *     sigfillset(&alrmset);     
 *     
 *     
 * 	   
 * 	pthread_sigmask(SIG_BLOCK, &alrmset, NULL);        
 *     
 *     pthread_mutex_init(&change_local_user_sn,NULL);
 *         
 * #endif    
 */

    rmcast_options.cur_state_server_is_up = 0;    
    
        
    initialized = 0;
    
        
    if (rmcast_options.log_file[0]!=0)
    {
        if ((logfile = fopen(rmcast_options.log_file,"w"))==NULL)
        {
    	    fprintf(stderr,"RM_initialize Error: could not open %s.",rmcast_options.log_file);
            exit(1);
        }
        
        if ( setvbuf( logfile, NULL, _IONBF, 0 ) !=0 )
            fprintf(stderr,"RM_initialize ERROR: setvbuf failed.\n");

        
    }
    else
    {
    	logfile = NULL;
    }
		
	for (i=0; i< MAX_IP_STRING_SIZE; i++ ) localhost_ip[i]=0;
	
	/* localhost_ip <- local ip */

	gethostname(localhost_name, 30);

    if (logfile!=NULL)
    {
  		fprintf (logfile,"host: %s\n", localhost_name );

    }

#ifdef DEBUG_NET_MSGS	
    fprintf (stderr,"DEBUG_NET_MSGS RM_initialize: local host name: %s\n", localhost_name );
#endif                
	

	localhost_ent  = gethostbyname(localhost_name);
	strcpy(localhost_ip, 
		inet_ntoa(*((struct in_addr*)(localhost_ent->h_addr_list[0]))));
	
	
	/* initialization of the local cache, event list and message queue */
	
	cacheInit(&cache);
	
	eventListInit(&event_list);  
    
	if (r_qid == -1 && (r_qid = init_queue() ) == -1)
	{	
		fprintf(stderr,"RM_initialize ERROR: Couldn't create message queue.\n");
		exit(1);
	} 
	
	/* initialization of local user info */
	
	pthread_mutex_lock(&change_local_user_sn);
	
	local_user_info.sn = 0;    
	
	pthread_mutex_unlock(&change_local_user_sn);
	
	local_user_info.member_id.pid = (int)getpid();		   
	strncpy(local_user_info.member_id.ip, localhost_ip, MAX_IP_STRING_SIZE);

	if (logfile!=NULL)	
    {
     	fprintf (logfile, "ip: %s\n",localhost_ip);
        fprintf (logfile, "pid: %d\n",local_user_info.member_id.pid);
		
		fprintf (logfile, "---------------------------------------------------------------------------------------------------------------------------\n");
		fprintf (logfile, "Time Snd/Rcv/Loss type sender_ip       sender_pid requested_ip    requested_pid  sn  [{base sn} {win size} {hmask} {lmask}]\n");
		fprintf (logfile, "---------------------------------------------------------------------------------------------------------------------------\n");
    
    }
    
#ifdef DEBUG_NET_MSGS
	fprintf(stderr, "DEBUG_NET_MSGS RM_initialize: local host ip: %s\n",localhost_ip);
#endif
	
}



#ifdef SUPPORT_FOR_NEW_USERS
static void recvfrom_alarm(int signo)
{
    return; /* just interrupt the recvfrom() */
}
#endif

/**************************************************************************************
 *
 * int	RM_getCurState (char *group, int port, CurState *c)
 *
 * When this routine is used, RM_initialize must have been called at least once.
 *  
 * Sets up multicast group information, creates the thread responsible for receiving 
 * packets and activates the alarm responsible for catching events from the event list.
 * If the CurState parameter is different from NULL, the current state of the multicast
 * group will be catched.
 *
 * Parameters: group, the multicast group ip;
 *			   port,  the multicast group port;
 *             c,     pointer to the location where the current state of 
 *                    the multicast group will be stored.
 *
 * Returns: the socket created.  If c is not equal to NULL, it will point to the current
 *          state of the group.
 *
 **************************************************************************************/

pthread_t pthread_cur_state_server;    

#ifndef SOLARIS        
#define SOCKLEN_TYPE       (socklen_t*)
#else
#define SOCKLEN_TYPE       
#endif

#ifdef SOLARIS            

int siginterrupt( int sig, int flag);

#endif      

int  RM_getCurState(char *group1, int port1, CurState *c)
{
#ifdef SOLARIS
    sigset_t     alrmset;
#endif    

    
    char *group; int port;

    if (((int)group1) == RM_USE_CURRENT_CONFIG)
    {
        group = rmcast_options.dest_ip;
    }      
    else
    {
        group = group1;
    }
    
    if (((int)port1) == RM_USE_CURRENT_CONFIG)
    {
        port = rmcast_options.dest_port;
    }      
    else
    {
        port = port1;
    }
    
 
        
	/* Now, we are setting the multicast info (multicast port and multicast ip), */
	
	local_user_info.group_id.port = port ;	  
	
	strncpy(local_user_info.group_id. ip, group, MAX_IP_STRING_SIZE) ;	  
	
	local_user_info.socket = rmcastCreateSocket  (port); 


	rmcastJoinGroup 	(local_user_info.socket,group);

	if (logfile!=NULL)	
    	fprintf(logfile, "joined to group\n");

#ifdef DEBUG_NET_MSGS
	fprintf(stderr,"DEBUG_NET_MSGS RM_getCurState: joined to group\n");
#endif
	

#ifdef SUPPORT_FOR_NEW_USERS
    if ( c!=NULL )
    {
        CurState retval;
        char *buffer;
        int buffsize;
        int timeout = 0;
        PACKET_INFO pckt_info;
        char message[MAX_BUFFER];
        struct sockaddr  sender;
        int len, nbytes;
        
        pckt_info.type = JOIN_REQUEST_PACKET_TYPE;
        pckt_info.sender_id = local_user_info.member_id;
        pckt_info.packet_size = 0;
        
                
        msgPcktMountMessage(&pckt_info, (BYTE**)&buffer, &buffsize);
        
        
        msgPcktShowMessage((BYTE*)buffer);        
        
        
        rmcastSendPacket	  (local_user_info.socket, (BYTE*)buffer, buffsize);        
        
        /* waiting to receive a "join accept packet" */
        
        
#ifdef SOLARIS                

/*
 *         sigemptyset( &alrmset );            
 *         sigaddset  ( &alrmset, SIGALRM );
 *     
 * 	    pthread_sigmask( SIG_UNBLOCK, &alrmset, NULL );    
 */

#endif        
        
        
        siginterrupt(SIGALRM, 1); /* please, see siginterrupt(3) */
        
        if (signal(SIGALRM, recvfrom_alarm)==SIG_ERR)
        {
            fprintf(stderr,"RM_getCurState ERROR: alarm failed");
            exit(1);
        }
        
        alarm ( 2 );
        
#ifdef  DEBUG_NET_MSGS      
	    fprintf(stderr, "DEBUG_NET_MSGS RM_getCurState: Waiting for a 'join accept message'\n");        
#endif
        
      	do {
        
            static int first = 1;
            
            
            if (!first)

#ifdef  DEBUG_NET_MSGS
                fprintf(stderr, "DEBUG_NET_MSGS RM_getCurState: Still waiting for a 'join accept message'\n");           
#endif                
                
            first = 0;
		    
    		bzero(message, MAX_BUFFER);
    		len  = sizeof(struct sockaddr_in);

            if ((nbytes = (int)recvfrom( local_user_info.socket, message, MAX_BUFFER, 0, 
                          (struct sockaddr *) &sender, SOCKLEN_TYPE &len)) < 0)
            {
                if (errno != EINTR)
                {
                    continue;
                }
            }
            
            if (nbytes>0)
    		{
                
#ifdef DEBUG_PCKT
                {
                    int cont;
                    fprintf(stderr,"DEBUG_PCKT RM_getCurState: Received message: \n");
                    for (cont = 0; cont < nbytes; cont++)
                    {
                        fprintf(stderr,"%d [%c] ", ((signed char*)(message))[cont], (message)[cont]);
                    }	
                }
#endif 
                
                
	    		msgPcktUnmountMessage(&pckt_info, (BYTE*)message);
                
#ifdef DEBUG_SHOW
                fprintf(stderr,"DEBUG_SHOW RM_getCurState: Showing message:\n");
                msgPcktShowMessage((BYTE*)message);
#endif
                if(!memberIsEqual(&local_user_info.member_id,&pckt_info.sender_id))
                {
                    timeout=0;
                }
            }
            else
            {
                timeout=1;
                pckt_info.type = -1;
            }
            
       	    
            
        } while (pckt_info.type != JOIN_ACCEPT_PACKET_TYPE && timeout == 0);
        
        siginterrupt(SIGALRM, 0);   /* the default behaviour in Linux */

#ifdef SOLARIS        

/*
 *         sigemptyset( &alrmset );            
 *         sigaddset  ( &alrmset, SIGALRM );
 * 
 *  
 * 	    pthread_sigmask( SIG_BLOCK, &alrmset, NULL );    
 */

#endif            
            
        if ( !timeout )
        {
        
#ifdef DEBUG_NET_MSGS
    	    fprintf(stderr,"DEBUG_NET_MSGS RM_getCurState: 'join accept packet' received\n");                
#endif
            if (rmcastReceiveTCPStatus(pckt_info.sender_id.ip, pckt_info.packet_data.join_accept_packet.port, &retval) == FALSE)
            {
#ifdef DEBUG_NET_MSGS    
                fprintf(stderr,"DEBUG_NET_MSGS RM_getCurState: TCP status received\n");   
#endif                
                retval.size = -1;
            }
        }
        else
        {
        
#ifdef DEBUG_NET_MSGS
            fprintf(stderr,"DEBUG_NET_MSGS RM_getCurState: timer expired probably I'm the first of the group!\n"); 
#endif            
            retval.size = -1;
        }
        
        *c = retval;
    }
    
#endif    

#ifndef SOLARIS /* if we are in LINUX -> that is, SOLARIS constant is NOT defined */


     sigfillset(&alrmset);            
     sigdelset(&alrmset, SIGTSTP);
     
 	pthread_sigmask(SIG_BLOCK, &alrmset, NULL);    

    
#endif
    
#ifdef SUPPORT_FOR_NEW_USERS

    if (pthread_create(&pthread_cur_state_server,NULL,rmcastSendTCPStatus,NULL)!=0)
    {
        fprintf(stderr,"RM_getCurState ERROR: Couldn't create the new thread to send the current status: the 'current status server'\n");
    }
    else
    {
        rmcast_options.cur_state_server_is_up = 1;
    }

#endif    

    initialized = 1;    
    
	return local_user_info.socket;   
}

/**************************************************************************************
 *
 * int	RM_joinGroup (char *group, int port)
 *
 * When this routine is used, RM_initialize must have been called at least once.
 *  
 * Sets up multicast group information, creates the thread responsible for receiving 
 * packets and activates the alarm responsible for catching events from the event list.
 *
 * Parameters: group, the multicast group ip;
 *			   port,  the multicast group port;
 *
 * Returns: the socket created.  If c is not equal to NULL, it will point to the current
 *          state of the group.
 *
 **************************************************************************************/


int  RM_joinGroup (char *group1, int port1)
{

#ifdef SOLARIS
    sigset_t   alrmset;
#endif
    
    char *group; int port;

    if (((int)group1) == RM_USE_CURRENT_CONFIG)
    {
        group = rmcast_options.dest_ip;
    }      
    else
    {
        group = group1;
    }
    
    if (((int)port1) == RM_USE_CURRENT_CONFIG)
    {
        port = rmcast_options.dest_port;
    }      
    else
    {
        port = port1;
    }

#ifdef DEBUG_NET_MSGS
    fprintf(stderr,"DEBUG_NET_MSGS RM_joinGroup: %s initialized, will try to get the current state.\n",initialized?"":"not");
#endif    
    
    if (!initialized)
    {
        RM_getCurState(group, port, NULL);
    }
    
   
	/* creating the thread responsible for receiving packets */
	
	pthread_create( &rcv_thread, NULL, rmcastReceivePackets, (void*)local_user_info.socket);
	
	pthread_create( &signal_handler_thread, NULL, rmcastHandleSignals, NULL);
    
	/* and activating the alarm responsible for catching events from the event list. */
     
    sleep(1);
 
   	eventListInsert(&event_list, &(local_user_info.member_id), REF_SND_WAIT, -1 );     

#ifdef SOLARIS

/*
 *     sigemptyset(&alrmset);            
 *     sigaddset(&alrmset, SIGINT);
 *     sigaddset(&alrmset, SIGTSTP);        
 *     pthread_sigmask(SIG_UNBLOCK, &alrmset, NULL);        
 */
    
#endif    
    

    return local_user_info.socket;    
}





int RM_SendCurStatus(int connfd, char *buff, int buffsize)
{

/* FIXME: this routine may consume a lot of time 
 *
 *         (it's necessary to implement it as a new thread or process) 
 *
 */
    int n;

    
   char *packed_cache; int cache_size;

    
    write(connfd, &buffsize, sizeof(int));

    n = write(connfd, buff, buffsize);

    if (n < 0)
	    perror("RM_SendCurStatus ERROR: write error");
        
#ifdef DEBUG_SHOW
    fprintf(stderr,"DEBUG_SHOW RM_SendCurStatus: Showing the cache...\n");

    cacheShow(cache);
#endif
    
    cachePack(&cache, &packed_cache, &cache_size);
         
    write(connfd, &cache_size, sizeof(int));
 
    n = write(connfd, packed_cache, cache_size);
 
    if (n < 0)
 	    perror("RM_SendCurStatus ERROR: write error");

    close(connfd);
    
    return n;
}     

void RM_getOption(int opt, void *return_value)      
{
    switch (opt)
    {
        case NEW_USER_SUPPORT:
            *((int *)return_value) = rmcast_options.new_user_support;
            break;
            
        case DEST_IP:
            strcpy((char *)return_value,rmcast_options.dest_ip);
            break;

        case DEST_PORT:
            *((int *)return_value) = rmcast_options.dest_port;
            break;
            
        default:
            fprintf(stderr,"RM_getOption warning: Trying to get an invalid option: %d\n",opt);
            break;
    }

}

void RM_setOption(int opt, void *opt_value)
{
    char logfilename[255], localhost_name[127];
    char pid[16];

    switch (opt)
    {
        case PIPE: /* address of the pipe used to make the connection between the application and the multicast library */
        
            rmcast_options.pipe = (int*)opt_value;
            
            break;

        case SHUT_DOWN_ROUTINE: /* pointer to the routine called when shutting down the protocol */
        
            rmcast_options.shut_down_routine = (void (*)(void))opt_value;
            
            break;
            
        case DEST_IP:
        
            strcpy( rmcast_options.dest_ip , (char*)opt_value );  break;
            
        case DEST_PORT:
        
            rmcast_options.dest_port = (int)opt_value ; break;
        
        case TTL:
        
            rmcast_options.ttl = (int)opt_value; break;
        
        case MICROSLEEP:
            
            rmcast_options.microsleep = (int)opt_value; break;
        
        case TIMER_LOWER:
        
            /* we multiply by 1000 because we need microsseconds instead of milisseconds */
            rmcast_options.timer_lower = 1000*(int)opt_value; break; 

        case TIMER_UPPER:

            /* we multiply by 1000 because we need microsseconds instead of milisseconds */
            rmcast_options.timer_upper = 1000*(int)opt_value; break;        

        case TIMER_DISTRIBUTION:
        
            rmcast_options.timer_distribution = (int)opt_value; break;        

        case MAX_NAK:
        
            rmcast_options.max_nak = (int)opt_value; break;
        
        case MAX_MEMBER_CACHE_SIZE:
        
            rmcast_options.max_member_cache_size = (int)opt_value; break;

        case VERSION:            
        
            rmcast_options.version = (int)opt_value; break;                
            
        case TRANSMISSION_MODE:            
        
            rmcast_options.transmission_mode = (int)opt_value; break;                
        
        case LOG_FILE:
        
            
            if (strcmp((char*)opt_value,"NULL")==0)
            {
                rmcast_options.log_file[0] = 0;
            }
            else {
            
                gethostname(localhost_name, 30);      

                strcpy(logfilename,"");
                strcat(logfilename,(char*)opt_value);
                strcat(logfilename,".");
                strcat(logfilename,localhost_name);
                strcat(logfilename,".");
                sprintf(pid,"%d",(int)getpid());
                strcat(logfilename,pid);

                strcpy ( rmcast_options.log_file, logfilename ); 
            }
            
            break;
            
           
        case NEW_USER_SUPPORT:
        
            rmcast_options.new_user_support = (int)opt_value; break;                
        
        case STATISTICS:
        
            rmcast_options.statistics = (int)opt_value; break;                
            
        case REFRESH_TIMER:
        
            rmcast_options.refresh_timer = (int)opt_value; break;                

        case LOSS_PROB:
        
            rmcast_options.loss_prob = (int)opt_value/(float)100; break;                
        
        default:
        
            fprintf(stderr,"RM_setOption warning: Trying to set and invalid option: %d\n",opt);
        
            break;

            
    }
}



/**************************************************************************************
 *
 * int	RM_leaveGroup(int  sock,   char *group)
 *
 * Used to leave from the multicast group.
 *
 * Parameters: sock, the multicast socket;
 *			   group, the multicast ip.
 *
 * Returns: always TRUE.
 *
 **************************************************************************************/

int  RM_leaveGroup(int	sock,	char *group1)
{
    char *group;

    if (((int)group1) == RM_USE_CURRENT_CONFIG)
    {
        group = rmcast_options.dest_ip;
    }      
    else
    {
        group = group1;
    }


	rmcastLeaveGroup  (sock, group);   
	
	remove_queues(0);
	
	return TRUE;
}

/**************************************************************************************
 *
 * void RM_terminate (void)
 *
 * Exits from the program. 
 *
 **************************************************************************************/

void RM_terminate (void)
{

#ifdef DEBUG_NET_MSGS
	fprintf(stderr,"DEBUG_NET_MSGS RM_terminate: Exiting...\n");
#endif
	
	exit(1);	
}

/**************************************************************************************
 *
 * int	RM_sendto	 (int socket, char *buffer, int buffsize)
 *
 * Sends a message, contained in buffer, to the multicast group.
 *
 * Parameters: socket, the socket number;
 *			   buffer, message to be sent;
 *			   buffsize, maximum number of chars to be sent.
 *
 * Returns: 1, on success;
 *			0, on error.
 *
 **************************************************************************************/

int  RM_sendto	  (int socket, char *buffer, int buffsize)
{
	PACKET_INFO pckt_info;
	
	BYTE* local_buffer = NULL;
	
	int final_message_size, retval;
	
	pckt_info.type = DATA_PACKET_TYPE;
	strcpy(pckt_info.sender_id.ip, local_user_info.member_id.ip);
	pckt_info.sender_id.pid = local_user_info.member_id.pid;
	
    lock_eventlist_and_cache();
    
	pthread_mutex_lock(&change_local_user_sn);	  

	pckt_info.packet_data.data_packet.sn = local_user_info.sn;
	
	pthread_mutex_unlock(&change_local_user_sn);
    	
	pckt_info.packet_data.data_packet.data_size = buffsize;
	
	pckt_info.packet_data.data_packet.data = (BYTE*)malloc((sizeof(char)*buffsize)+1);
	memcpy((char*)(pckt_info.packet_data.data_packet.data), buffer, buffsize);
	   
	retval = cacheInsertMessage(&cache, &(pckt_info.sender_id), &(pckt_info.packet_data.data_packet));
    
/*
    rmcastRemoveEvent(NULL,REF_SND_WAIT);
    retval = eventListInsert(&event_list, &(local_user_info.member_id), REF_SND_WAIT, 0 );		
*/
    
    unlock_eventlist_and_cache();
        
	if (retval == 0)
	{
		fprintf(stderr,"RM_sendto ERROR: inserting message in local cache.\n");
		exit(0);
	}
    
	   
	msgPcktMountMessage(&pckt_info, &local_buffer, &final_message_size);

#ifdef DEBUG_NET_MSGS
	fprintf(stderr,"DEBUG_NET_MSGS RM_sendto: Will send a message... \n");
#endif
	
	rmcastSendPacket (socket, local_buffer, final_message_size);
    
#ifdef DEBUG_NET_MSGS
	fprintf(stderr,"DEBUG_NET_MSGS RM_sendto: Message sent ok!\n");
#endif
	
#ifdef DEBUG_SHOW		
	msgPcktShowMessage(local_buffer);
#endif	  
	
	free(local_buffer);
	
	return TRUE;
}

/**************************************************************************************
 *
 * int	RM_recv 	 (int socket, char *buffer, int buffsize)
 *
 * Waits to receive a message.
 *
 * Parameters: socket, the socket number;
 *			   buffer, message received;
 *			   buffsize, maximum number of chars to be received.
 *
 * Returns: number of bytes received, on success;
 *			-1, on error.
 *
 **************************************************************************************/

int  RM_recv	  (int socket, char *buffer, int buffsize)
{
	struct q_entry r_entry = { 0 };
	int mlen = 0, size = 0, power = 0, ind = 0;

#ifdef DEBUG_NET_MSGS
	fprintf(stderr,"DEBUG_NET_MSGS RM_recv: Waiting to receive a \"reliable\" message...\n");
#endif
	
	/* waiting to receive a reliable message trhough the message queue */
	
	if ((mlen = msgrcv(r_qid, &r_entry, MAXOBN, 
		0, MSG_NOERROR)) == -1) {
		
#ifdef DEBUG_NET_MSGS
        perror("DEBUG_NET_MSGS RM_recv: msgrcv failed"); 
#endif        
    
        if ( errno == EIDRM || errno ==  EINVAL )
            return -1;
	}
	
	else {
		
		/* a reliable message was received */
		
		size =0;
		
		/* convert mtext from base 256 to base 10 - mtext[0] is the least signif. digit of mtext */
		
		for (power = 1, ind = 0; ind < MSG_QUEUE_HEADER_SIZE;  power = (int)pow(256,ind))
		{
			size += (((unsigned char)(r_entry.mtext[ind])) * power);
            
#ifdef DEBUG_NET_MSGS
			fprintf(stderr,"DEBUG_NET_MSGS RM_recv: power: %d r_entry.mtext[%d]=%d size=%d\n",power,ind,r_entry.mtext[ind],size);
#endif			  
			ind++;
		}
		
		if (size > buffsize)
		{
			fprintf(stderr,"RM_recv warning: Message greater than buffer. message size=%d, buffer size=%d\n", size, buffsize);
			exit(1);
		}
		
		else
		{
			memcpy(buffer,&(r_entry.mtext[MSG_QUEUE_HEADER_SIZE]),size);
		}
		
#ifdef DEBUG_NET_MSGS
		fprintf(stderr,"DEBUG_NET_MSGS RM_recv: A \"reliable\" message was received=%s!\n",buffer);
#endif

	}

#ifdef DEBUG_NET_MSGS
	fprintf(stderr,"DEBUG_NET_MSGS RM_recv: sending message to application. Message size=%d\n",size);
#endif	  
	
	return size;
}



/**************************************************************************************/

void RMDEBUG_setpidip(int pid, char *ip)
{
	local_user_info.member_id.pid = pid;		 
	strncpy(local_user_info.member_id.ip, ip, MAX_IP_STRING_SIZE);
}		 

/**************************************************************************************/


void RMDEBUG_setsn(int sn)
{
	pthread_mutex_lock(&change_local_user_sn);
	
	local_user_info.sn = sn;   
	
	pthread_mutex_unlock(&change_local_user_sn);
}
/**************************************************************************************/

#endif
