/***************************************************************************
                             rmmsgpckt.c
                             -------------------
    begin                : May 2001
    copyright            : (C) 2001 by Jorge Allyson Azevedo
                                       Milena Scanferla
                                       Magnos Martinello
                                       Daniel Sadoc
    email                : {allyson,milena,magnos,sadoc}@land.ufrj.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/******************************************************************************************************
 *																									  *
 * Reliable Multicast Protocol (rmmsgpckt.c)														  *
 *																									  *
 * Functions to manipulate (mount and unmount) message packets.										  *
 *																									  *
 * use tab spacing = 4																				  *
 *																									  *
 ******************************************************************************************************/

#ifndef RMMSGPCKT_C

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <ctype.h>

#include "rmstruct.h"
#include "rmmsgpckt.h"

#define RMMSGPCKT_C

#define MIN(a,b)  ((a<b)?a:b) 

#define get(dest, org, bytes) { bzero(dest,bytes); memcpy(dest,org, bytes); org+=bytes; }

#define getPCKInfo(dest, org,bytes) { bzero(dest,bytes); memcpy(dest,org, bytes); dest+=bytes; packet_size+=bytes; }

#define get_member_id(member_id, message)			\
{													\
	fprintf(stderr,"\tMember id:\n");				\
	get(&member_id,message,sizeof(MEMBER_ID));		\
	fprintf(stderr,"\t\tip: "); 					\
	for (k=0; k<MAX_IP_STRING_SIZE; k++)			\
	{												\
	if ((int)(member_id.ip[k])==0) break;			\
	fprintf(stderr,"%c",member_id.ip[k]);			\
	}												\
	fprintf(stderr,"\n");							\
	fprintf(stderr,"\t\tpid: %d\n",member_id.pid);	\
}													\
	
char *MESSAGE_TYPE_NAME[9] = {  "UN", /* "UNKNOWN PACKET TYPE" */
                                "DT", /* "DATA PACKET        ", */
                                "RT", /* "RETRANSMISSION PACKET ", */
                                "NK", /* "NAK PACKET         ", */
                                "SM", /* "SM PACKET          ", */
                                "UN", /* "UNKNOWN PACKET TYPE" */
                                "UN", /* "UNKNOWN PACKET TYPE" */
                                "UN", /* "UNKNOWN PACKET TYPE" */
                                "RF", /*"REFRESH PACKET     " */ };


/*******************************************************************************************************
 *
 * void msgPcktShowMessage(BYTE *message)
 *
 * Shows the message on the screen.
 *
 ******************************************************************************************************/

void msgPcktShowMessage(BYTE *message)
{

#ifdef DEBUG0

	char type;	
	
	MEMBER_ID member_id;  char string[1001];
	
	int sn, pckt_size, data_size, nlines, cache_status = 0, k, auxnlines;

#ifdef DEBUG1    
    int i;
#endif    
	
	SM_INFO sm_info;
	
	
	get(&type,message,sizeof(char));

	fprintf(stderr,"\ntype: %d\n",type);
	fprintf(stderr,"member info: \n");

	get_member_id(member_id, message);
	get(&pckt_size,message,sizeof(int));  

    fprintf(stderr,"packet size: %d\n",pckt_size);

	
	switch (type)
	{
		
	case DATA_PACKET_TYPE:
    
		fprintf(stderr,"data packet: \n");

		get(&sn,message,sizeof(int)); 

        fprintf(stderr,"\tsn: %d\n",sn);

		get(&data_size,message,sizeof(int)); 

        fprintf(stderr,"\tdata size: %d\n",data_size);

		memcpy(string,message, MIN((sizeof(BYTE)*data_size),1000)); string[MIN(data_size,1000)]='\0'; 
		//		fprintf(stderr,"\tdata: %s\n",string);

		fprintf(stderr,"\tdata: ");

#ifdef DEBUG1
		for(i=0; i<data_size; i++)
		{
			if (isalnum((int)string[i]))
			{
				fprintf(stderr,"%c ",string[i]);
			}
			else
			{
				fprintf(stderr,"\\%d ",string[i]);
			}
		}
#endif        
		fprintf(stderr,"\n\n");
        
		break;
		
	case RETRANSM_PACKET_TYPE: 
		
		get_member_id(member_id, message);
        
        get(&sn,message,sizeof(int)); 
		get(&data_size,message,sizeof(int)); 
		memcpy(string,message, sizeof(BYTE)*data_size); 
        string[data_size]='\0'; 

        fprintf(stderr,"retransmition packet: \n"); 
		fprintf(stderr,"\tdata packet: \n");
        fprintf(stderr,"\t\tsn: %d\n",sn);
        fprintf(stderr,"\t\tdata size: %d\n",data_size);
 		fprintf(stderr,"\tdata: %s\n",string);

#ifdef DEBUG1			
		for(i=0; i<data_size; i++)
		{
			if (isalnum((int)string[i]))
			{
				fprintf(stderr,"%c ",string[i]);
			}
			else
			{
				fprintf(stderr,"\\%d ",string[i]);
			}
		}
#endif        
		fprintf(stderr,"\n\n");


	break;
		
	case NAK_PACKET_TYPE:
		
		get_member_id(member_id, message);	
		
		get(&sn,message,sizeof(int)); 
        
        fprintf(stderr,"nak packet: \n");
        fprintf(stderr,"\tsn: %d\n",sn);
/*        
        get(&sn,message,sizeof(int)); 
        
        fprintf(stderr,"\tlast sn: %d\n",sn);
*/		
    break;
		
	case SM_PACKET_TYPE: 
		
        fprintf(stderr,"sm info packet: \n");
	
		get(&nlines, message, sizeof(int));
		
		fprintf(stderr,"\tnlines: %d\n", nlines);

		auxnlines = nlines;
		
		while (nlines > 0)
		{
			memcpy(&sm_info,message, sizeof(SM_INFO));
			message += sizeof(SM_INFO);
			
			fprintf(stderr,"\tLine %d\n",auxnlines-nlines);
			
			fprintf(stderr,"\t\tMember id:\n"); 
			
			fprintf(stderr,"\t\t\tip: ");			
			
			for (k=0; k<MAX_IP_STRING_SIZE; k++)		
				fprintf(stderr,"%c",sm_info.member_id.ip[k]);	
			
			fprintf(stderr,"\n");				
			fprintf(stderr,"\t\t\tpid: %d\n",sm_info.member_id.pid);	
			
			fprintf(stderr,"\t\tMember status:\n"); 
			
			fprintf(stderr,"\t\t\tFirst received: %d\n", sm_info.member_status.first_rcv);
			fprintf(stderr,"\t\t\tLast received: %d\n", sm_info.member_status.last_rcv);
			fprintf(stderr,"\t\t\tLast in seq received: %d\n", sm_info.member_status.last_seq_rcv);

			nlines --;
		}
		
		get(&cache_status, message, sizeof(int));

        fprintf(stderr,"\tcache_status: %d\n", cache_status);

		break;
		
	case LEAVE_GROUP_PACKET_TYPE:

		fprintf(stderr,"leave group packet: \n");

		break;
        
    case REFRESH_PACKET_TYPE:
    
		get(&sn,message,sizeof(int)); 
    	
        fprintf(stderr, "\tsequence number of last message sent: %d\n", sn);
    
		break;
	}

#endif

	return;
}


/*******************************************************************************************************
 *
 * void msgPcktUnmountMessage(PACKET_INFO *pckt_info, BYTE *message)
 *
 * Unmount the message come from the network, filling PACKET_INFO structure with the apropriate values.
 *
 * Arguments:	message,	a pointer to the message to be unmounted (source of information);
 *				pckt_info,	a pointer to the target strucutre, which will hold the message info.
 *							(this structure must have been previously allocated).
 *
 ******************************************************************************************************/

void msgPcktUnmountMessage(PACKET_INFO *pckt_info, BYTE *message)
{
	
	get(&(pckt_info->type),message,sizeof(char));
	get(&(pckt_info->sender_id),message,sizeof(MEMBER_ID));
	get(&(pckt_info->packet_size),message,sizeof(int));
	
	switch (pckt_info->type)
	{
	case DATA_PACKET_TYPE:
		
		get(&(pckt_info->packet_data.data_packet.sn),message,sizeof(int)); 
		get(&(pckt_info->packet_data.data_packet.data_size),message,sizeof(int));
		(pckt_info->packet_data.data_packet.data)=(BYTE*)message;
		break;
		
	case RETRANSM_PACKET_TYPE: 
		
		get(&(pckt_info->packet_data.retransm_packet.original_sender_id),message,sizeof(MEMBER_ID));
		get(&(pckt_info->packet_data.retransm_packet.data_packet.sn),message,sizeof(int));
		get(&(pckt_info->packet_data.retransm_packet.data_packet.data_size),message,sizeof(int));
		(pckt_info->packet_data.retransm_packet.data_packet.data) = (BYTE*)message;
		break;
		
	case NAK_PACKET_TYPE:
		
		get(&(pckt_info->packet_data.nak_packet.requested_member_id),message,sizeof(MEMBER_ID));	
		get(&(pckt_info->packet_data.nak_packet.sn),message,sizeof(int));	
//		get(&(pckt_info->packet_data.nak_packet.last_sn),message,sizeof(int));	
		break;
		
	case SM_PACKET_TYPE: 
		
		get(&(pckt_info->packet_data.sm_packet.nlines), message, sizeof(int));
		(pckt_info->packet_data.sm_packet.sm_line) =  (SM_INFO*)message;
		message+=(pckt_info->packet_data.sm_packet.nlines*sizeof(SM_INFO));
		get(&(pckt_info->packet_data.sm_packet.cache_status), message, sizeof(SM_INFO*));
		
		break;

#ifdef REFRESH
    
    case REFRESH_PACKET_TYPE:
    
    	get(&(pckt_info->packet_data.refresh_packet.sn_of_last_msg_sent), message, sizeof(int));
        
        break;
#endif
		
	case LEAVE_GROUP_PACKET_TYPE: 
		
		/* do nothing. We already have the sender id*/
		
		break;
        
    default:
    
        fprintf(stderr,"Unknown packet type: %d\n",pckt_info->type);
        
   
        break;
        
	}
	return; 
}

/*******************************************************************************************************
 *
 * void msgPcktMountMessage(PACKET_INFO *pckt_info, BYTE **message, int *msgsize)
 *
 * Mount the message to be sent over the network, getting data from the PACKET_INFO structure.
 *
 * Arguments:	pckt_info,	a pointer to the source strucutre, from which data will be retrieved;
 *				message,	a pointer to the message to be mounted - 
 *							(the space needed to store the message will be allocated as necessary);
 *				msgsize,	a pointer to an integer which will return the effective size of the message.
 *
 ******************************************************************************************************/

void msgPcktMountMessage(PACKET_INFO *pckt_info, BYTE **message, int *msgsize)
{
	BYTE *aux_message_pt;
	BYTE *common_header;
	
	int *sn_pointer;
	int common_header_size = sizeof(char)+sizeof(MEMBER_ID)+sizeof(int); 
	int msg_specific = 0, nlines = 0, data_size = 0;
	int packet_size = 0;
	
#ifdef DEBUG2	 
	fprintf(stderr,"mounting packet\n");
#endif
	
	/* info about type of the message, sender_id and packet_size */
	
	common_header = (BYTE*)malloc(common_header_size);
	
	aux_message_pt = common_header;

#ifdef DEBUG2	
	fprintf(stderr,"common_header allocated at: %p\n", common_header);
#endif	  
	
	getPCKInfo(aux_message_pt, &(pckt_info->type), sizeof(char));
	getPCKInfo(aux_message_pt, &(pckt_info->sender_id), sizeof(MEMBER_ID));
	getPCKInfo(aux_message_pt, &(pckt_info->packet_size), sizeof(int)); 
	
#ifdef DEBUG2		
	fprintf(stderr,"*1* message type: %d (%d) sn: (%d)\n", 
		*((char*)(common_header)), pckt_info->type,
		pckt_info->packet_data.data_packet.sn);
#endif	  
	
	packet_size = 0;
	
	/* specific info plus data */
	
	switch(pckt_info->type)
	{
		
	case DATA_PACKET_TYPE: 
		
		msg_specific = sizeof(DATA_PACKET)-sizeof(BYTE*);
		data_size = (pckt_info)->packet_data.data_packet.data_size;
		
		(*message) = (BYTE*)malloc(common_header_size+msg_specific+data_size);
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
		
#ifdef DEBUG2
		fprintf(stderr,"*2* message type: %d (%d) sn: (%d)\n", 
			*((char*)(*message)), pckt_info->type, pckt_info->packet_data.data_packet.sn);	 
#endif	  
		
		sn_pointer = (int*)aux_message_pt;

#ifdef DEBUG2
		fprintf(stderr,"sn is at: %p %p %u\n",(aux_message_pt), *message, (aux_message_pt) - (*message));
#endif
		getPCKInfo(aux_message_pt,&(pckt_info->packet_data.data_packet.sn), sizeof(int));
		
#ifdef DEBUG0
		fprintf(stderr,"message type: %d (%d) sn: %d (%d)\n", 
			*((char*)(*message)), pckt_info->type, *sn_pointer, pckt_info->packet_data.data_packet.sn);
#endif
		getPCKInfo(aux_message_pt,&(pckt_info->packet_data.data_packet.data_size), sizeof(int));
		getPCKInfo(aux_message_pt, (pckt_info->packet_data.data_packet.data), data_size);
		
		
		break;
		
	case RETRANSM_PACKET_TYPE: 
		
		msg_specific = sizeof(RETRANSM_PACKET)-sizeof(BYTE*);
		
		
		data_size = sizeof(BYTE)*(pckt_info)->packet_data.retransm_packet.data_packet.data_size;
		
		
		(*message) = (BYTE*)malloc(common_header_size+msg_specific+data_size);
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
		
		
		getPCKInfo(aux_message_pt,(void*)(&(pckt_info->packet_data.retransm_packet)), msg_specific);
		getPCKInfo(aux_message_pt,(pckt_info->packet_data.retransm_packet.data_packet.data), data_size);		   
		
		break;
		
	case NAK_PACKET_TYPE:
		
		(*message) = (BYTE*)malloc(common_header_size+sizeof(NAK_PACKET));
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
		getPCKInfo(aux_message_pt, &(pckt_info->packet_data.nak_packet.requested_member_id), sizeof(MEMBER_ID));        
		getPCKInfo(aux_message_pt, &(pckt_info->packet_data.nak_packet.sn), sizeof(int));
//		getPCKInfo(aux_message_pt, &(pckt_info->packet_data.nak_packet.last_sn), sizeof(int));		
		break;
		
	case SM_PACKET_TYPE: 
		
		msg_specific = sizeof(SM_PACKET)-sizeof(SM_PACKET*);
		nlines = ((pckt_info)->packet_data.sm_packet.nlines);
		data_size = sizeof(SM_INFO)*nlines;
		
		(*message) = (BYTE*)malloc(common_header_size+msg_specific+data_size);
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
		
		
		getPCKInfo(aux_message_pt, &nlines, sizeof(int));
		getPCKInfo(aux_message_pt, (pckt_info->packet_data.sm_packet.sm_line), data_size);
		getPCKInfo(aux_message_pt, &(pckt_info->packet_data.sm_packet.cache_status), sizeof(int));
		
		break;

#ifdef REFRESH

    case REFRESH_PACKET_TYPE:

		(*message) = (BYTE*)malloc(common_header_size+sizeof(int));
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
            	
        getPCKInfo(aux_message_pt, &(pckt_info->packet_data.refresh_packet.sn_of_last_msg_sent), sizeof(int));
    	
		break;    

#endif

	case LEAVE_GROUP_PACKET_TYPE:
		
		/* do nothing. The packet is already built */
		
		break;	
       
    	
	}
	
	packet_size++;
	
	
	aux_message_pt =  *message + sizeof(char) + sizeof(MEMBER_ID);
	memcpy(aux_message_pt, &packet_size,sizeof(int));
	
	*msgsize = packet_size;
	
#ifdef DEBUG2
	
	fprintf(stderr,"packet size: %d\n",packet_size);
	msgPcktShowMessage((BYTE*) *message);
	
	
	fprintf(stderr,"*4* message type: %d (%d) sn: %d (%d)\n", 
		*((char*)(*message)), pckt_info->type, *sn_pointer, pckt_info->packet_data.data_packet.sn);
	
	fprintf(stderr, "\n");			 
	
	fprintf(stderr,"Mounted message (bytes): \n");
	
	for (i=0; i<packet_size; i++)
		fprintf(stderr,"%d ",(*message)[i]);
	
	fprintf(stderr,"\n\nMounted message: \n");
	
	msgPcktShowMessage(*message);
	
	fprintf(stderr,"\n\nok!\n");
#endif	  
	
	free (common_header);
	
	return;
}
#endif
