/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1995, William Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgif" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgif" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgif", the right to sell derivative works of
 * "tgif", and the right to distribute "tgif" for a fee) are
 * reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /u/multimedia/william/X11/TGIF2/RCS/text.c,v 2.153 1995/05/15 01:23:27 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "attr.e"
#include "auxtext.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "copypaste.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#include "file.e"
#include "font.e"
#include "grid.e"
#include "mainloop.e"
#include "mark.e"
#include "msg.e"
#include "names.e"
#include "obj.e"
#include "pattern.e"
#include "poly.e"
#include "prtgif.e"
#include "raster.e"
#include "rect.e"
#include "ruler.e"
#include "scroll.e"
#include "select.e"
#include "setup.e"
#include "stretch.e"
#ifndef _NO_EXTERN
#include "text.e"
#endif
#include "xpixmap.e"

#define PAINT 1
#define FRONT_HIGHLIGHT 2
#define MID_HIGHLIGHT 4
#define BACK_HIGHLIGHT 8

#define ERASE 0
#define PAINT_NORM (PAINT)
#define PAINT_INV (PAINT|FRONT_HIGHLIGHT|MID_HIGHLIGHT|BACK_HIGHLIGHT)
#define PAINT_NORM_INV (PAINT|MID_HIGHLIGHT|BACK_HIGHLIGHT)
#define PAINT_INV_NORM (PAINT|FRONT_HIGHLIGHT)
#define PAINT_NORM_INV_NORM (PAINT|MID_HIGHLIGHT)

#ifndef XK_KP_Left
#define XK_KP_Left	0xFF96
#define XK_KP_Up	0xFF97
#define XK_KP_Right	0xFF98
#define XK_KP_Down	0xFF99
#endif /* ~XK_KP_LEFT */

int		textDrawn = FALSE;
int		curTextModified = FALSE;
int		textVSpace = 0;

int		textJust = JUST_L;
int		textCursorShown = FALSE;
int		textCursorH; /* UNSCALED height of the text cursor */
struct ObjRec	* curTextObj = NULL;

static struct ObjRec	* justDrawnTextObj = NULL;

static struct StrRec	* firstStr = NULL, * lastStr = NULL, * curStr = NULL;

static int	textOrigX = 20, textOrigY = 20, textCurX = 20, textCurY = 20;
		/* textOrigX, textOrigY, textCurX, textCurY */
		/*   are UNSCALED screen offsets */
static int	textW, textH; /* absolute for the current text font */
static int	textAbsX = INVALID, textAbsY = INVALID;
		/* textAbsX and textAbsY are absolute coordinates */
static int	textCurIndex = 0;
static int	curStrW = 0; /* UNSCALED width of the current string */
static int	editingText = FALSE;

static int	savedTextLtX, savedTextLtY, savedTextRbX, savedTextRbY;

static int	tmpAdjX, tmpAdjY;

/* the following static variables are for handling text highlight */
static int		textEndX, textEndY, textEndIndex, textHighlight = FALSE;
static int		endStrW;
static struct StrRec	* endStr = NULL;

static int	escPressed = FALSE;

#ifdef _NO_EXTERN
void	DrawTextObj ARGS_DECL((Window, int XOff, int YOff, struct ObjRec *));
void	RedrawCurText ARGS_DECL((void));
#endif /* _NO_EXTERN */

#define BLUR 32

void BlurText (Win, gc, XOff, YOff, W, H)
   Window	Win;
   GC		gc;
   int		XOff, YOff, W, H;
   /* XOff and YOff are screen offsets (scaled and translated) */
{
   XPoint	v[5];

   v[0].x = (short)XOff; v[0].y = (short)YOff;
   v[1].x = (short)XOff; v[1].y = (short)YOff+H+1;
   v[2].x = (short)XOff+W+1; v[2].y = (short)YOff+H+1;
   v[3].x = (short)XOff+W+1; v[3].y = (short)YOff;
   v[4].x = (short)XOff; v[4].y = (short)YOff;

   XFillPolygon (mainDisplay, Win, gc, v, 5, Convex, CoordModeOrigin);
}

static
void AddStr (PrevPtr, NextPtr, StrPtr)
   struct StrRec	* PrevPtr, * NextPtr, * StrPtr;
{
   StrPtr->prev = PrevPtr;
   StrPtr->next = NextPtr;

   if (PrevPtr == NULL)
      firstStr = StrPtr;
   else
      PrevPtr->next = StrPtr;

   if (NextPtr == NULL)
      lastStr = StrPtr;
   else
      NextPtr->prev = StrPtr;
}

void InitText ()
{
   XGCValues	values;

   textBackingPixmap = XCreatePixmap (mainDisplay, mainWindow, 10, 10, 1);
   if (textBackingPixmap==None) Error ("InitText()","Can not XCreatePixmap()");
   textBackingPixmapW = 10;
   textBackingPixmapH = 10;

   values.foreground = 1;
   values.background = 0;
   values.fill_style = FillSolid;
   values.function = GXcopy;
   rotateGC = XCreateGC (mainDisplay, textBackingPixmap,
         GCForeground | GCBackground | GCFillStyle | GCFunction,
         &values);
   if (rotateGC==NULL) Error ("InitText()", "Can not XCreateGC()");
}

void CleanUpText ()
{
   XFreePixmap (mainDisplay, textBackingPixmap);
   textBackingPixmap = None;
   XFreeGC (mainDisplay, rotateGC);
   rotateGC = NULL;
}

static
void PaintText (Win,Str,Just,Rotate,XOff,YOff,xfs,ColorIndex,Pen)
   register int	XOff, YOff;
   Window	Win;
   char		* Str;
   int		Just, Rotate, ColorIndex, Pen;
   XFontStruct	* xfs;
   /* XOff and YOff are UNSCALED screen offset */
{
   int		orig_w, orig_h, w, len;
   XGCValues	values;
   struct BBRec	text_bbox;

   if (Pen == NONEPAT) return;

   XOff = ZOOMED_SIZE(XOff);
   YOff = ZOOMED_SIZE(YOff);

   len = strlen (Str);

   orig_w = XTextWidth (xfs, Str, len);
   orig_h = textCursorH;
   w = ZOOMED_SIZE(orig_w);

   switch (Just)
   {
      case JUST_L: break;
      case JUST_C:
         switch (Rotate)
         {
            case ROTATE0: XOff -= w/2; break;
            case ROTATE90: YOff -= w/2; break;
            case ROTATE180: XOff += w/2; break;
            case ROTATE270: YOff += w/2; break;
         }
         break;
      case JUST_R:
         switch (Rotate)
         {
            case ROTATE0: XOff -= w; break;
            case ROTATE90: YOff -= w; break;
            case ROTATE180: XOff += w; break;
            case ROTATE270: YOff += w; break;
         }
         break;
   }
   text_bbox.ltx = ABS_X(XOff);
   text_bbox.lty = ABS_Y(YOff);
   text_bbox.rbx = ABS_X(XOff+w);
   text_bbox.rby = ABS_Y(YOff+ZOOMED_SIZE(orig_h));
   if (checkBBox && !BBoxIntersect (text_bbox, drawWinBBox)) return;

   values.foreground = colorPixels[ColorIndex];
   values.function = GXcopy;
   values.fill_style = FillOpaqueStippled;
   values.stipple = patPixmap[Pen];
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
   XDrawString (mainDisplay,Win,drawGC,XOff,YOff+canvasFontAsc,Str,len);
}

static
void PaintCurText (Win, gc, Str, Just, XOff, YOff, xfs, ColorIndex, Pen,
      Mode, FirstIndex, SecondIndex)
   register int	XOff, YOff;
   Window	Win;
   GC		gc;
   char		* Str;
   int		Just, ColorIndex, Pen, Mode, FirstIndex, SecondIndex;
   XFontStruct	* xfs;
   /* XOff and YOff are UNSCALED screen offset */
{
   int		w, h, left, right, len;
   XGCValues	values;
   struct BBRec	text_bbox;

   len = strlen (Str);

   w = XTextWidth (xfs, Str, len);
   h = textCursorH;

   switch (Just)
   {
      case JUST_L: break;
      case JUST_C: XOff -= w/2; break;
      case JUST_R: XOff -= w; break;
   }
   text_bbox.ltx = ABS_X(XOff);
   text_bbox.lty = ABS_Y(YOff);
   text_bbox.rbx = ABS_X(XOff+w);
   text_bbox.rby = ABS_Y(YOff+h);
   if (!BBoxIntersect (text_bbox, drawWinBBox)) return;

   if (Mode & PAINT)
   {
      unsigned long	xor_pixel;

      if (colorPixels[ColorIndex] == myBgPixel)
      {
         values.foreground = myFgPixel;
         xor_pixel = xorOne;
      }
      else
      {
         values.foreground = colorPixels[ColorIndex];
         xor_pixel = xorColorPixels[ColorIndex];
      }
      values.function = GXcopy;
      values.fill_style = FillSolid;
      XChangeGC (mainDisplay, gc,
            GCForeground | GCFunction | GCFillStyle, &values);
      XDrawString (mainDisplay, Win, gc, XOff, YOff+canvasFontAsc, Str, len);

      switch (Mode)
      {
         case PAINT_NORM: break;
         case PAINT_INV:
            XSetForeground (mainDisplay, revDefaultGC, xor_pixel);
            XFillRectangle (mainDisplay, Win, revDefaultGC, XOff, YOff,
                  w, h);
            XSetForeground (mainDisplay, revDefaultGC, xorOne);
            break;
         case PAINT_NORM_INV:
            left = XTextWidth (xfs, Str, FirstIndex);
            XSetForeground (mainDisplay, revDefaultGC, xor_pixel);
            XFillRectangle (mainDisplay, Win, revDefaultGC, XOff+left, YOff,
                  w-left, h);
            XSetForeground (mainDisplay, revDefaultGC, xorOne);
            break;
         case PAINT_INV_NORM:
            left = XTextWidth (xfs, Str, FirstIndex);
            XSetForeground (mainDisplay, revDefaultGC, xor_pixel);
            XFillRectangle (mainDisplay, Win, revDefaultGC, XOff, YOff,
                  left, h);
            XSetForeground (mainDisplay, revDefaultGC, xorOne);
            break;
         case PAINT_NORM_INV_NORM:
            left = XTextWidth (xfs, Str, FirstIndex);
            right = XTextWidth (xfs, Str, SecondIndex);
            XSetForeground (mainDisplay, revDefaultGC, xor_pixel);
            XFillRectangle (mainDisplay, Win, revDefaultGC, XOff+left, YOff,
                  right-left, h);
            XSetForeground (mainDisplay, revDefaultGC, xorOne);
            break;
      }
   }
   else
   {
      values.foreground = myBgPixel;
      values.function = GXcopy;
      values.fill_style = FillSolid;
      XChangeGC (mainDisplay, gc,
            GCForeground | GCFunction | GCFillStyle, &values);

      XFillRectangle (mainDisplay, Win, gc, XOff-1, YOff, w+1, h);
   }
}

void PutTextCursor ()
{
   XDrawLine (mainDisplay, drawWindow, defaultGC, textCurX,
         textCurY, textCurX, textCurY+textCursorH);
}

void EraseTextCursor ()
{
   XSetForeground (mainDisplay, revDefaultGC, myFgPixel^myBgPixel);
   XDrawLine (mainDisplay, drawWindow, revDefaultGC, textCurX,
         textCurY, textCurX, textCurY+textCursorH);
   XSetForeground (mainDisplay, revDefaultGC, xorOne);
}

static int	curTextIsNew=FALSE;

void NewCurText ()
{
   struct TextRec	* text_ptr;

   firstStr = lastStr = curStr =
         (struct StrRec *) calloc (1, sizeof(struct StrRec));

   if (textCursorH+textVSpace <= 0)
   {
      Msg ("Text vertical spacing too small.  Reset to 0.");
      textVSpace = 0;
      ShowTextVSpace ();
   }

   text_ptr = (struct TextRec *) calloc (1, sizeof(struct TextRec));
   text_ptr->font = curFont;
   text_ptr->style = curStyle;
   text_ptr->attr = NULL;
   text_ptr->size = curSize;
   text_ptr->just = textJust;
   text_ptr->v_space = textVSpace;
   text_ptr->rotate = curRotate;
   text_ptr->pen = penPat;
   text_ptr->fill = objFill;
   text_ptr->cached_bitmap = None;
   text_ptr->cached_zoom = 0;
   text_ptr->cached_rotate = ROTATE0;
   text_ptr->asc = canvasFontAsc;
   text_ptr->des = canvasFontDes;
   text_ptr->lines = 1;
   text_ptr->first = firstStr;
   text_ptr->last = lastStr;

   text_ptr->read_only = FALSE;
   text_ptr->orig_w = text_ptr->orig_h = 0;
   text_ptr->font_name = NULL;

   curTextObj = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   curTextObj->x = textAbsX;
   curTextObj->y = textAbsY;
   curTextObj->type = OBJ_TEXT;
   curTextObj->color = colorIndex;
   curTextObj->id = objId++;
   curTextObj->dirty = FALSE;
   curTextObj->rotation = 0;
   curTextObj->locked = FALSE;
   curTextObj->detail.t = text_ptr;
   curTextObj->fattr = curTextObj->lattr = NULL;
   AddObj (NULL, topObj, curTextObj);

   textW = 0;
   textH = textCursorH;

   textCursorShown = TRUE;
   textHighlight = FALSE;

   curTextIsNew = TRUE;
}

void FreeTextObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   if (ObjPtr->detail.t != NULL)
   {
      register struct StrRec	* s_ptr, * next_str;

      for (s_ptr = ObjPtr->detail.t->first; s_ptr != NULL; s_ptr = next_str)
      {
         next_str = s_ptr->next;
         cfree (s_ptr);
      }

      if (ObjPtr->detail.t->cached_bitmap != None)
         XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);

      if (ObjPtr->detail.t->font_name != NULL)
         cfree (ObjPtr->detail.t->font_name);
      cfree (ObjPtr->detail.t);
   }
   cfree (ObjPtr);
}

int CreateTextObj ()
   /* returns TRUE if something got created */
   /* returns FALSE otherwise */
{
   register int		i;
   struct StrRec	* s_ptr;
   struct AttrRec	* attr_ptr;
   int			max_len=0, w, ltx=0, lty, rbx=0, rby;
   int			scr_ltx=0, scr_lty;

   if (!textCursorShown) return (FALSE);
   EraseTextCursor ();

   if (firstStr == lastStr && curStr->s[0] == '\0')
   {  /* no text entered or all text erased */
      if ((attr_ptr = curTextObj->detail.t->attr) != NULL)
      {  /* the text being edited is an attribute */
         if (attr_ptr->nameshown)
         {
            UnlinkAttr (attr_ptr);
            FreeTextObj (curTextObj);
            FreeAttr (attr_ptr);
         }
         else
         {
            curTextObj->detail.t->lines = 1;
            UpdateAttr(curTextObj->detail.t, attr_ptr);
         }

         AdjObjBBox (attr_ptr->owner);
         if (outerSel != NULL) AdjAncestorsBBox ();
         if (curTextModified)
         {
            if (outerSel != NULL)
               RecordReplaceAnObj (outerSel->obj);
            else
               RecordReplaceAnObj (attr_ptr->owner);
         }
         else
            AbortPrepareCmd (CMD_REPLACE);
      }
      else
      {
         if (outerSel != NULL)
         {
            UnlinkCurTextFromInnerSel ();
            AdjAncestorsBBox ();
         }
         if (!curTextIsNew)
         {
            if (outerSel != NULL)
               RecordReplaceAnObj (outerSel->obj);
            else
               ChangeReplaceOneCmdToDeleteCmd ();
         }
         else
            AbortPrepareCmd (CMD_REPLACE);
         if (curTextObj != NULL)
         {
            if (outerSel != NULL)
               /* curTextObj already broken off from the main */
               /*	stream of objects, so just free it. */
               FreeObj (curTextObj);
            else
               DelObj (curTextObj);
         }
      }

      switch (textJust)
      {
         case JUST_L:
            scr_ltx = OFFSET_X(textAbsX-2);
            if (zoomedIn)
            {
               ltx = textAbsX-2-GRID_ABS_SIZE(2);
               rbx = textAbsX+textW+2+GRID_ABS_SIZE(2);
            }
            else
            {
               ltx = textAbsX-ABS_SIZE(2)-GRID_ABS_SIZE(2);
               rbx = textAbsX+ABS_SIZE(textW+2)+GRID_ABS_SIZE(2);
            }
            break;
         case JUST_C:
            scr_ltx = OFFSET_X(textAbsX)-textW/2-2;
            if (zoomedIn)
            {
               ltx = textAbsX-textW/2-2-GRID_ABS_SIZE(2);
               rbx = textAbsX+textW/2+2+GRID_ABS_SIZE(2);
            }
            else
            {
               ltx = textAbsX-ABS_SIZE(textW/2+2)-GRID_ABS_SIZE(2);
               rbx = textAbsX+ABS_SIZE(textW/2+2)+GRID_ABS_SIZE(2);
            }
            break;
         case JUST_R:
            scr_ltx = OFFSET_X(textAbsX)-textW-2;
            if (zoomedIn)
            {
               ltx = textAbsX-textW-2-GRID_ABS_SIZE(2);
               rbx = textAbsX+2+GRID_ABS_SIZE(2);
            }
            else
            {
               ltx = textAbsX-ABS_SIZE(textW+2)-GRID_ABS_SIZE(2);
               rbx = textAbsX+ABS_SIZE(2)+GRID_ABS_SIZE(2);
            }
            break;
      }
      scr_lty = OFFSET_Y(textAbsY)-2;
      if (zoomedIn)
      {
         lty = textAbsY-2-GRID_ABS_SIZE(2);
         rby = textAbsY+textH+2+GRID_ABS_SIZE(2);
      }
      else
      {
         lty = textAbsY-ABS_SIZE(2)-GRID_ABS_SIZE(2);
         rby = textAbsY+ABS_SIZE(textH+2)+GRID_ABS_SIZE(2);
      }

      if (editingText)
      {
         XClearArea (mainDisplay, drawWindow, scr_ltx, scr_lty,
               textW+5, textH+5, FALSE);
         RedrawAreas (botObj, savedTextLtX-GRID_ABS_SIZE(2),
               savedTextLtY-GRID_ABS_SIZE(2), savedTextRbX+GRID_ABS_SIZE(2),
               savedTextRbY+GRID_ABS_SIZE(2),
               ltx, lty, rbx, rby);
      }
      else
         RedrawAnArea (botObj, ltx, lty, rbx, rby);

      firstStr = lastStr = curStr = NULL;
      textCursorShown = FALSE;
      curTextObj = NULL;
      textCurIndex = 0;
      curStrW = 0;

      if (editingText)
      {
         PopCurFont ();
         ShowJust ();
         ShowColor (FALSE);
         ShowCurFont ();
         ShowTextVSpace ();
         ShowTextSize ();
         editingText = FALSE;
      }
      textDrawn = FALSE;
      justDrawnTextObj = NULL;
      textHighlight = FALSE;
      curTextIsNew = FALSE;
      return (FALSE);
   }

   for (s_ptr = firstStr, i = 0; s_ptr != NULL; s_ptr = s_ptr->next, i++)
   {
      w = XTextWidth (canvasFontPtr, s_ptr->s, strlen (s_ptr->s));
      if (w > max_len) max_len = w;
   }

   curTextObj->detail.t->last = lastStr;
   curTextObj->detail.t->lines = i;

   curTextObj->x = textAbsX-tmpAdjX;
   curTextObj->y = textAbsY-tmpAdjY;

   SetTextBBox (curTextObj, textJust, max_len, i*textCursorH+(i-1)*textVSpace,
         curRotate);

   switch (textJust)
   {
      case JUST_L:
         scr_ltx = OFFSET_X(textAbsX-2);
         if (zoomedIn)
         {
            ltx = textAbsX-2-GRID_ABS_SIZE(2);
            rbx = textAbsX+textW+2+GRID_ABS_SIZE(2);
         }
         else
         {
            ltx = textAbsX-ABS_SIZE(2)-GRID_ABS_SIZE(2);
            rbx = textAbsX+ABS_SIZE(textW+2)+GRID_ABS_SIZE(2);
         }
         break;
      case JUST_C:
         scr_ltx = OFFSET_X(textAbsX)-textW/2-2;
         if (zoomedIn)
         {
            ltx = textAbsX-textW/2-2-GRID_ABS_SIZE(2);
            rbx = textAbsX+textW/2+2+GRID_ABS_SIZE(2);
         }
         else
         {
            ltx = textAbsX-ABS_SIZE(textW/2+2)-GRID_ABS_SIZE(2);
            rbx = textAbsX+ABS_SIZE(textW/2+2)+GRID_ABS_SIZE(2);
         }
         break;
      case JUST_R:
         scr_ltx = OFFSET_X(textAbsX)-textW-2;
         if (zoomedIn)
         {
            ltx = textAbsX-textW-2-GRID_ABS_SIZE(2);
            rbx = textAbsX+2+GRID_ABS_SIZE(2);
         }
         else
         {
            ltx = textAbsX-ABS_SIZE(textW+2)-GRID_ABS_SIZE(2);
            rbx = textAbsX+ABS_SIZE(2)+GRID_ABS_SIZE(2);
         }
         break;
   }
   scr_lty = OFFSET_Y(textAbsY);
   if (zoomedIn)
   {
      lty = textAbsY-2-GRID_ABS_SIZE(2);
      rby = textAbsY+textH+2+GRID_ABS_SIZE(2);
   }
   else
   {
      lty = textAbsY-ABS_SIZE(2)-GRID_ABS_SIZE(2);
      rby = textAbsY+ABS_SIZE(textH+2)+GRID_ABS_SIZE(2);
   }

   if ((attr_ptr = curTextObj->detail.t->attr) != NULL)
   {
      UpdateAttr(curTextObj->detail.t, attr_ptr);
      textDrawn = FALSE;
      justDrawnTextObj = NULL;
      if (curTextModified && AutoCenterAttr (attr_ptr->owner))
      {
         struct BBRec	bbox;

         CenterObjInOBBox (attr_ptr->obj, attr_ptr->owner->obbox, &bbox);
         if (bbox.ltx < ltx) ltx = bbox.ltx;
         if (bbox.lty < lty) lty = bbox.lty;
         if (bbox.rbx > rbx) rbx = bbox.rbx;
         if (bbox.rby > rby) rby = bbox.rby;
      }
      AdjObjBBox (attr_ptr->owner);
      if (outerSel != NULL) AdjAncestorsBBox ();

      if (curTextModified)
      {
         if (outerSel != NULL)
            RecordReplaceAnObj (outerSel->obj);
         else
            RecordReplaceAnObj (attr_ptr->owner);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   else
   {
      if (outerSel != NULL)
      {
         textDrawn = FALSE;
         justDrawnTextObj = NULL;
         AdjAncestorsBBox ();
      }
      else
      {
         textDrawn = TRUE;
         justDrawnTextObj = curTextObj;
      }
      if (curTextIsNew)
         RecordNewObjCmd ();
      else if (curTextModified)
      {
         if (outerSel != NULL)
            RecordReplaceAnObj (outerSel->obj);
         else
            RecordReplaceAnObj (curTextObj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }

   firstStr = lastStr = curStr = NULL;
   textCursorShown = FALSE;
   textCurIndex = 0;

   if (editingText)
   {
      XClearArea (mainDisplay, drawWindow, scr_ltx, scr_lty,
            textW+5, textH+5, FALSE);
      RedrawAreas (botObj, savedTextLtX-GRID_ABS_SIZE(2),
            savedTextLtY-GRID_ABS_SIZE(2), savedTextRbX+GRID_ABS_SIZE(2),
            savedTextRbY+GRID_ABS_SIZE(2), ltx, lty, rbx, rby);
      if (curRotate != ROTATE0)
         DrawTextObj (drawWindow, drawOrigX, drawOrigY, curTextObj);
   }
   else
   {
      RedrawAnArea (botObj, ltx, lty, rbx, rby);
      if (curRotate != ROTATE0)
         DrawTextObj (drawWindow, drawOrigX, drawOrigY, curTextObj);
   }

   textOrigX = textOrigY = textCurX = textCurY = ABS_SIZE(20);
   curStrW = 0;
   textW = textH = 0;
   textAbsX = textOrigX + drawOrigX;
   textAbsY = textOrigY + drawOrigY;
   curTextObj = NULL;

   if (editingText)
   {
      PopCurFont ();
      ShowJust ();
      ShowColor (FALSE);
      ShowCurFont ();
      ShowTextVSpace ();
      ShowTextSize ();
      editingText = FALSE;
   }
   textHighlight = FALSE;
   curTextIsNew = FALSE;
   return (TRUE);
}

void HighLightJustDrawnText ()
{
   AddNewSelObj (justDrawnTextObj);
   UpdSelBBox ();
   HighLightAnObj (justDrawnTextObj);
   justDupped = FALSE;
}

static
void SetTextCurX ()
   /* set textCurX according to textCurIndex */
{
   register int	left = 0, w;
   char		s[MAXSTRING+1];

   strcpy (s, curStr->s);
   s[textCurIndex] = '\0';

   w = XTextWidth (canvasFontPtr, s, strlen (s));

   switch (textJust)
   {
      case JUST_L: left = textOrigX; break;
      case JUST_C: left = textOrigX-curStrW/2; break;
      case JUST_R: left = textOrigX-curStrW; break;
   }
   textCurX = left + w;
}

static struct StrRec	* highlightStartStr=NULL, * highlightEndStr=NULL;
static int		highlightStartIndex=0, highlightEndIndex=0;
static int		highlightStartY=0;

static
void XorText ()
{
   int			x=0, y, w, start_offset, end_offset, len;
   char			buf[MAXSTRING+1];
   unsigned long	xor_pixel;
   struct StrRec	* s_ptr;


   if (highlightStartStr==highlightEndStr &&
         highlightStartIndex==highlightEndIndex)
      return;

   if (colorPixels[colorIndex] == myBgPixel)
      xor_pixel = xorOne;
   else
      xor_pixel = xorColorPixels[colorIndex];

   y = highlightStartY;
   for (s_ptr = highlightStartStr; s_ptr != highlightEndStr->next;
         s_ptr = s_ptr->next)
   {
      len = strlen (s_ptr->s);
      w = XTextWidth (canvasFontPtr, s_ptr->s, len);

      switch (textJust)
      {
         case JUST_L: x = textOrigX; break;
         case JUST_C: x = textOrigX-w/2; break;
         case JUST_R: x = textOrigX-w; break;
      }

      if (s_ptr == highlightStartStr)
      {
         strcpy (buf, s_ptr->s);
         buf[highlightStartIndex] = '\0';
         len = strlen (buf);
         start_offset = XTextWidth (canvasFontPtr, buf, len);
      }
      else
         start_offset = 0;

      if (s_ptr == highlightEndStr)
      {
         strcpy (buf, s_ptr->s);
         buf[highlightEndIndex] = '\0';
         len = strlen (buf);
         end_offset = XTextWidth (canvasFontPtr, buf, len);
      }
      else
         end_offset = w;

      XSetForeground (mainDisplay, revDefaultGC, xor_pixel);
      XFillRectangle (mainDisplay, drawWindow, revDefaultGC,
            x+start_offset, y, end_offset-start_offset,
            textCursorH);
      XSetForeground (mainDisplay, revDefaultGC, xorOne);

      y += textCursorH+textVSpace;
   }
}

static
int PrepareEditExistingText (obj_ptr, abs_x, abs_y, x_off, y_off)
   struct ObjRec	* obj_ptr;
   int			abs_x, abs_y, * x_off, * y_off;
{
   int	dx, dy;

   curTextObj = obj_ptr;
   savedTextLtX = obj_ptr->bbox.ltx;
   savedTextLtY = obj_ptr->bbox.lty;
   savedTextRbX = obj_ptr->bbox.rbx;
   savedTextRbY = obj_ptr->bbox.rby;

   PushCurFont ();
   editingText = TRUE;

   curFont = obj_ptr->detail.t->font;
   curStyle = obj_ptr->detail.t->style;
   curSize = obj_ptr->detail.t->size;
   textJust = obj_ptr->detail.t->just;
   textVSpace = obj_ptr->detail.t->v_space;
   curRotate = obj_ptr->detail.t->rotate;
   penPat = obj_ptr->detail.t->pen;
   changingFontSizeFromRead = FALSE;
   SetCanvasFont ();
   changingFontSizeFromRead = TRUE;
   if (curSize != canvasFontSize)
   {
      char	msg[MAXSTRING+1];

      sprintf (msg, "Text size=%1d not available.  Can not edit.", curSize);
      MsgBox (msg, TOOL_NAME, INFO_MB);
      PopCurFont ();
      SetCanvasFont ();
      editingText = FALSE;
      curTextModified = FALSE;
      return (FALSE);
   }
   ShowJust ();
   ShowPen ();
   colorIndex = obj_ptr->color;
   ShowColor (FALSE);
   ShowCurFont ();
   ShowTextVSpace ();
   ShowTextSize ();
   CurFontMsg ();

   firstStr = curStr = obj_ptr->detail.t->first;
   lastStr = obj_ptr->detail.t->last;

   textAbsX = obj_ptr->x;
   textAbsY = obj_ptr->y;
   textOrigX = OFFSET_X(obj_ptr->x);
   textOrigY = OFFSET_Y(obj_ptr->y);

   switch (curRotate)
   {
      case ROTATE0:
         textW = obj_ptr->obbox.rbx - obj_ptr->obbox.ltx;
         textH = obj_ptr->obbox.rby - obj_ptr->obbox.lty;
         switch (textJust)
         {
            case JUST_L: tmpAdjX = (textW-ABS_SIZE(textW))/2; break;
            case JUST_C: tmpAdjX = 0; break;
            case JUST_R: tmpAdjX = (ABS_SIZE(textW)-textW)/2; break;
         }
         tmpAdjY = (textH-ABS_SIZE(textH))/2;
         break;
      case ROTATE90:
         dx = textAbsX - abs_x;
         dy = textAbsY - abs_y;
         abs_x = textAbsX - dy;
         abs_y = textAbsY + dx;
         textW = obj_ptr->obbox.rby - obj_ptr->obbox.lty;
         textH = obj_ptr->obbox.rbx - obj_ptr->obbox.ltx;
         switch (textJust)
         {
            case JUST_L:
               tmpAdjX = -(ABS_SIZE(textW)+textH)/2;
               tmpAdjY = (textW-ABS_SIZE(textH))/2;
               break;
            case JUST_C:
               tmpAdjX = -textH/2;
               tmpAdjY = -ABS_SIZE(textH)/2;
               break;
            case JUST_R:
               tmpAdjX = (ABS_SIZE(textW)-textH)/2;
               tmpAdjY = -(textW+ABS_SIZE(textH))/2;
               break;
         }
         break;
      case ROTATE180:
         abs_x = 2*textAbsX - abs_x;
         abs_y = 2*textAbsY - abs_y;
         textW = obj_ptr->obbox.rbx - obj_ptr->obbox.ltx;
         textH = obj_ptr->obbox.rby - obj_ptr->obbox.lty;
         switch (textJust)
         {
            case JUST_L: tmpAdjX = -(textW+ABS_SIZE(textW))/2; break;
            case JUST_C: tmpAdjX = 0; break;
            case JUST_R: tmpAdjX = (textW+ABS_SIZE(textW))/2; break;
         }
         tmpAdjY = -(textH+ABS_SIZE(textH))/2;
         break;
      case ROTATE270:
         dx = textAbsX - abs_x;
         dy = textAbsY - abs_y;
         abs_x = textAbsX + dy;
         abs_y = textAbsY - dx;
         textW = obj_ptr->obbox.rby - obj_ptr->obbox.lty;
         textH = obj_ptr->obbox.rbx - obj_ptr->obbox.ltx;
         switch (textJust)
         {
            case JUST_L:
               tmpAdjX = -(ABS_SIZE(textW)-textH)/2;
               tmpAdjY = -(textW+ABS_SIZE(textH))/2;
               break;
            case JUST_C:
               tmpAdjX = textH/2;
               tmpAdjY = -ABS_SIZE(textH)/2;
               break;
            case JUST_R:
               tmpAdjX = (ABS_SIZE(textW)+textH)/2;
               tmpAdjY = (textW-ABS_SIZE(textH))/2;
               break;
         }
         break;
   }
   textAbsX += tmpAdjX; textOrigX = OFFSET_X(textAbsX);
   textAbsY += tmpAdjY; textOrigY = OFFSET_Y(textAbsY);
   abs_x += tmpAdjX; *x_off = OFFSET_X(abs_x);
   abs_y += tmpAdjY; *y_off = OFFSET_Y(abs_y);
   if (outerSel != NULL)
      PrepareToReplaceAnObj (outerSel->obj);
   else if (obj_ptr->detail.t->attr == NULL)
      PrepareToReplaceAnObj (obj_ptr);
   else
      PrepareToReplaceAnObj (obj_ptr->detail.t->attr->owner);
   return (TRUE);
}

static
int PartOfAWord (ch)
   char	ch;
{
   return (ch == '_' || (ch >= '0' && ch <= '9') ||
         (ch >= 'a' && ch <= 'z') || (ch >= 'A' && ch <= 'Z'));
}

static Time	lastClickTime;
static int	textJustClicked=FALSE;

static
void HandleButton (Button_Ev)
   XButtonEvent	* Button_Ev;
{
   int			grid_x, grid_y, x_off, y_off, amount;
   int                  left, pressed_in_same_text = FALSE, x = 0;
   int                  abs_x, abs_y, tmp_x, tmp_y;
   struct ObjRec	* obj_ptr = NULL;
   struct AttrRec       * attr_ptr;
   char			* c_ptr, s[2];
   int			done=FALSE, saved_end_index;
   XEvent		input, ev;
   struct StrRec	* saved_end_str;


   escPressed = FALSE;
   if (Button_Ev->button == Button1 &&
         !(Button_Ev->state & (ShiftMask|ControlMask)))
   {
      x_off = Button_Ev->x; abs_x = ABS_X(x_off);
      y_off = Button_Ev->y; abs_y = ABS_Y(y_off);

      if (textCursorShown)
      {
         switch (textJust)
         {
            case JUST_L: x = textOrigX-2; break;
            case JUST_C: x = textOrigX-textW/2-2; break;
            case JUST_R: x = textOrigX-textW-2; break;
         }
         if (x_off >= x && x_off <= x+textW+4 &&
               y_off >= textOrigY-2 && y_off <= textOrigY+textH+2)
            pressed_in_same_text = TRUE;
         else
         {
            CreateTextObj (); /* end editing on the old text */
            curTextModified = FALSE;
         }
      }
      else
      {
         editingText = FALSE;
         curTextModified = FALSE;
      }

      if (!pressed_in_same_text &&
            (obj_ptr = FindTextObj (x_off, y_off)) == NULL)
      {  /* cursor not within any existing text object */
         if (curSize != canvasFontSize)
         {
            char	msg[MAXSTRING+1];

            sprintf (msg, "Text size=%1d not available.  %s.", curSize,
                  "Please try a different size");
            MsgBox (msg, TOOL_NAME, INFO_MB);
         }
         else
         {
            GridXY (x_off, y_off, &grid_x, &grid_y);

            textOrigX = textCurX = grid_x;
            textOrigY = textCurY = grid_y;
            textAbsX = ABS_X(grid_x);
            textAbsY = ABS_Y(grid_y);
            tmpAdjX = tmpAdjY = 0;
            NewCurText ();
            RedrawCurText ();
            CurFontMsg ();
            PrepareToReplaceAnObj (curTextObj);
         }
      }
      else
      {  /* cursor inside an existing text object */
         Time	click_time=Button_Ev->time;
         int	double_clicked=FALSE;
         int	inherited_attr_left_index=(-1);

         double_clicked = (pressed_in_same_text && textJustClicked &&
               (click_time-lastClickTime) < doubleClickInterval);
         if (pressed_in_same_text)
         {
            obj_ptr = curTextObj;
            curStr = obj_ptr->detail.t->first;
            if (textJustClicked && (click_time-lastClickTime) <
                  doubleClickInterval)
               double_clicked = TRUE;
         }
         else if (!PrepareEditExistingText (obj_ptr,abs_x,abs_y,&x_off,&y_off))
            return;

         textCurY = textOrigY;
         if (pressed_in_same_text)
         {
            while (y_off >= textCurY+textCursorH+textVSpace &&
                  curStr->next != NULL)
            {
               textCurY += textCursorH+textVSpace;
               curStr = curStr->next;
            }
         }
         else
         {
            tmp_y = textAbsY;
            while (ABS_Y(y_off)>=tmp_y+textCursorH+textVSpace &&
                  curStr->next != NULL)
            {
               textCurY += textCursorH+textVSpace;
               tmp_y += textCursorH+textVSpace;
               curStr = curStr->next;
            }
         }

         curStrW = amount = XTextWidth (canvasFontPtr, curStr->s,
               strlen (curStr->s));

         textCurIndex = 0;
         textCurX = textOrigX;

         switch (textJust)
         {
            case JUST_L: break;
            case JUST_C: textCurX -= (amount>>1); break;
            case JUST_R: textCurX -= amount; break;
         }
         left = textCurX;

         s[1] = '\0';
         c_ptr = curStr->s;
         if (pressed_in_same_text)
         {
            for ( ; *c_ptr != '\0'; c_ptr++)
            {
               s[0] = *c_ptr;
               amount = XTextWidth (canvasFontPtr, s, 1);

               if (double_clicked)
               {
                  if (x_off < textCurX+amount)
                     /* make sure it stops on the left side of the character */
                     break;
                  else
                     textCurX += amount;
               }
               else
               {
                  if (x_off < textCurX+(amount>>1))
                     /* stops on the right side of the clicked */
                     /*		character if the cursor is */
                     /*		closer to the right side */
                     break;
                  else
                     textCurX += amount;
               }
               textCurIndex++;
            }
         }
         else
         {
            tmp_x = textAbsX;
            switch (textJust)
            {
               case JUST_L: break;
               case JUST_C: tmp_x -= (amount>>1); break;
               case JUST_R: tmp_x -= amount; break;
            }

            for ( ; *c_ptr != '\0'; c_ptr++)
            {
               s[0] = *c_ptr;
               amount = XTextWidth (canvasFontPtr, s, 1);

               if (ABS_X(x_off)< tmp_x+(amount>>1))
                  break;
               else
               {
                  textCurX += amount;
                  tmp_x += amount;
               }
               textCurIndex++;
            }
         }

         attr_ptr = obj_ptr->detail.t->attr;
         if (attr_ptr != NULL && attr_ptr->inherited && textCurY == textOrigY &&
               attr_ptr->shown && attr_ptr->nameshown)
         {
            inherited_attr_left_index = strlen (attr_ptr->name);
            if (textCurIndex < inherited_attr_left_index)
            {  /* clicked in the name of an inherited attribute */
               textCurIndex = strlen (attr_ptr->name);
               textCurX = left + XTextWidth (canvasFontPtr, attr_ptr->name,
                     textCurIndex);
            }
         }
         textCursorShown = TRUE;
         textHighlight = FALSE;

         if (double_clicked)
         {
            char	cur_char=curStr->s[textCurIndex];
            int		right_index=strlen(curStr->s);

            highlightStartStr = highlightEndStr = endStr = curStr;
            highlightStartY = textEndY = textCurY;
            endStrW = curStrW;
            textEndIndex = textCurIndex;

            if (PartOfAWord (cur_char))
            {
               saved_end_index = textCurIndex;

               if (inherited_attr_left_index != (-1))
               {
                  while (textCurIndex > inherited_attr_left_index)
                     if (PartOfAWord (curStr->s[textCurIndex-1]))
                        textCurIndex--;
                     else
                        break;
               }
               else
               {
                  while (textCurIndex > 0)
                     if (PartOfAWord (curStr->s[textCurIndex-1]))
                        textCurIndex--;
                     else
                        break;
               }
               SetTextCurX ();

               highlightStartIndex = highlightEndIndex = textCurIndex;

               textEndIndex = saved_end_index+1;
               while (textEndIndex < right_index)
                  if (PartOfAWord (curStr->s[textEndIndex]))
                     textEndIndex++;
                  else
                     break;

               highlightEndIndex = textEndIndex;
            }
            else if (cur_char != '\0')
            {
               saved_end_index = textCurIndex;

               if (inherited_attr_left_index != (-1))
               {
                  while (textCurIndex > inherited_attr_left_index)
                     if (curStr->s[textCurIndex-1] == cur_char)
                        textCurIndex--;
                     else
                        break;
               }
               else
               {
                  while (textCurIndex > 0)
                     if (curStr->s[textCurIndex-1] == cur_char)
                        textCurIndex--;
                     else
                        break;
               }
               SetTextCurX ();

               highlightStartIndex = highlightEndIndex = textCurIndex;

               textEndIndex = saved_end_index+1;
               while (textEndIndex < right_index)
                  if (curStr->s[textEndIndex] == cur_char)
                     textEndIndex++;
                  else
                     break;

               highlightEndIndex = textEndIndex;
            }
            textHighlight = !(endStr==curStr && textEndIndex==textCurIndex);
            RedrawCurText ();
            textJustClicked = FALSE;
            return;
         }
         RedrawCurText ();
         textJustClicked = TRUE;
         lastClickTime = click_time;

         XGrabPointer (mainDisplay, drawWindow, FALSE,
               PointerMotionMask | ButtonReleaseMask,
               GrabModeAsync, GrabModeAsync, None, textCursor, CurrentTime);

         saved_end_index = textEndIndex = textCurIndex;
         saved_end_str = endStr = curStr;

         highlightStartStr = highlightEndStr = curStr;
         highlightStartIndex = highlightEndIndex = textCurIndex;
         highlightStartY = textCurY;

         while (!done)
         {
            XNextEvent (mainDisplay, &input);

            if (input.type == Expose || input.type == VisibilityNotify)
               ExposeEventHandler (&input, TRUE);
            else if (input.type == ButtonRelease)
            {
               XUngrabPointer (mainDisplay, CurrentTime);
               done = TRUE;
            }
            else if (input.type == MotionNotify)
            {
               x_off = input.xmotion.x;
               y_off = input.xmotion.y;

               obj_ptr = curTextObj;
               endStr = obj_ptr->detail.t->first;

               textEndY = textOrigY;
               while (y_off >= textEndY+textCursorH+textVSpace &&
                     endStr->next != NULL)
               {
                  textEndY += textCursorH+textVSpace;
                  endStr = endStr->next;
               }

               endStrW = amount = XTextWidth (canvasFontPtr, endStr->s,
                     strlen (endStr->s));

               textEndX = textOrigX;
               switch (textJust)
               {
                  case JUST_L: break;
                  case JUST_C: textEndX -= (amount>>1); break;
                  case JUST_R: textEndX -= amount; break;
               }
               textEndIndex = 0;
               left = textEndX;

               if (y_off >= textOrigY-2)
               {
                  if (y_off > textEndY+textCursorH+textVSpace+2 &&
                        endStr->next == NULL)
                  {
                     textEndIndex = strlen (endStr->s);
                     textEndX += endStrW;
                  }
                  else
                  {
                     s[1] = '\0';
                     c_ptr = endStr->s;
                     for ( ; *c_ptr != '\0'; c_ptr++)
                     {
                        s[0] = *c_ptr;
                        amount = XTextWidth (canvasFontPtr, s, 1);

                        if (x_off < textEndX+(amount>>1))
                           break;
                        else
                           textEndX += amount;
                        textEndIndex++;
                     }
                  }
               }

               attr_ptr = obj_ptr->detail.t->attr;
               if (attr_ptr != NULL && attr_ptr->inherited &&
                     textEndY == textOrigY &&
                     attr_ptr->shown && attr_ptr->nameshown &&
                     textEndIndex < ((int) strlen (attr_ptr->name)))
               {
                  textEndIndex = strlen (attr_ptr->name);
                  textEndX = left + XTextWidth (canvasFontPtr,
                        attr_ptr->name, textEndIndex);
               }
               textHighlight = !(endStr==curStr&&textEndIndex==textCurIndex);

               if (endStr!=saved_end_str || textEndIndex!=saved_end_index)
               {
                  XorText ();

                  if (textEndY == textCurY)
                  {
                     highlightStartStr = highlightEndStr = curStr;
                     highlightStartY = textCurY;
                     if (textEndIndex > textCurIndex)
                     {
                        highlightStartIndex = textCurIndex;
                        highlightEndIndex = textEndIndex;
                     }
                     else
                     {
                        highlightStartIndex = textEndIndex;
                        highlightEndIndex = textCurIndex;
                     }
                  }
                  else if (textEndY > textCurY)
                  {
                     highlightStartStr = curStr;
                     highlightStartIndex = textCurIndex;
                     highlightEndStr = endStr;
                     highlightEndIndex = textEndIndex;
                     highlightStartY = textCurY;
                  }
                  else
                  {
                     highlightStartStr = endStr;
                     highlightStartIndex = textEndIndex;
                     highlightEndStr = curStr;
                     highlightEndIndex = textCurIndex;
                     highlightStartY = textEndY;
                  }

                  saved_end_str = endStr;
                  saved_end_index = textEndIndex;
                  XorText ();
               }

               while (XCheckMaskEvent (mainDisplay,PointerMotionMask,&ev)) ;
            }
         }
      }
   }
   else if (Button_Ev->button == Button1 &&
         (Button_Ev->state & (ShiftMask|ControlMask)))
   {
      if (!textCursorShown) return;

      x_off = Button_Ev->x;
      y_off = Button_Ev->y;

      switch (textJust)
      {
         case JUST_L: x = textOrigX-2; break;
         case JUST_C: x = textOrigX-textW/2-2; break;
         case JUST_R: x = textOrigX-textW-2; break;
      }

      if (!(x_off >= x && x_off <= x+textW+2 &&
            y_off >= textOrigY-2 && y_off <= textOrigY+textH+2))
         return;

      obj_ptr = curTextObj;
      endStr = obj_ptr->detail.t->first;

      textEndY = textOrigY;
      while (y_off >= textEndY+textCursorH+textVSpace && endStr->next != NULL)
      {
         textEndY += textCursorH+textVSpace;
         endStr = endStr->next;
      }

      endStrW = amount = XTextWidth (canvasFontPtr, endStr->s,
            strlen (endStr->s));

      textEndX = textOrigX;
      switch (textJust)
      {
         case JUST_L: break;
         case JUST_C: textEndX -= (amount>>1); break;
         case JUST_R: textEndX -= amount; break;
      }
      textEndIndex = 0;
      left = textEndX;

      s[1] = '\0';
      c_ptr = endStr->s;
      for ( ; *c_ptr != '\0'; c_ptr++)
      {
         s[0] = *c_ptr;
         amount = XTextWidth (canvasFontPtr, s, 1);

         if (x_off < textEndX+(amount>>1))
            break;
         else
            textEndX += amount;
         textEndIndex++;
      }

      attr_ptr = obj_ptr->detail.t->attr;
      if (attr_ptr != NULL && attr_ptr->inherited && textEndY == textOrigY &&
            attr_ptr->shown && attr_ptr->nameshown &&
            textEndIndex < ((int) strlen (attr_ptr->name)))
      {
         textEndIndex = strlen (attr_ptr->name);
         textEndX = left + XTextWidth (canvasFontPtr, attr_ptr->name,
               textEndIndex);
      }
      textHighlight = !(endStr == curStr && textEndIndex == textCurIndex);
      RedrawCurText ();
   }
}

static
int NumLines ()
{
   register struct StrRec	* str_ptr;
   register int			i = 0;

   for (str_ptr = firstStr; str_ptr != NULL; str_ptr = str_ptr->next, i++) ;
   return (i);
}

static
void JustDeleteHighlightedText ()
{
   struct StrRec	* s_ptr, * s_ptr1, * new_cur_str = NULL;
   int			highlighting = FALSE, new_cur_index = 0;
   int			second_index, len;
   int			i, y, new_cur_y = 0, len1, len2;
   char			* s, msg[80];

   if (!textHighlight) return;

   y = textOrigY;
   for (s_ptr = firstStr; s_ptr != NULL; y += textCursorH+textVSpace)
   {
      if (highlighting)
      {  /* started deleting already */
         if (s_ptr == curStr || s_ptr == endStr)
         {
            second_index = (s_ptr == curStr) ? textCurIndex : textEndIndex;
            len1 = strlen (new_cur_str->s);
            len2 = strlen (&(s_ptr->s[second_index]));
            if (len1+len2 >= MAXSTRING)
            {
               sprintf (msg, "String length exceeds %1d.  String truncated.",
                     MAXSTRING);
               Msg (msg);
               s_ptr->s[MAXSTRING-len1+second_index] = '\0';
            }
            strcat (new_cur_str->s, &(s_ptr->s[second_index]));
            if (s_ptr == lastStr)
               lastStr = s_ptr->prev;
            else
               s_ptr->next->prev = s_ptr->prev;
            s_ptr->prev->next = s_ptr->next;
            cfree (s_ptr);
            break;
         }
         else
         {  /* delete the whole line */
            s_ptr1 = s_ptr->next;
            s_ptr1->prev = s_ptr->prev;
            cfree (s_ptr);
            s_ptr = s_ptr1;
         }
      }
      else
      {  /* looking for the beginning ... */
         if (s_ptr == curStr && s_ptr == endStr)
         {  /* the whole string to be deleted is within s_ptr */
            new_cur_str = s_ptr;
            new_cur_index = min(textCurIndex,textEndIndex);
            new_cur_y = y;
            second_index = max(textCurIndex,textEndIndex);
            s = new_cur_str->s;
            len = strlen (&(s[second_index]));
            for (i = 0; i <= len; i++) s[new_cur_index+i] = s[second_index+i];
            break;
         }
         else if (s_ptr == curStr || s_ptr == endStr)
         {  /* found the beginning */
            new_cur_str = s_ptr;
            new_cur_index = (s_ptr == curStr) ? textCurIndex : textEndIndex;
            new_cur_y = y;
            s_ptr->s[new_cur_index] = '\0';
            highlighting = TRUE;
            s_ptr = s_ptr->next;
         }
         else
         {  /* still looking */
            s_ptr = s_ptr->next;
         }
      }
   }
   textHighlight = FALSE;
   curStr = new_cur_str;
   textCurIndex = new_cur_index;
   curStrW = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));
   if (curStrW > textW) textW = curStrW;
   textCurY = new_cur_y;
   SetTextCurX ();
}

static
void HandleCRLF (ColorIndex)
   int	ColorIndex;
{
   register int		i, y;
   struct StrRec	* str_ptr;
   int			need_redraw = textHighlight, w, max_len=0;

   escPressed = FALSE;

   curTextModified = TRUE;
   if (textHighlight) JustDeleteHighlightedText ();

   y = textCurY;
   for (str_ptr = curStr; str_ptr != NULL; str_ptr = str_ptr->next)
   {
      PaintCurText (drawWindow, drawGC, str_ptr->s, textJust, textOrigX,
            y, canvasFontPtr, ColorIndex, penPat, ERASE, INVALID, INVALID);
      y += textCursorH+textVSpace;
   }

   str_ptr = (struct StrRec *) calloc (1, sizeof(struct StrRec));
   strcpy (str_ptr->s, &curStr->s[textCurIndex]);
   curStr->s[textCurIndex] = '\0';
   AddStr (curStr, curStr->next, str_ptr);
   curStr = str_ptr;
   textCurY += textCursorH+textVSpace;
   textCurIndex = 0;
   curStrW = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));
   SetTextCurX ();
   i = textCursorH*NumLines () + textVSpace*(NumLines ()-1);

   if (i > textH)
   {
      textH = i;
      RedrawCurText ();
   }
   else if (need_redraw)
      RedrawCurText ();
   else
   {
      y = textCurY - textCursorH - textVSpace;
      for (str_ptr = curStr->prev; str_ptr != NULL; str_ptr = str_ptr->next)
      {
         PaintCurText (drawWindow, drawGC, str_ptr->s, textJust,
               textOrigX, y, canvasFontPtr, ColorIndex, penPat, PAINT_NORM,
               INVALID, INVALID);
         y += textCursorH+textVSpace;
      }
   }

   if (curTextObj->detail.t->cached_bitmap != None)
      XFreePixmap (mainDisplay, curTextObj->detail.t->cached_bitmap);

   curTextObj->detail.t->cached_zoom = 0;
   curTextObj->detail.t->cached_bitmap = None;

   if (zoomScale != 0 || curTextObj->detail.t->rotate != ROTATE0)
   {
      curTextObj->detail.t->lines = 0;
      for (str_ptr=firstStr, i=0; str_ptr!=NULL; str_ptr=str_ptr->next, i++)
      {
         w = XTextWidth (canvasFontPtr, str_ptr->s, strlen (str_ptr->s));
         if (w > max_len) max_len = w;
         curTextObj->detail.t->lines++;
      }
      SetTextBBox (curTextObj, textJust, max_len,
            i*textCursorH+(i-1)*textVSpace, curRotate);
   }
   ScrollTo (textCurX, textCurY);
}

static
void HandleBS (ColorIndex)
   int	ColorIndex;
{
   register int		i, y;
   register char	* s;
   struct StrRec	* str_ptr;
   struct AttrRec       * attr_ptr;
   int			len1, len2, w, max_len=0;
   char			msg[80];

   escPressed = FALSE;

   if (textHighlight)
   {
      curTextModified = TRUE;
      JustDeleteHighlightedText ();
      RedrawCurText ();

      if (curTextObj->detail.t->cached_bitmap != None)
         XFreePixmap (mainDisplay, curTextObj->detail.t->cached_bitmap);

      curTextObj->detail.t->cached_zoom = 0;
      curTextObj->detail.t->cached_bitmap = None;

      if (zoomScale != 0 || curTextObj->detail.t->rotate != ROTATE0)
      {
         curTextObj->detail.t->lines = 0;
         for (str_ptr=firstStr, i=0; str_ptr!=NULL; str_ptr=str_ptr->next, i++)
         {
            w = XTextWidth (canvasFontPtr, str_ptr->s, strlen (str_ptr->s));
            if (w > max_len) max_len = w;
            curTextObj->detail.t->lines++;
         }
         SetTextBBox (curTextObj, textJust, max_len,
               i*textCursorH+(i-1)*textVSpace, curRotate);
      }
      ScrollTo (textCurX, textCurY);
      return;
   }

   attr_ptr = curTextObj->detail.t->attr;
   if (curStr == firstStr && attr_ptr != NULL && attr_ptr->inherited &&
         attr_ptr->nameshown && textCurIndex == strlen(attr_ptr->name))
      return;

   curTextModified = TRUE;
   if (textCurIndex != 0)
   {
      PaintCurText (drawWindow, drawGC, curStr->s, textJust,
            textOrigX, textCurY, canvasFontPtr, ColorIndex, penPat, ERASE,
            INVALID, INVALID);

      s = curStr->s;
      for (i = textCurIndex; i <= ((int)strlen(curStr->s)); i++) s[i-1] = s[i];
         textCurIndex--;

      curStrW = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));
      PaintCurText (drawWindow, drawGC, curStr->s, textJust, textOrigX,
            textCurY, canvasFontPtr, ColorIndex, penPat, PAINT_NORM,
            INVALID, INVALID);
      SetTextCurX ();
   }
   else
   {
      if (curStr->prev != NULL)
      {
         struct StrRec	* prev_str;

         y = textCurY - textCursorH - textVSpace;
         for (str_ptr=curStr->prev; str_ptr!=NULL; str_ptr=str_ptr->next)
         {
            PaintCurText (drawWindow, drawGC, str_ptr->s, textJust,
                  textOrigX, y, canvasFontPtr, ColorIndex, penPat, ERASE,
                  INVALID, INVALID);
            y += textCursorH+textVSpace;
         }

         if (curStr->next == NULL)
            lastStr = curStr->prev;
         else
            curStr->next->prev = curStr->prev;
         curStr->prev->next = curStr->next;
         textCurIndex = strlen (curStr->prev->s);
         len1 = strlen (curStr->prev->s);
         len2 = strlen (curStr->s);
         if (len1+len2 >= MAXSTRING)
         {
            sprintf (msg, "String length exceeds %1d.  String truncated.",
                  MAXSTRING);
            Msg (msg);
            curStr->s[MAXSTRING-len1] = '\0';
         }
         strcat (curStr->prev->s, curStr->s);
         prev_str = curStr->prev;
         cfree (curStr);
         curStr = prev_str;
         curStrW = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));
         textCurY -= textCursorH+textVSpace;
         SetTextCurX ();

         y = textCurY;
         for (str_ptr = curStr; str_ptr != NULL; str_ptr = str_ptr->next)
         {
            PaintCurText (drawWindow, drawGC, str_ptr->s, textJust,
                  textOrigX, y, canvasFontPtr, ColorIndex, penPat,
                  PAINT_NORM, INVALID, INVALID);
            y += textCursorH+textVSpace;
         }
         if (curStrW > textW)
         {
            textW = curStrW;
            RedrawCurText ();
         }
      }
   }

   if (curTextObj->detail.t->cached_bitmap != None)
      XFreePixmap (mainDisplay, curTextObj->detail.t->cached_bitmap);

   curTextObj->detail.t->cached_zoom = 0;
   curTextObj->detail.t->cached_bitmap = None;

   if (zoomScale != 0 || curTextObj->detail.t->rotate != ROTATE0)
   {
      curTextObj->detail.t->lines = 0;
      for (str_ptr=firstStr, i=0; str_ptr!=NULL; str_ptr=str_ptr->next, i++)
      {
         w = XTextWidth (canvasFontPtr, str_ptr->s, strlen (str_ptr->s));
         if (w > max_len) max_len = w;
         curTextObj->detail.t->lines++;
      }
      SetTextBBox (curTextObj, textJust, max_len,
            i*textCursorH+(i-1)*textVSpace, curRotate);
   }
   ScrollTo (textCurX, textCurY);
}

static
void HandleChar (Str, ColorIndex)
   char	* Str;
   int	ColorIndex;
{
   register int		i;
   register char	* s;
   register int		amount;
   int			need_redraw, w, max_len=0;
   char			msg[80];
   struct StrRec	* str_ptr;
   XEvent		ev;

   if (escPressed)
   {
      Str[0] |= 0x80;
      escPressed = FALSE;
   }

   if (((*Str)&0x80) && curFont != FONT_SYM && !ValidCharCode (Str)) return;

   if ((need_redraw = textHighlight))
   {
      curTextModified = TRUE;
      JustDeleteHighlightedText ();
   }

   if (textCurIndex+((int) strlen (&(curStr->s[textCurIndex]))) >= MAXSTRING)
   {
      sprintf (msg, "String length exceeds %1d.  Character ignored.",MAXSTRING);
      Msg (msg);
      RedrawCurText ();
      while (XCheckWindowEvent (mainDisplay, drawWindow, KeyPressMask, &ev)) ;
      return;
   }

   curTextModified = TRUE;
   amount = XTextWidth (canvasFontPtr, Str, 1);
 
   if (textJust != JUST_L || textCurIndex != strlen (curStr->s))
      PaintCurText (drawWindow, drawGC, curStr->s, textJust, textOrigX,
            textCurY, canvasFontPtr, ColorIndex, penPat, ERASE,
            INVALID, INVALID);

   s = curStr->s;
   for (i = strlen (curStr->s); i >= textCurIndex; i--) s[i+1] = s[i];
   s[textCurIndex++] = *Str;

   curStrW += amount;
   if (!need_redraw)
      PaintCurText (drawWindow, drawGC, curStr->s, textJust, textOrigX,
            textCurY, canvasFontPtr, ColorIndex, penPat, PAINT_NORM,
            INVALID, INVALID);
   SetTextCurX ();
   if (curStrW > textW)
   {
      textW = curStrW;
      RedrawCurText ();
   }
   else if (need_redraw)
      RedrawCurText ();

   if (curTextObj->detail.t->cached_bitmap != None)
      XFreePixmap (mainDisplay, curTextObj->detail.t->cached_bitmap);

   curTextObj->detail.t->cached_zoom = 0;
   curTextObj->detail.t->cached_bitmap = None;

   if (zoomScale != 0 || curTextObj->detail.t->rotate != ROTATE0)
   {
      curTextObj->detail.t->lines = 0;
      for (str_ptr=firstStr, i=0; str_ptr!=NULL; str_ptr=str_ptr->next, i++)
      {
         w = XTextWidth (canvasFontPtr, str_ptr->s, strlen (str_ptr->s));
         if (w > max_len) max_len = w;
         curTextObj->detail.t->lines++;
      }
      SetTextBBox (curTextObj, textJust, max_len,
            i*textCursorH+(i-1)*textVSpace, curRotate);
   }
   ScrollTo (textCurX, textCurY);
}

static
void HandleTAB (key_ev, ColorIndex)
   XKeyEvent	* key_ev;
   int		ColorIndex;
{
   struct AttrRec	* attr_ptr, * new_attr;
   struct ObjRec	* obj_ptr;
   int			abs_x, abs_y, x_off, y_off, left, amount;

   escPressed = FALSE;

   if ((attr_ptr = curTextObj->detail.t->attr) == NULL)
   {
      Msg ("Can not TAB out of a non-attribute text.");
      return;
   }
   CreateTextObj ();
   curTextModified = FALSE;

   if (key_ev->state & Mod1Mask)
   {  /* new_attr will be the next attribute */
      if (attr_ptr->next == NULL)
         new_attr = attr_ptr->owner->fattr;
      else
         new_attr = attr_ptr->next;
      while (!new_attr->shown)
      {
         if (new_attr->next == NULL)
            new_attr = new_attr->owner->fattr;
         else
            new_attr = new_attr->next;
      }
   }
   else
   {  /* new_attr will be the previous attribute */
      if (attr_ptr->prev == NULL)
         new_attr = attr_ptr->owner->lattr;
      else
         new_attr = attr_ptr->prev;
      while (!new_attr->shown)
      {
         if (new_attr->prev == NULL)
            new_attr = new_attr->owner->lattr;
         else
            new_attr = new_attr->prev;
      }
   }
   obj_ptr = new_attr->obj;
   abs_x = obj_ptr->x; x_off = OFFSET_X(abs_x);
   abs_y = obj_ptr->y; y_off = OFFSET_Y(abs_y);

   if (!PrepareEditExistingText (obj_ptr, abs_x, abs_y, &x_off, &y_off))
      return;

   textCurIndex = 0;
   textCurY = textOrigY;
   textCurX = textOrigX;
   curStrW = amount = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));

   switch (textJust)
   {
      case JUST_L: break;
      case JUST_C: textCurX -= (amount>>1); break;
      case JUST_R: textCurX -= amount; break;
   }
   left = textCurX;

   attr_ptr = obj_ptr->detail.t->attr;
   if (attr_ptr != NULL && attr_ptr->inherited && textCurY == textOrigY &&
         attr_ptr->shown && attr_ptr->nameshown &&
         textCurIndex < ((int) strlen (attr_ptr->name)))
   {  /* clicked in the name of an inherited attribute */
      textCurIndex = strlen (attr_ptr->name);
      textCurX = left + XTextWidth (canvasFontPtr, attr_ptr->name,
            textCurIndex);
   }
   textCursorShown = TRUE;
   textHighlight = FALSE;

   textEndIndex = textCurIndex;

   highlightStartStr = endStr;
   highlightStartIndex = textEndIndex;
   highlightEndStr = curStr;
   highlightEndIndex = textCurIndex;
   highlightStartY = textEndY;

   textEndY = textOrigY;
   endStr = obj_ptr->detail.t->first;
   while (endStr->next != NULL)
   {
      textEndY += textCursorH+textVSpace;
      endStr = endStr->next;
   }

   endStrW = amount = XTextWidth (canvasFontPtr, endStr->s, strlen (endStr->s));
   left = textEndX = textOrigX;
   switch (textJust)
   {
      case JUST_L: break;
      case JUST_C: textEndX -= (amount>>1); break;
      case JUST_R: textEndX -= amount; break;
   }
   textEndIndex = strlen (endStr->s);
   textEndX += endStrW;

   attr_ptr = obj_ptr->detail.t->attr;
   if (attr_ptr != NULL && attr_ptr->inherited && textEndY == textOrigY &&
         attr_ptr->shown && attr_ptr->nameshown &&
         textEndIndex < ((int) strlen (attr_ptr->name)))
   {
      textEndIndex = strlen (attr_ptr->name);
      textEndX = left + XTextWidth (canvasFontPtr,
            attr_ptr->name, textEndIndex);
   }
   textHighlight = !(endStr==curStr && textEndIndex==textCurIndex);

   RedrawCurText ();
   ScrollTo (textCurX, textCurY);
}

static
void HandleLeft ()
{
   struct AttrRec       * attr_ptr;

   escPressed = FALSE;

   attr_ptr = curTextObj->detail.t->attr;
   if (curStr == firstStr && attr_ptr != NULL && attr_ptr->inherited &&
         attr_ptr->nameshown && textCurIndex == strlen(attr_ptr->name))
   {
      if (textHighlight)
      {
         textHighlight = FALSE;
         RedrawCurText ();
      }
      return;
   }
   if (textCurIndex != 0)
   {
      textCurIndex--;
      SetTextCurX ();
   }
   else
   {
      if (curStr->prev != NULL)
      {
         textCurIndex = strlen (curStr->prev->s);
         curStr = curStr->prev;
         curStrW = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));
         textCurY -= textCursorH+textVSpace;
         SetTextCurX ();
      }
      else if (!textHighlight)
         return;
   }
   textHighlight = FALSE;
   RedrawCurText ();
   ScrollTo (textCurX, textCurY);
}

static
void HandleUp ()
{
   int			x_off, left, amount;
   char			* c_ptr, s[2];
   struct AttrRec       * attr_ptr;

   escPressed = FALSE;
   if (curStr == firstStr)
   {
      if (!textHighlight) return;
      textHighlight = FALSE;
      RedrawCurText ();
      return;
   }

   x_off = textCurX;
   curStr = curStr->prev;
   curStrW = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));
   textCurY -= textCursorH+textVSpace;

   textCurX = textOrigX;
   switch (textJust)
   {
      case JUST_L: break;
      case JUST_C: textCurX -= (curStrW/2); break;
      case JUST_R: textCurX -= curStrW; break;
   }
   textCurIndex = 0;
   left = textCurX;

   s[1] = '\0';
   for (c_ptr = curStr->s; *c_ptr != '\0'; c_ptr++)
   {
      s[0] = *c_ptr;
      amount = XTextWidth (canvasFontPtr, s, 1);

      if (x_off < textCurX+(amount>>1))
         break;
      else
         textCurX += amount;
      textCurIndex++;
   }

   attr_ptr = curTextObj->detail.t->attr;
   if (curStr == firstStr && attr_ptr != NULL && attr_ptr->inherited &&
         attr_ptr->nameshown && textCurIndex < ((int) strlen(attr_ptr->name)))
   {
      textCurIndex = strlen (attr_ptr->name);
      textCurX = left + XTextWidth (canvasFontPtr, attr_ptr->name,
            textCurIndex);
   }
   textHighlight = FALSE;
   RedrawCurText ();
   ScrollTo (textCurX, textCurY);
}
 
static
void HandleRight ()
{
   int	len;

   escPressed = FALSE;

   len = strlen (curStr->s);
   if (textCurIndex != len)
   {
      textCurIndex++;
      SetTextCurX ();
   }
   else
   {
      if (curStr->next != NULL)
      {
         textCurIndex = 0;
         curStr = curStr->next;
         curStrW = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));
         textCurY += textCursorH+textVSpace;
         SetTextCurX ();
      }
      else if (!textHighlight)
         return;
   }
   textHighlight = FALSE;
   RedrawCurText ();
   ScrollTo (textCurX, textCurY);
}
 
static
void HandleDown ()
{
   int			x_off, amount;
   char			* c_ptr, s[2];

   escPressed = FALSE;
   if (curStr == lastStr)
   {
      if (!textHighlight) return;
      textHighlight = FALSE;
      RedrawCurText ();
      return;
   }

   x_off = textCurX;
   curStr = curStr->next;
   curStrW = XTextWidth (canvasFontPtr, curStr->s, strlen (curStr->s));
   textCurY += textCursorH+textVSpace;

   textCurX = textOrigX;
   switch (textJust)
   {
      case JUST_L: break;
      case JUST_C: textCurX -= (curStrW/2); break;
      case JUST_R: textCurX -= curStrW; break;
   }
   textCurIndex = 0;

   s[1] = '\0';
   for (c_ptr = curStr->s; *c_ptr != '\0'; c_ptr++)
   {
      s[0] = *c_ptr;
      amount = XTextWidth (canvasFontPtr, s, 1);

      if (x_off < textCurX+(amount>>1))
         break;
      else
         textCurX += amount;
      textCurIndex++;
   }
   textHighlight = FALSE;
   RedrawCurText ();
   ScrollTo (textCurX, textCurY);
}

static XComposeStatus	c_stat;

void DrawText (input)
   XEvent	* input;
{
   register char	* c_ptr;
   char			s[80], * cut_buffer=NULL;
   int			xfree_cut_buffer=FALSE, have_ch;
   XKeyEvent		* key_ev;
   KeySym		key_sym;

   if (input->type == ButtonPress)
      HandleButton (&(input->xbutton));
   else if (input->type == KeyPress)
   {
      if (!textCursorShown) return;

      if (pasteInDrawTextMode)
      {
         int	len;

         pasteInDrawTextMode = FALSE;
         if (pasteFromFileInDrawTextMode)
         {
            FILE	* fp;
            char	msg[MAXSTRING+1], inbuf[MAXSTRING+1];
            int		size=0;

            pasteFromFileInDrawTextMode = FALSE;
            if ((fp = fopen (pasteFromFileName, "r")) == NULL)
            {
               sprintf (msg, "Can not open '%s' for read.", pasteFromFileName);
               MsgBox (msg, TOOL_NAME, INFO_MB);
               return;
            }
            while (fgets (inbuf, MAXSTRING, fp) != NULL) size += strlen (inbuf);
            fclose (fp);
            if (size == 0)
            {
               sprintf (msg, "File '%s' is empty.", pasteFromFileName);
               MsgBox (msg, TOOL_NAME, INFO_MB);
               return;
            }
            cut_buffer = (char *) calloc (size+2, sizeof(char));
            if (cut_buffer == NULL)
            {
               sprintf (msg, "Can not malloc %1d bytes.", size+2);
               MsgBox (msg, TOOL_NAME, INFO_MB);
               return;
            }
            if ((fp = fopen (pasteFromFileName, "r")) == NULL)
            {
               sprintf (msg, "Can not open '%s' for read.", pasteFromFileName);
               MsgBox (msg, TOOL_NAME, INFO_MB);
               cfree (cut_buffer);
               return;
            }
            len = 0;
            while (fgets (&cut_buffer[len], MAXSTRING, fp) != NULL)
               len += strlen (&cut_buffer[len]);
            fclose (fp);
         }
         else
         {
            cut_buffer = (char *) FetchCutBuffer (&len);
            if (len == 0) { Msg ("Cut buffer is empty."); return; }
            xfree_cut_buffer = TRUE;
         }

         if (escPressed)
         {
            escPressed = FALSE;
            Msg ("An <ESC> key press is ignored.");
         }
         s[1] = '\0';
         EraseTextCursor ();
         for (c_ptr = cut_buffer; *c_ptr != '\0'; c_ptr++)
         {
            s[0] = *c_ptr;
            switch (s[0])
            {
               case '\r':
               case '\n': HandleCRLF (colorIndex); break;

               case '\177': Msg ("Can not paste <DEL>."); break;
               case '\b': Msg ("Can not paste <BS>."); break;
               case '\033': Msg ("Can not paste <ESC>."); break;
               case '\t': Msg ("Can not paste <TAB>."); break;

               default: HandleChar (s, colorIndex); break;
            }
         }
         PutTextCursor ();
         MarkRulers (textCurX, textCurY);
         SetFileModified (TRUE);
         if (xfree_cut_buffer)
            XFree (cut_buffer);
         else
            cfree (cut_buffer);
         return;
      }
      else if (copyInDrawTextMode)
      {
         int		copy_failed=FALSE, highlighting=FALSE;
         int		first_index=0, second_index;
         int		cut_buffer_size=0;
         struct StrRec	* s_ptr, * tmp_s_ptr, * first_str=NULL;
         char		msg[80];

         copyInDrawTextMode = FALSE;

         if (!textHighlight) return;

         if (escPressed)
         {
            escPressed = FALSE;
            Msg ("An <ESC> key press is ignored.");
         }
         for (s_ptr = firstStr; s_ptr != NULL; )
         {
            if (highlighting)
            {
               if (s_ptr == curStr || s_ptr == endStr)
               {
                  second_index = (s_ptr==curStr) ? textCurIndex : textEndIndex;
                  cut_buffer_size += second_index+1;
                  cut_buffer = (char *)calloc(cut_buffer_size+1,sizeof(char));

                  c_ptr = cut_buffer;
                  strcpy (cut_buffer, &first_str->s[first_index]);
                  c_ptr += strlen(&first_str->s[first_index]);
                  *c_ptr++ = '\n';
                  for (tmp_s_ptr = first_str->next; TRUE;
                        tmp_s_ptr = tmp_s_ptr->next)
                  {
                     if (tmp_s_ptr == s_ptr)
                     {
                        strncpy (c_ptr, s_ptr->s, second_index);
                        c_ptr[second_index] = '\0';
                        break;
                     }
                     else
                     {
                        strcpy (c_ptr, tmp_s_ptr->s);
                        c_ptr += strlen (tmp_s_ptr->s);
                        *c_ptr++ = '\n';
                     }
                  }
                  break;
               }
               else
               {  /* include the whole line */
                  cut_buffer_size += strlen(s_ptr->s)+1;
                  s_ptr = s_ptr->next;
               }
            }
            else
            {  /* looking for the beginning ... */
               if (s_ptr == curStr && s_ptr == endStr)
               {  /* the whole string to be copied is within s_ptr */
                  if (textCurIndex == textEndIndex) return;

                  first_index = min(textCurIndex,textEndIndex);
                  second_index = max(textCurIndex,textEndIndex);
                  cut_buffer_size = second_index - first_index;
                  cut_buffer = (char *)calloc(cut_buffer_size+1,sizeof(char));
                  strncpy (cut_buffer, &s_ptr->s[first_index], cut_buffer_size);
                  cut_buffer[cut_buffer_size] = '\0';
                  break;
               }
               else if (s_ptr == curStr || s_ptr == endStr)
               {  /* found the beginning */
                  first_str = s_ptr;
                  first_index = (s_ptr==curStr) ? textCurIndex : textEndIndex;
                  cut_buffer_size = strlen(&s_ptr->s[first_index]);
                  highlighting = TRUE;
                  s_ptr = s_ptr->next;
               }
               else
               {  /* still looking */
                  s_ptr = s_ptr->next;
               }
            }
         }
         copyingToCutBuffer = TRUE;
         XStoreBytes (mainDisplay, cut_buffer, cut_buffer_size);
         XSync (mainDisplay, False);
         if (copyingToCutBuffer == INVALID)
         {
            sprintf (msg, "%s  %s", "Copy to cut buffer fails.",
                  "Selected string may be too long.");
            copy_failed = TRUE;
         }
         else
            sprintf (msg, "Copy buffer updated.");
         copyingToCutBuffer = FALSE;
         Msg (msg);
         if (copy_failed)
         {
            *cut_buffer = '\0';
            XStoreBytes (mainDisplay, cut_buffer, 1);
         }
         cfree (cut_buffer);
         return;
      }

      key_ev = &(input->xkey);
      have_ch = XLookupString (key_ev, s, 80-1, &key_sym, &c_stat);
      TranslateKeys (s, &key_sym);

      if (!((s[0]=='\r' && (key_sym & 0xff)=='\r') ||
            (s[0]=='\n' && (key_sym & 0xff)=='\n') ||
            (s[0]=='\b' && (key_sym & 0xff)=='\b') ||
            (s[0]=='\b' && (key_sym & 0xff)=='h' &&
            (key_ev->state & ControlMask)) ||
            key_sym==XK_Left || key_sym==XK_Up ||
            key_sym==XK_Right || key_sym==XK_Down ||
            key_sym==XK_KP_Left || key_sym==XK_KP_Up ||
            key_sym==XK_KP_Right || key_sym==XK_KP_Down ||
            (key_sym==XK_Tab && (key_ev->state & (ShiftMask | Mod1Mask))) ||
            (s[0]=='\033' && (key_sym & 0xff)=='\033') ||
            (s[0]=='\177' && (key_sym & 0x7f)=='\177') ||
            ((key_ev->state & ControlMask)==0 && (s[0]&0xff)!=0 &&
            key_sym>='\040' && key_sym<='\177') ||
            ((s[0]&0xff) && key_sym>0xa0 && key_sym<=0xff)))
         return;

      EraseTextCursor ();
      switch (key_sym)
      {
         case XK_Left: HandleLeft (); break;
         case XK_KP_Left: HandleLeft (); break;
         case XK_Up: HandleUp (); break;
         case XK_KP_Up: HandleUp (); break;
         case XK_Right: HandleRight (); break;
         case XK_KP_Right: HandleRight (); break;
         case XK_Down: HandleDown (); break;
         case XK_KP_Down: HandleDown (); break;
         default:
            s[1] = '\0';
            switch (s[0])
            {
               case '\r':
               case '\n': HandleCRLF (colorIndex); break;

               case '\177': /* <DEL> */
                  if (escPressed)
                     HandleChar (s, colorIndex);
                  else
                     HandleBS (colorIndex);
                  break;

               case '\b': HandleBS (colorIndex); break;

               case '\033': /* <ESC> */
                  if (!escPressed) escPressed = TRUE;
                  break;

               case '\t': HandleTAB (key_ev, colorIndex); break;

               default: if (have_ch) HandleChar (s, colorIndex); break;
            }
            break;
      }
      if (textCursorShown)
      {
         PutTextCursor ();
         MarkRulers (textCurX, textCurY);
         SetFileModified (TRUE);
      }
   }
}

void DeleteHighLightedText ()
{
   if (!textCursorShown) return;

   EraseTextCursor ();
   HandleBS (colorIndex);
   if (textCursorShown)
   {
      PutTextCursor ();
      MarkRulers (textCurX, textCurY);
      SetFileModified (TRUE);
   }
}

void DumpOneStr (FP, FontIndex, Str)
   register FILE	* FP;
   int			FontIndex;
   register char	* Str;
{
   register char	* c_ptr;

   for ( ; *Str != '\0'; Str++)
   {
      switch (*Str)
      {
         case '(':
         case ')':
         case '\\': fprintf (FP, "\\"); break;
      }
      if ((*Str) & 0x80)
      {
         if (FontIndex != FONT_SYM && (c_ptr = CharCodeTranslate (Str)) != NULL)
         {
            if (*c_ptr == '\\')
               fprintf (FP, "%s", c_ptr);
            else if (*c_ptr == '8')
               fprintf (FP, "\\%c%c%c", c_ptr[2], c_ptr[3], c_ptr[4]);
         }
         else
            fprintf (FP, "\\%o", (*Str)&0xff);
      }
      else
         fprintf (FP, "%c", *Str);
   }
}

void DrawTextObj (Win, XOff, YOff, ObjPtr)
   Window		Win;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   struct TextRec	* text_ptr = ObjPtr->detail.t;
   struct StrRec	* s_ptr;
   int			x, y, xinc=0, yinc=0, fill, pen, draw_it=FALSE;
   XGCValues		values;

   for (s_ptr=text_ptr->first; !draw_it && s_ptr!=NULL; s_ptr=s_ptr->next)
      if (*s_ptr->s != '\0')
         draw_it = TRUE;

   if (!draw_it) return;

   SaveCurFont ();
   curFont = text_ptr->font;
   curStyle = text_ptr->style;
   curSize = text_ptr->size;
   textJust = text_ptr->just;
   textVSpace = text_ptr->v_space;
   curRotate = text_ptr->rotate;
   SetCanvasFont ();

   pen = text_ptr->pen;

   if ((zoomScale!=0 || curRotate!=ROTATE0 || text_ptr->read_only) &&
         text_ptr->cached_bitmap==None)
   {
      int	watch_cursor = watchCursorOnMainWindow;

      if (!watch_cursor)
      {
         SetWatchCursor (drawWindow);
         SetWatchCursor (mainWindow);
      }
      MakeCachedTextBitmap (ObjPtr);
      if (!watch_cursor)
      {
         SetDefaultCursor (mainWindow);
         ShowCursor ();
      }
   }

   x = ObjPtr->x - XOff;
   y = ObjPtr->y - YOff;

   switch (curRotate)
   {
      case ROTATE0: xinc = 0; yinc = textCursorH+textVSpace; break;
      case ROTATE90: xinc = -(textCursorH+textVSpace); yinc = 0; break;
      case ROTATE180: xinc = 0; yinc = -(textCursorH+textVSpace); break;
      case ROTATE270: xinc = textCursorH+textVSpace; yinc = 0; break;
   }

   fill = text_ptr->fill;
   if (fill != NONEPAT)
   {
      int	real_x_off, real_y_off, ltx, lty, rbx, rby;

      real_x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
      real_y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);
      ltx = ZOOMED_SIZE(ObjPtr->bbox.ltx - real_x_off);
      lty = ZOOMED_SIZE(ObjPtr->bbox.lty - real_y_off);
      rbx = ZOOMED_SIZE(ObjPtr->bbox.rbx - real_x_off);
      rby = ZOOMED_SIZE(ObjPtr->bbox.rby - real_y_off);

      values.foreground = (fill == BACKPAT) ? myBgPixel :
            colorPixels[ObjPtr->color];
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[fill];
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
      XFillRectangle (mainDisplay, Win, drawGC, ltx, lty, rbx-ltx, rby-lty);
   }

   if (zoomScale!=0 || curRotate!=ROTATE0 || text_ptr->read_only)
   {
      if (ObjPtr->obbox.ltx!=ObjPtr->obbox.rbx &&
            ObjPtr->obbox.lty!=ObjPtr->obbox.rby)
      {
         int	ltx, lty, w, h;

         ltx = OFFSET_X(ObjPtr->obbox.ltx);
         lty = OFFSET_Y(ObjPtr->obbox.lty);
         w = ZOOMED_SIZE(ObjPtr->obbox.rbx-ObjPtr->obbox.ltx);
         h = ZOOMED_SIZE(ObjPtr->obbox.rby-ObjPtr->obbox.lty);

         values.foreground = (pen == BACKPAT) ? myBgPixel :
               colorPixels[ObjPtr->color];
         values.function = GXcopy;
         values.fill_style = FillOpaqueStippled;
         values.stipple = patPixmap[pen];
         values.clip_mask = text_ptr->cached_bitmap;
         values.clip_x_origin = ltx;
         values.clip_y_origin = lty;
         XChangeGC (mainDisplay, drawGC,
               GCForeground | GCFunction | GCFillStyle | GCStipple |
               GCClipMask | GCClipXOrigin | GCClipYOrigin, &values);
         XFillRectangle (mainDisplay, Win, drawGC, ltx, lty, w, h);

         values.clip_mask = None;
         values.clip_x_origin = 0;
         values.clip_y_origin = 0;
         XChangeGC (mainDisplay, drawGC,
               GCClipMask | GCClipXOrigin | GCClipYOrigin, &values);
         if (numClipRecs > 0)
            XSetClipRectangles (mainDisplay, drawGC, 0, 0, clipRecs,
               numClipRecs, clipOrdering);
      }
   }
   else
   {
      for (s_ptr = text_ptr->first; s_ptr != NULL; s_ptr = s_ptr->next)
      {
         PaintText (Win, s_ptr->s, textJust, curRotate, x, y,
               canvasFontPtr, ObjPtr->color, pen);
         x += xinc;
         y += yinc;
      }
   }
   RestoreCurFont ();
}

#define FONTS_PER_DPI (((MAXFONTS-1)*MAXFONTSTYLES+1)*MAXFONTSIZES)

static
int OldFontIndex (dpi_index, font_index, size_index, style_index)
   register int dpi_index, font_index, size_index, style_index;
   /* obsoleted procedure, kept to remain compatible with old versions */
{
   if (font_index == FONT_SYM)
      return (size_index+MAXFONTSIZES*(MAXFONTSTYLES*font_index) +
            dpi_index*FONTS_PER_DPI);
   else
      return (size_index+MAXFONTSIZES*(style_index+MAXFONTSTYLES*font_index) +
            dpi_index*FONTS_PER_DPI);
}

#define GETVALUE(val,name) ScanValue("%d", &(val), name, "text")
#define GETSTRNG(val,name) ScanValue("%s", (val), name, "text")

void ReadTextObj (FP, Inbuf, ObjPtr)
   FILE			* FP;
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   register int		i, max_len = 0, len;
   struct StrRec	* s_ptr;
   struct TextRec	* text_ptr;
   char			color_str[80], * s, * c_ptr, font_str[80];
   char			* tmp_str, inbuf[MAXSTRING+1], * line;
   int			num_lines, x, y, font, style, size, new_alloc, id=0;
   int			text_just, rotate, pen, cur_size, done, rotation;
   int			bbox_w, bbox_h, dpi, asc, des, fill, allocated, v_space;
   int			locked=FALSE;

   dpi = FONT_DPI_75;
   fill = NONEPAT;
   v_space = 0;

   *ObjPtr = NULL;

   s = FindChar ((int)'(', Inbuf);
   s = ParseStr (s, (int)',', color_str, sizeof(color_str));

   InitScan (s, ", \t\n");

   rotate = 0;
   pen = 1;
   rotation = 0;
   if (fileVersion <= 2)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID)
      {
         return;
      }
      id = objId++;
   }
   else if (fileVersion <= 6)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID)
      {
         return;
      }
      id = objId++;
   }
   else if (fileVersion <= 7)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID ||
          GETVALUE (bbox_w,    "bbox w") == INVALID ||
          GETVALUE (bbox_h,    "bbox h") == INVALID)
      {
         return;
      }
      id = objId++;
   }
   else if (fileVersion <= 9)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID ||
          GETVALUE (bbox_w,    "bbox w") == INVALID ||
          GETVALUE (bbox_h,    "bbox h") == INVALID ||
          GETVALUE (id,        "id") == INVALID ||
          GETVALUE (dpi,       "dpi") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 10)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID ||
          GETVALUE (bbox_w,    "bbox w") == INVALID ||
          GETVALUE (bbox_h,    "bbox h") == INVALID ||
          GETVALUE (id,        "id") == INVALID ||
          GETVALUE (dpi,       "dpi") == INVALID ||
          GETVALUE (asc,       "ascent") == INVALID ||
          GETVALUE (des,       "descent") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 12)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID ||
          GETVALUE (bbox_w,    "bbox w") == INVALID ||
          GETVALUE (bbox_h,    "bbox h") == INVALID ||
          GETVALUE (id,        "id") == INVALID ||
          GETVALUE (dpi,       "dpi") == INVALID ||
          GETVALUE (asc,       "ascent") == INVALID ||
          GETVALUE (des,       "descent") == INVALID ||
          GETVALUE (fill,      "fill") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 13)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID ||
          GETVALUE (bbox_w,    "bbox w") == INVALID ||
          GETVALUE (bbox_h,    "bbox h") == INVALID ||
          GETVALUE (id,        "id") == INVALID ||
          GETVALUE (dpi,       "dpi") == INVALID ||
          GETVALUE (asc,       "ascent") == INVALID ||
          GETVALUE (des,       "descent") == INVALID ||
          GETVALUE (fill,      "fill") == INVALID ||
          GETVALUE (v_space,   "vertical spacing") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 25)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID ||
          GETVALUE (bbox_w,    "bbox w") == INVALID ||
          GETVALUE (bbox_h,    "bbox h") == INVALID ||
          GETVALUE (id,        "id") == INVALID ||
          GETVALUE (dpi,       "dpi") == INVALID ||
          GETVALUE (asc,       "ascent") == INVALID ||
          GETVALUE (des,       "descent") == INVALID ||
          GETVALUE (fill,      "fill") == INVALID ||
          GETVALUE (v_space,   "vertical spacing") == INVALID ||
          GETVALUE (rotation,  "rotation") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 29)
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETVALUE (font,      "font") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID ||
          GETVALUE (bbox_w,    "bbox w") == INVALID ||
          GETVALUE (bbox_h,    "bbox h") == INVALID ||
          GETVALUE (id,        "id") == INVALID ||
          GETVALUE (dpi,       "dpi") == INVALID ||
          GETVALUE (asc,       "ascent") == INVALID ||
          GETVALUE (des,       "descent") == INVALID ||
          GETVALUE (fill,      "fill") == INVALID ||
          GETVALUE (v_space,   "vertical spacing") == INVALID ||
          GETVALUE (rotation,  "rotation") == INVALID ||
          GETVALUE (locked,    "locked") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else
   {
      if (GETVALUE (x,         "x") == INVALID ||
          GETVALUE (y,         "y") == INVALID ||
          GETSTRNG (font_str,  "font_str") == INVALID ||
          GETVALUE (style,     "style") == INVALID ||
          GETVALUE (size,      "size") == INVALID ||
          GETVALUE (num_lines, "num_lines") == INVALID ||
          GETVALUE (text_just, "text_just") == INVALID ||
          GETVALUE (rotate,    "rotate") == INVALID ||
          GETVALUE (pen,       "pen") == INVALID ||
          GETVALUE (bbox_w,    "bbox w") == INVALID ||
          GETVALUE (bbox_h,    "bbox h") == INVALID ||
          GETVALUE (id,        "id") == INVALID ||
          GETVALUE (dpi,       "dpi") == INVALID ||
          GETVALUE (asc,       "ascent") == INVALID ||
          GETVALUE (des,       "descent") == INVALID ||
          GETVALUE (fill,      "fill") == INVALID ||
          GETVALUE (v_space,   "vertical spacing") == INVALID ||
          GETVALUE (rotation,  "rotation") == INVALID ||
          GETVALUE (locked,    "locked") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   fill = UpgradePenFill (fill);
   pen = UpgradePenFill (pen);

   * ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));

   text_ptr = (struct TextRec *) calloc (1, sizeof(struct TextRec));
   text_ptr->lines = num_lines;
   text_ptr->cached_bitmap = None;
   text_ptr->cached_zoom = 0;
   text_ptr->cached_rotate = ROTATE0;

   if (!PRTGIF) SaveCurFont ();

   if (PRTGIF)
   {
      if (fileVersion <= 29)
      {
         curFont = text_ptr->font = font;
         text_ptr->font_name = NULL;
      }
      else
      {
         len = strlen (font_str);
         text_ptr->font_name = (char *) calloc (len+1, sizeof(char));
         if (*font_str == '\'' && font_str[len-1] == '\'')
         {
            font_str[len-1] = '\0';
            strcpy (text_ptr->font_name, &font_str[1]);
         }
         else
            strcpy (text_ptr->font_name, font_str);
         curFont = text_ptr->font =
               GetFontIndex (text_ptr->font_name, style, FALSE);
      }
   }
   else
   {
      text_ptr->font_name = NULL;
      if (fileVersion <= 29)
         curFont = text_ptr->font = font;
      else
      {
         char	*s;

         len = strlen (font_str);
         if (*font_str == '\'' && font_str[len-1] == '\'')
         {
            font_str[len-1] = '\0';
            s = &font_str[1];
         }
         else
            s = font_str;
         curFont = text_ptr->font = GetFontIndex (s, style, TRUE);
         if (curFont == INVALID)
         {
            char	msg[MAXSTRING];

            sprintf (msg, "Can not find screen font for '%s'.", s);
            TwoLineMsg (msg, "    Use Times instead.");
            SetFileModified (TRUE);
            curFont = text_ptr->font = FONT_TIM;
         }
      }
   }
   curStyle = text_ptr->style = style;
   curSize = text_ptr->size =
         (fileVersion<=29 ? GetCompatibleSize (dpi, size) : size);
   textJust = text_ptr->just = text_just;
   textVSpace = text_ptr->v_space = v_space;
   curRotate = text_ptr->rotate = rotate;
   penPat = text_ptr->pen = pen;
   objFill = text_ptr->fill = fill;

   if (PRTGIF)
   {
      if (fileVersion < 10)
      {
         canvasFontAsc =
               pDrawFontAsc[OldFontIndex(dpi,curFont,size,curStyle)];
         canvasFontDes =
               pDrawFontDes[OldFontIndex(dpi,curFont,size,curStyle)];
      }
      else
      {
         canvasFontAsc = asc;
         canvasFontDes = des;
      }
      textCursorH = canvasFontAsc + canvasFontDes;
      text_ptr->read_only = FALSE;
   }
   else
   {
      SetCanvasFont ();
      text_ptr->read_only = (curSize != canvasFontSize);
   }
   if (text_ptr->read_only)
   {
      text_ptr->asc = asc;
      text_ptr->des = des;
      switch (rotate)
      {
         case ROTATE0:
         case ROTATE180:
            text_ptr->orig_w = bbox_w;
            text_ptr->orig_h = bbox_h;
            break;
         case ROTATE90:
         case ROTATE270:
            text_ptr->orig_w = bbox_h;
            text_ptr->orig_h = bbox_w;
            break;
      }
   }
   else
   {
      text_ptr->asc = canvasFontAsc;
      text_ptr->des = canvasFontDes;
      text_ptr->orig_w = text_ptr->orig_h = 0;
   }
   for (i = 0; i < num_lines; i++)
   {
      fgets (inbuf, MAXSTRING, FP);
      scanLineNum++;
      allocated = FALSE;

      len = strlen(inbuf);
      if (inbuf[len-1] != '\r' && inbuf[len-1] != '\n')
      {  /* line longer than MAXSTRING characters */
         cur_size = 2*MAXSTRING-1;
         line = (char *) calloc (cur_size+1, sizeof(char));
         strcpy (line, inbuf);
         c_ptr = &(line[MAXSTRING-1]);
         allocated = TRUE;

         done = FALSE;
         while (!done && fgets (inbuf, MAXSTRING, FP) != NULL)
         {
            scanLineNum++;
            len = strlen(inbuf);
            if (inbuf[len-1] == '\r' || inbuf[len-1] == '\n')
            {
               done = TRUE;
               inbuf[len-1] = '\0';
               strcpy (c_ptr, inbuf);
            }
            else
            {
               int	n = c_ptr - line;	/* NMH */

               cur_size += MAXSTRING-1;
               line = (char *) realloc (line, cur_size+1);
               c_ptr = line + n;		/* NMH */
               strcpy (c_ptr, inbuf);
               c_ptr += MAXSTRING-1;
            }
         }
         if (!done)
         {
            sprintf (inbuf, "%s, %d:  EOF in ReadTextObj ().  Read aborted!",
                  scanFileName, scanLineNum);
            if (PRTGIF)
               fprintf (stderr, "%s\n", inbuf);
            else
               Msg (inbuf);

            cfree (line);
            return;
         }
      }
      else
      {
         line = inbuf;
         line[len-1] = '\0';
      }

      tmp_str = FindChar ((int)'"', line);
      s = ReadString (tmp_str);
      *(--s) = '\0';
      s_ptr = (struct StrRec *) calloc (1, sizeof(struct StrRec));
      strcpy (s_ptr->s, tmp_str);
      AddStr (lastStr, (struct StrRec *)NULL, s_ptr);
      if (PRTGIF)
         len = strlen (tmp_str); /* assume string width = 1 pixel per char */
      else
         len = XTextWidth (canvasFontPtr, tmp_str, strlen (tmp_str));
      if (len > max_len) max_len = len;

      if (allocated) cfree (line);
   }
   text_ptr->first = firstStr;
   text_ptr->last = lastStr;

   firstStr = lastStr = NULL;
   textCurIndex = 0;

   (*ObjPtr)->x = x;
   (*ObjPtr)->y = y;

   if (PRTGIF && fileVersion > 6)
   {
      switch (rotate)
      {
         case ROTATE0:
         case ROTATE180:
            SetTextBBox (*ObjPtr, text_just, bbox_w, bbox_h, rotate);
            break;
         case ROTATE90:
         case ROTATE270:
            SetTextBBox (*ObjPtr, text_just, bbox_h, bbox_w, rotate);
            break;
      }
   }
   else if (!PRTGIF && text_ptr->read_only)
      SetTextBBox (*ObjPtr, text_just, text_ptr->orig_w, text_ptr->orig_h,
            rotate);
   else
      SetTextBBox (*ObjPtr, text_just, max_len,
            num_lines*textCursorH+(num_lines-1)*textVSpace, rotate);

   (*ObjPtr)->type = OBJ_TEXT;
   (*ObjPtr)->color = QuickFindColorIndex (color_str, &new_alloc, TRUE);
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->id = id;
   (*ObjPtr)->rotation = rotation;
   (*ObjPtr)->locked = locked;
   (*ObjPtr)->detail.t = text_ptr;

   if (!PRTGIF) RestoreCurFont ();
}

void RedrawCurText ()
{
   register int		x = 0, y;
   struct StrRec	* s_ptr;
   int			highlighting = FALSE, mode;
   int			first_index = INVALID, second_index = INVALID;

   if (!textCursorShown) return;

   switch (textJust)
   {
      case JUST_L: x = textOrigX-2; break;
      case JUST_C: x = textOrigX-textW/2-2; break;
      case JUST_R: x = textOrigX-textW-2; break;
   }
   XClearArea (mainDisplay, drawWindow, x, textOrigY-2,
         textW+5, textH+5, FALSE);
   XDrawRectangle (mainDisplay, drawWindow, defaultGC, x, textOrigY-2,
         textW+4, textH+4);

   y = textOrigY;

   for (s_ptr = curTextObj->detail.t->first; s_ptr != NULL; s_ptr = s_ptr->next)
   {
      if (textHighlight)
      {
         first_index = second_index = INVALID;
         if (highlighting)
         {
            if (s_ptr == curStr || s_ptr == endStr)
            {
               mode = PAINT_INV_NORM;
               first_index = (s_ptr == curStr) ? textCurIndex : textEndIndex;
               highlighting = FALSE;
            }
            else
            {
               mode = PAINT_INV;
               first_index = (s_ptr == curStr) ? textCurIndex : textEndIndex;
            }
         }
         else
         {
            if (s_ptr == curStr && s_ptr == endStr)
            {
               mode = PAINT_NORM_INV_NORM;
               first_index = min(textCurIndex,textEndIndex);
               second_index = max(textCurIndex,textEndIndex);
            }
            else if (s_ptr == curStr || s_ptr == endStr)
            {
               mode = PAINT_NORM_INV;
               first_index = (s_ptr == curStr) ? textCurIndex : textEndIndex;
               highlighting = TRUE;
            }
            else
               mode = PAINT_NORM;
         }
      }
      else
         mode = PAINT_NORM;

      PaintCurText (drawWindow, drawGC, s_ptr->s, textJust, textOrigX, y,
            canvasFontPtr, colorIndex, penPat, mode, first_index, second_index);
      y += textCursorH+textVSpace;
   }

   PutTextCursor ();
}

void AdjustCurText (XOff, YOff)
   int	XOff, YOff;
{
   textOrigX += XOff;
   textOrigY += YOff;
   textCurX += XOff;
   textCurY += YOff;
}

void PrepareZoomCurText (AbsXc, AbsYc)
   int	* AbsXc, * AbsYc;
{
   switch (textJust)
   {
      case JUST_L: *AbsXc = ABS_X(textOrigX+(textW>>1)); break;
      case JUST_C: *AbsXc = ABS_X(textOrigX); break;
      case JUST_R: *AbsXc = ABS_X(textOrigX-(textW>>1)); break;
   }
   *AbsYc = ABS_Y(textOrigY+(textH>>1));
}

void PostZoomCurText (AbsXc, AbsYc)
   int	AbsXc, AbsYc;
{
   int	x=0, y;

   switch (textJust)
   {
      case JUST_L: x = OFFSET_X(AbsXc)-(textW>>1); break;
      case JUST_C: x = OFFSET_X(AbsXc); break;
      case JUST_R: x = OFFSET_X(AbsXc)+(textW>>1); break;
   }
   y = OFFSET_Y(AbsYc)-(textH>>1);
   AdjustCurText (x-textOrigX, y-textOrigY);
   textAbsX -= tmpAdjX;
   textAbsY -= tmpAdjY;
   switch (curRotate)
   {
      case ROTATE0:
         switch (textJust)
         {
            case JUST_L: tmpAdjX = (textW-ABS_SIZE(textW))/2; break;
            case JUST_C: tmpAdjX = 0; break;
            case JUST_R: tmpAdjX = (ABS_SIZE(textW)-textW)/2; break;
         }
         tmpAdjY = (textH-ABS_SIZE(textH))/2;
         break;
      case ROTATE90:
         switch (textJust)
         {
            case JUST_L:
               tmpAdjX = -(ABS_SIZE(textW)+textH)/2;
               tmpAdjY = (textW-ABS_SIZE(textH))/2;
               break;
            case JUST_C:
               tmpAdjX = -textH/2;
               tmpAdjY = -ABS_SIZE(textH)/2;
               break;
            case JUST_R:
               tmpAdjX = (ABS_SIZE(textW)-textH)/2;
               tmpAdjY = -(textW+ABS_SIZE(textH))/2;
               break;
         }
         break;
      case ROTATE180:
         switch (textJust)
         {
            case JUST_L: tmpAdjX = -(textW+ABS_SIZE(textW))/2; break;
            case JUST_C: tmpAdjX = 0; break;
            case JUST_R: tmpAdjX = (textW+ABS_SIZE(textW))/2; break;
         }
         tmpAdjY = -(textH+ABS_SIZE(textH))/2;
         break;
      case ROTATE270:
         switch (textJust)
         {
            case JUST_L:
               tmpAdjX = -(ABS_SIZE(textW)-textH)/2;
               tmpAdjY = -(textW+ABS_SIZE(textH))/2;
               break;
            case JUST_C:
               tmpAdjX = textH/2;
               tmpAdjY = -ABS_SIZE(textH)/2;
               break;
            case JUST_R:
               tmpAdjX = (ABS_SIZE(textW)+textH)/2;
               tmpAdjY = (textW-ABS_SIZE(textH))/2;
               break;
         }
         break;
   }
   textAbsX += tmpAdjX;
   textAbsY += tmpAdjY;
   RedrawCurText ();
}
