/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1994, William Cheng.
 *
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights (including the right to sell "tgif" and the right to sell derivative
 * works of tgif) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/maui/tangram/u/william/X11/TGIF2/RCS/color.c,v 2.68 1994/05/01 23:22:57 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "types.h"

#include "choice.e"
#include "cmd.e"
#ifndef _NO_EXTERN
#include "color.e"
#endif
#include "drawing.e"
#include "mainloop.e"
#include "mark.e"
#include "menu.e"
#include "msg.e"
#include "obj.e"
#include "pattern.e"
#include "raster.e"
#include "select.e"
#include "setup.e"
#include "text.e"
#include "xpixmap.e"

extern int	atoi ARGS_DECL((char *));

#define COLORSTRLEN 80

#define WHITE_COLOR_INDEX 8
#define BLACK_COLOR_INDEX 9

int	maxColors = MAXCOLORS;
int	defaultColorIndex = 9;
int	colorIndex = 0;
int	* colorPixels = NULL;
int	* xorColorPixels = NULL;
char	myFgColorStr[COLORSTRLEN];
char	myBgColorStr[COLORSTRLEN];
char	* * colorMenuItems = NULL;
XColor	* tgifColors=NULL;
XColor	* tgifRequestedColors=NULL;
XColor	myBgColor;
int	maxRGB = 0;
int	colorDump = FALSE;

static int	defaultColorIndexInXDefaults = FALSE;

static int	allocatedMaxColors = MAXCOLORS;

static char	* defaultColorMenuItems[MAXCOLORS] =
{
   "magenta", "red", "green", "blue", "yellow", "pink", "cyan", "CadetBlue",
   "white", "black", "DarkSlateGray"
};

void DefaultColorArrays (Entries, ForePixels, Valid, InitRV, StatusStr)
   int	Entries, * * ForePixels, * * Valid, * * InitRV;
   char	* * * StatusStr;
{
   register int	i, * fore_pixels, pixel, * valid, * init_rv;

   pixel = myFgPixel;
   *ForePixels = fore_pixels = (int *) calloc (Entries, sizeof(int));
   *Valid = valid = (int *) calloc (Entries, sizeof(int));
   *InitRV = init_rv = (int *) calloc (Entries, sizeof(int));
   for (i = 0; i < Entries; i++)
   {
      *fore_pixels++ = pixel;
      *valid++ = TRUE;
      *init_rv++ = FALSE;
   }
   if (StatusStr != NULL)
   {
      *StatusStr = (char **) calloc (Entries, sizeof(char *));
      for (i=0; i < Entries; i++)
      {
         (*StatusStr)[i] = (char *) calloc (MAXSTRING+1, sizeof(char));
         *(*StatusStr)[i] = '\0';
      }
   }
}

static
int ParseAndAllocColorByName (colorname, color, red_req, green_req, blue_req)
   char			* colorname;
   XColor		* color;
   unsigned short	* red_req, * green_req, * blue_req;
{
   Colormap	colormap;

   if (!XParseColor(mainDisplay, mainColormap, colorname, color))
   {
      fprintf (stderr, "Warning:  can not parse color '%s'\n", colorname);
      return (FALSE);
   }
   if (red_req != NULL) *red_req = color->red;
   if (green_req != NULL) *green_req = color->green;
   if (blue_req != NULL) *blue_req = color->blue;
   if (!XAllocColor(mainDisplay, mainColormap, color))
   {
      if (newColormapUsed)
      {
         fprintf (stderr, "Warning:  can not allocate color '%s'\n", colorname);
         return (FALSE);
      }
      colormap = XCopyColormapAndFree (mainDisplay, mainColormap);
      mainColormap = colormap;
      newColormapUsed = TRUE;
      if (mainWindow != None)
         XSetWindowColormap (mainDisplay, mainWindow, mainColormap);
      if (!XAllocColor(mainDisplay, mainColormap, color))
      {
         fprintf (stderr, "Warning:  can not allocate color '%s'\n", colorname);
         return (FALSE);
      }
   }
   return (TRUE);
}

static
int MyStrCmp (s1, s2)
   register char	* s1, * s2;
{
   for ( ; *s1 != '\0' && *s2 != '\0'; s1++, s2++)
      if (*s1 != *s2)
      {
         if (*s1 >= 'A' && *s1 <= 'Z')
         {
            if (*s1-'A'+'a' != *s2) return (*s1 - *s2);
         }
         else if (*s2 >= 'A' && *s2 <= 'Z')
         {
            if (*s2-'A'+'a' != *s1) return (*s1 - *s2);
         }
         else
            return (*s1 - *s2);
      }
   return (0);
}

void InitColor ()
{
   register int	i, index;
   XColor	color, exact_def;
   char		buf[80], * c_ptr, fg_color[80], bg_color[80], brdr_color[80];
   long		bg_gray=(long)0;
   int		looking_for_default_color;
   int		color_in_x_default = FALSE;
   int		bg_allocated=FALSE, fg_allocated=FALSE, brdr_allocated=FALSE;
   int		tmp_max;

   if (cmdLineRV != INVALID)
      reverseVideo = cmdLineRV;
   else if ((c_ptr = XGetDefault(mainDisplay,TOOL_NAME,"ReverseVideo")) != NULL)
   {
      reverseVideo = FALSE;
      if (strcmp (c_ptr, "on") == 0 || strcmp (c_ptr, "On") == 0 ||
            strcmp (c_ptr, "True") == 0 || strcmp (c_ptr, "true") == 0)
         reverseVideo = TRUE;
   }

   colorDump = FALSE;
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"InitialPrintInColor")) != NULL)
      if ((strcmp (c_ptr, "True") == 0) || (strcmp (c_ptr, "true") == 0))
         colorDump = TRUE;

   if ((c_ptr = XGetDefault (mainDisplay, TOOL_NAME, "DefaultColorIndex"))
         != NULL)
   {
      defaultColorIndexInXDefaults = TRUE;
      defaultColorIndex = atoi (c_ptr);
   }
   else
   {
      defaultColorIndexInXDefaults = FALSE;
      defaultColorIndex = 9;
   }

   if ((c_ptr = XGetDefault (mainDisplay, TOOL_NAME, "MaxColors")) != NULL)
   {
      color_in_x_default = TRUE;
      maxColors = atoi (c_ptr);
   }

   if (colorDisplay)
   {
      if ((cmdLineForeground != NULL && cmdLineBackground == NULL ||
            cmdLineForeground == NULL && cmdLineBackground != NULL) &&
            reverseVideo)
      {
         fprintf (stderr, "Normal video mode assumed since %s is %s.\n",
               cmdLineForeground == NULL ? "-bg" : "-fg",
               "specified in the command line");
         reverseVideo = FALSE;
      }
      if (cmdLineForeground != NULL)
         strcpy (fg_color, cmdLineForeground);
      else if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"Foreground")) != NULL)
      {
         if (reverseVideo)
            strcpy (bg_color, c_ptr);
         else
            strcpy (fg_color, c_ptr);
      }
      else if (reverseVideo)
         strcpy (fg_color, "white");
      else
         strcpy (fg_color, "black");

      if (cmdLineBackground != NULL)
         strcpy (bg_color, cmdLineBackground);
      else if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"Background")) != NULL)
      {
         if (reverseVideo)
            strcpy (fg_color, c_ptr);
         else
            strcpy (bg_color, c_ptr);
      }
      else if (reverseVideo)
         strcpy (bg_color, "black");
      else
         strcpy (bg_color, "white");

      if (cmdLineBorder != NULL)
         strcpy (brdr_color, cmdLineBorder);
      else if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"BorderColor")) != NULL)
         strcpy (brdr_color, c_ptr);
      else
         strcpy (brdr_color, fg_color);
   }
   else
   {
      if (reverseVideo)
      {
         strcpy (fg_color, "white");
         strcpy (bg_color, "black");
      }
      else
      {
         strcpy (fg_color, "black");
         strcpy (bg_color, "white");
      }
      strcpy (brdr_color, fg_color);
   }

   if (maxColors <= 0)
   {
      fprintf (stderr, "0 colors specified (must be at least 1).\n");
      exit (-1);
   }

   colorPixels = (int *) calloc (maxColors, sizeof (int));
   xorColorPixels = (int *) calloc (maxColors, sizeof (int));
   colorMenuItems = (char * *) calloc (maxColors, sizeof (char *));

   tgifColors = (XColor *) calloc (maxColors, sizeof (XColor));
   tgifRequestedColors = (XColor *) calloc (maxColors, sizeof (XColor));

   allocatedMaxColors = maxColors;
   for (i = 0; i < allocatedMaxColors; i++)
      colorMenuItems[i] = (char *) calloc (COLORSTRLEN, sizeof(char));

   if (color_in_x_default)
   {
      for (i = 0; i < maxColors; i++)
      {
         sprintf (buf, "Color%1d", i);
         if ((c_ptr = XGetDefault (mainDisplay, TOOL_NAME, buf)) != NULL)
            strcpy (colorMenuItems[i], c_ptr);
         else
         {
            fprintf (stderr, "Could not GetDefault %s*%s\n", TOOL_NAME, buf);
            exit (-1);
         }
      }
   }
   else
      for (i = 0; i < maxColors; i++)
         strcpy (colorMenuItems[i], defaultColorMenuItems[i]);

   if (colorDisplay)
   {
      index = 0;
      for (i = 0; i < maxColors; i++)
      {
         unsigned short	red_req, green_req, blue_req;

         if (!ParseAndAllocColorByName (colorMenuItems[i], &color,
               &red_req, &green_req, &blue_req))
         {
            fprintf (stderr, "%s %1d:  '%s'!  %s aborted!\n",
                  "Fail to allocate color number", i, colorMenuItems[i],
                  TOOL_NAME);
            exit (-1);
         }

         exact_def.red = color.red;
         exact_def.green = color.green;
         exact_def.blue = color.blue;

         if (i != index) strcpy (colorMenuItems[index], colorMenuItems[i]);

         colorPixels[index] = color.pixel;

         tgifColors[index].red = color.red;
         tgifColors[index].green = color.green;
         tgifColors[index].blue = color.blue;

         tgifRequestedColors[index].red = red_req;
         tgifRequestedColors[index].green = green_req;
         tgifRequestedColors[index].blue = blue_req;

         if (MyStrICmp (colorMenuItems[i], fg_color) == 0)
         {
            myFgPixel = color.pixel;
            strcpy (myFgColorStr, fg_color);

            fg_allocated = TRUE;
         }
         if (MyStrICmp (colorMenuItems[i], bg_color) == 0)
         {
            myBgPixel = color.pixel;
            myBgColor.pixel = color.pixel;
            myBgColor.red = color.red;
            myBgColor.green = color.green;
            myBgColor.blue = color.blue;
            strcpy (myBgColorStr, bg_color);

            bg_allocated = TRUE;
            bg_gray = (((long)tgifColors[i].red)<<2) +
                  (((long)tgifColors[i].green)<<3) + ((long)tgifColors[i].blue);
         }
         if (MyStrICmp (colorMenuItems[i], brdr_color) == 0)
         {
            myBorderPixel = color.pixel;
            brdr_allocated = TRUE;
         }
         index++;
      }
      maxColors = index;
      if (maxColors <= 0)
      {
         fprintf (stderr, "0 colors specified (must be at least 1).\n");
         exit (-1);
      }
      looking_for_default_color = FALSE;
      if (defaultColorIndexInXDefaults)
      {
         if (defaultColorIndexInXDefaults && defaultColorIndex >= maxColors)
         {
            fprintf (stderr, "Warning:  DefaultColorIndex >= MaxColors, ");
            fprintf (stderr, "Use 0 for DefaultColorIndex\n");
            defaultColorIndex = 0;
         }
      }
      else
      {
         looking_for_default_color = TRUE;
         for (i = 0; i < maxColors; i++)
         {
            if (MyStrCmp (fg_color, colorMenuItems[i]) == 0)
            {
               defaultColorIndex = i;
               looking_for_default_color = FALSE;
               break;
            }
         }
      }
      if (!fg_allocated)
      {
         if (!ParseAndAllocColorByName (fg_color, &color, NULL, NULL, NULL))
         {
            fprintf (stderr, "Fail to allocate the '%s' color!  Abort!\n",
                  fg_color);
            exit (-1);
         }
         myFgPixel = color.pixel;
         strcpy (myFgColorStr, fg_color);
      }
      if (!bg_allocated)
      {
         if (!ParseAndAllocColorByName (bg_color, &color, NULL, NULL, NULL))
         {
            fprintf (stderr, "Fail to allocate the '%s' color!  Abort!\n",
                  bg_color);
            exit (-1);
         }
         myBgPixel = color.pixel;
         myBgColor.pixel = color.pixel;
         myBgColor.red = color.red;
         myBgColor.green = color.green;
         myBgColor.blue = color.blue;
         strcpy (myBgColorStr, bg_color);
         if (looking_for_default_color)
            bg_gray = (((long)color.red)<<2) + (((long)color.green)<<3) +
                  ((long)color.blue);
      }
      if (looking_for_default_color)
      {
         long	val, val1;

         defaultColorIndex = 0;
         val = (((long)tgifColors[0].red)<<2) +
               (((long)tgifColors[0].green)<<3) + ((long)tgifColors[0].blue);
         if (bg_gray >= 0x67ff9) /* (0xffff<<1) + (0xffff<<2) + (0x7fff) */
         {
            for (i = 1; i < maxColors; i++)
            {
               val1 = (((long)tgifColors[i].red)<<2) +
                     (((long)tgifColors[i].green)<<3) +
                     ((long)tgifColors[i].blue);
               if (val > val1)
               {
                  val = val1;
                  defaultColorIndex = i;
               }
            }
         }
         else
         {
            for (i = 1; i < maxColors; i++)
            {
               val1 = (((long)tgifColors[i].red)<<2) +
                     (((long)tgifColors[i].green)<<3) +
                     ((long)tgifColors[i].blue);
               if (val < val1)
               {
                  val = val1;
                  defaultColorIndex = i;
               }
            }
         }
      }
      if (!brdr_allocated)
      {
         if (!ParseAndAllocColorByName (brdr_color, &color, NULL, NULL, NULL))
         {
            fprintf (stderr, "Fail to allocate the '%s' color!  Abort!\n",
                  brdr_color);
            exit (-1);
         }
         myBorderPixel = color.pixel;
      }
      for (i = 0; i < maxColors; i++)
         xorColorPixels[i] = colorPixels[i] ^ myBgPixel;
      colorIndex = defaultColorIndex;
   }
   else
   {
      if (!ParseAndAllocColorByName (fg_color, &color, NULL, NULL, NULL))
      {
         fprintf (stderr, "Fail to allocate the '%s' color!  Abort!\n",
               fg_color);
         exit (-1);
      }
      myFgPixel = color.pixel;
      strcpy (myFgColorStr, fg_color);

      if (!ParseAndAllocColorByName (bg_color, &color, NULL, NULL, NULL))
      {
         fprintf (stderr, "Fail to allocate the '%s' color!  Abort!\n",
               bg_color);
         exit (-1);
      }
      myBgPixel = color.pixel;
      strcpy (myBgColorStr, bg_color);

      if (!ParseAndAllocColorByName (brdr_color, &color, NULL, NULL, NULL))
      {
         fprintf (stderr, "Fail to allocate the '%s' color!  Abort!\n",
               brdr_color);
         exit (-1);
      }
      myBorderPixel = color.pixel;

      for (i = 0; i < maxColors; i++)
      {
         colorPixels[i] = myFgPixel;
         xorColorPixels[i] = myFgPixel ^ myBgPixel;
      }
      colorIndex = defaultColorIndex;
   }
   if (!ParseAndAllocColorByName ("black", &color, NULL, NULL, NULL))
   {
      fprintf (stderr, "Fail to allocate the 'black' color!  Abort!\n");
      exit (-1);
   }
   tmp_max = max(color.red,max(color.green,color.blue));
#ifndef DONTFREECOLORS
   XFreeColors (mainDisplay, mainColormap, &(color.pixel), 1, 0);
#endif
   if (tmp_max > maxRGB) maxRGB = tmp_max;
   if (!ParseAndAllocColorByName ("white", &color, NULL, NULL, NULL))
   {
      fprintf (stderr, "Fail to allocate the 'white' color!  Abort!\n");
      exit (-1);
   }
   tmp_max = max(color.red,max(color.green,color.blue));
#ifndef DONTFREECOLORS
   XFreeColors (mainDisplay, mainColormap, &(color.pixel), 1, 0);
#endif
   if (tmp_max > maxRGB) maxRGB = tmp_max;
   if (tmp_max == 0)
      fprintf (stderr, "Warning:  Unexpected maximum RGB intensity of 0.");

   if ((c_ptr = XGetDefault (mainDisplay,TOOL_NAME,"RubberBandColor"))!=NULL &&
            ParseAndAllocColorByName (c_ptr, &color, NULL, NULL, NULL))
      xorOne = color.pixel^myBgPixel;
   else
      xorOne = myFgPixel^myBgPixel;

   xorZero = myBgPixel;
}

int OneColorObject (ObjPtr, ColorIndex)
   struct ObjRec	* ObjPtr;
   int			* ColorIndex;
{
   register struct ObjRec	* obj_ptr;

   for (obj_ptr=ObjPtr->detail.r->last; obj_ptr!=NULL; obj_ptr=obj_ptr->prev)
      switch (obj_ptr->type)
      {
         case OBJ_POLY:
            if (*ColorIndex == INVALID)
            {
               if (obj_ptr->detail.p->fill != NONEPAT &&
                     obj_ptr->detail.p->fill != BACKPAT ||
                     obj_ptr->detail.p->pen != NONEPAT &&
                     obj_ptr->detail.p->pen != BACKPAT)
                  *ColorIndex = obj_ptr->color;
            }
            else if (obj_ptr->color != *ColorIndex)
            {
               if (obj_ptr->detail.p->fill != NONEPAT &&
                     obj_ptr->detail.p->fill != BACKPAT ||
                     obj_ptr->detail.p->pen != NONEPAT &&
                     obj_ptr->detail.p->pen != BACKPAT)
                  return (FALSE);
            }
            break;
         case OBJ_BOX:
            if (*ColorIndex == INVALID)
            {
               if (obj_ptr->detail.b->fill != NONEPAT &&
                     obj_ptr->detail.b->fill != BACKPAT ||
                     obj_ptr->detail.b->pen != NONEPAT &&
                     obj_ptr->detail.b->pen != BACKPAT)
                  *ColorIndex = obj_ptr->color;
            }
            else if (obj_ptr->color != *ColorIndex)
            {
               if (obj_ptr->detail.b->fill != NONEPAT &&
                     obj_ptr->detail.b->fill != BACKPAT ||
                     obj_ptr->detail.b->pen != NONEPAT &&
                     obj_ptr->detail.b->pen != BACKPAT)
                  return (FALSE);
            }
            break;
         case OBJ_OVAL:
            if (*ColorIndex == INVALID)
            {
               if (obj_ptr->detail.o->fill != NONEPAT &&
                     obj_ptr->detail.o->fill != BACKPAT ||
                     obj_ptr->detail.o->pen != NONEPAT &&
                     obj_ptr->detail.o->pen != BACKPAT)
                  *ColorIndex = obj_ptr->color;
            }
            else if (obj_ptr->color != *ColorIndex)
            {
               if (obj_ptr->detail.o->fill != NONEPAT &&
                     obj_ptr->detail.o->fill != BACKPAT ||
                     obj_ptr->detail.o->pen != NONEPAT &&
                     obj_ptr->detail.o->pen != BACKPAT)
                  return (FALSE);
            }
            break;
         case OBJ_TEXT:
            if (*ColorIndex == INVALID)
            {
               if (obj_ptr->detail.t->fill != NONEPAT &&
                     obj_ptr->detail.t->fill != BACKPAT ||
                     obj_ptr->detail.t->pen != NONEPAT &&
                     obj_ptr->detail.t->pen != BACKPAT)
                  *ColorIndex = obj_ptr->color;
            }
            else if (obj_ptr->color != *ColorIndex)
            {
               if (obj_ptr->detail.t->fill != NONEPAT &&
                     obj_ptr->detail.t->fill != BACKPAT ||
                     obj_ptr->detail.t->pen != NONEPAT &&
                     obj_ptr->detail.t->pen != BACKPAT)
                  return (FALSE);
            }
            break;
         case OBJ_POLYGON:
            if (*ColorIndex == INVALID)
            {
               if (obj_ptr->detail.g->fill != NONEPAT &&
                     obj_ptr->detail.g->fill != BACKPAT ||
                     obj_ptr->detail.g->pen != NONEPAT &&
                     obj_ptr->detail.g->pen != BACKPAT)
                  *ColorIndex = obj_ptr->color;
            }
            else if (obj_ptr->color != *ColorIndex)
            {
               if (obj_ptr->detail.g->fill != NONEPAT &&
                     obj_ptr->detail.g->fill != BACKPAT ||
                     obj_ptr->detail.g->pen != NONEPAT &&
                     obj_ptr->detail.g->pen != BACKPAT)
                  return (FALSE);
            }
            break;
         case OBJ_ARC:
            if (*ColorIndex == INVALID)
            {
               if (obj_ptr->detail.a->fill != NONEPAT &&
                     obj_ptr->detail.a->fill != BACKPAT ||
                     obj_ptr->detail.a->pen != NONEPAT &&
                     obj_ptr->detail.a->pen != BACKPAT)
                  *ColorIndex = obj_ptr->color;
            }
            else if (obj_ptr->color != *ColorIndex)
            {
               if (obj_ptr->detail.a->fill != NONEPAT &&
                     obj_ptr->detail.a->fill != BACKPAT ||
                     obj_ptr->detail.a->pen != NONEPAT &&
                     obj_ptr->detail.a->pen != BACKPAT)
                  return (FALSE);
            }
            break;
         case OBJ_RCBOX:
            if (*ColorIndex == INVALID)
            {
               if (obj_ptr->detail.rcb->fill != NONEPAT &&
                     obj_ptr->detail.rcb->fill != BACKPAT ||
                     obj_ptr->detail.rcb->pen != NONEPAT &&
                     obj_ptr->detail.rcb->pen != BACKPAT)
                  *ColorIndex = obj_ptr->color;
            }
            else if (obj_ptr->color != *ColorIndex)
            {
               if (obj_ptr->detail.rcb->fill != NONEPAT &&
                     obj_ptr->detail.rcb->fill != BACKPAT ||
                     obj_ptr->detail.rcb->pen != NONEPAT &&
                     obj_ptr->detail.rcb->pen != BACKPAT)
                  return (FALSE);
            }
            break;
         case OBJ_XBM:
            if (*ColorIndex == INVALID)
            {
               if (obj_ptr->detail.xbm->fill != NONEPAT &&
                     obj_ptr->detail.xbm->fill != BACKPAT)
                  *ColorIndex = obj_ptr->color;
            }
            else if (obj_ptr->color != *ColorIndex)
            {
               if (obj_ptr->detail.xbm->fill != NONEPAT &&
                     obj_ptr->detail.xbm->fill != BACKPAT)
                  return (FALSE);
            }
            break;

         case OBJ_XPM: return (FALSE);

         case OBJ_GROUP:
         case OBJ_SYM:
         case OBJ_ICON:
            if (!OneColorObject (obj_ptr, ColorIndex))
               return (FALSE);
            break;

      }
   return (TRUE);
}

int ChangeObjColor (ObjPtr, ColorIndex)
   struct ObjRec	* ObjPtr;
   int			ColorIndex;
{
   register struct ObjRec	* obj_ptr;
   int				changed=FALSE, icon_color_index;

   switch (ObjPtr->type)
   {
      case OBJ_POLY:
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_TEXT:
      case OBJ_POLYGON:
      case OBJ_ARC:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
         if (ObjPtr->color != ColorIndex)
         {
            ObjPtr->color = ColorIndex;
            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjColor (obj_ptr, ColorIndex))
               changed = TRUE;
         break;

      case OBJ_ICON:
         icon_color_index = INVALID;
         if (OneColorObject (ObjPtr, &icon_color_index) &&
               icon_color_index != ColorIndex)
            for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
                  obj_ptr = obj_ptr->prev)
               if (ChangeObjColor (obj_ptr, ColorIndex))
                  changed = TRUE;
         break;
   }
   return (changed);
}

void ChangeAllSelColor (ColorIndex)
   int	ColorIndex;
{
   register struct SelRec	* sel_ptr;
   int				text_obj_created, text_cursor_shown;
   int				changed=FALSE;
   XGCValues			values;

   if (topSel == NULL || stickyMenuSelection)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      colorIndex = ColorIndex;
      ShowColor (TRUE);
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      if (topSel == NULL) return;
   }

   values.foreground = colorPixels[ColorIndex];
   values.function = GXcopy;
   values.fill_style = FillSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle, &values);

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjColor (sel_ptr->obj, ColorIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      SetFileModified (TRUE);
      RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
   }
   HighLightForward ();
}

void SetUpColorMenuPixmap (fore_colors, init_rv, pixmap, rows, cols)
   int		* * fore_colors, * * init_rv, * rows, * cols;
   Pixmap	* * pixmap;
{
   register int	i;

   *pixmap = (Pixmap *) calloc (maxColors, sizeof (Pixmap));
   *fore_colors = (int *) calloc (maxColors, sizeof(int));
   *init_rv = (int *) calloc (maxColors, sizeof(int));
   for (i = 0; i < maxColors; i++)
   {
      (*pixmap)[i] = patPixmap[SOLIDPAT];
      (*fore_colors)[i] = colorPixels[i];
      (*init_rv)[i] = FALSE;
   }
   *cols = ((maxColors % 10)==0) ? (int)(maxColors/10) : (int)(maxColors/10)+1;
   *rows = (maxColors <= 10) ? maxColors : 10;
}

int ColorMenu (X, Y, TrackMenubar)
   int	X, Y, TrackMenubar;
{
   register int	i;
   Pixmap	* pixmap;
   int		index, * fore_colors, * init_rv, rows, cols;
   char		* * desc;

   if (!colorDisplay) return (INVALID);

   desc = (char **) calloc (maxColors+1, sizeof(char*));
   if (desc == NULL) fprintf (stderr, "Can not caloc().\n");
   for (i=0; i < maxColors; i++)
   {
      desc[i] = (char*) calloc (80, sizeof(char));
      if (desc[i] == NULL) fprintf (stderr, "Can not caloc().\n");
      sprintf (desc[i], "Set color to '%s'", colorMenuItems[i]);
   }
   desc[i] = NULL;
   SetUpColorMenuPixmap (&fore_colors, &init_rv, &pixmap, &rows, &cols);
   activeMenu = MENU_COLOR;
   index = PxMpMenuLoop (X, Y, choiceImageW, choiceImageH, rows, cols,
         maxColors, fore_colors, pixmap, init_rv, desc, MULTICOLOR,
         TrackMenubar);
   cfree (pixmap);

   if (index >= 0) ChangeAllSelColor (index);
   if (desc != NULL)
   {
      for (i=0; i < maxColors; i++)
         if (desc[i] != NULL) cfree (desc[i]);
      cfree (desc);
   }
   return (index);
}

void CleanUpColors ()
{
   register int		i;

   cfree (colorPixels);
   cfree (xorColorPixels);
   cfree (tgifColors);
   cfree (tgifRequestedColors);
   for (i = 0; i < allocatedMaxColors; i++) cfree (colorMenuItems[i]);
   cfree (colorMenuItems);

   maxColors = MAXCOLORS;
   defaultColorIndex = 9;
   colorIndex = 0;
}

static
int UpdatePixel (ObjPtr, OldColorStr)
   struct ObjRec	* ObjPtr;
   char			* * OldColorStr;
{
   register int		c, i, r;
   int			ncolors, new_alloc, index, changed, len;
   int			picture_changed=FALSE;
   char			msg[MAXSTRING];
   unsigned long	pixel, * from_pixels, * to_pixels;
   struct ObjRec	* obj_ptr;
   struct XPmRec	* xpm_ptr;

   switch (ObjPtr->type)
   {
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         for (obj_ptr=ObjPtr->detail.r->last; obj_ptr!=NULL;
               obj_ptr=obj_ptr->prev)
            if (UpdatePixel (obj_ptr, OldColorStr))
               picture_changed = TRUE;
         break;
      case OBJ_XPM:
         changed = FALSE;
         xpm_ptr = ObjPtr->detail.xpm;
         ncolors = xpm_ptr->ncolors;
         from_pixels = (unsigned long *) calloc (ncolors,sizeof(unsigned long));
         to_pixels = (unsigned long *) calloc (ncolors,sizeof(unsigned long));
         for (i = 0; i < ncolors; i++)
         {
            from_pixels[i] = xpm_ptr->pixels[i];
            index = QuickFindColorIndex (xpm_ptr->color_str[i], &new_alloc);
            if (index == INVALID)
            {
               sprintf (msg, "Can not allocate color '%s', use '%s' instead.",
                     xpm_ptr->color_str[i], colorMenuItems[colorIndex]);
               Msg (msg);

               len = strlen (colorMenuItems[colorIndex]);
               cfree (xpm_ptr->color_str[i]);
               xpm_ptr->color_str[i] = (char *) calloc (len+1, sizeof(char));
               strcpy(xpm_ptr->color_str[i], colorMenuItems[colorIndex]);

               xpm_ptr->pixels[i] = colorPixels[colorIndex];
            }
            else
            {
               if (xpm_ptr->pixels[i] != colorPixels[index]) changed = TRUE;
               xpm_ptr->pixels[i] = colorPixels[index];
            }
            to_pixels[i] = xpm_ptr->pixels[i];
         }
         if (changed)
         {
            int		image_w=xpm_ptr->image_w, image_h=xpm_ptr->image_h;
            XImage	* image = xpm_ptr->image;
            Pixmap	pixmap = xpm_ptr->pixmap;

            if (xpm_ptr->cached_pixmap != None)
               XFreePixmap (mainDisplay, xpm_ptr->cached_pixmap);
            xpm_ptr->cached_pixmap = None;
            xpm_ptr->cached_zoom = 0;
            xpm_ptr->cached_rotate = INVALID;

            for (r=0; r<image_h; r++)
               for (c=0; c<image_w; c++)
               {
                  pixel = XGetPixel (image, c, r);
                  for (i=0; i < ncolors; i++)
                     if (from_pixels[i] == pixel)
                     {
                        XPutPixel (image, c, r, to_pixels[i]);
                        break;
                     }
               }
            XPutImage (mainDisplay,pixmap,xpmGC,image,0,0,0,0,image_w,image_h);
            picture_changed = TRUE;
         }
         cfree (from_pixels);
         cfree (to_pixels);
         break;
      default:
         index = QuickFindColorIndex (OldColorStr[ObjPtr->color], &new_alloc);
         if (index != ObjPtr->color) picture_changed = TRUE;
         if (index == INVALID)
         {
            sprintf (msg, "Can not allocate color '%s', use '%s' instead.",
                  OldColorStr[ObjPtr->color], colorMenuItems[colorIndex]);
            Msg (msg);
            ObjPtr->color = colorIndex;
         }
         else
            ObjPtr->color = index;
         break;
   }
   return (picture_changed);
}

static
void UpdateXPmObjects (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   struct ObjRec	* obj_ptr;
   struct XPmRec	* xpm_ptr;

   switch (ObjPtr->type)
   {
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         for (obj_ptr=ObjPtr->detail.r->last; obj_ptr!=NULL;
               obj_ptr=obj_ptr->prev)
            UpdateXPmObjects (obj_ptr);
         break;
      case OBJ_XPM:
         xpm_ptr = ObjPtr->detail.xpm;
         if (xpm_ptr->image == NULL)
            xpm_ptr->image = XGetImage (mainDisplay, xpm_ptr->pixmap, 0, 0,
                  xpm_ptr->image_w, xpm_ptr->image_h, AllPlanes, ZPixmap);
         break;
   }
}

int FlushColormap ()
{
   register int		i;
   int			go_ahead=TRUE, old_allocated_colors, changed=FALSE;
   char			* * old_color_str;
   struct ObjRec	* obj_ptr;

#ifdef DONT_FREE_COLORMAP
   go_ahead = FALSE;
#endif
   if (newColormapUsed)
   {
      if (!go_ahead)
      {
         Msg ("Colormap not reseted due to DONT_FREE_COLORMAP.");
         return (FALSE);
      }
      for (obj_ptr=botObj; obj_ptr!=NULL; obj_ptr=obj_ptr->prev)
         UpdateXPmObjects (obj_ptr);

      old_color_str = (char * *) calloc (maxColors, sizeof(char *));
      old_allocated_colors = maxColors;
      for (i = 0; i < maxColors; i++)
      {
         old_color_str[i] = (char *) calloc (strlen(colorMenuItems[i])+1,
               sizeof(char));
         strcpy (old_color_str[i], colorMenuItems[i]);
      }
      CleanUpColors ();
      XFreeColormap (mainDisplay, mainColormap);
      mainColormap = DefaultColormap (mainDisplay, mainScreen);
      XSetWindowColormap (mainDisplay, mainWindow, mainColormap);
      newColormapUsed = FALSE;
      InitColor ();
      ShowColor (TRUE);

      for (obj_ptr=botObj; obj_ptr!=NULL; obj_ptr=obj_ptr->prev)
         if (UpdatePixel (obj_ptr, old_color_str))
            changed = TRUE;

      for (i = 0; i < old_allocated_colors; i++) cfree (old_color_str[i]);
      cfree (old_color_str);
      if (changed) ClearAndRedrawDrawWindow ();
      return (TRUE);
   }
   return (FALSE);
}
