/***************************************************************************
                             rmmsg.c
                             -------------------
    begin                : May 2001
    copyright            : (C) 2001 by Jorge Allyson Azevedo
                                       Milena Scanferla
                                       Magnos Martinello
                                       Daniel Sadoc
    email                : {allyson,milena,magnos,sadoc}@land.ufrj.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/******************************************************************************************************
 *																									  *
 * Reliable Multicast Protocol (rmmsg.c)															  *
 *																									  *
 * Functions to manipulate the message queue.														  *
 *																									  *
 * Note:	the message queue is a resource of the operating system.  It is used  to establish		  *
 *			the communication between the application that uses the RMP and the RMP itself. 		  *
 *			When a message is available to the application, the message queue is responsible for	  * 
 *			storing that message data until the application makes a requests for it.				  *
 *																									  *
 * use tab spacing = 4																				  *
 *																									  *
 ******************************************************************************************************/

#ifndef MSG_C
#define MSG_C

#include <pthread.h>
#include "rmmsg.h"

int s_qid = -1; 	/* message queue identifier 
					 * -1: queue not initialized
					 * -2: queue removed
					 * otherwise: queue identifier 
					 */

pthread_mutex_t remove_queues_lock = PTHREAD_MUTEX_INITIALIZER;

int proc_obj(struct q_entry *msg)
{
#ifdef DEBUG0
	fprintf(stderr,"\npriority: %ld \n name: %s\n",msg->mtype, msg->mtext);
#endif    
	return 0;
}

/*****************************************************************************************************
 *
 * int init_queue()
 * 
 * Initializes the message queue.
 *
 * Side effects: updates s_qid. 
 *
 ******************************************************************************************************/

int init_queue()
{
	int queue_id;
	
	int QKEY;
	
	QKEY = QKEY_BASE;
	

	while ((queue_id = msgget( QKEY, IPC_CREAT | QPERM | IPC_EXCL )) == -1)
	{
		if	(errno == EEXIST)
		{
#ifdef DEBUG0
            fprintf(stderr,"Message queue key %d already used. Trying %d. \n",
				QKEY,
				QKEY+1);
#endif			
			QKEY++;
		}
		else
		{
			perror("msgget failed");
			exit(1);
		}
	}
	
	s_qid = queue_id;
	
	return(queue_id);
}

/*****************************************************************************************************
 *
 * void remove_queues(int i)
 * 
 * Removes the message queue.
 *
 * Side effects: sets s_qid to -2 on success.
 *
 ******************************************************************************************************/

void remove_queues(int i)
{	 
	
	pthread_mutex_lock(&remove_queues_lock);		

	if (s_qid!=-2)
	{
		if (msgctl(s_qid, IPC_RMID, NULL)!=-1)
		  
			fprintf(stderr,"Queue %d removed.\n",s_qid);
	
		else
	
			fprintf(stderr,"Failed to remove the queue %d.\n",s_qid);
	}
	
	s_qid = -2;


	pthread_mutex_unlock(&remove_queues_lock);

}

/*****************************************************************************************************
 *
 * int messageQueueEnter(char *objname, int len)
 * 
 * Puts a message of size len in the message queue.
 *
 * Arguments:	objname,	the message to be enqueued;
 *				len,		the size of the message.
 * 
 * Return value: 0 on success;
 *				-1 on error.
 *
 ******************************************************************************************************/

int messageQueueEnter(char *objname, int len)
{
	struct q_entry s_entry;
	int aux, ind;
	
	pthread_mutex_lock(&remove_queues_lock);
	
	if (len > MAXOBN)
	{
		fprintf(stderr,"Internal error: Name too long (messageQueueEnter)\n");
		return(-1);
	}
	
	if (s_qid == -1)
		s_qid = init_queue();
	
	if (s_qid == -1 || s_qid == -2)
		return(-1);
		
#ifdef DEBUG2	
	fprintf(stderr,"preparing to send data to app. (messageQueueEnter)\n");
#endif	  
		
	s_entry.mtype = (long)1;

	/* converts len from base 10 to base 256 - least signif. digit is stored in mtext[0] */
	
	for (aux = len, ind=0; aux!=0; aux = aux/256, ind++)
		s_entry.mtext[ind] = aux % 256;
	

	/* FIXME: the following test is really necessary? */
	
	if (ind > MSG_QUEUE_HEADER_SIZE)
	{
			fprintf(stderr,"Error. You must increment the MSG_QUEUE_HEADER_SIZE. ind: %d (messageQueueEnter)\n",ind);
			return(-1);
	}
	
	memcpy(&(s_entry.mtext[MSG_QUEUE_HEADER_SIZE]), objname, len);

#ifdef DEBUG2	 
	fprintf(stderr,"ok\n");
#endif
	
	if (msgsnd(s_qid, &s_entry, len + MSG_QUEUE_HEADER_SIZE, 0) == -1)
	{
		perror("msgsnd failed");
		return(-1);
	}
#ifdef DEBUG0	
	fprintf(stderr,"data sent to app.\n");
#endif    
	
	pthread_mutex_unlock(&remove_queues_lock);
	
	return 0;
}


#endif
