/*
 * Author:      William Chia-Wei Cheng (bill.cheng@acm.org)
 *
 * Copyright (C) 1999, William Chia-Wei Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgifcal" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgifcal" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgifcal", the right to sell or distribute derivative
 * works of "tgifcal", the right to distribute "tgifcal" for a fee, and
 * the right to include "tgifcal" or derivative works of "tgifcal" in a
 * for-sale product) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * @(#)$Header: /mm/src/tgif/v4/tgifcal/RCS/tgifintf.c,v 1.1 1999/03/09 20:15:23 william Exp $
 */

#define _INCLUDE_FROM_TGIFINTF_C_

#include "../tgifdefs.h"

#include "version.h"

#include "../attr.e"
#include "../auxtext.e"
#include "../box.e"
#include "../color.e"
#include "../dialog.e"
#include "../file.e"
#include "../font.e"
#include "../ftp.e"
#include "../grid.e"
#include "../group.e"
#include "../http.e"
#include "../imgproc.e"
#include "../import.e"
#include "../mainloop.e"
#include "../miniline.e"
#include "../move.e"
#include "../msg.e"
#include "../names.e"
#include "../obj.e"
#include "../oval.e"
#include "../pattern.e"
#include "../poly.e"
#include "../setup.e"
#include "../select.e"
#include "../stretch.e"
#include "../text.e"
#include "../util.e"
#include "../xpixmap.e"

static char THIS_TOOL_NAME[]="TGIFCAL";

int	lastFile=TRUE;

short pDrawFontAsc[] = {
    8, 10, 12, 14, 17, 22, 
    8, 10, 12, 14, 17, 22, 
    8, 10, 12, 14, 17, 23, 
    8, 10, 12, 14, 17, 22, 
    8,  9, 11, 13, 15, 19, 
    7,  9, 11, 12, 15, 20, 
    7,  9, 11, 13, 15, 19, 
    7,  9, 11, 12, 15, 20, 
    9, 11, 12, 14, 18, 24, 
    9, 11, 12, 14, 17, 24, 
    9, 11, 12, 14, 17, 24, 
    9, 11, 12, 14, 17, 24, 
    8, 11, 12, 14, 18, 23, 
    8, 11, 12, 15, 18, 24, 
    8, 11, 12, 14, 16, 23, 
    8, 11, 12, 14, 16, 24, 
    8, 10, 12, 14, 18, 24, 
    9, 14, 15, 18, 23, 30, 
   10, 14, 16, 17, 23, 30, 
    9, 13, 15, 17, 22, 30, 
    9, 13, 15, 18, 22, 30, 
    9, 12, 14, 15, 19, 26, 
    9, 12, 14, 15, 21, 26, 
    9, 12, 14, 15, 19, 26, 
    9, 12, 14, 15, 20, 26, 
   11, 14, 16, 18, 24, 31, 
   11, 14, 16, 18, 24, 31, 
   11, 14, 16, 18, 24, 31, 
   11, 14, 16, 18, 24, 31, 
   11, 14, 16, 19, 24, 32, 
   11, 15, 16, 19, 24, 33, 
   11, 14, 16, 18, 23, 32, 
   11, 15, 16, 19, 24, 32, 
   11, 12, 13, 14, 19, 27 
};
short pDrawFontDes[] = {
    2,  3,  3,  4,  4,  6, 
    2,  3,  3,  4,  4,  6, 
    2,  3,  3,  4,  5,  6, 
    2,  3,  3,  3,  4,  6, 
    2,  2,  3,  3,  4,  5, 
    2,  2,  3,  4,  5,  5, 
    2,  2,  3,  3,  4,  5, 
    2,  2,  3,  4,  5,  5, 
    2,  2,  3,  3,  4,  5, 
    2,  2,  3,  3,  5,  5, 
    2,  2,  3,  3,  5,  5, 
    2,  2,  3,  3,  5,  5, 
    2,  2,  3,  3,  4,  5, 
    2,  2,  3,  3,  4,  5, 
    2,  2,  3,  3,  6,  5, 
    2,  2,  3,  3,  6,  5, 
    3,  4,  4,  6,  7,  8, 
    3,  3,  4,  4,  6,  7, 
    3,  3,  4,  4,  6,  7, 
    3,  4,  4,  5,  6,  7, 
    3,  3,  4,  4,  6,  7, 
    2,  3,  3,  4,  5,  6, 
    2,  4,  3,  5,  5,  7, 
    2,  3,  4,  4,  5,  6, 
    2,  4,  4,  5,  5,  7, 
    2,  3,  4,  4,  5,  7, 
    2,  3,  4,  5,  5,  7, 
    2,  3,  4,  5,  5,  7, 
    2,  3,  4,  5,  5,  7, 
    2,  3,  3,  4,  5,  7, 
    2,  3,  3,  4,  5,  7, 
    2,  3,  3,  4,  5,  7, 
    2,  3,  3,  4,  5,  7, 
    4,  3,  4,  5,  5,  7 
};

/* ======================= Init & CleanUp ======================= */

void CleanUpTgifInterface()
{
   CleanUp();
   CleanUpPaperSize();
}

int InitTgifInterface()
{
   int new_alloc=FALSE;

   PRTGIF = TRUE;
   cmdLineOpenDisplay = TRUE;
   InitPaperSize();
   JustInit(NULL, NULL);

   /* things that never change */
   textJust = JUST_L;
   textVSpace = 0;
   penPat = SOLIDPAT;
   lineStyle = LS_RIGHT;
   curDash = 0;
   lineWidth = 0; /* this is the line width index */
   colorIndex = QuickFindColorIndex(NULL, "Black", &new_alloc, TRUE);

   zoomedIn = FALSE;
   zoomScale = 1;

   printMag = (float)180.0;
   UpdPageStyle(PORTRAIT);

   debugHttp = debugFtp = 0;

   return TRUE;
}

/* ======================= Text Justification ======================= */

int TgifTextJust(psz_just)
   char *psz_just;
{
   if (UtilStrICmp(psz_just, "left") == 0) {
      textJust = JUST_L;
   } else if (UtilStrICmp(psz_just, "center") == 0) {
      textJust = JUST_C;
   } else if (UtilStrICmp(psz_just, "right") == 0) {
      textJust = JUST_R;
   } else {
      fprintf(stderr, "\n%s Error in %s - Unrecognized justification: '%s'.\n",
            THIS_TOOL_NAME, "TgifTextJust()", psz_just);
      return FALSE;
   }
   return TRUE;
}

/* ======================= Create Attr ======================= */

int TgifAttr(x, baseline_y, font_name, font_size, center_justified, visible,
      name_visible, attr_name, attr_value, color_str)
   float *x, *baseline_y;
   int font_size, center_justified, visible, name_visible;
   char *font_name, *attr_name, *attr_value, *color_str;
{
   int new_alloc=FALSE, saved_color_index=colorIndex;
   int saved_font=curFont, saved_style=curStyle, saved_sz_unit=curSzUnit;
   struct AttrRec *attr_ptr=NULL;

   if (topObj->type == OBJ_TEXT) {
      fprintf(stderr, "\n%s Error in %s - %s.\n",
            THIS_TOOL_NAME, "TgifAttr()",
            "Cannot create attr when topObj is of type: text");
      return FALSE;
   }
   if (strcmp(font_name, "Courier") == 0) {
      curFont = FONT_COU;
      curStyle = STYLE_NR;
   } else if (strcmp(font_name, "Times-BoldItalic") == 0) {
      curFont = FONT_TIM;
      curStyle = STYLE_BI;
   } else if (strcmp(font_name, "Times-Bold") == 0) {
      curFont = FONT_TIM;
      curStyle = STYLE_BR;
   } else {
      fprintf(stderr, "\n%s Error in %s - Unrecognized font_name: '%s'.\n",
            THIS_TOOL_NAME, "TgifAttr()", font_name);
      return FALSE;
   }
   curSzUnit = FontSizeToSzUnit(font_size);
   SetCanvasFont();

   attr_ptr = AddAttrByNameAndValue(topObj, attr_name, attr_value);
   if (attr_ptr == NULL) {
      return FALSE;
   }
   attr_ptr->obj->detail.t->minilines.first->first_block->seg->color =
         QuickFindColorIndex(NULL, color_str, &new_alloc, TRUE);
   attr_ptr->obj->detail.t->minilines.just =
         (center_justified ? JUST_C : JUST_L);
   attr_ptr->obj->detail.t->fill = NONEPAT;
   attr_ptr->obj->detail.t->pen = SOLIDPAT;
   attr_ptr->shown = visible;
   attr_ptr->nameshown = name_visible;
   RecalcTextMetrics(attr_ptr->obj->detail.t, attr_ptr->obj->x,
         attr_ptr->obj->detail.t->baseline_y);
   MoveObj(attr_ptr->obj, round(*x)-attr_ptr->obj->x,
         round(*baseline_y)-attr_ptr->obj->detail.t->baseline_y);

   UpdAttr(attr_ptr);
   AdjObjBBox(attr_ptr->obj);

   curFont = saved_font;
   curStyle = saved_style;
   curSzUnit = saved_sz_unit;
   SetCanvasFont();
   colorIndex = saved_color_index;

   return TRUE;
}

int TgifCenterAttr(ltx, lty, rbx, rby, font_name, font_size, visible,
      name_visible, attr_name, attr_value, color_str)
   float *ltx, *lty, *rbx, *rby;
   int font_size, visible, name_visible;
   char *font_name, *attr_name, *attr_value, *color_str;
{
   int new_alloc=FALSE, saved_color_index=colorIndex;
   int saved_font=curFont, saved_style=curStyle, saved_sz_unit=curSzUnit;
   float final_ltx=(float)0, final_lty=(float)0;
   struct AttrRec *attr_ptr=NULL;

   if (topObj->type == OBJ_TEXT) {
      fprintf(stderr, "\n%s Error in %s - %s.\n",
            THIS_TOOL_NAME, "TgifAttr()",
            "Cannot create attr when topObj is of type: text");
      return FALSE;
   }
   if (strcmp(font_name, "Courier") == 0) {
      curFont = FONT_COU;
      curStyle = STYLE_NR;
   } else if (strcmp(font_name, "Times-BoldItalic") == 0) {
      curFont = FONT_TIM;
      curStyle = STYLE_BI;
   } else if (strcmp(font_name, "Times-Bold") == 0) {
      curFont = FONT_TIM;
      curStyle = STYLE_BR;
   } else {
      fprintf(stderr, "\n%s Error in %s - Unrecognized font_name: '%s'.\n",
            THIS_TOOL_NAME, "TgifAttr()", font_name);
      return FALSE;
   }
   curSzUnit = FontSizeToSzUnit(font_size);
   SetCanvasFont();

   attr_ptr = AddAttrByNameAndValue(topObj, attr_name, attr_value);
   if (attr_ptr == NULL) {
      return FALSE;
   }
   attr_ptr->obj->detail.t->minilines.first->first_block->seg->color =
         QuickFindColorIndex(NULL, color_str, &new_alloc, TRUE);
   attr_ptr->obj->detail.t->minilines.just = JUST_C;
   attr_ptr->obj->detail.t->fill = NONEPAT;
   attr_ptr->obj->detail.t->pen = SOLIDPAT;
   attr_ptr->shown = visible;
   attr_ptr->nameshown = name_visible;
   UpdAttr(attr_ptr);
   AdjObjBBox(attr_ptr->obj);

   final_ltx = (*ltx) + ((((*rbx)-(*ltx)) -
      (float)(attr_ptr->obj->obbox.rbx-attr_ptr->obj->obbox.ltx)) / 2.0);
   final_lty = (*lty) + ((((*rby)-(*lty)) -
      (float)(attr_ptr->obj->obbox.rby-attr_ptr->obj->obbox.lty)) / 2.0);
   MoveObj(attr_ptr->obj, round(final_ltx)-attr_ptr->obj->obbox.ltx,
         round(final_lty)-attr_ptr->obj->obbox.lty);

   curFont = saved_font;
   curStyle = saved_style;
   curSzUnit = saved_sz_unit;
   SetCanvasFont();
   colorIndex = saved_color_index;

   return TRUE;
}

/* ======================= Create Text ======================= */

int TgifText(x, baseline_y, font_name, font_size, buf, color_str)
   float *x, *baseline_y;
   int font_size;
   char *font_name, *buf, *color_str;
{
   int saved_fill=objFill, saved_color_index=colorIndex, new_alloc=FALSE;
   int saved_font=curFont, saved_style=curStyle, saved_sz_unit=curSzUnit;

   if (strcmp(font_name, "Courier") == 0) {
      curFont = FONT_COU;
      curStyle = STYLE_NR;
   } else if (strcmp(font_name, "Times-BoldItalic") == 0) {
      curFont = FONT_TIM;
      curStyle = STYLE_BI;
   } else if (strcmp(font_name, "Times-Bold") == 0) {
      curFont = FONT_TIM;
      curStyle = STYLE_BR;
   } else {
      fprintf(stderr, "\n%s Error in %s - Unrecognized font_name: '%s'\n",
            THIS_TOOL_NAME, "TgifText()", font_name);
      return FALSE;
   }
   curSzUnit = FontSizeToSzUnit(font_size);
   objFill = NONEPAT;
   colorIndex = QuickFindColorIndex(NULL, color_str, &new_alloc, TRUE);

   SetCanvasFont();
   NewCurText();
   DynStrSet(&topObj->detail.t->minilines.first->first_block->seg->dyn_str,
         buf);
   RecalcTextMetrics(topObj->detail.t, topObj->x, topObj->detail.t->baseline_y);
   UpdTextBBox(topObj);
   MoveObj(topObj, round(*x)-topObj->x,
         round(*baseline_y)-topObj->detail.t->baseline_y);

   curFont = saved_font;
   curStyle = saved_style;
   curSzUnit = saved_sz_unit;
   SetCanvasFont();
   objFill = saved_fill;
   colorIndex = saved_color_index;

   return TRUE;
}

/* ======================= Create Circles ======================= */

int TgifOval(ltx, lty, rbx, rby, color_str)
   float *ltx, *lty, *rbx, *rby;
   char *color_str;
{
   int saved_color_index=colorIndex, new_alloc=FALSE;
   int saved_fill=objFill;
   struct BBRec bbox;

   objFill = NONEPAT;
   colorIndex = QuickFindColorIndex(NULL, color_str, &new_alloc, TRUE);
   bbox.ltx = round(*ltx); bbox.lty = round(*lty);
   bbox.rbx = round(*rbx); bbox.rby = round(*rby);
   CreateOvalObj(&bbox, TRUE);

   objFill = saved_fill;
   colorIndex = saved_color_index;

   return TRUE;
}

int TgifMarkedOval(inside_ltx, inside_lty, inside_rbx, inside_rby,
      ltx, lty, rbx, rby, color_str)
   float *inside_ltx, *inside_lty, *inside_rbx, *inside_rby;
   float *ltx, *lty, *rbx, *rby;
   char *color_str;
{
   struct ObjRec *outside_oval=NULL, *inside_oval=NULL;
   struct BBRec bbox;
   int saved_fill=objFill;
   int saved_color_index=colorIndex, new_alloc=FALSE;

   colorIndex = QuickFindColorIndex(NULL, color_str, &new_alloc, TRUE);
   bbox.ltx = round(*ltx); bbox.lty = round(*lty);
   bbox.rbx = round(*rbx); bbox.rby = round(*rby);
   CreateOvalObj(&bbox, TRUE);
   outside_oval = topObj;
   UnlinkObj(outside_oval);

   objFill = SOLIDPAT;
   bbox.ltx = round(*inside_ltx); bbox.lty = round(*inside_lty);
   bbox.rbx = round(*inside_rbx); bbox.rby = round(*inside_rby);
   CreateOvalObj(&bbox, TRUE);
   objFill = saved_fill;
   inside_oval = topObj;
   UnlinkObj(inside_oval);

   inside_oval->prev = NULL;
   inside_oval->next = outside_oval;
   outside_oval->prev = inside_oval;
   outside_oval->next = NULL;

   CreateGroupObj(inside_oval, outside_oval);

   colorIndex = saved_color_index;

   return TRUE;
}

/* ======================= Create Box ======================= */

int TgifBox(ltx, lty, rbx, rby, solid, rotate, width_spec, color_str)
   float *ltx, *lty, *rbx, *rby;
   int solid, rotate;
   char *width_spec, *color_str;
{
   int saved_color_index=colorIndex, new_alloc=FALSE;
   int saved_fill=objFill;
   struct BBRec bbox;

   objFill = (solid ? SOLIDPAT : NONEPAT);
   colorIndex = QuickFindColorIndex(NULL, color_str, &new_alloc, TRUE);
   bbox.ltx = round(*ltx); bbox.lty = round(*lty);
   bbox.rbx = round(*rbx); bbox.rby = round(*rby);
   CreateBoxObj(bbox.ltx, bbox.lty, bbox.rbx, bbox.rby, TRUE);
   if (rotate != 0) {
      fprintf(stderr, "\n%s Error in %s - %s: %1d (yet).\n",
            THIS_TOOL_NAME, "TgifBox()",
            "Don't know how to handle rotate having value", rotate);
      return FALSE;
   }
   topObj->detail.b->width = atoi(width_spec);
   UtilStrCpyN(topObj->detail.b->width_spec,
         sizeof(topObj->detail.b->width_spec), width_spec);
   objFill = saved_fill;
   colorIndex = saved_color_index;

   return TRUE;
}

/* ======================= Create Poly ======================= */

static int gnNumVertices=0;

int TgifStartVertex(x, y)
   float *x, *y;
{
   if (gnNumVertices != 0) {
      fprintf(stderr, "\n%s Error in %s - %s.\n",
            THIS_TOOL_NAME, "TgifStartVertex()", "gnNumVertices is not 0");
      return FALSE;
   }
   gnNumVertices = 0;
   ResetCreatePoly();
   if (x != NULL && y != NULL) {
      AddPtToCreatePoly(round(*x), round(*y));
      gnNumVertices++;
   }
   return TRUE;
}

int TgifAddVertex(x, y)
   float *x, *y;
{
   AddPtToCreatePoly(round(*x), round(*y));
   gnNumVertices++;

   return TRUE;
}

int TgifPoly(width_spec, arrow_width_spec, arrow_height_spec, color_str)
   char *width_spec, *arrow_width_spec, *arrow_height_spec, *color_str;
{
   int saved_color_index=colorIndex, new_alloc=FALSE;
   int saved_fill=objFill, saved_style=lineStyle, saved_dash=curDash;
   int saved_width=curWidthOfLine[lineWidth];
   int saved_aw=curArrowHeadW[lineWidth];
   int saved_ah=curArrowHeadW[lineWidth];
   char *saved_width_spec=curWidthOfLineSpec[lineWidth];
   char *saved_aw_spec=curArrowHeadWSpec[lineWidth];
   char *saved_ah_spec=curArrowHeadHSpec[lineWidth];
   float width=(float)0, aw=(float)0, ah=(float)0;

   if (gnNumVertices <= 1) {
      fprintf(stderr, "\n%s Error in %s - %s: %1d.\n",
            THIS_TOOL_NAME, "TgifPoly()",
            "Too few points", gnNumVertices);
      return FALSE;
   }
   if (sscanf(width_spec, "%f", &width) != 1) {
      fprintf(stderr, "\n%s Error in %s - %s: '%s'.\n",
            THIS_TOOL_NAME, "TgifPoly()",
            "Fail to evaluate width_spec", width_spec);
      return FALSE;
   }
   if (sscanf(arrow_width_spec, "%f", &aw) != 1) {
      fprintf(stderr, "\n%s Error in %s - %s: '%s'.\n",
            THIS_TOOL_NAME, "TgifPoly()",
            "Fail to evaluate arrow_width_spec", arrow_width_spec);
      return FALSE;
   }
   if (sscanf(arrow_height_spec, "%f", &ah) != 1) {
      fprintf(stderr, "\n%s Error in %s - %s: '%s'.\n",
            THIS_TOOL_NAME, "TgifPoly()",
            "Fail to evaluate arrow_height_spec", arrow_height_spec);
      return FALSE;
   }
   objFill = NONEPAT;
   colorIndex = QuickFindColorIndex(NULL, color_str, &new_alloc, TRUE);
   lineStyle = LS_RIGHT;
   curDash = 0; /* are there other types? */
   curWidthOfLineSpec[lineWidth] = UtilStrDup(width_spec);
   curArrowHeadWSpec[lineWidth] = UtilStrDup(arrow_width_spec);
   curArrowHeadHSpec[lineWidth] = UtilStrDup(arrow_height_spec);
   curWidthOfLine[lineWidth] = width;
   curArrowHeadW[lineWidth] = aw;
   curArrowHeadH[lineWidth] = ah;

   CreatePolyObj(gnNumVertices, TRUE);
   gnNumVertices = 0;

   UtilFree(curWidthOfLineSpec[lineWidth]);
   UtilFree(curArrowHeadWSpec[lineWidth]);
   UtilFree(curArrowHeadHSpec[lineWidth]);
   objFill = saved_fill;
   lineStyle = saved_style;
   curDash = saved_dash;
   curWidthOfLine[lineWidth] = saved_width;
   curArrowHeadW[lineWidth] = saved_aw;
   curArrowHeadH[lineWidth] = saved_ah;
   curWidthOfLineSpec[lineWidth] = saved_width_spec;
   curArrowHeadWSpec[lineWidth] = saved_aw_spec;
   curArrowHeadHSpec[lineWidth] = saved_ah_spec;

   colorIndex = saved_color_index;

   return TRUE;
}

static
int HorizontalSegment(pv0, pv1)
   IntPoint *pv0, *pv1;
{
   static double min_flag_angle=(double)0, max_flag_angle=(double)0;
   static int initialized=FALSE;
   int dx=pv1->x-pv0->x;
   int dy=pv1->y-pv0->y;
   double radian=(double)0;

   if (!initialized) {
      min_flag_angle = (double)((((double)10.0)/((double)180.0))*M_PI);
      max_flag_angle = (double)((((double)170.0)/((double)180.0))*M_PI);
      initialized = TRUE;
   }
   radian = (double)atan2(((double)dy),((double)dx));
   radian = (double)fabs(radian);
   return (radian <= min_flag_angle || radian >= max_flag_angle);
}

#ifdef _NOT_DEFINED
static
int VerticalSegment(pv0, pv1)
   IntPoint *pv0, *pv1;
{
   static double min_flag_angle=(double)0, max_flag_angle=(double)0;
   static int initialized=FALSE;
   int dx=pv1->x-pv0->x;
   int dy=pv1->y-pv0->y;
   double radian=(double)0;

   if (!initialized) {
      min_flag_angle = (double)((((double)80.0)/((double)180.0))*M_PI);
      max_flag_angle = (double)((((double)100.0)/((double)180.0))*M_PI);
      initialized = TRUE;
   }
   radian = (double)atan2(((double)dy),((double)dx));
   radian = (double)fabs(radian);
   return (radian >= min_flag_angle && radian <= max_flag_angle);
}
#endif /* _NOT_DEFINED */

static
double SegmentLength(pv1, pv2)
   IntPoint *pv1, *pv2;
{
   double dx=(double)(pv1->x-pv2->x), dy=(double)(pv1->y-pv2->y);

   return (double)sqrt((double)(dx*dx+dy*dy));
}

static
void GetFractionVector(pv_from, pv_to, fraction, pv_result)
   IntPoint *pv_from, *pv_to, *pv_result;
   double fraction;
{
   double dx=(double)(((double)(pv_to->x-pv_from->x))*fraction);
   double dy=(double)(((double)(pv_to->y-pv_from->y))*fraction);
   double final_x=(double)0, final_y=(double)0;

   final_x = ((double)pv_from->x) + dx;
   final_y = ((double)pv_from->y) + dy;
   pv_result->x = round(final_x);
   pv_result->y = round(final_y);
}

#define HEAD_PIVOT_TOL 24
#define TAIL_PIVOT_TOL 8

static
int GetThreeNewHoriVertices(at_head, pn_num_pts, ppv, psmooth)
   int at_head, *pn_num_pts;
   IntPoint **ppv;
   char **psmooth;
{
   IntPoint *pv=(*ppv), v1, v2, v3, *new_vs=NULL;
   int i, dx=0, dy=0, num_pts=(*pn_num_pts);
   double x=(double)0, pivot_dx=(double)0, x_dir=(double)0;
   double length=(double)0, radius=(double)0;
   char *smooth=(*psmooth);

   if (at_head) {
      dx = pv[num_pts-1].x-pv[num_pts-2].x;
      dy = pv[num_pts-1].y-pv[num_pts-2].y;
      if (abs(dx) >= HEAD_PIVOT_TOL) {
         pivot_dx = ((double)HEAD_PIVOT_TOL);
      } else {
         pivot_dx = fabs((double)dx);
      }
   } else {
      dx = pv[1].x-pv[0].x;
      dy = pv[1].y-pv[0].y;
      if (abs(dy) > abs(dx)) {
         if (abs(dx) >= HEAD_PIVOT_TOL) {
            pivot_dx = ((double)HEAD_PIVOT_TOL);
         } else {
            pivot_dx = fabs((double)dx);
         }
      } else {
         if (abs(dx) >= TAIL_PIVOT_TOL) {
            pivot_dx = ((double)TAIL_PIVOT_TOL);
         } else {
            pivot_dx = fabs((double)dx);
         }
      }
   }
   x_dir = (dx > 0 ? ((double)1) : ((double)(-1)));
   num_pts += 3;
   new_vs = (IntPoint*)malloc((num_pts+1)*sizeof(IntPoint));
   if (new_vs == NULL) FailAllocMessage();
   memset(new_vs, 0, (num_pts+1)*sizeof(IntPoint));

   smooth = (char*)malloc((num_pts+1)*sizeof(char));
   if (smooth == NULL) FailAllocMessage();
   if (at_head) {
      x = ((double)(pv[*pn_num_pts-1].x))-(pivot_dx/2.0*x_dir);
      v3.x = round(x);
      v3.y = pv[*pn_num_pts-1].y;
      x = ((double)(pv[*pn_num_pts-1].x))-(pivot_dx*x_dir);
      v2.x = round(x);
      v2.y = pv[*pn_num_pts-1].y;
      radius = (double)abs(v2.x-v3.x);
      length = SegmentLength(&pv[*pn_num_pts-2], &v2);
      if (length/((double)2.0) > radius+((double)2.0)) {
         GetFractionVector(&v2, &pv[*pn_num_pts-2], (double)(radius/length),
               &v1);
         memcpy(&new_vs[num_pts-4], &v1, sizeof(IntPoint));
         smooth[num_pts-4] = FALSE;
         for (i=0; i < num_pts-4; i++) {
            memcpy(&new_vs[i], &pv[i], sizeof(IntPoint));
            if (*psmooth == NULL) {
               smooth[i] = FALSE;
            } else {
               smooth[i] = (*psmooth)[i];
            }
         }
      } else {
         num_pts--;
         for (i=0; i < num_pts-3; i++) {
            memcpy(&new_vs[i], &pv[i], sizeof(IntPoint));
            if (*psmooth == NULL) {
               smooth[i] = FALSE;
            } else {
               smooth[i] = (*psmooth)[i];
            }
         }
      }
      memcpy(&new_vs[num_pts-3], &v2, sizeof(IntPoint));
      memcpy(&new_vs[num_pts-2], &v3, sizeof(IntPoint));
      memcpy(&new_vs[num_pts-1], &pv[*pn_num_pts-1], sizeof(IntPoint));

      smooth[num_pts-3] = TRUE;
      smooth[0] = smooth[num_pts-2] = smooth[num_pts-1] = FALSE;
   } else {
      x = ((double)(pv[0].x))+(pivot_dx/2.0*x_dir);
      v1.x = round(x);
      v1.y = pv[0].y;
      x = ((double)(pv[0].x))+(pivot_dx*x_dir);
      v2.x = round(x);
      v2.y = pv[0].y;
      radius = (double)abs(v2.x-v1.x);
      length = SegmentLength(&pv[1], &v2);
      if (length/((double)2.0) > radius+((double)2.0)) {
         GetFractionVector(&v2, &pv[1], (double)(radius/length), &v3);
         memcpy(&new_vs[3], &v3, sizeof(IntPoint));
         smooth[3] = FALSE;
         for (i=4; i < num_pts; i++) {
            memcpy(&new_vs[i], &pv[i-3], sizeof(IntPoint));
            if (*psmooth == NULL) {
               smooth[i] = FALSE;
            } else {
               smooth[i] = (*psmooth)[i-3];
            }
         }
      } else {
         num_pts--;
         for (i=3; i < num_pts; i++) {
            memcpy(&new_vs[i], &pv[i-3], sizeof(IntPoint));
            if (*psmooth == NULL) {
               smooth[i] = FALSE;
            } else {
               smooth[i] = (*psmooth)[i-3];
            }
         }
      }
      memcpy(&new_vs[0], &pv[0], sizeof(IntPoint));
      memcpy(&new_vs[1], &v1, sizeof(IntPoint));
      memcpy(&new_vs[2], &v2, sizeof(IntPoint));

      smooth[2] = TRUE;
      smooth[0] = smooth[1] = smooth[num_pts-1] = FALSE;
   }
   *pn_num_pts = num_pts;
   *ppv = new_vs;
   *psmooth = smooth;

   return TRUE;
}

static
int GetTwoNewHoriVertices(at_head, pn_num_pts, ppv, psmooth)
   int at_head, *pn_num_pts;
   IntPoint **ppv;
   char **psmooth;
{
   IntPoint *pv=(*ppv), v1, v2, v3, *new_vs=NULL;
   int i, dx=0, dy=0, num_pts=(*pn_num_pts);
   double x=(double)0, pivot_dx=(double)0, x_dir=(double)0;
   double length=(double)0, radius=(double)0;
   char *smooth=(*psmooth);

   if (at_head) {
      dx = pv[num_pts-1].x-pv[num_pts-2].x;
      dy = pv[num_pts-1].y-pv[num_pts-2].y;
      pivot_dx = fabs((double)dx);
   } else {
      dx = pv[1].x-pv[0].x;
      dy = pv[1].y-pv[0].y;
      pivot_dx = fabs((double)dx);
   }
   x_dir = (dx > 0 ? ((double)1) : ((double)(-1)));
   num_pts += 2;
   new_vs = (IntPoint*)malloc((num_pts+1)*sizeof(IntPoint));
   if (new_vs == NULL) FailAllocMessage();
   memset(new_vs, 0, (num_pts+1)*sizeof(IntPoint));

   smooth = (char*)malloc((num_pts+1)*sizeof(char));
   if (smooth == NULL) FailAllocMessage();
   if (at_head) {
      x = ((double)(pv[*pn_num_pts-1].x))-(pivot_dx/2.0*x_dir);
      v3.x = round(x);
      v3.y = pv[*pn_num_pts-1].y;
      v2.x = pv[*pn_num_pts-2].x;
      v2.y = pv[*pn_num_pts-2].y;
      radius = (double)abs(v2.x-v3.x);
      length = SegmentLength(&pv[*pn_num_pts-3], &v2);
      if (length/((double)2.0) > radius+((double)2.0)) {
         GetFractionVector(&v2, &pv[*pn_num_pts-3], (double)(radius/length),
               &v1);
         memcpy(&new_vs[num_pts-4], &v1, sizeof(IntPoint));
         smooth[num_pts-4] = FALSE;
         for (i=0; i < num_pts-4; i++) {
            memcpy(&new_vs[i], &pv[i], sizeof(IntPoint));
            if (*psmooth == NULL) {
               smooth[i] = FALSE;
            } else {
               smooth[i] = (*psmooth)[i];
            }
         }
      } else {
         num_pts--;
         for (i=0; i < num_pts-3; i++) {
            memcpy(&new_vs[i], &pv[i], sizeof(IntPoint));
            if (*psmooth == NULL) {
               smooth[i] = FALSE;
            } else {
               smooth[i] = (*psmooth)[i];
            }
         }
      }
      memcpy(&new_vs[num_pts-3], &v2, sizeof(IntPoint));
      memcpy(&new_vs[num_pts-2], &v3, sizeof(IntPoint));
      memcpy(&new_vs[num_pts-1], &pv[*pn_num_pts-1], sizeof(IntPoint));

      smooth[num_pts-3] = TRUE;
      smooth[0] = smooth[num_pts-2] = smooth[num_pts-1] = FALSE;
   } else {
      x = ((double)(pv[0].x))+(pivot_dx/2.0*x_dir);
      v1.x = round(x);
      v1.y = pv[0].y;
      v2.x = pv[1].x;
      v2.y = pv[1].y;
      radius = (double)abs(v2.x-v1.x);
      length = SegmentLength(&pv[2], &v2);
      if (length/((double)2.0) > radius+((double)2.0)) {
         GetFractionVector(&v2, &pv[2], (double)(radius/length), &v3);
         memcpy(&new_vs[3], &v3, sizeof(IntPoint));
         smooth[3] = FALSE;
      } else {
         num_pts--;
      }
      memcpy(&new_vs[0], &pv[0], sizeof(IntPoint));
      memcpy(&new_vs[1], &v1, sizeof(IntPoint));
      memcpy(&new_vs[2], &v2, sizeof(IntPoint));
      for (i=4; i < num_pts; i++) {
         memcpy(&new_vs[i], &pv[i-2], sizeof(IntPoint));
         if (*psmooth == NULL) {
            smooth[i] = FALSE;
         } else {
            smooth[i] = (*psmooth)[i-2];
         }
      }
      smooth[2] = TRUE;
      smooth[0] = smooth[1] = smooth[num_pts-1] = FALSE;
   }
   *pn_num_pts = num_pts;
   *ppv = new_vs;
   *psmooth = smooth;

   return TRUE;
}

int TgifSmooth(at_head, horizontal_transition, rotate)
   int at_head, horizontal_transition, rotate;
{
   struct PolyRec *poly_ptr=topObj->detail.p;
   int num_pts=poly_ptr->n;
   IntPoint *pv=poly_ptr->vlist;
   char *smooth=poly_ptr->smooth;

   if (rotate != 0) {
      fprintf(stderr, "\n%s Error in %s - %s: %1d (yet).\n",
            THIS_TOOL_NAME, "TgifSmooth()",
            "Don't know how to handle rotate having value", rotate);
      return FALSE;
   }
   if (horizontal_transition) {
      /* this part not tested */
      fprintf(stderr, "\n%s Error in %s - %s.\n",
            THIS_TOOL_NAME, "TgifSmooth()",
            "Don't know how to handle horizonal transitions");
      return FALSE;
#ifdef _NOT_DEFINED
      if (num_pts == 2) {
         if (VerticalSegment(&pv[0], &pv[1])) {
            /* too thin, no need to smooth */
            return TRUE;
         }
      } else {
      }
#endif /* _NOT_DEFINED */
   } else {
      if (num_pts == 2) {
         if (HorizontalSegment(&pv[0], &pv[1])) {
            /* too flat, no need to smooth */
            return TRUE;
         }
         GetThreeNewHoriVertices(at_head, &num_pts, &pv, &smooth);
      } else {
         if (at_head) {
            if (HorizontalSegment(&pv[num_pts-1], &pv[num_pts-2])) {
               /* just need to adjust */
               GetTwoNewHoriVertices(at_head, &num_pts, &pv, &smooth);
            } else {
               GetThreeNewHoriVertices(at_head, &num_pts, &pv, &smooth);
            }
         } else {
            if (HorizontalSegment(&pv[0], &pv[1])) {
               /* just need to adjust */
               GetTwoNewHoriVertices(at_head, &num_pts, &pv, &smooth);
            } else {
               GetThreeNewHoriVertices(at_head, &num_pts, &pv, &smooth);
               /*
                * Although we are smoothing the tail here, we still
                *       need to look at the head and see if we also need
                *       to smooth it.
                * There is an asymmetry here and it's intensional.
                */
               if (poly_ptr->n != num_pts) {
                  poly_ptr->n = num_pts;
                  if (poly_ptr->vlist != NULL) free(poly_ptr->vlist);
                  if (poly_ptr->smooth != NULL) free(poly_ptr->smooth);
                  poly_ptr->vlist = pv;
                  poly_ptr->smooth = smooth;
                  poly_ptr->style = LT_SPLINE;

                  AdjObjSplineVs(topObj);
                  UpdPolyBBox(topObj, poly_ptr->n, poly_ptr->vlist);
               }
               if (HorizontalSegment(&pv[num_pts-1], &pv[num_pts-2])) {
                  GetTwoNewHoriVertices(TRUE, &num_pts, &pv, &smooth);
               }
            }
         }
      }
      if (poly_ptr->n != num_pts) {
         poly_ptr->n = num_pts;
         if (poly_ptr->vlist != NULL) free(poly_ptr->vlist);
         if (poly_ptr->smooth != NULL) free(poly_ptr->smooth);
         poly_ptr->vlist = pv;
         poly_ptr->smooth = smooth;
         poly_ptr->style = LT_SPLINE;

         AdjObjSplineVs(topObj);
         UpdPolyBBox(topObj, poly_ptr->n, poly_ptr->vlist);
      }
   }
   return TRUE;
}

/* ======================= TgifImport... ======================= */

#define TOL_ASPECT_RATIO (0.04)

static
int FinishImport(path, x, y, w, h, gamma, fit)
   char *path;
   float *x, *y, *w, *h;
   char *gamma;
   int fit;
{
   int nW=round(*w), nH=round(*h), final_x=0, final_y=0, final_w=0, final_h=0;
   float orig_aspect_ratio=(float)0, aspect_ratio=(*w)/(*h);

   orig_aspect_ratio = ((float)(topObj->detail.xpm->image_w)) /
         ((float)(topObj->detail.xpm->image_h));
   final_w = nW;
   final_h = nH;
   final_x = round(*x);
   final_y = round(*y);
   if (fit) {
      if (fabs(orig_aspect_ratio-aspect_ratio) > TOL_ASPECT_RATIO) {
         fprintf(stderr, "%s: aspect ratio adjusted from %.2f to %.2f\n",
               THIS_TOOL_NAME, orig_aspect_ratio, aspect_ratio);
      }
   } else {
      if (topObj->detail.xpm->image_w == nW &&
            topObj->detail.xpm->image_h == nH) {
      } else if (orig_aspect_ratio > aspect_ratio) {
         final_w = nW;
         final_h = round(((float)nW) / orig_aspect_ratio);
         final_x = round(*x);
         final_y = round(*y) + ((nH-final_h)>>1);
      } else {
         final_h = nH;
         final_w = round(((float)nH) * orig_aspect_ratio);
         final_y = round(*y);
         final_x = round(*x) + ((nW-final_w)>>1);
      }
   }
   SelectTopObj();
   if (topObj->detail.xpm->image_w != nW || topObj->detail.xpm->image_h != nH) {
      fprintf(stderr, "%s: resizing image...\n", THIS_TOOL_NAME);
      SizeAllSelToGivenWidthHeight(final_w, final_h);
      MoveObj(topObj, final_x-topObj->x, final_y-topObj->y);
   }
   if (gamma != NULL && *gamma != '\0') {
      fprintf(stderr, "%s: applying gamma correction...\n", THIS_TOOL_NAME);
      Gamma(gamma);
   }
   MakeQuiescent();
   AdjObjBBox(topObj);

   return TRUE;
}

int TgifImportGIF(path, x, y, w, h, gamma, fit)
   char *path;
   float *x, *y, *w, *h;
   char *gamma;
   int fit;
{
   int saved_color_index=colorIndex, rc=TRUE;

   fprintf(stderr, "%s: importing image...\n", THIS_TOOL_NAME);
   if (!ImportSpecifiedFileType(path, "GIF")) {
      fprintf(stderr, "Fail to import '%s'.\n", path);
      return FALSE;
   }
   rc = FinishImport(path, x, y, w, h, gamma, fit);
   colorIndex = saved_color_index;

   return rc;
}

int TgifImportJPEG(path, x, y, w, h, gamma, fit)
   char *path;
   float *x, *y, *w, *h;
   char *gamma;
   int fit;
{
   int saved_color_index=colorIndex, rc=TRUE;

   fprintf(stderr, "%s: importing image...\n", THIS_TOOL_NAME);
   if (!ImportSpecifiedFileType(path, "JPEG")) {
      fprintf(stderr, "Fail to import '%s'.\n", path);
      return FALSE;
   }
   rc = FinishImport(path, x, y, w, h, gamma, fit);
   colorIndex = saved_color_index;

   return rc;
}

/* ======================= Generate ======================= */

int TgifGenerate(fp, psz_format)
   FILE *fp;
   char *psz_format;
{
   if (UtilStrICmp(psz_format, "obj") == 0) {
      fprintf(stderr, "%s: generating tgif file...\n", THIS_TOOL_NAME);
      Save(fp, botObj, 0, 1);
      fprintf(stderr, "%s: Done.\n", THIS_TOOL_NAME);
   } else if (UtilStrICmp(psz_format, "ps") == 0) {
      char saved_print_command[MAXSTRING+1];

      UtilStrCpyN(saved_print_command, sizeof(saved_print_command),
            printCommand);
      PRTGIF = FALSE;
      whereToPrint = PRINTER;
      colorDump = TRUE;
      strcpy(printCommand, "cat");
      fprintf(stderr, "%s: generating PS file...\n", THIS_TOOL_NAME);
      Dump("");
      fprintf(stderr, "%s: Done.\n", THIS_TOOL_NAME);
      strcpy(printCommand, saved_print_command);
      PRTGIF = TRUE;
   } else {
      fprintf(stderr, "Unrecognized format '%s'.\n",
            (psz_format==NULL ? "(unknown)" : psz_format));
      return FALSE;
   }
   return TRUE;
}

/* ----------------------- Utility Functions ----------------------- */

void TgifFree(pszStr)
   char *pszStr;
{
   UtilFree(pszStr);
}

int TgifStrCpyN(pszDest, nMaxDestSz, pszSrc)
   char *pszDest, *pszSrc;
   int nMaxDestSz;
{
   return UtilStrCpyN(pszDest, nMaxDestSz, pszSrc);
}

void TgifTrimBlanks(pszStr)
   char *pszStr;
{
   UtilTrimBlanks(pszStr);
}

void TgifStrLower(pszStr)
   char *pszStr;
{
   UtilStrLower(pszStr);
}

char *TgifStrRChr(pszStr, int_ch)
   char *pszStr; 
   int int_ch;
{
   return UtilStrRChr(pszStr, int_ch);
}

char *TgifGetALine(lpFile)
   FILE *lpFile;
{
   return UtilGetALine(lpFile);
}

int TgifPathExists(pszPath)
   char *pszPath;
{
   return UtilPathExists(pszPath);
}

/* ======================= TgifGetCurDir ======================= */

int TgifGetCurDir(buf, buf_sz)
   char *buf;
   int buf_sz;
{
   UtilStrCpyN(buf, buf_sz, curDir);
   return TRUE;
}

/* ======================= TgifDownloadURL ======================= */

int TgifDownloadURL(psz_url, tmp_fname)
   char *psz_url, *tmp_fname;
{
   int is_html=FALSE;

   *tmp_fname = '\0';
   if (!DownloadRemoteFile(psz_url, NULL, NULL, &is_html, tmp_fname,
         NULL, 0) || *tmp_fname == '\0') {
      return FALSE;
   }
   return TRUE;
}
