
#include <stdio.h>
#include "frame.h"
#include "vector.h"
#include "povproto.h"
#include "colour.h"
#include "objects.h"
#include "povray.h"
#include "ray.h"
#include "matrices.h"
#include "render.h"
#include "texture.h"
#include "pigment.h"

#ifdef PostProcessPatch
#include "postproc.h"

extern char Actual_Output_Name[FILE_NAME_LENGTH];
static void setup_output_file_name(char *filename);

static void FreePostProcessTasks(void);

static FILE* post_file;
static PP_DATA_LISTS Post_Lines;
static COLOUR **Input_Lines;
static COLOUR **Output_Lines;
static static_height;


#define ALLOCATE_GRID(name, t, h, w) {int y; name = \
  (t**)POV_MALLOC((h) * sizeof(t*),"post process"); \
  for(y=0; y<h; y++) \
    name[y] = (t*)POV_MALLOC((w)*sizeof(t),"post process");}

#define FREE_GRID(name, h) {int y; if(name){ \
  for(y=0; y<h; y++) \
    POV_FREE(name[y]); \
  POV_FREE(name); \
  name=NULL;}}

/*****************************************************************************

 FUNCTION

  void Initialize_PostProcessing(void)
    Initializes pointer so that we know if we need to free memory after an
    interruptoin

  Preconditions:
    DoPostProcess was not started without finishing

  Postconditions:
    various pointers are set to NULL

******************************************************************************/
void Initialize_PostProcessing(void)
{

 static_height = 0;
  Input_Lines=NULL;
  Output_Lines=NULL;
  post_file=NULL;
  /* only allocate what we need */
  Post_Lines.Depth = NULL;
  Post_Lines.Colour = NULL;
  Post_Lines.INormal =  NULL;
  Post_Lines.PNormal = NULL;
  Post_Lines.Iuv = NULL;
  Post_Lines.IPoint = NULL;
  Post_Lines.Object = NULL;
#ifdef PostProcessKeepDataPatch
  strcpy(opts.post_process_data_file,opts.Output_Path);
  strcat(opts.post_process_data_file,opts.Scene_Name);
  strcat(opts.post_process_data_file,".ppd");
#endif
}

/*****************************************************************************

 FUNCTION

  void Deinitialize_PostProcessing(void)
    frees memory (if necessary), closes the post-data file (if necessary),
    and frees the linked-list of post-process tasks

  Preconditions:
    Initialize_PostProcessing was called at some point prior to this.
    If any of the pointer were allocated and freed, they should have been
    set to null so that we don't accidently free them again.

  Postconditions:
    all post-processing data is cleaned up, memory is freed, and pointers
    set to zero

******************************************************************************/
void Deinitialize_PostProcessing(void)
{
  int height=static_height;

  FreePostProcessTasks();
  Close_Post_File();

  if (height)
  {
    /* free this stuff to handle interruptions */
    FREE_GRID(Input_Lines, height);
    FREE_GRID(Output_Lines, height);
    FREE_GRID(Post_Lines.Depth, height);
    FREE_GRID(Post_Lines.Colour, height);
    FREE_GRID(Post_Lines.IPoint, height);
    FREE_GRID(Post_Lines.INormal, height);
    FREE_GRID(Post_Lines.PNormal, height);
    FREE_GRID(Post_Lines.Iuv, height);
    FREE_GRID(Post_Lines.Object, height);
  }
}

/*****************************************************************************

 FUNCTION

  void Open_Post_File()
    opens the post-processing file (static post_file) for
    write

  Preconditions:
    Initialize_PostProcessing was called at some point prior to this.
    The file "static post_file" is not currently open.
    opts.postProcessFlags contains flags for all necessary post-data.
    If post processing will be done, opts.postProcess is not NULL.

  Postconditions:
    the post-processing file (static variable post_file) is open for write

******************************************************************************/
void Open_Post_File()
{
  if (post_file) fclose(post_file); /* just in case */

  if (!(opts.Options & DISKWRITE) ||  /* won't be able to read image */
      !(opts.postProcess) ||          /* no post processing tasks */
      !(opts.postProcessFlags))       /* no need for any extra data */
  {
    post_file = NULL;
    return;
  }
#ifdef PostProcessKeepDataPatch
  if (opts.Options & CONTINUE_TRACE)
    post_file = fopen(opts.post_process_data_file,APPEND_BINFILE_STRING);
  else
    post_file = fopen(opts.post_process_data_file,WRITE_BINFILE_STRING);

#else
  if (opts.Options & CONTINUE_TRACE)
    post_file = fopen("postdata.tmp",APPEND_BINFILE_STRING);
  else
    post_file = fopen("postdata.tmp",WRITE_BINFILE_STRING);
#endif
}

/*****************************************************************************

 FUNCTION

  void Open_Post_Input()
    opens the post-processing file (static variable post_file) for
    read

  Preconditions:
    Initialize_PostProcessing was called at some point prior to this.
    The file "static post_file" is not currently open.
    opts.postProcessFlags contains flags for all necessary post-data.
    If post processing will be done, opts.postProcess is not NULL.

  Postconditions:
    the post-processing file (static variable post_file) is open for read

******************************************************************************/
void Open_Post_Input()
{
  if (post_file) 	fclose(post_file); /* just in case */
  
  if (!(opts.Options & DISKWRITE) ||  /* won't be able to read image */
      !(opts.postProcess) ||          /* no post processing tasks */
      !(opts.postProcessFlags))       /* no need for any extra data */
  {
    post_file = NULL;
    return;
  }


#ifdef PostProcessKeepDataPatch
  post_file = fopen(opts.post_process_data_file,READ_BINFILE_STRING);
	if ( post_file == NULL )
		Error("No post process data file found!\n");

	if( fseek(post_file,1,SEEK_SET) != 0)
		Error("The post process data file is empty! Use 'keep_data_file' in the post_process block { }\n to allow the use of the continue option!\n");
	rewind(post_file);
#else
  post_file = fopen("postdata.tmp",READ_BINFILE_STRING);
#endif
}

/*****************************************************************************

 FUNCTION

  void Write_Post_Line(PP_DATA *Line, int width)
    writes a line of post-process data to static post_file

  Preconditions:
    Initialize_PostProcessing was called at some point prior to this.
    The file "static post_file" is open for write.
    opts.postProcessFlags contains flags for all necessary post-data.

    Line is a single-dim array that contains the post-process data for a
      single horizontal line of the image.
    width is the size of the array (the width of the image)

  Postconditions:
    necessary data (indicated by opts.postProcessFlags) is written to
    "static post_file"

******************************************************************************/
void Write_Post_Line(PP_DATA *Line, int width)
{
  int x;

  if (!post_file) return;

  /* only write the stuff we need */
  for(x=0; x<width; x++)
  {
    if (opts.postProcessFlags & PP_FLAG_DEPTH)
      fwrite(&Line[x].Depth,sizeof(DBL),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_COLOUR)
      fwrite(&Line[x].Colour,sizeof(COLOUR),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_IPOINT)
      fwrite(&Line[x].IPoint,sizeof(VECTOR),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_INORMAL)
      fwrite(&Line[x].INormal,sizeof(VECTOR),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_PNORMAL)
      fwrite(&Line[x].PNormal,sizeof(VECTOR),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_IUV)
      fwrite(&Line[x].Iuv,sizeof(UV_VECT),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_OBJECT)
      fwrite(&Line[x].Object,sizeof(int),1,post_file);
  }
}

/*****************************************************************************

 FUNCTION

  void Read_Post_Line(PP_DATA_LISTS Data, int y, int width)
    reads a line of post-process data from static post_file

  Preconditions:
    Initialize_PostProcessing was called at some point prior to this.
    The file "static post_file" is open for read.
    opts.postProcessFlags contains flags for all necessary post-data.
      These flags must be exactly the same flags that were set when the
      file was written, or the data will be corrupt.

    Data is a struct that contains pointers to grids (2-dim arrays)
      for receiving the post-process data
    width the width of the image (width of the 2-dim arrays)
    y indicates which line (in Line) the data is to be stored in

  Postconditions:
    necessary data (indicated by opts.postProcessFlags) is read from
    "static post_file" and 

******************************************************************************/
void Read_Post_Line(PP_DATA_LISTS Data, int y, int width)
{
  int size;
  int x;

  if (!post_file) return;

  /* only read the stuff we need */
  for(x=0; x<width; x++)
  {
    if (opts.postProcessFlags & PP_FLAG_DEPTH)
      size = fread(&Data.Depth[y][x],sizeof(DBL),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_COLOUR)
      size = fread(&Data.Colour[y][x],sizeof(COLOUR),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_IPOINT)
      size = fread(&Data.IPoint[y][x],sizeof(VECTOR),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_INORMAL)
      size = fread(&Data.INormal[y][x],sizeof(VECTOR),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_PNORMAL)
      size = fread(&Data.PNormal[y][x],sizeof(VECTOR),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_IUV)
      size = fread(&Data.Iuv[y][x],sizeof(UV_VECT),1,post_file);
    if (opts.postProcessFlags & PP_FLAG_OBJECT)
      size = fread(&Data.Object[y][x],sizeof(int),1,post_file);
  }
}

/*****************************************************************************

 FUNCTION

  void Close_Post_File()
    closes the post-processing file (static post_file)

  Preconditions:
    Initialize_PostProcessing was called at some point prior to this.
    The file "static post_file" is open.

  Postconditions:
    the post-processing file (static variable post_file) is closed, and
    the variable is set to NULL

******************************************************************************/
void Close_Post_File()
{
  if (!(opts.Options & DISKWRITE)) return;
  if (!post_file) return;

  fclose(post_file);
  post_file = NULL;

}

/*****************************************************************************

 FUNCTION

  void Delete_Post_File()
    deletes the temporary post-processing file

  Preconditions:
    Initialize_PostProcessing was called at some point prior to this.
    The file "static post_file" is not open.

  Postconditions:
    the file "postdata.tmp" is deleted

******************************************************************************/
void Delete_Post_File()
{
#ifdef PostProcessKeepDataPatch
	if ( !opts.keep_post_process_data_file)
  		DELETE_FILE(opts.post_process_data_file);
#else
	DELETE_FILE("postdata.tmp");
 #endif
}

/*****************************************************************************

 FUNCTION

  void Assign_PP_Data(PP_DATA *pp_data, INTERSECTION *Inter, 
                      COLOUR Colour, DBL depth)
    assigns data to the PP_DATA struct

  Preconditions:
    Inter points to the intersection continaing data to be stored
    Colour is the true color of the pixel (before gamma-correct and clipping)
    depth is the depth of the trace ray

  Postconditions:
    the data in pp_data has been filled in from the data provided by the
      input variables

******************************************************************************/
void Assign_PP_Data(PP_DATA *pp_data, INTERSECTION *Inter, COLOUR Colour, DBL depth)
{
  Assign_Colour(pp_data->Colour, Colour);
  Assign_Vector(pp_data->IPoint, Inter->IPoint);
  Assign_Vector(pp_data->INormal ,Inter->INormal);
  Assign_UV_Vect(pp_data->Iuv ,Inter->Iuv);
  Assign_Vector(pp_data->PNormal ,Inter->PNormal);
  pp_data->Depth = depth;

  /* pp_data->Object = Inter->Object->Post_Process_Id; */
}

/*****************************************************************************

 FUNCTION

  void DoPostProcess(char* filename)
    Post-process the file "filename" using the data in "postdata.tmp"
    and the tasks in opts.postProcess

  Preconditions:
    Initialize_PostProcessing was called at some point prior to this.
    The file "static post_file" is not open.
    filename is the full path+filename for the output image
    Output_File_Handle (static) is the handle used to create that file
    opts.postProcess points to the start of a linked list of post tasks
    opts.postProcessFlags indicates what kind of data is necessary for all
      of the tasks in the linked list
    that data (and only that data) was saved in the file "postdata.tmp" 
      during ray-tracing (using Open/Write/Close with thos same flags set)

  Postconditions:
    the image is post-processed using the tasks in opts.postProcess

******************************************************************************/

void DoPostProcess(char* filename)
{
  int x,y,tmp;
  int width, height;
  PP_TASK *pp_task;
  COLOUR **temp_swap;

  if (!opts.postProcess) return;

  COOPERATE_0
  /* YS added this to refresh the image window for MacMegaPOV
      because not every pixel is drawn and therefore a large portion
      of the image is not visible while doing post proc.
      Should be defined in config.h
  */
  REDRAW_IMAGE_WINDOW		
  /* open image */
if (  Open_File(Output_File_Handle, filename, &width, &height, 0, READ_MODE) !=1)
     Error ("Error opening post process input file.");

  /* save the size of all of this data in case we are interrupted */
  static_height = height;

  /* allocate memory */
  ALLOCATE_GRID(Input_Lines, COLOUR, height, width);
  ALLOCATE_GRID(Output_Lines, COLOUR, height, width);

  /* only allocate what we need */
  Post_Lines.Depth = NULL;
  Post_Lines.Colour = NULL;
  Post_Lines.INormal =  NULL;
  Post_Lines.PNormal = NULL;
  Post_Lines.Iuv = NULL;
  Post_Lines.IPoint = NULL;
  Post_Lines.Object = NULL;

  if (opts.postProcessFlags & PP_FLAG_DEPTH)
    ALLOCATE_GRID(Post_Lines.Depth, DBL, height, width);
  if (opts.postProcessFlags & PP_FLAG_COLOUR)
    ALLOCATE_GRID(Post_Lines.Colour, COLOUR, height, width);
  if (opts.postProcessFlags & PP_FLAG_IPOINT)
    ALLOCATE_GRID(Post_Lines.IPoint, VECTOR, height, width);
  if (opts.postProcessFlags & PP_FLAG_INORMAL)
    ALLOCATE_GRID(Post_Lines.INormal, VECTOR, height, width);
  if (opts.postProcessFlags & PP_FLAG_PNORMAL)
    ALLOCATE_GRID(Post_Lines.PNormal, VECTOR, height, width);
  if (opts.postProcessFlags & PP_FLAG_IUV)
    ALLOCATE_GRID(Post_Lines.Iuv, UV_VECT, height, width);
  if (opts.postProcessFlags & PP_FLAG_OBJECT)
    ALLOCATE_GRID(Post_Lines.Object, int, height, width);

  /* read image */
  tmp = -1; /*fixed this for Macintosch Pict format */
            /* was 0. Tip from Thorsten Froehlich*/
  for(y=0; y<height; y++)
  {
    COOPERATE_1
    Read_Line(Output_File_Handle, Input_Lines[y], &tmp);
  }
  Close_File(Output_File_Handle);

  COOPERATE_0

  /* open post-data for input */
  Open_Post_Input();
  for(y=0; y<height; y++)
  {
    COOPERATE_1
    Read_Post_Line(Post_Lines, y, width);
  }
  Close_Post_File();

  COOPERATE_0

  /* post-process the image */
  for(pp_task = (PP_TASK*)opts.postProcess; pp_task != NULL; pp_task=pp_task->next)
  {
    (*(pp_task->doPostProcess))(pp_task,Input_Lines,Output_Lines,Post_Lines,height,width);

    /* if not last node, then swap */
    if (pp_task->next)
    {
      temp_swap = Input_Lines;
      Input_Lines = Output_Lines;
      Output_Lines = temp_swap;
    }
  }

  /* the data to output is in Input_Lines, since it was swapped */

  COOPERATE_0

  /* write the image file and update the GUI display */
  if(!opts.postProcessOverwriteOriginal)
  {
    setup_output_file_name(filename);
  }
  MAKEFILESPEC(filename);
  if (Open_File(Output_File_Handle, filename, &width, &height, 0, WRITE_MODE) != 1)
  {
     Error ("Error opening post process output file.");
  }
  CHANGEFILETYPE

  for(y=0; y<height; y++)
  {
    Write_Line(Output_File_Handle, Output_Lines[y], y);

    /* do GUI output */
    for(x=0; x<width; x++)
    {
      POV_ASSIGN_PIXEL (x, y, Output_Lines[y][x]);
      plot_pixel(x, y, Output_Lines[y][x]);
      COOPERATE_1
    }

    POV_WRITE_LINE (Output_Lines[y], y)
  }
  Close_File(Output_File_Handle);

  COOPERATE_0

  FREE_GRID(Input_Lines, height);
  FREE_GRID(Output_Lines, height);

  FREE_GRID(Post_Lines.Depth, height);
  FREE_GRID(Post_Lines.Colour, height);
  FREE_GRID(Post_Lines.IPoint, height);
  FREE_GRID(Post_Lines.INormal, height);
  FREE_GRID(Post_Lines.PNormal, height);
  FREE_GRID(Post_Lines.Iuv, height);
  FREE_GRID(Post_Lines.Object, height);

}


static void setup_output_file_name(char *filename)
{
  char separator_string[2] = {FILENAME_SEPARATOR, 0} ;
  char *plast_period;
  int available_characters;
  int ilast_period;
  int fname_chars;
  char tempOut[FILE_NAME_LENGTH];

  available_characters = POV_NAME_MAX-2;  /* we will add PP to the name */
  plast_period = strrchr(opts.Output_Numbered_Name, '.');
  if (plast_period == NULL)
  {
    Error("Illegal file name %s -- no extension.\n", opts.Output_Numbered_Name);
  }
  ilast_period = plast_period - opts.Output_Numbered_Name;
  fname_chars = ilast_period;
  if (fname_chars > available_characters)
  {
     Warning(0, "Need to cut the output filename by %d characters.\n",   ilast_period - available_characters);
     fname_chars = available_characters;
  }
  /* Perform actual generation of filename */
  strncpy(tempOut, opts.Output_Numbered_Name, (unsigned)fname_chars);
  /* strncpy doesn't terminate if strlen(opts.Output_File_Name)<fname_chars */
  tempOut[fname_chars]='\0';
  strcat(tempOut, "PP");
  strcat(tempOut, &opts.Output_Numbered_Name[ilast_period]);
  if (strlen (opts.Output_Path) == 0)
  {
    getcwd (opts.Output_Path, sizeof (opts.Output_Path) - 1) ;
    /* on some systems (MacOS) getcwd adds the path separator on the end */
    /* so only add it if it isn't already there...  [esp]                */
    if (opts.Output_Path[strlen(opts.Output_Path)-1] != FILENAME_SEPARATOR)
        strcat (opts.Output_Path, separator_string) ;
  }
  strncpy (filename,opts.Output_Path, FILE_NAME_LENGTH);
  strncat (filename,tempOut, FILE_NAME_LENGTH);
}



/*****************************************************************************

 FUNCTION

  static void FreePostProcessTasks(void)
    frees the linked list of tasks stored in opts.postProcess

  Preconditions:
    if opts.postProcess is not NULL, then it points to the start of
      a linked list that contains post-processing tasks
    the "next" pointer in the last node of the list is NULL

  Postconditions:
    the linked list is freed, and opts.postProcess is set to NULL

******************************************************************************/
static void FreePostProcessTasks(void)
{
  /* free the post-process list */
  PP_TASK *pp_task = (PP_TASK*)opts.postProcess;
  while(pp_task != NULL)
  {
    PP_TASK *pp_task2 = pp_task;
    pp_task=pp_task->next;
#ifdef HuffPostProcessPatches
    (*(pp_task2->DestroyPostProcess))(pp_task2);
#endif
    POV_FREE(pp_task2);
  }
  opts.postProcess = NULL;
}



#ifdef PostProcessPatternBlurPatch
/* ============================================================ */
/*  pattern_blur post process */
/* ============================================================ */

PP_PATTERN_BLUR *createPostPatternBlur(void)
{
  PP_PATTERN_BLUR *New;
  New = (PP_PATTERN_BLUR *)POV_MALLOC(sizeof(PP_PATTERN_BLUR),"pattern_blur post process");
  New->next = NULL;
  New->doPostProcess = doPostPatternBlur;
  New->DestroyPostProcess = DestroyPostPatternBlur;
  
  New->Radius = 0.2;
  New->Levelling = 0;
  New->Pig = Create_Pigment();
  
  return New;
}

void DestroyPostPatternBlur(PP_TASK * task)
{
    PP_PATTERN_BLUR * Task = (PP_PATTERN_BLUR *)task;
    if(Task->Pig != NULL)
    	Destroy_Pigment (Task->Pig);
}

void doPostPatternBlur(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_PATTERN_BLUR * Task;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'pattern_blur' effect: 0%%\r");
#else
  Status_Info("\nApplying 'pattern_blur' effect: 0%%");
#endif
  Task = (PP_PATTERN_BLUR *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'pattern_blur' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'pattern_blur' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        DBL tmpR = Task->Levelling;
        DBL tmpG = Task->Levelling;
        DBL tmpB = Task->Levelling;
        int j,l;
        int k=0;
        COLOUR Weight;
        VECTOR Loc;
        
        Loc[0] = (DBL)x/width;
        Loc[1] = (DBL)y/height;
        Loc[2] = 0;
	    Compute_Pigment (Weight, Task->Pig, Loc, NULL);
        
        for(j=-Task->Radius*Weight[0]; j<=Task->Radius*Weight[0]; j++)
        {
            for(l=-Task->Radius*Weight[0]; l<=Task->Radius*Weight[0]; l++)
            {
                if(((x+l >= 0)&&(x+l < width)) && (((y+j >= 0)&&(y+j < height))))
                {
                    tmpR += in[y+j][x+l][0];
                    tmpG += in[y+j][x+l][1];
                    tmpB += in[y+j][x+l][2];
                }
                k++;
/*                if((sqrt(sqr(j)+sqr(l)) < pp_edges->Radius))*/
            }
        }
        tmpR /= Task->Div + k;
        tmpG /= Task->Div + k;
        tmpB /= Task->Div + k;
        
        out[y][x][0]=tmpR;
        out[y][x][1]=tmpG;
        out[y][x][2]=tmpB;
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
        
    }
  }
}
#endif


#ifdef PostProcessStarsPatch

/* ============================================================ */
/*  stars post process */
/* ============================================================ */

PP_STARS *createPostStars(void)
{
  PP_STARS *New;
  New = (PP_STARS *)POV_MALLOC(sizeof(PP_STARS),"stars post process");
  New->next = NULL;
  New->doPostProcess = doPostStars;
  New->DestroyPostProcess = DestroyPostStars;
  
  New->colRangeMin[0] = 0;
  New->colRangeMin[1] = 0;
  New->colRangeMin[2] = 0;
  New->colRangeMin[3] = 0;
  New->colRangeMin[4] = 0;
  
  New->colRangeMax[0] = 1;
  New->colRangeMax[1] = 1;
  New->colRangeMax[2] = 1;
  New->colRangeMax[3] = 0;
  New->colRangeMax[4] = 0;
  
  New->Density = 0.2;
  
  return New;
}

void DestroyPostStars(PP_TASK * task)
{
}

void doPostStars(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_STARS * Task;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'stars' effect: 0%%\r");
#else
  Status_Info("\nApplying 'stars' effect: 0%%");
#endif
  Task = (PP_STARS *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'stars' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'stars' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        if(Post_Lines.Depth[y][x]>=Max_Distance)
        {
            if(FRAND() < Task->Density)
            {
                out[y][x][0] = FRAND()*(Task->colRangeMax[0] - Task->colRangeMin[0]) + Task->colRangeMin[0];
                out[y][x][1] = FRAND()*(Task->colRangeMax[1] - Task->colRangeMin[1]) + Task->colRangeMin[1];
                out[y][x][2] = FRAND()*(Task->colRangeMax[2] - Task->colRangeMin[2]) + Task->colRangeMin[2];
            }
            else
            {
                out[y][x][0] = 0;
                out[y][x][1] = 0;
                out[y][x][2] = 0;
            }
        }
        else
        {
            out[y][x][0]=in[y][x][0];
            out[y][x][1]=in[y][x][1];
            out[y][x][2]=in[y][x][2];
            out[y][x][3]=in[y][x][3];
            out[y][x][4]=in[y][x][4];
        }
    }
  }
}
#endif

#ifdef 	PostProcessStepsPatch
/* ============================================================ */
/*  steps post process */
/* ============================================================ */

PP_STEPS *createPostSteps(void)
{
  PP_STEPS *New;
  New = (PP_STEPS *)POV_MALLOC(sizeof(PP_STEPS),"steps post process");
  New->next = NULL;
  New->doPostProcess = doPostSteps;
  New->DestroyPostProcess = DestroyPostSteps;
  
  New->numSteps[0] = 10;
  New->numSteps[1] = 10;
  New->numSteps[2] = 10;
  New->numSteps[3] = 10;
  New->numSteps[4] = 10;
  
  return New;
}

void DestroyPostSteps(PP_TASK * task)
{
}

void doPostSteps(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_STEPS * Task;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'steps' effect: 0%%\r");
#else
  Status_Info("\nApplying 'steps' effect: 0%%");
#endif
  Task = (PP_STEPS *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'steps' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'steps' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        out[y][x][0] = (int)(in[y][x][0] * Task->numSteps[0])/Task->numSteps[0];
        out[y][x][1] = (int)(in[y][x][1] * Task->numSteps[1])/Task->numSteps[1];
        out[y][x][2] = (int)(in[y][x][2] * Task->numSteps[2])/Task->numSteps[2];
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif

#ifdef PostProcesNormalPatch
/* ============================================================ */
/*  normal post process */
/* ============================================================ */

PP_NORMAL *createPostNormal(void)
{
  PP_NORMAL *New;
  New = (PP_NORMAL *)POV_MALLOC(sizeof(PP_NORMAL),"normal post process");
  New->next = NULL;
  New->doPostProcess = doPostNormal;
  New->DestroyPostProcess = DestroyPostNormal;
  
  return New;
}

void DestroyPostNormal(PP_TASK * task)
{
}

void doPostNormal(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_NORMAL * Task;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'normal' effect: 0%%\r");
#else
  Status_Info("\nApplying 'normal' effect: 0%%");
#endif
  Task = (PP_NORMAL *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'normal' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'normal' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        if(Post_Lines.Depth[y][x] < Max_Distance)
        {
            out[y][x][0] = (Post_Lines.INormal[y][x][0] + 1)/2;
            out[y][x][1] = (Post_Lines.INormal[y][x][1] + 1)/2;
            out[y][x][2] = (Post_Lines.INormal[y][x][2] + 1)/2;
        }
        else
        {
            out[y][x][0] = 0;
            out[y][x][1] = 0;
            out[y][x][2] = 0;
        }
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif

#ifdef PostProcessMathPatch
/* ============================================================ */
/*  min, max, add, subtract, multiply, divide, and exponent post processes */
/* ============================================================ */

PP_MATH *createPostMath(void)
{
  PP_MATH *New;
  New = (PP_MATH *)POV_MALLOC(sizeof(PP_MATH),"math post process");
  New->next = NULL;
  New->doPostProcess = NULL;
  New->DestroyPostProcess = DestroyPostMath;
  
  New->Pig = Create_Pigment();
  
  return New;
}

void DestroyPostMath(PP_TASK * task)
{
    PP_MATH * Task = (PP_MATH *)task;
    if(Task->Pig != NULL)
    	Destroy_Pigment (Task->Pig);
}

#ifdef PostProcessMinPatch
void doPostMin(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_MATH * Task;

 #ifdef ProgressOnOneLine
 Status_Info("\nApplying 'min' effect: 0%%\r");
#else
 Status_Info("\nApplying 'min' effect: 0%%");
#endif
  Task = (PP_MATH *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'min' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'min' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        COLOUR Vals;
        VECTOR Loc;
        
        Loc[0] = (DBL)x/width;
        Loc[1] = 1-((DBL)y/height);
        Loc[2] = 0;
	    Compute_Pigment (Vals, Task->Pig, Loc, NULL);
	    
        out[y][x][0] = min(in[y][x][0], Vals[0]);
        out[y][x][1] = min(in[y][x][1], Vals[1]);
        out[y][x][2] = min(in[y][x][2], Vals[2]);
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif

#ifdef PostProcessMaxPatch
void doPostMax(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_MATH * Task;

 #ifdef ProgressOnOneLine
 Status_Info("\nApplying 'max' effect: 0%%\r");
#else
 Status_Info("\nApplying 'max' effect: 0%%");
#endif
  Task = (PP_MATH *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'max' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'max' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        COLOUR Vals;
        VECTOR Loc;
        
        Loc[0] = (DBL)x/width;
        Loc[1] = 1-((DBL)y/height);
        Loc[2] = 0;
	    Compute_Pigment (Vals, Task->Pig, Loc, NULL);
	    
        out[y][x][0] = max(in[y][x][0], Vals[0]);
        out[y][x][1] = max(in[y][x][1], Vals[1]);
        out[y][x][2] = max(in[y][x][2], Vals[2]);
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif
#ifdef PostProcessAddPatch
void doPostAdd(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_MATH * Task;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'add' effect: 0%%\r");
#else
  Status_Info("\nApplying 'add' effect: 0%%");
#endif
  Task = (PP_MATH *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'add' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'add' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        COLOUR addVals;
        VECTOR Loc;
        
        Loc[0] = (DBL)x/width;
        Loc[1] = 1-((DBL)y/height);
        Loc[2] = 0;
	    Compute_Pigment (addVals, Task->Pig, Loc, NULL);
	    
        out[y][x][0] = in[y][x][0] + addVals[0];
        out[y][x][1] = in[y][x][1] + addVals[1];
        out[y][x][2] = in[y][x][2] + addVals[2];
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif
#ifdef PostProcessSubtractPatch
void doPostSub(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_MATH * Task;

 #ifdef ProgressOnOneLine
 Status_Info("\nApplying 'subtract' effect: 0%%\r");
#else
 Status_Info("\nApplying 'subtract' effect: 0%%");
#endif
  Task = (PP_MATH *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'subtract' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'subtract' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        COLOUR Vals;
        VECTOR Loc;
        
        Loc[0] = (DBL)x/width;
        Loc[1] = 1-((DBL)y/height);
        Loc[2] = 0;
	    Compute_Pigment (Vals, Task->Pig, Loc, NULL);
	    
        out[y][x][0] = in[y][x][0] - Vals[0];
        out[y][x][1] = in[y][x][1] - Vals[1];
        out[y][x][2] = in[y][x][2] - Vals[2];
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif
#ifdef PostProcessMultiplyPatch
void doPostMult(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_MATH * Task;

 #ifdef ProgressOnOneLine
 Status_Info("\nApplying 'multiply' effect: 0%%\r");
#else
 Status_Info("\nApplying 'multiply' effect: 0%%");
#endif
  Task = (PP_MATH *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'multiply' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'multiply' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        COLOUR multVals;
        VECTOR Loc;
        
        Loc[0] = (DBL)x/width;
        Loc[1] = 1-((DBL)y/height);
        Loc[2] = 0;
	    Compute_Pigment (multVals, Task->Pig, Loc, NULL);
	    
        out[y][x][0] = in[y][x][0] * multVals[0];
        out[y][x][1] = in[y][x][1] * multVals[1];
        out[y][x][2] = in[y][x][2] * multVals[2];
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif
#ifdef PostProcessDividePatch
void doPostDiv(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_MATH * Task;

 #ifdef ProgressOnOneLine
 Status_Info("\nApplying 'divide' effect: 0%%\r");
#else
 Status_Info("\nApplying 'divide' effect: 0%%");
#endif
  Task = (PP_MATH *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'divide' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'divide' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        COLOUR Vals;
        VECTOR Loc;
        
        Loc[0] = (DBL)x/width;
        Loc[1] = 1-((DBL)y/height);
        Loc[2] = 0;
	    Compute_Pigment (Vals, Task->Pig, Loc, NULL);
	    
        out[y][x][0] = in[y][x][0] / Vals[0];
        out[y][x][1] = in[y][x][1] / Vals[1];
        out[y][x][2] = in[y][x][2] / Vals[2];
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif
#ifdef PostProcessExponentPatch
void doPostExp(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_MATH * Task;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'exponent' effect: 0%%\r");
#else
  Status_Info("\nApplying 'exponent' effect: 0%%");
#endif
  Task = (PP_MATH *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
   Status_Info("\rApplying 'exponent' effect: %d%%",(y+1)*100/height);
#else 
   Status_Info("\nApplying 'exponent' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        COLOUR expVals;
        VECTOR Loc;
        
        Loc[0] = (DBL)x/width;
        Loc[1] = 1-((DBL)y/height);
        Loc[2] = 0;
	    Compute_Pigment (expVals, Task->Pig, Loc, NULL);
	    
        out[y][x][0] = pow(in[y][x][0], expVals[0]);
        out[y][x][1] = pow(in[y][x][1], expVals[1]);
        out[y][x][2] = pow(in[y][x][2], expVals[2]);
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif
#endif

#ifdef PostProcessClipColorsPatch
/* ============================================================ */
/*  clip_color post process */
/* ============================================================ */

PP_CLIP_COLORS *createPostClipColors(void)
{
  PP_CLIP_COLORS *New;
  New = (PP_CLIP_COLORS *)POV_MALLOC(sizeof(PP_CLIP_COLORS),"clip_colors post process");
  New->next = NULL;
  New->doPostProcess = doPostClipColors;
  New->DestroyPostProcess = DestroyPostClipColors;
  
  New->ClipMin[0] = 0;
  New->ClipMin[1] = 0;
  New->ClipMin[2] = 0;
  New->ClipMin[3] = 0;
  New->ClipMin[4] = 0;
  
  New->ClipMax[0] = 1;
  New->ClipMax[1] = 1;
  New->ClipMax[2] = 1;
  New->ClipMax[3] = 1;
  New->ClipMax[4] = 1;
  
  return New;
}

void DestroyPostClipColors(PP_TASK * task)
{
}

void doPostClipColors(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_CLIP_COLORS * clip;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'clip_colors' effect: 0%%\r");
#else
  Status_Info("\nApplying 'clip_colors' effect: 0%%");
#endif
  clip = (PP_CLIP_COLORS *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'clip_colors' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'clip_colors' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        out[y][x][0] = max(min(in[y][x][0], clip->ClipMax[0]), clip->ClipMin[0]);
        out[y][x][1] = max(min(in[y][x][1], clip->ClipMax[1]), clip->ClipMin[1]);
        out[y][x][2] = max(min(in[y][x][2], clip->ClipMax[2]), clip->ClipMin[2]);
        out[y][x][3]=in[y][x][3];
        out[y][x][4]=in[y][x][4];
    }
  }
}
#endif


#ifdef PostProcessInvertPatch
/* ============================================================ */
/*  invert post process */
/* ============================================================ */
PP_INVERT *createPostInvert(void)
{
  PP_INVERT *New;
  New = (PP_INVERT *)POV_MALLOC(sizeof(PP_INVERT),"invert post process");
  New->next = NULL;
  New->doPostProcess = doPostInvert;
  New->DestroyPostProcess = DestroyPostInvert;
  
  return New;
}

void DestroyPostInvert(PP_TASK * task)
{
}

void doPostInvert(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_INVERT * invert;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'invert' effect: 0%%\r");
#else
  Status_Info("\nApplying 'invert' effect: 0%%");
#endif
  invert = (PP_INVERT *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'invert' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'invert' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
      out[y][x][0] = 1-in[y][x][0];
      out[y][x][1] = 1-in[y][x][1];
      out[y][x][2] = 1-in[y][x][2];
      out[y][x][3]=in[y][x][3];
      out[y][x][4]=in[y][x][4];
    }
  }
}
#endif

#ifdef PostProcessFindEdgesPatch
/* ============================================================ */
/*  mark_edges post process */
/* ============================================================ */

PP_EDGES *createPostEdges(void)
{
  PP_EDGES *New;
  New = (PP_EDGES *)POV_MALLOC(sizeof(PP_EDGES),"edges post process");
  New->next = NULL;
  New->doPostProcess = doPostEdges;
  New->DestroyPostProcess = DestroyPostEdges;
  
  New->ThreshDepthMax = 0.6;
  
  New->ThreshNormMax = 0.6;
  
  New->ThreshColMax = 0.6;
  
  New->Radius = 2.0;

  New->Sharpness = 1.0;
  
  New->LinePig = Create_Pigment();
  /*New->LineCol[0] = 0;
  New->LineCol[1] = 0;
  New->LineCol[2] = 0;
  New->LineCol[3] = 0;
  New->LineCol[4] = 0;*/
  return New;
}

void DestroyPostEdges(PP_TASK * task)
{
  PP_EDGES *edges = (PP_EDGES*)task;
  Destroy_Pigment(edges->LinePig);
}

void doPostEdges(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_EDGES *pp_edges;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'find_edges' effect: 0%%\r");
#else
  Status_Info("\nApplying 'find_edges' effect: 0%%");
#endif
  pp_edges = (PP_EDGES *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'find_edges' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'find_edges' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
      DBL diff;
      DBL avgNormDiff = 0;
      DBL avgColDiff = 0;
      DBL avgDepthDiff = 0;
      DBL weight = 0.0;
      VECTOR centerNorm;
      COLOUR centerCol;
      DBL centerDepth = Post_Lines.Depth[y][x];
      Assign_Vector(centerNorm, Post_Lines.PNormal[y][x]);
      Assign_Colour(centerCol, Post_Lines.Colour[y][x]);

      {/*average depth difference*/
        DBL tmp = 0;
        int j,l;
        weight = 0.0;
        for(j=-pp_edges->Radius; j<=pp_edges->Radius; j++)
        {
            for(l=-pp_edges->Radius; l<=pp_edges->Radius; l++)
            {
                if((l != 0 || j != 0)&&(sqrt(sqr(j)+sqr(l)) <= pp_edges->Radius))
                {
                    if(((x+l >= 0)&&(x+l < width)) && (((y+j >= 0)&&(y+j < height))))
                    {
                        diff = fabs(centerDepth-Post_Lines.Depth[y+j][x+l]);
                        
                        if (diff>pp_edges->ThreshDepthMax)
                          avgDepthDiff++;
                        
                        weight++;
                    }
                }
            }
        }
        if(weight>0) avgDepthDiff /= weight;
      }
      
      if (Post_Lines.Depth[y][x]<Max_Distance)
      {/*average normal difference*/
        DBL tmp = 0;
        int j,l;
        weight = 0.0;
        for(j=-pp_edges->Radius; j<=pp_edges->Radius; j++)
        {
            for(l=-pp_edges->Radius; l<=pp_edges->Radius; l++)
            {
                if((l != 0 || j != 0)&&(sqrt(sqr(j)+sqr(l)) <= pp_edges->Radius))
                {
                    if(((x+l >= 0)&&(x+l < width))&&(((y+j >= 0)&&(y+j < height)))
                       &&(Post_Lines.Depth[y+j][x+l]<Max_Distance))
                    {
                        VDot(tmp, centerNorm, Post_Lines.PNormal[y+j][x+l])
                        diff = (1.0-tmp)*0.5;

                        if (diff>pp_edges->ThreshNormMax)
                          avgNormDiff++;

                        weight++;
                    }
                }
            }
        }
        if(weight>0) avgNormDiff /= weight;
      }
      
      {/*average color difference*/
        int j,l;
        int k=0;
        int xOff = -pp_edges->Radius;
        int yOff = -pp_edges->Radius;
        weight = 0.0;
        for(j=-pp_edges->Radius; j<=pp_edges->Radius; j++)
        {
            for(l=-pp_edges->Radius; l<=pp_edges->Radius; l++)
            {
                if((l != 0 || j != 0)&&(sqrt(sqr(j)+sqr(l)) <= pp_edges->Radius))
                {
                    if(((x+l >= 0)&&(x+l < width))&&(((y+j >= 0)&&(y+j < height))))
                    {
                        diff = (fabs(centerCol[0] - Post_Lines.Colour[y+j][x+l][0])+
                                fabs(centerCol[1] - Post_Lines.Colour[y+j][x+l][1])+
                                fabs(centerCol[2] - Post_Lines.Colour[y+j][x+l][2])) / 3.0;

                        if (diff>pp_edges->ThreshColMax)
                          avgColDiff++;

                        weight++;
                    }
                }
            }
        }
        if(weight>0) avgColDiff /= weight;
      }
      
      
      if((pp_edges->ThreshDepthMax != 0)||
         (pp_edges->ThreshNormMax  != 0)||
         (pp_edges->ThreshColMax   != 0))
      {
          COLOUR LineCol;
          VECTOR Loc;
          DBL filt = 1;
          DBL sharpness = pp_edges->Sharpness * (pp_edges->Radius - 1.0) + 1.0;

          if(pp_edges->ThreshColMax)
          {
            filt = max(min(filt, 1.0 - avgColDiff*sharpness),0.0);
          }
          
          if(pp_edges->ThreshNormMax)
          {
            filt = max(min(filt, 1.0 - avgNormDiff*sharpness),0.0);
          }
          
          if(pp_edges->ThreshDepthMax)
          {
            filt = max(min(filt, 1.0 - avgDepthDiff*sharpness),0.0);
          }

          Loc[0] = (DBL)x/width;
          Loc[1] = 1-((DBL)y/height);
          Loc[2] = 0;
	        Compute_Pigment (LineCol, pp_edges->LinePig, Loc, NULL);

          if(filt != 0)
          {
              out[y][x][0] = LineCol[0]*(1-filt) + (in[y][x][0]*filt);
              out[y][x][1] = LineCol[1]*(1-filt) + (in[y][x][1]*filt);
              out[y][x][2] = LineCol[2]*(1-filt) + (in[y][x][2]*filt);
          }
          else
          {
              out[y][x][0] = LineCol[0];
              out[y][x][1] = LineCol[1];
              out[y][x][2] = LineCol[2];
          }
          out[y][x][3]=in[y][x][3];
          out[y][x][4]=in[y][x][4];
      }
      else
      {
          out[y][x][0] = in[y][x][0];
          out[y][x][1] = in[y][x][1];
          out[y][x][2] = in[y][x][2];
          out[y][x][3] = in[y][x][3];
          out[y][x][4] = in[y][x][4];
      }
      if(pp_edges->ThreshColMax == 0 && pp_edges->ThreshNormMax == 0 && pp_edges->ThreshDepthMax == 0)
      {
          out[y][x][0] = avgDepthDiff;
          out[y][x][1] = avgNormDiff;
          out[y][x][2] = avgColDiff;
          out[y][x][3] = in[y][x][3];
          out[y][x][4] = in[y][x][4];
      }
    }
  }
}
#endif

#ifdef PostProcessBlurMatrixPatch
/* ============================================================ */
/*  blur_matrix post process */
/* ============================================================ */

PP_BLUR_MATRIX * createPostBlur_Matrix(void)
{
  PP_BLUR_MATRIX * blur_matrix;
  blur_matrix = (PP_BLUR_MATRIX *)POV_MALLOC(sizeof(PP_BLUR_MATRIX),"blur_matrix process");
  blur_matrix->next = NULL;
  blur_matrix->doPostProcess = doPostBlur_Matrix;
  blur_matrix->DestroyPostProcess = DestroyPostBlur_Matrix;
  
  blur_matrix->Matrix = NULL;
  blur_matrix->xSize = 0;
  blur_matrix->ySize = 0;
  
  blur_matrix->Div = 1;
  blur_matrix->Levelling = 0;
  
  return blur_matrix;
}

void DestroyPostBlur_Matrix(PP_TASK * task)
{
    PP_BLUR_MATRIX * tmp = (PP_BLUR_MATRIX *)task;
    POV_FREE(tmp->Matrix);
}

void doPostBlur_Matrix(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_BLUR_MATRIX * pp_matrix;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'blur_matrix' effect: 0%%\r");
#else
  Status_Info("\nApplying 'blur_matrix' effect: 0%%");
#endif
  pp_matrix = (PP_BLUR_MATRIX *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'blur_matrix' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'blur_matrix' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
        DBL totalR = 0;
        DBL totalG = 0;
        DBL totalB = 0;
        DBL totalF = 0;
        DBL totalT = 0;
        int j,l;
        int k=0;
        int xOff = -pp_matrix->xSize/2;
        int yOff = -pp_matrix->ySize/2;
        for(j=0; j<pp_matrix->ySize; j++)
        {
            for(l=0; l<pp_matrix->xSize; l++)
            {
                if(((x+xOff+l>=0)&&(x+xOff+l<width))&&(((y+yOff+j>=0)&&(y+yOff+j<height))))
                {
                    totalR += pp_matrix->Matrix[k]*in[y+yOff+j][x+xOff+l][0];
                    totalG += pp_matrix->Matrix[k]*in[y+yOff+j][x+xOff+l][1];
                    totalB += pp_matrix->Matrix[k]*in[y+yOff+j][x+xOff+l][2];
                    totalF += pp_matrix->Matrix[k]*in[y+yOff+j][x+xOff+l][3];
                    totalT += pp_matrix->Matrix[k]*in[y+yOff+j][x+xOff+l][4];
                }
                k++;
            }
        }
        out[y][x][0] = (totalR+pp_matrix->Levelling)/pp_matrix->Div;
        out[y][x][1] = (totalG+pp_matrix->Levelling)/pp_matrix->Div;
        out[y][x][2] = (totalB+pp_matrix->Levelling)/pp_matrix->Div;
        out[y][x][3] = (totalF+pp_matrix->Levelling)/pp_matrix->Div;
        out[y][x][4] = (totalT+pp_matrix->Levelling)/pp_matrix->Div;
    }
  }
}
#endif

#ifdef PostProcessColorMatrixPatch
/* ============================================================ */
/*  color_matrix post process */
/* ============================================================ */

PP_COLOR_MATRIX * createPostColor_Matrix(void)
{
  PP_COLOR_MATRIX * color_matrix;
  color_matrix = (PP_COLOR_MATRIX *)POV_MALLOC(sizeof(PP_COLOR_MATRIX),"color_matrix process");
  color_matrix->next = NULL;
  color_matrix->doPostProcess = doPostColor_Matrix;
  color_matrix->DestroyPostProcess = DestroyPostColor_Matrix;
  color_matrix->Matrix[0][0] = 1; color_matrix->Matrix[1][0] = 0; color_matrix->Matrix[2][0] = 0;
  color_matrix->Matrix[0][1] = 0; color_matrix->Matrix[1][1] = 1; color_matrix->Matrix[2][1] = 0;
  color_matrix->Matrix[0][2] = 0; color_matrix->Matrix[1][2] = 0; color_matrix->Matrix[2][2] = 1;
  return color_matrix;
}

void DestroyPostColor_Matrix(PP_TASK * task)
{
}

void doPostColor_Matrix(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_COLOR_MATRIX * pp_matrix;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'color_matrix' effect: 0%%\r");
#else
  Status_Info("\nApplying 'color_matrix' effect: 0%%");
#endif
  pp_matrix = (PP_COLOR_MATRIX *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'color_matrix' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'color_matrix' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
      out[y][x][0] = pp_matrix->Matrix[0][0]*in[y][x][0] + pp_matrix->Matrix[1][0]*in[y][x][1] + pp_matrix->Matrix[2][0]*in[y][x][2];
      out[y][x][1] = pp_matrix->Matrix[0][1]*in[y][x][0] + pp_matrix->Matrix[1][1]*in[y][x][1] + pp_matrix->Matrix[2][1]*in[y][x][2];
      out[y][x][2] = pp_matrix->Matrix[0][2]*in[y][x][0] + pp_matrix->Matrix[1][2]*in[y][x][1] + pp_matrix->Matrix[2][2]*in[y][x][2];
      out[y][x][3]=in[y][x][3];
      out[y][x][4]=in[y][x][4];
    }
  }
}
#endif

/* ============================================================ */
/*  Focal Blur post process */
/* ============================================================ */

PP_FOCAL_BLUR *createPostFocalBlur(void)
{
  PP_FOCAL_BLUR *blur;
  blur = (PP_FOCAL_BLUR *)POV_MALLOC(sizeof(PP_FOCAL_BLUR),"focal blur");
  blur->doPostProcess = doPostFocalBlur;
  blur->DestroyPostProcess = DestroyPostFocalBlur;
  blur->next = NULL;
  blur->fieldStart = 0;
  blur->fieldDepth = BOUND_HUGE;
  blur->maxPixelBlur = 0;
  blur->Weights = NULL;
  blur->keepAA = 0.0;
  return blur;
}

void DestroyPostFocalBlur(PP_TASK *task)
{
  PP_FOCAL_BLUR *blur = (PP_FOCAL_BLUR *)task;
  FREE_GRID(blur->Weights, static_height);
}


void doPostFocalBlur(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  PP_FOCAL_BLUR *blur;
  DBL depth, blur_amount, dist;
  DBL pixel_dist;
  DBL weight1, weight2, w;
  int x,y;
  int xa,xb,ya,yb;
  DBL start,mid;
  DBL field_depth;
  DBL depth2;
  int iblur;
  int x2,y2;
  DBL selfweight;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'focal blur' effect: 0%%\r");
#else
  Status_Info("\nApplying 'focal blur' effect: 0%%");
#endif
  blur = (PP_FOCAL_BLUR *)pp_task;

  start = blur->fieldStart;
  field_depth = blur->fieldDepth;
  mid = start+field_depth/2.0;

  ALLOCATE_GRID(blur->Weights, DBL, height, width);

  /* initialize weights and output data */
  for(y=0; y<height; y++)
  {
    for(x=0; x<width; x++)
    {
      blur->Weights[y][x] = 0.0;
      out[y][x][0] = 
      out[y][x][1] = 
      out[y][x][2] = 0.0;
    }
  }


  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'focal blur' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'focal blur' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1

    for(x=0; x<width; x++)
    {
      depth = data.Depth[y][x];

#define FIND_BLUR(d) (max((sqrt(((d)-mid)*2.0/field_depth) - 1.0),0.0)*(width/640.0));

      if (depth>mid)
      {
        if (depth > mid+field_depth/2)
        {
          /*dist = depth-mid;*/
          /*blur_amount = sqrt(dist*2.0/field_depth) - 1.0;*/
          blur_amount = FIND_BLUR(depth);
        }
        else
          blur_amount = 0.0;
      }
      else
      {
        if (depth < start)
        {
          dist = mid - field_depth/2;
          blur_amount = dist/depth - 1.0;

          /* compensate for resolution, assuming square pixels */
          blur_amount *= (width/640.0);
        }
        else
          blur_amount = 0.0;
      }

      /* don't go below zero for any reason */
      blur_amount = max(blur_amount, 0.0);

      if(blur_amount < 0.1)  /* maybe 0.5 should be the threshold? */
      {
        out[y][x][0] += in[y][x][0];
        out[y][x][1] += in[y][x][1];
        out[y][x][2] += in[y][x][2];
        blur->Weights[y][x] += 1.0;
      }
      else
      {
        /* maximum blur amount */
        blur_amount = max(min(blur_amount,blur->maxPixelBlur),0);

        iblur = (int)(blur_amount+0.5);
        /*iblur = max(min(blur->maxPixelBlur,iblur),0);*/
        xa = x-iblur; xb=x+iblur;
        ya = y-iblur; yb=y+iblur;

        if(depth>mid)
        {
          /* too far away */
          weight1=0.0;
          selfweight=0.0;
          for (x2=xa; x2<=xb; x2++)
          {
            for (y2=ya; y2<=yb; y2++)
            {
              if (!(y2>=0 && y2<height && x2>=0 && x2<width)) continue;
              /*if( (x2!=x || y2!= y) && fabs(data.Depth[y2][x2]-mid)<field_depth/2 ) continue; */

              pixel_dist = sqrt((DBL)(x-x2)*(x-x2)+(DBL)(y-y2)*(y-y2));

#if usegauss
              w = 1.0/(2*M_PI*blur_amount) * exp(-pixel_dist*pixel_dist / (2*blur_amount));
#else
              w = max(0.0, blur_amount-pixel_dist)/blur_amount;
#endif
              /* blend more if closer to same depth */
              depth2 = data.Depth[y2][x2];
              if (depth2 > mid+field_depth/2)
              {
                DBL blur_amount2 = FIND_BLUR(depth2);
                /*selfweight += w * (1.0 - min(min(blur_amount2,blur->maxPixelBlur) / min(blur_amount,blur->maxPixelBlur),1.0));*/
                selfweight += max((1.41-pixel_dist),0.0) * w *
                              (1.0-min(min(blur_amount2,blur->maxPixelBlur) / min(blur_amount,blur->maxPixelBlur),1.0));
                w *= min(min(blur_amount2,blur->maxPixelBlur) / min(blur_amount,blur->maxPixelBlur),1.0);
              }
              else 
              {
                selfweight += max((1.41-pixel_dist)*w,0.0);
                w = 0.0;
              }

              /* compensate for AA */
              /*if (w == 0.0 && (abs(y2-y)==1 || abs(x2-x)==1)) w=0.1;*/

              weight1 += w;
            }
          }
          selfweight *= blur_amount * blur->keepAA * 3.0;
          weight1 += selfweight;

          for (x2=xa; x2<=xb; x2++)
          {
            for (y2=ya; y2<=yb; y2++)
            {
              if (!(y2>=0 && y2<height && x2>=0 && x2<width)) continue;
              if( (x2!=x || y2!= y) && fabs(data.Depth[y2][x2]-mid)<field_depth/2 ) continue;

              pixel_dist = sqrt((DBL)(x-x2)*(x-x2)+(DBL)(y-y2)*(y-y2));

#if usegauss
              w = 1.0/(2*M_PI*blur_amount) * exp(-pixel_dist*pixel_dist / (2*blur_amount));
              w /= weight1;
#else
              w = 
                max(0.0, blur_amount-pixel_dist) / (blur_amount * weight1);
#endif
              /* blend more if closer to same depth */
              depth2 = data.Depth[y2][x2];
              if (depth2 > mid+field_depth/2)
              {
                DBL blur_amount2 = FIND_BLUR(depth2);
                w *= min(min(blur_amount2,blur->maxPixelBlur) / min(blur_amount,blur->maxPixelBlur),1.0);
              }
              else w = 0.0;

              if (x==x2 && y==y2) w += selfweight/weight1;

              /* compensate for AA */
              /*if (w == 0.0 && (abs(y2-y)==1 || abs(x2-x)==1)) w=0.1/weight1;*/

              if (w==0.0) continue;

              out[y][x][0] += in[y2][x2][0]*w;
              out[y][x][1] += in[y2][x2][1]*w;
              out[y][x][2] += in[y2][x2][2]*w;
              blur->Weights[y][x] += w;
            }
          }
        }
        else
        {
          /* too close */
          weight1=weight2=0.0;
          for (x2=xa; x2<=xb; x2++)
          {
            for (y2=ya; y2<=yb; y2++)
            {
              pixel_dist = sqrt((DBL)(x-x2)*(x-x2)+(DBL)(y-y2)*(y-y2));
#if usegauss
              w = 1.0/(2*M_PI*blur_amount) * exp(-pixel_dist*pixel_dist / (2*blur_amount));
#else
              w = max(0.0, blur_amount-pixel_dist)/blur_amount;
#endif

              weight2 += w;

              if (!(y2>=0 && y2<height && x2>=0 && x2<width)) continue;

              weight1 += w;
            }
          }

          for (x2=xa; x2<=xb; x2++)
          {
            for (y2=ya; y2<=yb; y2++)
            {
              if (!(y2>=0 && y2<height && x2>=0 && x2<width)) continue;
              pixel_dist = sqrt((DBL)(x-x2)*(x-x2)+(DBL)(y-y2)*(y-y2));
              
#if usegauss
              w = 1.0/(2*M_PI*blur_amount) * exp(-pixel_dist*pixel_dist / (2*blur_amount));
              w /= weight1;
#else
              w = max(0.0, blur_amount-pixel_dist) / (blur_amount*weight1);
#endif

              if (w==0.0) continue;
              out[y][x][0] += in[y2][x2][0]*w;
              out[y][x][1] += in[y2][x2][1]*w;
              out[y][x][2] += in[y2][x2][2]*w;
              blur->Weights[y][x] += w;

#if usegauss
              w = 1.0/(2*M_PI*blur_amount) * exp(-pixel_dist*pixel_dist / (2*blur_amount));
              w /= weight2;
#else
              w = max(0.0, blur_amount-pixel_dist) / (weight2);
#endif
              out[y2][x2][0] += in[y][x][0]*w;
              out[y2][x2][1] += in[y][x][1]*w;
              out[y2][x2][2] += in[y][x][2]*w;
              blur->Weights[y2][x2] += w;
            }
          }
        }
      }
    }
  }

  for(y=0; y<height; y++)
  {
    COOPERATE_1
    for(x=0; x<width; x++)
    {
      Scale_Colour(out[y][x],out[y][x],1.0/blur->Weights[y][x]);
      out[y][x][3]=in[y][x][3];
      out[y][x][4]=in[y][x][4];
    }
  }

  FREE_GRID(blur->Weights, height);
}

/* ============================================================ */
/*  Depth post process */
/* ============================================================ */

PP_DEPTH *createPostDepth(void)
{
  PP_DEPTH *depth;
  depth = (PP_DEPTH *)POV_MALLOC(sizeof(PP_DEPTH),"depth post process");
  depth->next = NULL;
  depth->doPostProcess = doPostDepth;
 #ifdef HuffPostProcessPatches
 depth->DestroyPostProcess = DestroyPostDepth;
#endif
  depth->fieldDepth = BOUND_HUGE;
  depth->fieldStart = 0;
  return depth;
}

void DestroyPostDepth(PP_TASK * task)
{
}
void doPostDepth(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y;
  PP_DEPTH *pp_depth;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'depth' effect: 0%%\r");
#else
  Status_Info("\nApplying 'depth' effect: 0%%");
#endif
  pp_depth = (PP_DEPTH *)pp_task;

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'depth' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'depth' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
      /* for now, convert to depth */
      DBL depth;
      DBL start = pp_depth->fieldStart;
      DBL field_depth = pp_depth->fieldDepth;

      depth = Post_Lines.Depth[y][x];
      out[y][x][0] = min(1.0,max(0.0,1.0-((depth-start)/field_depth)));
      out[y][x][1] = min(1.0,max(0.0,1.0-((depth-start)/field_depth)));
      out[y][x][2] = min(1.0,max(0.0,1.0-((depth-start)/field_depth)));
      out[y][x][3]=in[y][x][3];
      out[y][x][4]=in[y][x][4];
    }
  }
}

/* ============================================================ */
/*  Soft Glow post process */
/* ============================================================ */
PP_SOFT_GLOW *createPostSoftGlow(void)
{
  PP_SOFT_GLOW *glow;
  glow = (PP_SOFT_GLOW *)POV_MALLOC(sizeof(PP_SOFT_GLOW),"soft glow");
  glow->next = NULL;
  glow->doPostProcess = doPostSoftGlow;
#ifdef HuffPostProcessPatches
  glow->DestroyPostProcess = DestroyPostSoftGlow;
#endif
  glow->blurAmount = 3;
  glow->glowAmount = 1.0;
  return glow;
}

void DestroyPostSoftGlow(PP_TASK * task)
{
}
void doPostSoftGlow(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y,x2,y2;
  int xa,xb,ya,yb;
  DBL pixel_dist2;
  PP_SOFT_GLOW *glow;
  DBL weight,w;
  int i;
  DBL **gauss_w;
  int iblur;

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'soft glow' effect: 0%%\r");
#else
  Status_Info("\nApplying 'soft glow' effect: 0%%");
#endif
  glow = (PP_SOFT_GLOW *)pp_task;

  iblur = (int)glow->blurAmount;

  /* precompute gaussian weights, since they are used a lot */

  /* allocate the gaussian-blur weights */
  ALLOCATE_GRID(gauss_w, DBL, (iblur*2+1), (iblur*2+1));

  for(y=0; y<=iblur*2; y++)
    for(x=0; x<=iblur*2; x++)
    {
      pixel_dist2 = (DBL)(x-iblur)*(x-iblur)+(DBL)(y-iblur)*(y-iblur);
      gauss_w[y][x] = 1.0/(2*M_PI*glow->blurAmount) * exp(-pixel_dist2 / (2*glow->blurAmount));
    }


  /* now, post-process the image */

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'soft glow' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'soft glow' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
      /* do a gaussian blur first */
      xa = x-iblur; xb=x+iblur;
      ya = y-iblur; yb=y+iblur;

      weight = 0.0;
      out[y][x][0] = out[y][x][1] = out[y][x][2] = 0.0;

      for (x2=xa; x2<=xb; x2++)
        for (y2=ya; y2<=yb; y2++)
        {
          if (!(y2>=0 && y2<height && x2>=0 && x2<width)) continue;

          w = gauss_w[y2-y+iblur][x2-x+iblur];
          out[y][x][0] += in[y2][x2][0]*w;
          out[y][x][1] += in[y2][x2][1]*w;
          out[y][x][2] += in[y2][x2][2]*w;
          weight += w;
        }
      Scale_Colour(out[y][x],out[y][x],1.0/weight);

      if(glow->glowAmount>0.0)
      {
        /* now, combine with original using LIGHTEN combination */
        out[y][x][0] *= (glow->glowAmount);
        out[y][x][1] *= (glow->glowAmount);
        out[y][x][2] *= (glow->glowAmount);

        for(i=0; i<3; i++)
        {
          if (in[y][x][i]>out[y][x][i])  /* input is brighter - keep it */
            out[y][x][i]=in[y][x][i];
          else if (glow->glowAmount < 1.0)  /* must blend */
            out[y][x][i] = out[y][x][i] + in[y][x][i] * (1.0-glow->glowAmount);
        }
      }
      else
      {
        /* now, combine with original using DARKEN combination */
        out[y][x][0] *= -(glow->glowAmount);
        out[y][x][1] *= -(glow->glowAmount);
        out[y][x][2] *= -(glow->glowAmount);

        for(i=0; i<3; i++)
        {
          if (in[y][x][i]<out[y][x][i])  /* input is darker - keep it */
            out[y][x][i]=in[y][x][i];
          else if (glow->glowAmount > -1.0)  /* must blend */
            out[y][x][i] = out[y][x][i] + in[y][x][i] * (1.0+glow->glowAmount);
        }
      }

      out[y][x][3]=in[y][x][3];
      out[y][x][4]=in[y][x][4];

      Clip_Colour(out[y][x], out[y][x]);
    }
  }

  FREE_GRID(gauss_w, (iblur*2+1));
}

#ifdef PostCartoonPatch

/* ============================================================ */
/*  Cartoonize post process */
/* ============================================================ */

PP_CARTOON *createPostCartoon(void)
{
  PP_CARTOON *cartoon;
  cartoon = (PP_CARTOON *)POV_MALLOC(sizeof(PP_CARTOON),"cartoon");
  cartoon->next = NULL;
  cartoon->doPostProcess = doPostCartoon;
  cartoon->DestroyPostProcess = DestroyPostCartoon;
  
  cartoon->blurAmount = 4.0;
  cartoon->hueThreshold = 0.1;
  cartoon->lumThreshold = 0.2;

  /*Make_Colour(cartoon->lineColour,0.0,0.0,0.0);
  cartoon->lineHueThreshold = 0.1;
  cartoon->lineWidth = 2.0;*/

  return cartoon;
}

void DestroyPostCartoon(PP_TASK * task)
{
}

void doPostCartoon(PP_TASK *pp_task, COLOUR **in, COLOUR **out, PP_DATA_LISTS data, int height, int width)
{
  int x,y,x2,y2;
  int xa,xb,ya,yb;
  DBL pixel_dist2;
  PP_CARTOON *cartoon;
  DBL weight,w;
  DBL **gauss_w;
  int iblur;
  DBL hue1,hue2,lum1,lum2;
  /*COLOUR edgeColor;
  DBL huediff;*/

#ifdef ProgressOnOneLine
  Status_Info("\nApplying 'cartoon' effect: 0%%\r");
#else
  Status_Info("\nApplying 'cartoon' effect: 0%%");
#endif
  cartoon = (PP_CARTOON *)pp_task;

  iblur = (int)cartoon->blurAmount;

  /* precompute gaussian weights, since they are used a lot */

  /* allocate the gaussian-blur weights */
  ALLOCATE_GRID(gauss_w, DBL, (iblur*2+1), (iblur*2+1));

  for(y=0; y<=iblur*2; y++)
    for(x=0; x<=iblur*2; x++)
    {
      pixel_dist2 = (DBL)(x-iblur)*(x-iblur)+(DBL)(y-iblur)*(y-iblur);
      gauss_w[y][x] = 1.0/(2*M_PI*cartoon->blurAmount) * exp(-pixel_dist2 / (2*cartoon->blurAmount));
    }


  /* now, post-process the image */

  for(y=0; y<height; y++)
  {
#ifdef ProgressOnOneLine
    Status_Info("\rApplying 'cartoon' effect: %d%%",(y+1)*100/height);
#else
    Status_Info("\nApplying 'cartoon' effect: %d%%",(y+1)*100/height);
#endif
    COOPERATE_1
    for(x=0; x<width; x++)
    {
      /* do a gaussian blur first */
      xa = x-iblur; xb=x+iblur;
      ya = y-iblur; yb=y+iblur;

      weight = 0.0;
      out[y][x][0] = out[y][x][1] = out[y][x][2] = 0.0;

      hue1 = RGBtoHue(in[y][x]);

      lum1 = GREY_SCALE(in[y][x]);

      for (x2=xa; x2<=xb; x2++)
        for (y2=ya; y2<=yb; y2++)
        {
          if (!(y2>=0 && y2<height && x2>=0 && x2<width)) continue;

          w = gauss_w[y2-y+iblur][x2-x+iblur];
          hue2 = RGBtoHue(in[y2][x2]);
          lum2 = GREY_SCALE(in[y2][x2]);
          
          /* blur only same hue */
          if (fabs(hue1-hue2)<cartoon->hueThreshold &&
              fabs(lum1-lum2)<cartoon->lumThreshold)
          {
            out[y][x][0] += in[y2][x2][0]*w;
            out[y][x][1] += in[y2][x2][1]*w;
            out[y][x][2] += in[y2][x2][2]*w;
            weight += w;
          }
        }

      Scale_Colour(out[y][x],out[y][x],1.0/weight);

#if(0)
      /* now, subtract a grey-scaled edge detection */
      xa = x-cartoon->lineWidth; xb=x+cartoon->lineWidth;
      ya = y-cartoon->lineWidth; yb=y+cartoon->lineWidth;

      w = -((cartoon->lineWidth*2+1)*(cartoon->lineWidth*2+1)-1);

      edgeColor[0] = edgeColor[1] = edgeColor[2] = 0.0;
      huediff = 0.0;
      weight = 0.0;

      for (x2=xa; x2<=xb; x2++)
        for (y2=ya; y2<=yb; y2++)
        {
          if (!(y2>=0 && y2<height && x2>=0 && x2<width)) continue;
          if (y2==y && x2==x)
          {
            edgeColor[0] += in[y2][x2][0]*w;
            edgeColor[1] += in[y2][x2][1]*w;
            edgeColor[2] += in[y2][x2][2]*w;
          }
          else
          {
            edgeColor[0] += in[y2][x2][0];
            edgeColor[1] += in[y2][x2][1];
            edgeColor[2] += in[y2][x2][2];

            hue2 = RGBtoHue(in[y2][x2]);
            huediff += fabs(hue1-hue2);
            weight++;
          }
        }
      
      if (huediff/weight > cartoon->lineHueThreshold)
      {
        /* clip edgeColor */
        edgeColor[0] = min(max(edgeColor[0],0.0),1.0);
        edgeColor[1] = min(max(edgeColor[1],0.0),1.0);
        edgeColor[2] = min(max(edgeColor[2],0.0),1.0);

        lum1 = GREY_SCALE(edgeColor);

        out[y][x][0]-=lum1;
        out[y][x][1]-=lum1;
        out[y][x][2]-=lum1;

        /* clip final color */
        out[y][x][0] = min(max(out[y][x][0],0.0),1.0);
        out[y][x][1] = min(max(out[y][x][1],0.0),1.0);
        out[y][x][2] = min(max(out[y][x][2],0.0),1.0);

        out[y][x][0]+=lum1*cartoon->lineColour[0];
        out[y][x][1]+=lum1*cartoon->lineColour[1];
        out[y][x][2]+=lum1*cartoon->lineColour[2];
      }
#endif
      out[y][x][3]=in[y][x][3];
      out[y][x][4]=in[y][x][4];

      /* clip final color */
      out[y][x][0] = min(max(out[y][x][0],0.0),1.0);
      out[y][x][1] = min(max(out[y][x][1],0.0),1.0);
      out[y][x][2] = min(max(out[y][x][2],0.0),1.0);

    }
  }

  FREE_GRID(gauss_w, (iblur*2+1));
}

#endif

#else
static char dummy[2];
#endif
