/****************************************************************************
*                parstxtr.c
*
*  -------------------------------------------------------
*  ATTENTION:
*  This is an unofficial version of partxtr.c modified by
*  Ryoichi Suzuki, rsuzuki@etl.go.jp for use with
*  "isosurface" shape type.
*  -------------------------------------------------------
*  This module parses textures and atmosphere effects.
*
*  from Persistence of Vision(tm) Ray Tracer
*  Copyright 1996,1999 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file.
*  If POVLEGAL.DOC is not available or for more info please contact the POV-Ray
*  Team Coordinator by email to team-coord@povray.org or visit us on the web at
*  http://www.povray.org. The latest version of POV-Ray may be found at this site.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
* Modifications by Thomas Willhalm, March 1999, used with permission.
*
*****************************************************************************/

#include "frame.h"
#include "vector.h"
#include "povproto.h"
#include "parse.h"
#include "parstxtr.h"
#include "colour.h"
#include "express.h"
#ifndef BurnAllGifs
#include "gif.h"
#endif
#include "iff.h"
#include "image.h"
#include "matrices.h" 
#include "media.h"
#include "normal.h"
#include "pigment.h"  
#include "povray.h"
#include "pgm.h"
#include "ppm.h"
#include "targa.h"
#include "png_pov.h"
#include "texture.h"
#include "tokenize.h"
#include "lightgrp.h"
#ifdef BlobPatternPatch
	#include "blob.h"
#endif
/** poviso:  Apr. 97 R.S. **/
#ifdef POVISO
#include "isosrf.h"
FUNCTION *Parse_Function (void);
FUNCTION *TFunc;
extern   DBL iso_p[ISOSURFACE_MAXPARM];
#endif
/** --- **/
#ifdef ObjectPatternPatch
#include "objects.h"
#endif

#ifdef PatternPatch
extern int token_count , line_count;
#endif

/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

#define ADD_TNORMAL if (Tnormal == NULL) {if ((Default_Texture->Tnormal) != NULL) \
 Tnormal = Copy_Tnormal ((Default_Texture->Tnormal)); else Tnormal = Create_Tnormal ();\
 Texture->Tnormal=Tnormal;};

/*****************************************************************************
* Local typedefs
******************************************************************************/



/*****************************************************************************
* Local variables
******************************************************************************/

TEXTURE *Default_Texture;



/*****************************************************************************
* Static functions
******************************************************************************/

/* NK 1998 */
static void Parse_Image_Pattern (TPATTERN *TPattern);
/* NK ---- */
static void Parse_Bump_Map (TNORMAL *Tnormal);
static void Parse_Image_Map (PIGMENT *Pigment);
static void Parse_Slope (TPATTERN *New);
static void Parse_Pattern (TPATTERN *New, int TPat_Type);
static TEXTURE *Parse_Vers1_Texture (void);
static TEXTURE *Parse_Tiles (void);
static TEXTURE *Parse_Material_Map (void);
static void Parse_Texture_Transform (TEXTURE *Texture);
static TURB *Check_Turb (WARP **Warps_Ptr);
#ifndef GlowPatch
static void Parse_Warp (WARP **Warp_Ptr);
#endif
static void Check_BH_Parameters (BLACK_HOLE *bh);
static void Warn_Interior (char *s);

#ifdef BlobPatternPatch
static BLOB_PATTERN_DATA * Create_Blob_Pattern_Component();/*Chris Huff blob pattern*/
static void Add_Blob_Pattern_Component(TPATTERN * pat, BLOB_PATTERN_DATA * comp);/*Chris Huff blob pattern*/
static void Parse_Blob_Pattern_Component_Mods(BLOB_PATTERN_DATA * comp);/*Chris Huff blob pattern*/
static void Parse_Blob_Pigment_Component_Mods(BLOB_PATTERN_DATA * comp);/*Chris Huff blob pattern*/
#endif
/** poviso: July 12, '96 R.S. **/
#ifdef POVISO
   extern int tempfunc;
#endif
/** --- **/
#ifdef PatternPatch
/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Ronald Parker
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/
static void Make_Pattern_Image ( IMAGE *Image, PIGMENT *Pigment, int Gray16 )
{
	int i = 0;
	int j = 0;
	int k = 0;
	DBL val = 0;
	IMAGE_LINE *line_data = NULL;
	COLOUR colour;
	VECTOR Point;
	INTERSECTION Dummy_Intersect;
	unsigned int color16;

	Image->iwidth  = Image->width;
	Image->iheight = Image->height;
	Image->Colour_Map_Size = 0;
	Image->Colour_Map = NULL;
	Image->data.rgb_lines = (IMAGE_LINE *)POV_MALLOC(Image->iheight * sizeof(IMAGE_LINE), "pattern image");

	k = Image->iwidth * sizeof(unsigned char);
	Point[Y] = 0;
	for (i = 0; i < Image->iheight; i++)
	{ 
		Point[Z] = 1-((DBL)(i) / (Image->height-1));
		line_data = &Image->data.rgb_lines[i];
		line_data->red    = (unsigned char *)POV_MALLOC((size_t)k, "pattern image line");
		line_data->green  = (unsigned char *)POV_MALLOC((size_t)k, "pattern image line");
		line_data->blue   = (unsigned char *)POV_MALLOC((size_t)k, "pattern image line");
		if ( Gray16 ) 
		{
			line_data->transm = NULL;
		}
		else 
		{
			line_data->transm = (unsigned char *)POV_MALLOC((size_t)k, "pattern image line");
		}
		for ( j = 0; j < Image->iwidth; j++ ) 
		{
			Point[X] = ((DBL)j / (Image->width-1));
			Compute_Pigment( colour, Pigment, Point, &Dummy_Intersect );
			if ( Gray16 ) 
			{
				color16=65535*colour[RED];
				line_data->red[j] = (color16&0xff00)>>8;
				line_data->green[j] = color16 & 0xff; 
				line_data->blue[j] = 0;
			}
			else 
			{
				line_data->red[j]   = 255*colour[RED];
				line_data->blue[j]  = 255*colour[BLUE];
				line_data->green[j] = 255*colour[GREEN];
				line_data->transm[j] = 255*colour[TRANSM];
			}
		}
		token_count++;
		if (token_count > 15)
		{
			token_count = 0;
			COOPERATE_0
			Check_User_Abort(FALSE);
			Status_Info(".");
			line_count++;
			if (line_count > 78)
			{
				line_count = 0;
				Status_Info ("\n");
			}
		}
	}
}
#endif
/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

IMAGE *Parse_Image (int Legal)
{
	IMAGE *Image;
	VECTOR Local_Vector;
	char *Name;
	#ifdef PatternPatch
		PIGMENT *Local_Pigment;
		int Gray16;
	#endif
	Image = Create_Image ();
	Image->Image_Type = Legal;
	if (Legal & GRAD_FILE)
	{
		EXPECT
			CASE_VECTOR
				Warn(150, "Old style orientation vector or map type not supported.  Ignoring value.");
				Parse_Vector (Local_Vector);
			END_CASE
	
			OTHERWISE
				UNGET
				EXIT
			END_CASE
		END_EXPECT
	}

	EXPECT
		#ifdef PatternPatch
			CASE (PATTERN_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				Image->File_Type = TGA_FILE;
				Image->width = Parse_Float();
				Parse_Comma();
				Image->height = Parse_Float();
				Local_Pigment = Create_Pigment();
				Parse_Begin ();
				EXPECT
					CASE (HF_GRAY_16_TOKEN)
						Gray16=1;
						EXIT
					END_CASE

					OTHERWISE
						Gray16=0;
						UNGET
						EXIT
					END_CASE
				END_EXPECT
				Parse_Pigment( &Local_Pigment );
				Parse_End ();
				Post_Pigment( Local_Pigment );

				Make_Pattern_Image( Image, Local_Pigment, Gray16 );
				Destroy_Pigment( Local_Pigment );
				EXIT
			END_CASE
		#endif
		CASE (IFF_TOKEN)
			Image->File_Type = IFF_FILE;
			Name=Parse_String();
			Read_Iff_Image (Image, Name);
			POV_FREE(Name);
			EXIT
		END_CASE

		CASE (GIF_TOKEN)
			#ifdef BurnAllGifs
				Error("Support for GIF and POT images has been removed due to potential patent \nissues.  This has been done to legally protect the authors of \nMegaPov and POV-Ray.  We are sorry for this inconvenience.  You can \nuse gif2png (which can currently be dowlnoaded from \nhttp://www.tuxedo.org/~esr/gif2png/ or\nhttp://pegasus.uni-paderborn.de/enno/burnallgifs/)\nto convert your GIF images to PNG images.\n");
			#else
				Image->File_Type = GIF_FILE;
				Name=Parse_String();
				Read_Gif_Image(Image, Name);
				POV_FREE(Name);
				EXIT
			#endif
		END_CASE

		CASE (POT_TOKEN)
			#ifdef BurnAllGifs
				Error("Support for GIF and POT images has been removed due to potential patent \nissues.  This has been done to legally protect the authors of \nMegaPov and POV-Ray.  We are sorry for this inconvenience.  You can \nuse gif2png (which can currently be dowlnoaded from \nhttp://www.tuxedo.org/~esr/gif2png/ or\nhttp://pegasus.uni-paderborn.de/enno/burnallgifs/)\nto convert your GIF images to PNG images.\n");
			#else
				Image->File_Type = POT_FILE;
				Name=Parse_String();
				Read_Gif_Image(Image, Name);
				POV_FREE(Name);
				EXIT
			#endif
		END_CASE

		CASE (SYS_TOKEN)
			Image->File_Type = SYS_FILE;
			Name=Parse_String();
			READ_SYS_IMAGE(Image, Name);
			POV_FREE(Name);
			EXIT
		END_CASE

		CASE (TGA_TOKEN)
			Image->File_Type = TGA_FILE;
			Name=Parse_String();
			Read_Targa_Image(Image, Name);
			POV_FREE(Name);
			EXIT
		END_CASE

		CASE (PNG_TOKEN)
			Image->File_Type = PNG_FILE;
			Name=Parse_String();
			Read_Png_Image(Image, Name);
			POV_FREE(Name);
			EXIT
		END_CASE

		CASE (PGM_TOKEN)
			Image->File_Type = PGM_FILE;
			Name=Parse_String();
			Read_PGM_Image(Image, Name);
			POV_FREE(Name);
			EXIT
		END_CASE

		CASE (PPM_TOKEN)
			Image->File_Type = PPM_FILE;
			Name=Parse_String();
			Read_PPM_Image(Image, Name);
			POV_FREE(Name);
			EXIT
		END_CASE

		OTHERWISE
			Parse_Error_Str ("map file spec");
		END_CASE
	END_EXPECT
	/* MWW 2000 - added error check */
	if(Image->width == 0 || Image->height == 0)
		Error("Image file has a dimension of 0.");
		/* MWW ---- */
	if (!(Image->File_Type & Legal))
		Error ("File type not supported here.");
	return (Image);
  }



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static void Parse_Image_Map (PIGMENT *Pigment)
{
	int reg;
	IMAGE *Image;
	Parse_Begin();
	Image = Parse_Image (IMAGE_FILE);
	Image->Use_Colour_Flag = TRUE;

	EXPECT                   /* Look for image_attribs */
		CASE (ONCE_TOKEN)
			Image->Once_Flag=TRUE;
		END_CASE

		CASE (INTERPOLATE_TOKEN)
			Image->Interpolation_Type = (int)Parse_Float();
		END_CASE
	
		#ifdef ImagePhasePatch
			CASE (FREQUENCY_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(60);
				#endif
				Pigment->Frequency = Parse_Float ();
			END_CASE

			CASE (PHASE_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(60);
				#endif
				Pigment->Phase =  Parse_Float ();
			END_CASE
		#endif
		
		CASE (MAP_TYPE_TOKEN)
			Image->Map_Type = (int) Parse_Float ();
		END_CASE

		CASE (USE_COLOUR_TOKEN)
			Image->Use_Colour_Flag = TRUE;
		END_CASE

		CASE (USE_INDEX_TOKEN)
			Image->Use_Colour_Flag = FALSE;
		END_CASE

		CASE (ALPHA_TOKEN)
			Warn(155, "Keyword ALPHA discontinued. Use FILTER instead.");

		CASE (COLOUR_KEY_TOKEN)
			switch(Token.Function_Id)
			{
				CASE (FILTER_TOKEN)
					EXPECT
					CASE (ALL_TOKEN)
					{
						DBL filter;
						filter = Parse_Float();
						#ifdef AllTransmitPatch
							Image->AllFilter = filter;
						#endif
						for (reg = 0 ; reg < Image->Colour_Map_Size ; reg++)
							Image->Colour_Map[reg].Filter = (unsigned short) (filter *255.0);
					}
					EXIT
				END_CASE
		
				OTHERWISE
					UNGET
					reg = (int)(Parse_Float() + 0.01);
					if (Image->Colour_Map == NULL)
						Not_With ("filter","images without a color palette");
					if ((reg < 0) || (reg >= Image->Colour_Map_Size))
						Error ("FILTER color register value out of range.");
					Parse_Comma();
					Image->Colour_Map[reg].Filter   = (unsigned short) (255.0 * Parse_Float());
					EXIT
				END_CASE
			END_EXPECT
			Pigment->Flags |= HAS_FILTER;
			break;

		CASE (TRANSMIT_TOKEN)
			EXPECT
				CASE (ALL_TOKEN)
					{
						DBL transmit;
						transmit = Parse_Float();
						#ifdef AllTransmitPatch
							Image->AllTransmit = transmit;
						#endif
						for (reg = 0 ; reg < Image->Colour_Map_Size ; reg++)
							Image->Colour_Map[reg].Transmit  = (unsigned short) (transmit *255.0);
					}
					EXIT
					END_CASE
				OTHERWISE
					UNGET
					reg = (int)(Parse_Float() + 0.01);
					if (Image->Colour_Map == NULL)
						Not_With ("transmit","non color-mapped image");
					if ((reg < 0) || (reg >= Image->Colour_Map_Size))
						Error ("TRANSMIT color register value out of range.");
					Parse_Comma();
					Image->Colour_Map[reg].Transmit = (unsigned short) (255.0 * Parse_Float());
					EXIT
				END_CASE
			END_EXPECT
			Pigment->Flags |= HAS_FILTER;
			break;
			OTHERWISE
				UNGET
				Parse_Error_Str ("filter or transmit");
			END_CASE
		}
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT
	Pigment->Vals.Image=Image;
	Parse_End();
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static void Parse_Bump_Map (TNORMAL *Tnormal)
{
	IMAGE *Image;

	Parse_Begin();

	Image = Parse_Image(IMAGE_FILE);
	Image->Use_Colour_Flag = TRUE;

	EXPECT
		CASE (ONCE_TOKEN)
			Image->Once_Flag=TRUE;
		END_CASE

		CASE (MAP_TYPE_TOKEN)
			Image->Map_Type = (int) Parse_Float ();
		END_CASE

		CASE (INTERPOLATE_TOKEN)
			Image->Interpolation_Type = (int)Parse_Float();
		END_CASE

		CASE (BUMP_SIZE_TOKEN)
			Tnormal->Amount = Parse_Float ();
		END_CASE

		CASE (USE_COLOUR_TOKEN)
			Image->Use_Colour_Flag = TRUE;
		END_CASE

		CASE (USE_INDEX_TOKEN)
			Image->Use_Colour_Flag = FALSE;
		END_CASE

		OTHERWISE
		UNGET
		EXIT
		END_CASE
	END_EXPECT
	Tnormal->Vals.Image=Image;
	Parse_End();
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Nathan Kopp
*
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static void Parse_Image_Pattern (TPattern)
TPATTERN *TPattern;
  {
	IMAGE *Image;
	Parse_Begin();
	Image = Parse_Image(IMAGE_FILE);
	Image->Use_Colour_Flag = TRUE;

	EXPECT
		CASE (ONCE_TOKEN)
			Image->Once_Flag=TRUE;
		END_CASE

		CASE (MAP_TYPE_TOKEN)
			Image->Map_Type = (int) Parse_Float ();
		END_CASE

		CASE (INTERPOLATE_TOKEN)
			Image->Interpolation_Type = (int)Parse_Float();
		END_CASE

		CASE (USE_COLOUR_TOKEN)
			Image->Use_Colour_Flag = USE_COLOUR;
		END_CASE

		CASE (USE_INDEX_TOKEN)
			Image->Use_Colour_Flag = USE_INDEX;
		END_CASE

		CASE (USE_ALPHA_TOKEN)
			Image->Use_Colour_Flag = USE_ALPHA;
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT
	TPattern->Vals.Image=Image;
	Parse_End();
}


/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

void Parse_Pigment (PIGMENT **Pigment_Ptr)
{
	EXPECT            /* Look for [pigment_id] */
		CASE (PIGMENT_ID_TOKEN)
			Destroy_Pigment(*Pigment_Ptr);
			*Pigment_Ptr = Copy_Pigment ((PIGMENT *) Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT    /* End pigment_id */
	Parse_Pattern((TPATTERN *)(*Pigment_Ptr),PIGMENT_TYPE);
	if (Not_In_Default && ((*Pigment_Ptr)->Type == NO_PATTERN))
	{
		Warn(170, "Pigment type unspecified or not 1st item.");
	}
}


/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   H.-D. Fink Jan 99
*
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static void Parse_Slope(TPATTERN *New)
{
	int i;
	SNGL Tot_Len;

	New->Type = SLOPE_PATTERN;
	Parse_Vector (New->Vals.Slope.Slope_Vector);
	New->Vals.Slope.Altit_Vector[X] = 0.0;
	New->Vals.Slope.Altit_Vector[Y] = 0.0;
	New->Vals.Slope.Altit_Vector[Z] = 0.0;
	New->Vals.Slope.Slope_Mod[U]    = 0.0;
	New->Vals.Slope.Altit_Mod[U]    = 0.0;
	New->Vals.Slope.Slope_Mod[V]    = 0.0;
	New->Vals.Slope.Altit_Mod[V]    = 0.0;
	EXPECT
		CASE(COMMA_TOKEN)
			Parse_Vector (New->Vals.Slope.Altit_Vector);
			EXPECT
				CASE(COMMA_TOKEN)
					Parse_UV_Vect(New->Vals.Slope.Slope_Mod);
					Parse_Comma();
					Parse_UV_Vect(New->Vals.Slope.Altit_Mod);
					if (New->Vals.Slope.Slope_Mod[V] == New->Vals.Slope.Slope_Mod[U])
					{
						New->Vals.Slope.Slope_Mod[U] = 0.0;
						New->Vals.Slope.Slope_Mod[V] = 0.0;
						Warn (0, "Zero slope range, ignoring.");
					}
					else
					{
						New->Vals.Slope.Slope_Mod[V] -= New->Vals.Slope.Slope_Mod[U];
					}
					if (New->Vals.Slope.Altit_Mod[V] == New->Vals.Slope.Altit_Mod[U])
					{
						New->Vals.Slope.Altit_Mod[U] = 0.0;
						New->Vals.Slope.Altit_Mod[V] = 0.0;
						Warn (0, "Zero gradient range, ignoring.");
					}
					else
					{
						New->Vals.Slope.Altit_Mod[V] -= New->Vals.Slope.Altit_Mod[U];
					}
					EXIT
				END_CASE
	
				OTHERWISE
					UNGET
					EXIT
				END_CASE
			END_EXPECT
	
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	VLength(New->Vals.Slope.Slope_Len, New->Vals.Slope.Slope_Vector)
	VLength(New->Vals.Slope.Altit_Len, New->Vals.Slope.Altit_Vector)
	if (New->Vals.Slope.Slope_Len != 0.0) 
		VNormalizeEq(New->Vals.Slope.Slope_Vector);
	if (New->Vals.Slope.Altit_Len != 0.0) 
		VNormalizeEq(New->Vals.Slope.Altit_Vector);
	Tot_Len = New->Vals.Slope.Slope_Len + New->Vals.Slope.Altit_Len;
	if (Tot_Len != 0.0)
	{
		New->Vals.Slope.Slope_Len /= Tot_Len;
		New->Vals.Slope.Altit_Len  = 1.0 - New->Vals.Slope.Slope_Len;
	}
	else
	{
		Error ("Zero length for both slope and gradient.");
	}
	New->Vals.Slope.Slope_Base = New->Vals.Slope.Altit_Base = 0; /* mark unused */
	for (i=X; i<=Z; i++)
	{
		if (New->Vals.Slope.Slope_Vector[i] == 1.0)
		{
			New->Vals.Slope.Slope_Base = i+1;
			break;
		}
		else if (New->Vals.Slope.Slope_Vector[i] == -1.0)
		{
			New->Vals.Slope.Slope_Base = -(i+1);
			break;
		}
	}
	for (i=X; i<=Z; i++)
	{
		if (New->Vals.Slope.Altit_Vector[i] == 1.0)
		{
			New->Vals.Slope.Altit_Base = i+1;
			break;
		}
		else if (New->Vals.Slope.Altit_Vector[i] == -1.0)
		{
			New->Vals.Slope.Altit_Base = -(i+1);
			break;
		}
	}
}

/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static void SetFractalDefaults(TPATTERN *New)
{
	New->Vals.Fractal.interior_type = DEFAULT_FRACTAL_INTERIOR_TYPE;
	New->Vals.Fractal.exterior_type = DEFAULT_FRACTAL_EXTERIOR_TYPE;
	New->Vals.Fractal.efactor = DEFAULT_FRACTAL_EXTERIOR_FACTOR;
	New->Vals.Fractal.ifactor = DEFAULT_FRACTAL_INTERIOR_FACTOR;
}


/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static void Parse_Pattern (TPATTERN *New, int TPat_Type)
{
	VECTOR Local_Vector;
	COLOUR Local_Colour;
	MATRIX Local_Matrix;
	TRANSFORM Local_Trans;
	TURB *Local_Turb;
	unsigned short Old_Type=New->Type;
	IMAGE *Old_Image = NULL;
	DENSITY_FILE *Old_Density_File = NULL;
	#ifdef ProximityPatch
		int temp;
	#endif
	#ifdef BlobPatternPatch
		BLOB_PATTERN_DATA * newComponent;
	#endif
	if (Old_Type==BITMAP_PATTERN)
		Old_Image=New->Vals.Image;

	if (Old_Type==DENSITY_FILE_PATTERN)
	Old_Density_File=New->Vals.Density_File;

	EXPECT
		#ifdef PigmentPatternPatch
			CASE(PIGMENT_PATTERN_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(50);
				#endif
				Parse_Begin();
				New->Type = PIGMENT_PATTERN;
				New->Vals.Pigment = Create_Pigment();
				Parse_Pigment(&(New->Vals.Pigment));
				Post_Pigment(New->Vals.Pigment);
				Parse_End();
			END_CASE
		#endif

		#ifdef BlobPatternPatch
			CASE (BLOB_TOKEN)/*Chris Huff blob pattern*/
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				Parse_Begin();
				New->Type = BLOB_PATTERN;
				New->Vals.Blob.blob_threshold = 0;
				New->Vals.Blob.max_density = 1;
				New->Vals.Blob.blob_dat = NULL;
				EXPECT
					CASE (BLOB_TOKEN)
						#ifdef UnofficialBlocking
							parseUnofficialFeature(60);
						#endif
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 4;
						Add_Blob_Pattern_Component(New, newComponent);
						Parse_Begin();
						EXPECT
							CASE(OBJECT_ID_TOKEN)
								newComponent->blob =Copy_Object((OBJECT *)Token.Data);
								if ( ((OBJECT*)newComponent->blob)->Type!=BLOB_OBJECT)
									Error("\nOnly blob allowed.\n");
								EXIT
							END_CASE

							OTHERWISE
								UNGET
								EXIT
							END_CASE
						END_EXPECT
		              Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();

						/*YS sept 18 2000 bugfix */
						newComponent->pigment = Create_Pigment();
						/*YS*/               
						Parse_Blob_Pigment_Component_Mods(newComponent);
						Parse_End();
					END_CASE
			
					CASE (PIGMENT_TOKEN)
						#ifdef UnofficialBlocking
							parseUnofficialFeature(40);
						#endif
						Parse_Begin();
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 3;
						Add_Blob_Pattern_Component(New, newComponent);

						newComponent->pigment = Create_Pigment();
						Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();
						Parse_Pigment(&newComponent->pigment);
						Post_Pigment(newComponent->pigment);
						Parse_Blob_Pattern_Component_Mods(newComponent);
						Parse_End();
					END_CASE

					CASE (BOX_TOKEN)
						#ifdef UnofficialBlocking
							parseUnofficialFeature(60);
						#endif
						Parse_Begin();
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 2;
						Add_Blob_Pattern_Component(New, newComponent);
						Parse_Vector(newComponent->center);
						Parse_Comma();
						Parse_Vector(newComponent->pointB);
						Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();
						Parse_Blob_Pattern_Component_Mods(newComponent);
						Parse_End();
					END_CASE

					CASE (CYLINDER_TOKEN)
						Parse_Begin();
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 1;
						Add_Blob_Pattern_Component(New, newComponent);
						Parse_Vector(newComponent->center);
						Parse_Comma();
						Parse_Vector(newComponent->pointB);
						Parse_Comma();
						newComponent->radius = Parse_Float();
						Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();
						Parse_Blob_Pattern_Component_Mods(newComponent);
						Parse_End();
					END_CASE

					CASE (SPHERE_TOKEN)
						Parse_Begin();
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 0;
						Add_Blob_Pattern_Component(New, newComponent);
						Parse_Vector(newComponent->center);
						Parse_Comma();
						newComponent->radius = Parse_Float();
						Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();
						Parse_Blob_Pattern_Component_Mods(newComponent);
						Parse_End();
					END_CASE

					CASE (THRESHOLD_TOKEN)
						New->Vals.Blob.blob_threshold = Parse_Float();
					END_CASE

					CASE (MAX_DENSITY_TOKEN)
						New->Vals.Blob.max_density = Parse_Float();
					END_CASE

					OTHERWISE
						UNGET
						EXIT
					END_CASE
				END_EXPECT
				Parse_End();
				EXIT
			END_CASE
		#endif
		CASE (AGATE_TOKEN)
			New->Type = AGATE_PATTERN;
			Check_Turb(&(New->Warps));
			New->Vals.Agate_Turb_Scale = 1.0;
			EXIT
		END_CASE

		CASE (BOZO_TOKEN)
			New->Type = BOZO_PATTERN;
			EXIT
		END_CASE

		CASE (GRANITE_TOKEN)
			New->Type = GRANITE_PATTERN;
			EXIT
		END_CASE

		CASE (LEOPARD_TOKEN)
			New->Type = LEOPARD_PATTERN;
			EXIT
		END_CASE

		CASE (MARBLE_TOKEN)
			New->Type = MARBLE_PATTERN;
			New->Wave_Type = TRIANGLE_WAVE;
			EXIT
		END_CASE

		CASE (MAGNET1M_TOKEN)
			New->Type = MAGNET1M_PATTERN;
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (MAGNET1J_TOKEN)
			New->Type = MAGNET1J_PATTERN;
			Parse_UV_Vect(New->Vals.Fractal.Coord);
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (MAGNET2M_TOKEN)
			New->Type = MAGNET2M_PATTERN;
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (MAGNET2J_TOKEN)
			New->Type = MAGNET2J_PATTERN;
			Parse_UV_Vect(New->Vals.Fractal.Coord);
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (MANDEL_TOKEN)
			New->Type = MANDEL_PATTERN;
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (MANDEL3_TOKEN)
			New->Type = MANDEL3_PATTERN;
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (MANDEL4_TOKEN)
			New->Type = MANDEL4_PATTERN;
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (JULIA_TOKEN)
			New->Type = JULIA_PATTERN;
			Parse_UV_Vect(New->Vals.Fractal.Coord);
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (JULIA3_TOKEN)
			New->Type = JULIA3_PATTERN;
			Parse_UV_Vect(New->Vals.Fractal.Coord);
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (JULIA4_TOKEN)
			New->Type = JULIA4_PATTERN;
			Parse_UV_Vect(New->Vals.Fractal.Coord);
			New->Vals.Fractal.Iterations = (int)Parse_Float();
			SetFractalDefaults(New);
			EXIT
		END_CASE

		CASE (ONION_TOKEN)
			New->Type = ONION_PATTERN;
			EXIT
		END_CASE

		CASE (SPIRAL1_TOKEN)
			New->Type = SPIRAL1_PATTERN;
			New->Vals.Arms = (short)Parse_Float ();
			New->Wave_Type = TRIANGLE_WAVE;
			EXIT
		END_CASE

		CASE (SPIRAL2_TOKEN)
			New->Type = SPIRAL2_PATTERN;
			New->Vals.Arms = (short)Parse_Float ();
			New->Wave_Type = TRIANGLE_WAVE;
			EXIT
		END_CASE

		CASE (SPOTTED_TOKEN)
			New->Type = SPOTTED_PATTERN;
			EXIT
		END_CASE

		CASE (WOOD_TOKEN)
			New->Type = WOOD_PATTERN;
			New->Wave_Type = TRIANGLE_WAVE;
			EXIT
		END_CASE

		CASE (GRADIENT_TOKEN)
			New->Type = GRADIENT_PATTERN;
			Parse_Vector (New->Vals.Gradient);
			EXIT
		END_CASE

		CASE (RADIAL_TOKEN)
			New->Type = RADIAL_PATTERN;
			EXIT
		END_CASE
		#ifdef PolaricalPatch
			CASE (POLARICAL_TOKEN)             /* Added by bajcik */
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = POLARICAL_PATTERN;
				EXIT
			END_CASE
		#endif
		
		CASE (CRACKLE_TOKEN)
			New->Type = CRACKLE_PATTERN;
			#ifdef CracklePatch
				New->Vals.Crackle.IsSolid = 0;
				New->Vals.Crackle.Form[X] = -1;
				New->Vals.Crackle.Form[Y] = 1;
				New->Vals.Crackle.Form[Z] = 0;
				New->Vals.Crackle.Metric[X] = 
				New->Vals.Crackle.Metric[Y] = 
				New->Vals.Crackle.Metric[Z] = 2;
				New->Vals.Crackle.Offset = 0;
				New->Vals.Crackle.Dim = 3;
				New->Vals.Crackle.cv = POV_MALLOC( 125*sizeof(VECTOR), "crackle cache");
				New->Vals.Crackle.lastseed = 0x8000000; 
			#endif
			EXIT
		END_CASE

		CASE_COLOUR
			if ((TPat_Type != PIGMENT_TYPE) && (TPat_Type != DENSITY_TYPE))
			{
				Only_In("color","pigment or density");
			}
			New->Type = PLAIN_PATTERN;
			Parse_Colour (((PIGMENT *)New)->Colour);
			EXIT
		END_CASE

		/** poviso: Apr. '97 R.S. **/
		#ifdef POVISO
			CASE (FUNCTION_TOKEN)
				New->Type = FUNCTION_PATTERN;
				TFunc=Parse_Function();
				New->Vals.Function=TFunc;
				Load_Function((FUNCTION *)(New->Vals.Function), TFunc->func_name);
				tempfunc=TRUE;
				EXIT
			END_CASE
		#endif
		/** --- **/

		CASE (CHECKER_TOKEN)
			New->Type = CHECKER_PATTERN;
			New->Frequency = 0.0;
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Blend_List(2,&Check_Default_Map,TPat_Type);
			/* NK delta */
			if (TPat_Type == NORMAL_TYPE)
				((TNORMAL *)New)->Delta = 0.02;
			/* NK ---- */
			EXIT
		END_CASE

		#ifdef ProximityPatch
			CASE (PROXIMITY_TOKEN)/*Chris Huff proximity pattern*/
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				Parse_Begin();
				New->Type = PROXIMITY_PATTERN;
				New->Vals.Proximity.max_density   = 0;
				New->Vals.Proximity.samples   = 5;
				New->Vals.Proximity.sides   = 2;
				New->Vals.Proximity.proxCalcMthd   = 1;
				New->Vals.Proximity.sample_bailout   = 0;
				New->Vals.Proximity.proxDist   = 1;
				New->Vals.Proximity.sampleMthd = 2;
				/*YS oct 12 Bug fix set to zero otherwise random if not specified*/
				Make_Vector(New->Vals.Proximity.sampleWeight,0,0,0);
				/*YS*/
				EXPECT
					CASE (OBJECT_ID_TOKEN)
					New->Vals.Proximity.proxObject = Copy_Object((OBJECT *) Token.Data);
					EXIT
					END_CASE
			
					OTHERWISE
						UNGET /* NK moved UNGET and changed NULL to Parse_Object() */
						New->Vals.Proximity.proxObject = Parse_Object();
						EXIT
					END_CASE
				END_EXPECT
				if (New->Vals.Proximity.proxObject == NULL) 
				{
					Error ("Object or object dentifier Expected.");
				}
				Parse_Comma();
				New->Vals.Proximity.proxDist = Parse_Float();		
				EXPECT
					CASE (SIDES_TOKEN)
						temp = (int)Parse_Float();
						if (temp<0 || temp>2)
							Error("sides must be 0, 1, or 2.\n");
						New->Vals.Proximity.sides = temp;
					END_CASE
		
					CASE (SAMPLE_BAILOUT_TOKEN)
						New->Vals.Proximity.sample_bailout = (int)Parse_Float();
					END_CASE

					CASE (SAMPLE_WEIGHTING_TOKEN)
						#ifdef UnofficialBlocking
							parseUnofficialFeature(60);
						#endif
						Parse_Vector(New->Vals.Proximity.sampleWeight);
					END_CASE

					CASE (SAMPLES_TOKEN)
						New->Vals.Proximity.samples = Parse_Float();
					END_CASE

					CASE (MAX_DENSITY_TOKEN)
						New->Vals.Proximity.max_density = Parse_Float();
					END_CASE

					CASE (TYPE_TOKEN)
						temp = (int)Parse_Float();
						if (temp<0 || temp>3)
							Error("proximity type must be 0, 1, 2, or 3.\n");
						New->Vals.Proximity.proxCalcMthd = temp;
						if(New->Vals.Proximity.proxCalcMthd == 3)
						{
							Parse_Comma();
							New->Vals.Proximity.falloff = Parse_Float();
						}
					END_CASE

					CASE (METHOD_TOKEN)
						temp = (int)Parse_Float();
						if (temp<0 || temp>3)
							Error("proximity method must be 0, 1, 2, or 3.\n");
						New->Vals.Proximity.sampleMthd = temp;
					END_CASE

					OTHERWISE
						UNGET
						EXIT
					END_CASE
				END_EXPECT
				Parse_End();
				if(New->Vals.Proximity.sample_bailout == 0)
					New->Vals.Proximity.sample_bailout = 5*New->Vals.Proximity.samples;
				EXIT
			END_CASE
		#endif

		#ifdef ObjectPatternPatch
			CASE (OBJECT_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				Parse_Begin();
				EXPECT
					CASE (OBJECT_ID_TOKEN)
						New->Vals.Object = Copy_Object((OBJECT *) Token.Data);
						EXIT
					END_CASE

					OTHERWISE
						UNGET /* NK moved UNGET and changed NULL to Parse_Object() */
						New->Vals.Object = Parse_Object();
						EXIT
					END_CASE
				END_EXPECT

				if (New->Vals.Object == NULL) 
				{
					Error ("object or object identifier expected.");
				}
				New->Type = OBJECT_PATTERN;
				New->Frequency = 0.0;
				Destroy_Blend_Map(New->Blend_Map);
				New->Blend_Map = Parse_Blend_List(2,&Check_Default_Map,TPat_Type);
				Parse_End();
				EXIT
			END_CASE
		#endif   

		CASE (BRICK_TOKEN)
			if (New->Type!=BRICK_PATTERN)
			{
				Make_Vector(New->Vals.Brick.Size,8.0,3.0,4.5);
				New->Vals.Brick.Mortar=0.5-Small_Tolerance*2.0;
				New->Type = BRICK_PATTERN;
			}
			New->Frequency = 0.0;
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Blend_List(2,&Brick_Default_Map,TPat_Type);
			/* NK delta */
			if (TPat_Type == NORMAL_TYPE)
				((TNORMAL *)New)->Delta = 0.02;
			/* NK ---- */
			EXIT
		END_CASE

		CASE (HEXAGON_TOKEN)
			New->Type = HEXAGON_PATTERN;
			New->Frequency = 0.0;
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Blend_List(3,&Hex_Default_Map,TPat_Type);
			/* NK delta */
			if (TPat_Type == NORMAL_TYPE)
				((TNORMAL *)New)->Delta = 0.02;
			/* NK ---- */
			EXIT
		END_CASE

		#ifdef TrianglulairSquarePatch 
			CASE (SQUARE_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = SQUARE_PATTERN;
				New->Frequency = 0.0;
				Destroy_Blend_Map(New->Blend_Map);
				New->Blend_Map = Parse_Blend_List(4,&Square_Default_Map,TPat_Type);
				EXIT
			END_CASE

			CASE (TRIANGULAR_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = TERNAIRE_PATTERN;
				New->Frequency = 0.0;
				Destroy_Blend_Map(New->Blend_Map);
				New->Blend_Map = Parse_Blend_List(6,&Ter_Default_Map,TPat_Type);
				EXIT
			END_CASE
			#endif
			CASE (IMAGE_MAP_TOKEN)
				if ((TPat_Type != PIGMENT_TYPE) && (TPat_Type != DENSITY_TYPE))
				{
					Only_In("image_map","pigment");
				}

				if (Old_Type==BITMAP_PATTERN) 
				{
					Destroy_Image(Old_Image);
				}   

				New->Type = BITMAP_PATTERN;
				#ifndef ImagePhasePatch
					New->Frequency = 0.0;
				#else
					New->Frequency = 1.0;
					New->Phase = 0.0;
				#endif
				/*1999 June 10 Bugfix from  Edna Dornblazer*/
				New->Vals.Image=NULL;
				/*---*/

				Parse_Image_Map ((PIGMENT *)New);
				EXIT
			END_CASE

			#ifdef BlobPatternPatch
				CASE (BLOB_PIGMENT_TOKEN)/*Chris Huff blob pigment*/
					#ifdef UnofficialBlocking
						parseUnofficialFeature(30);
					#endif
					if((TPat_Type != PIGMENT_TYPE)&&(TPat_Type != DENSITY_TYPE))
					{
					Only_In("blob_pigment","pigment");
					}

					Parse_Begin();
					New->Type = BLOB_PIGMENT;
					New->Vals.Blob.blob_threshold = 0;
					New->Vals.Blob.max_density = 1;
					New->Vals.Blob.blob_dat = NULL;

					EXPECT
						CASE (BLOB_TOKEN)
						#ifdef UnofficialBlocking
							parseUnofficialFeature(60);
						#endif
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 4;
						Add_Blob_Pattern_Component(New, newComponent);
						Parse_Begin();
						EXPECT
							CASE(OBJECT_ID_TOKEN)
								newComponent->blob =Copy_Object((OBJECT *)Token.Data);
								if ( ((OBJECT*)newComponent->blob)->Type!=BLOB_OBJECT)
									Error("\nOnly blob allowed.\n");
								EXIT
							END_CASE

							OTHERWISE
								UNGET
								EXIT
							END_CASE
						END_EXPECT
		              Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();

						/*YS sept 18 2000 bugfix */
						newComponent->pigment = Create_Pigment();
						/*YS*/               
						Parse_Blob_Pigment_Component_Mods(newComponent);
						Parse_End();
					END_CASE


					CASE (PIGMENT_TOKEN)
						#ifdef UnofficialBlocking
							parseUnofficialFeature(60);
						#endif
						newComponent = Create_Blob_Pattern_Component();
						Parse_Begin();
						newComponent->type = 3;
						Add_Blob_Pattern_Component(New, newComponent);
						if(newComponent->pigment == NULL)
							newComponent->pigment = Create_Pigment();
						Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();
						Parse_Pigment(&newComponent->pigment);
						Post_Pigment(newComponent->pigment);
						Parse_Blob_Pigment_Component_Mods(newComponent);
						Parse_End();
					END_CASE

					CASE (BOX_TOKEN)
						#ifdef UnofficialBlocking
							parseUnofficialFeature(60);
						#endif
						Parse_Begin();
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 2;
						Add_Blob_Pattern_Component(New, newComponent);
						Parse_Vector(newComponent->center);
						Parse_Comma();
						Parse_Vector(newComponent->pointB);
						Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();
						newComponent->pigment = Create_Pigment();
						Parse_Blob_Pigment_Component_Mods(newComponent);
						Parse_End();
					END_CASE

					CASE (CYLINDER_TOKEN)
						Parse_Begin();
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 1;
						Add_Blob_Pattern_Component(New, newComponent);
						Parse_Vector(newComponent->center);
						Parse_Comma();
						Parse_Vector(newComponent->pointB);
						Parse_Comma();
						newComponent->radius = Parse_Float();
						Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();
						newComponent->pigment = Create_Pigment();
						Parse_Blob_Pigment_Component_Mods(newComponent);
						Parse_End();
					END_CASE

					CASE (SPHERE_TOKEN)
						Parse_Begin();
						newComponent = Create_Blob_Pattern_Component();
						newComponent->type = 0;
						Add_Blob_Pattern_Component(New, newComponent);
						Parse_Vector(newComponent->center);
						Parse_Comma();
						newComponent->radius = Parse_Float();
						Parse_Comma();
						GET(STRENGTH_TOKEN)
						newComponent->strength = Parse_Float();
						newComponent->pigment = Create_Pigment();
						Parse_Blob_Pigment_Component_Mods(newComponent);
						Parse_End();
					END_CASE

					CASE (THRESHOLD_TOKEN)
						New->Vals.Blob.blob_threshold = Parse_Float();
					END_CASE

					CASE (MAX_DENSITY_TOKEN)
						New->Vals.Blob.max_density = Parse_Float();
					END_CASE

					OTHERWISE
						UNGET
						EXIT
					END_CASE
				END_EXPECT
		
				Parse_End();
				EXIT
			END_CASE
		#endif

		#ifdef NoisePigmentPatch
			CASE (NOISE_PIGMENT_TOKEN)/*Chris Huff noise pigment*/
				#ifdef UnofficialBlocking
					parseUnofficialFeature(60);
				#endif
				if((TPat_Type != PIGMENT_TYPE)&&(TPat_Type != DENSITY_TYPE))
				{
				Only_In("noise_pigment","pigment");
				}
				Parse_Begin();
				New->Type = NOISE_PIGMENT;
				New->Vals.NoisePigment.NoiseType = Parse_Float();
				Parse_Comma();
				Parse_Colour(New->Vals.NoisePigment.Min);
				Parse_Comma();
				Parse_Colour(New->Vals.NoisePigment.Max);
				Parse_End();
				EXIT
			END_CASE
	#endif

		CASE (BUMP_MAP_TOKEN)
			if (TPat_Type != NORMAL_TYPE)
			{
				Only_In("bump_map","normal");
			}
			if (Old_Type==BITMAP_PATTERN) 
			{
				Destroy_Image(Old_Image);
			}   
			New->Type = BITMAP_PATTERN;
			New->Frequency = 0.0;
			Parse_Bump_Map ((TNORMAL *)New);
			EXIT
		END_CASE

		CASE (WAVES_TOKEN)
			New->Type = WAVES_PATTERN;
			EXIT
		END_CASE

		CASE (RIPPLES_TOKEN)
			New->Type = RIPPLES_PATTERN;
			EXIT
		END_CASE

		CASE (WRINKLES_TOKEN)
			New->Type = WRINKLES_PATTERN;
			EXIT
		END_CASE

		CASE (BUMPS_TOKEN)
			New->Type = BUMPS_PATTERN;
			EXIT
		END_CASE

		#ifdef CellsPatch
			CASE (CELLS_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = CELLS_PATTERN;
				EXIT
			END_CASE
		#endif

		#ifdef VanSicklePatternPatch
			CASE (BLOTCHES_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = BLOTCHES_PATTERN;
				EXIT
			END_CASE

			CASE (BANDS_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = BANDS_PATTERN;
				EXIT
			END_CASE

			CASE (SHEET_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = SHEET_PATTERN;
				EXIT
			END_CASE
		#endif

		CASE (DENTS_TOKEN)
			New->Type = DENTS_PATTERN;
			EXIT
		END_CASE

		CASE (QUILTED_TOKEN)
			New->Type = QUILTED_PATTERN;
			New->Vals.Quilted.Control0 = 1.0;
			New->Vals.Quilted.Control1 = 1.0;
			New->Frequency = 0.0;
			EXIT
		END_CASE

		#ifdef CracklePatch
			CASE (FACETS_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = FACETS_PATTERN;
				New->Vals.Facets.Size = 0.1;
				New->Vals.Facets.UseCoords = 0;
				New->Vals.Facets.Metric[X] = 
				New->Vals.Facets.Metric[Y] = 
				New->Vals.Facets.Metric[Z] = 2;
				EXIT
			END_CASE
		#endif

		CASE (AVERAGE_TOKEN)
			New->Type = AVERAGE_PATTERN;
			EXIT
		END_CASE

		/* NK 1998 */
		CASE (IMAGE_PATTERN_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(10);
			#endif
			if ((Old_Type==BITMAP_PATTERN) || (Old_Type==IMAGE_PATTERN))
			{
				Destroy_Image(Old_Image);
			}
			New->Type = IMAGE_PATTERN;
			New->Frequency = 0.0;
			Parse_Image_Pattern (New);
			EXIT
		END_CASE
		/* NK 1998 */

		CASE (PLANAR_TOKEN)
			New->Type = PLANAR_PATTERN;
			EXIT
		END_CASE

		CASE (BOXED_TOKEN)
			New->Type = BOXED_PATTERN;
			EXIT
		END_CASE

		CASE (SPHERICAL_TOKEN)
			New->Type = SPHERICAL_PATTERN;
			EXIT
		END_CASE

		CASE (CYLINDRICAL_TOKEN)
			New->Type = CYLINDRICAL_PATTERN;
			EXIT
		END_CASE

		CASE (DENSITY_FILE_TOKEN)
			if (Old_Type==DENSITY_FILE_PATTERN) 
			{
				Destroy_Density_File(Old_Density_File);
			}   
			New->Type = DENSITY_FILE_PATTERN;
			New->Vals.Density_File = Create_Density_File();
			GET(DF3_TOKEN);
			New->Vals.Density_File->Data->Name = Parse_String();
			Read_Density_File(New->Vals.Density_File);
			EXIT
		END_CASE

		#ifdef SolidPatternPatch
			/*Chris Huff solid pattern*/
			CASE (SOLID_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New->Type = SOLID_PATTERN;
				New->Vals.SolidVal = Parse_Float ();
				EXIT
			END_CASE
		#endif
	
		#ifdef ClothPatternPatch
			/*Chris Huff cloth pattern*/
			CASE (CLOTH_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				New->Type = CLOTH_PATTERN;
				EXIT
			END_CASE
			/*Chris Huff cloth2 pattern*/
		
			CASE (CLOTH2_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				New->Type = CLOTH2_PATTERN;
				EXIT
			END_CASE
		#endif
	
		#ifdef TorodialPatch
			/*Chris Huff toroidal pattern*/
			CASE (TOROIDAL_SPIRAL_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				New->Type = TOROIDAL_SPIRAL_PATTERN;
				New->Vals.SolidVal = Parse_Float ();
				EXIT
			END_CASE
		#endif

		CASE (SLOPE_TOKEN)
			Parse_Slope(New);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT     /* Concludes pattern_body */

	if ((Old_Type==BITMAP_PATTERN) && (New->Type!=BITMAP_PATTERN))
	{
		Destroy_Image(Old_Image);
	}   

	if ((Old_Type==DENSITY_FILE_PATTERN) && (New->Type!=DENSITY_FILE_PATTERN))
	{
		Destroy_Density_File(Old_Density_File);
	}   

	if (TPat_Type == NORMAL_TYPE) 
	{
		Parse_Comma();
		((TNORMAL *)New)->Amount = Allow_Float (((TNORMAL *)New)->Amount );
	}

	EXPECT         /* Look for pattern_modifier */
		/* NK delta */
		CASE (ACCURACY_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			if(TPat_Type != NORMAL_TYPE)
			{
				Error("delta can only be used with normal patterns.\n");
			}
			((TNORMAL *)New)->Delta = Parse_Float();
		END_CASE
		/* NK ---- */

		#ifdef CracklePatch
			CASE (SOLID_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				if (New->Type != CRACKLE_PATTERN ) 
				{
					Only_In("solid", "crackle");
				}
				New->Vals.Crackle.IsSolid = 1;
			END_CASE

			CASE (COORDS_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				if (New->Type != FACETS_PATTERN ) 
				{
				Only_In("coords", "facets");
				}
				New->Vals.Facets.UseCoords = Parse_Float();
			END_CASE

			CASE (SIZE_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				if (New->Type == FACETS_PATTERN ) 
				{
					New->Vals.Facets.Size = Parse_Float();
				}
				else 
				{
					Only_In("size", "facets");
				}
			END_CASE

			CASE (METRIC_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				if (New->Type == FACETS_PATTERN ) 
				{
					Parse_Vector( New->Vals.Facets.Metric );
				}
				else if ( New->Type == CRACKLE_PATTERN ) 
				{
					Parse_Vector( New->Vals.Crackle.Metric );
				}
				else 
				{
					Only_In("metric", "facets or crackle");
				}
			END_CASE

			CASE (FORM_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				if (New->Type != CRACKLE_PATTERN ) 
				{
					Only_In("form", "crackle");
				}
				Parse_Vector( New->Vals.Crackle.Form );
			END_CASE

			CASE (OFFSET_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				if (New->Type != CRACKLE_PATTERN ) 
				{
					Only_In("offset", "crackle");
				}
				New->Vals.Crackle.Offset = Parse_Float();
			END_CASE
		#endif
	
		CASE (TURBULENCE_TOKEN)
			Local_Turb=Check_Turb(&(New->Warps));
			Parse_Vector(Local_Turb->Turbulence);
		END_CASE

		CASE (COLOUR_MAP_TOKEN)
			if ((TPat_Type != PIGMENT_TYPE) && (TPat_Type != DENSITY_TYPE))
			{
				Only_In("color_map","pigment");
			}
			if (New->Type == CHECKER_PATTERN ||
			New->Type == BRICK_PATTERN ||	New->Type == HEXAGON_PATTERN ||
			#ifdef TrianglulairSquarePatch 
				New->Type == TERNAIRE_PATTERN ||	New->Type == SQUARE_PATTERN ||
			#endif
			New->Type == PLAIN_PATTERN ||	New->Type == AVERAGE_PATTERN ||
			#ifdef ObjectPatternPatch
				New->Type == OBJECT_PATTERN ||/*Chris Huff*/
			#endif
			New->Type == BITMAP_PATTERN)
			{
				Error("Cannot use color_map with this pattern type.");
			}
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Colour_Map ();
		END_CASE

		CASE (PIGMENT_MAP_TOKEN)
			if (TPat_Type != PIGMENT_TYPE)
			{
				Only_In("pigment_map","pigment");
			}
			if (New->Type == CHECKER_PATTERN ||
			#ifdef ObjectPatternPatch
				New->Type == OBJECT_PATTERN ||/*Chris Huff*/
			#endif
			New->Type == BRICK_PATTERN ||	New->Type == HEXAGON_PATTERN ||
			#ifdef TrianglulairSquarePatch 
				New->Type == TERNAIRE_PATTERN ||	New->Type == SQUARE_PATTERN ||
			#endif
			New->Type == PLAIN_PATTERN || New->Type == BITMAP_PATTERN)
			Not_With ("pigment_map","this pigment type");
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Blend_Map (PIGMENT_TYPE,New->Type);
		END_CASE

		CASE (DENSITY_MAP_TOKEN)
			if (TPat_Type != DENSITY_TYPE)
			{
				Only_In("density_map","density");
			}
			if (New->Type == CHECKER_PATTERN ||
			#ifdef ObjectPatternPatch
				New->Type == OBJECT_PATTERN ||/*Chris Huff*/
			#endif
			New->Type == BRICK_PATTERN ||	New->Type == HEXAGON_PATTERN ||
			#ifdef TrianglulairSquarePatch 
				New->Type == TERNAIRE_PATTERN || 	New->Type == SQUARE_PATTERN ||
			#endif
			New->Type == PLAIN_PATTERN || New->Type == BITMAP_PATTERN)
			Not_With ("density_map","this density type");
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Blend_Map (DENSITY_TYPE,New->Type);
		END_CASE

		CASE (SLOPE_MAP_TOKEN)
			if (TPat_Type != NORMAL_TYPE)
			{
				Only_In("slope_map","normal");
			}
			if (New->Type == CHECKER_PATTERN ||
			#ifdef ObjectPatternPatch
				New->Type == OBJECT_PATTERN ||/*Chris Huff*/
			#endif
			New->Type == BRICK_PATTERN || New->Type == HEXAGON_PATTERN ||
			#ifdef TrianglulairSquarePatch 
				New->Type == TERNAIRE_PATTERN || New->Type == SQUARE_PATTERN ||
			#endif
			New->Type == PLAIN_PATTERN || New->Type == AVERAGE_PATTERN ||
			New->Type == BITMAP_PATTERN)
			Not_With ("slope_map","this normal type");
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Blend_Map (SLOPE_TYPE,New->Type);
		END_CASE

		CASE (NORMAL_MAP_TOKEN)
			if (TPat_Type != NORMAL_TYPE)
			{
				Only_In("normal_map","normal");
			}
			if (New->Type == CHECKER_PATTERN ||
			#ifdef ObjectPatternPatch
				New->Type == OBJECT_PATTERN ||/*Chris Huff*/
			#endif
			New->Type == BRICK_PATTERN || New->Type == HEXAGON_PATTERN ||
			#ifdef TrianglulairSquarePatch 
				New->Type == TERNAIRE_PATTERN || New->Type == SQUARE_PATTERN ||
			#endif
			New->Type == PLAIN_PATTERN || New->Type == BITMAP_PATTERN)
			Not_With ("normal_map","this normal type");
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Blend_Map (NORMAL_TYPE,New->Type);
		END_CASE

		CASE (TEXTURE_MAP_TOKEN)
			if (TPat_Type != TEXTURE_TYPE)
			{
				Only_In("texture_map","texture");
			}
			if (New->Type == CHECKER_PATTERN ||
			#ifdef ObjectPatternPatch
				New->Type == OBJECT_PATTERN ||/*Chris Huff*/
			#endif
			New->Type == BRICK_PATTERN || New->Type == HEXAGON_PATTERN ||
			#ifdef TrianglulairSquarePatch 
				New->Type == TERNAIRE_PATTERN || New->Type == SQUARE_PATTERN ||
			#endif
			New->Type == PLAIN_PATTERN || New->Type == BITMAP_PATTERN)
			Not_With ("texture_map","this pattern type");
			Destroy_Blend_Map(New->Blend_Map);
			New->Blend_Map = Parse_Blend_Map (TEXTURE_TYPE,New->Type);
		END_CASE

		CASE (QUICK_COLOUR_TOKEN)
			if (TPat_Type != PIGMENT_TYPE)
			{
				Only_In("quick_color","pigment");
			}
			Parse_Colour (Local_Colour);
			if (opts.Quality_Flags & Q_QUICKC)
			{
				New->Type = PLAIN_PATTERN;
				Assign_Colour(((PIGMENT *)New)->Colour,Local_Colour);
			}
		END_CASE

		CASE (CONTROL0_TOKEN)
			if (New->Type == QUILTED_PATTERN) 
			{
				New->Vals.Quilted.Control0 = Parse_Float ();
			}
			else
			{
				Not_With ("control0","this normal");
			}
		END_CASE

		CASE (CONTROL1_TOKEN)
			if (New->Type == QUILTED_PATTERN)
			{
				New->Vals.Quilted.Control1 = Parse_Float ();
			}
			else
			{
				Not_With ("control1","this normal");
			}
		END_CASE

		CASE (OCTAVES_TOKEN)
			Local_Turb=Check_Turb(&(New->Warps));
			Local_Turb->Octaves = (int)Parse_Float();
			if(Local_Turb->Octaves < 1)
				Local_Turb->Octaves = 1;
			if(Local_Turb->Octaves > 10)  /* Avoid DOMAIN errors */
				Local_Turb->Octaves = 10;
		END_CASE

		CASE (OMEGA_TOKEN)
			Local_Turb=Check_Turb(&(New->Warps));
			Local_Turb->Omega = Parse_Float();
		END_CASE

		CASE (LAMBDA_TOKEN)
			Local_Turb=Check_Turb(&(New->Warps));
			Local_Turb->Lambda = Parse_Float();
		END_CASE

		CASE (FREQUENCY_TOKEN)
			New->Frequency = Parse_Float();
		END_CASE

		CASE (RAMP_WAVE_TOKEN)
			New->Wave_Type = RAMP_WAVE;
		END_CASE

		CASE (TRIANGLE_WAVE_TOKEN)
			New->Wave_Type = TRIANGLE_WAVE;
		END_CASE

		CASE (SINE_WAVE_TOKEN)
			New->Wave_Type = SINE_WAVE;
		END_CASE

		CASE (SCALLOP_WAVE_TOKEN)
			New->Wave_Type = SCALLOP_WAVE;
		END_CASE

		CASE (CUBIC_WAVE_TOKEN)
			New->Wave_Type = CUBIC_WAVE;
		END_CASE

		CASE (POLY_WAVE_TOKEN)
			New->Wave_Type = POLY_WAVE;
			New->Exponent  = Allow_Float(New->Exponent);
		END_CASE

		#ifdef AtanWavePatch
			CASE (ATAN_WAVE_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(60);
				#endif
				New->Wave_Type = ATAN_WAVE;
			END_CASE
		#endif

		CASE (PHASE_TOKEN)
			New->Phase = Parse_Float();
		END_CASE

		CASE (BUMP_SIZE_TOKEN)
			if (TPat_Type != NORMAL_TYPE)
				Only_In ("bump_size","normal");
			((TNORMAL *)New)->Amount = Parse_Float ();
		END_CASE

		CASE (AGATE_TURB_TOKEN)
			if (New->Type != AGATE_PATTERN)
				Not_With ("agate_turb","non-agate");
			New->Vals.Agate_Turb_Scale = Parse_Float();
			Check_Turb(&(New->Warps));   /* agate needs Octaves, Lambda etc. */
		END_CASE

		CASE (BRICK_SIZE_TOKEN)
			if (New->Type != BRICK_PATTERN)
				Not_With ("brick_size","non-brick");
			Parse_Vector(New->Vals.Brick.Size);
		END_CASE

		CASE (MORTAR_TOKEN)
			if (New->Type != BRICK_PATTERN)
				Not_With ("mortar","non-brick");
			New->Vals.Brick.Mortar = Parse_Float()-Small_Tolerance*2.0;
		END_CASE

		CASE (INTERPOLATE_TOKEN)
			if (New->Type != DENSITY_FILE_PATTERN)
				Not_With ("interpolate","non-density_file");
			New->Vals.Density_File->Interpolation = (int)Parse_Float();
		END_CASE

		CASE (FRACTAL_EXTERIOR_TYPE_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			if(New->Type < FIRST_FRACTAL_PATTERN || New->Type > LAST_FRACTAL_PATTERN)
				Not_With ("fractal_exterior_type","non-fractal pattern");
			New->Vals.Fractal.exterior_type = (int)Parse_Float();
			Parse_Comma();
			New->Vals.Fractal.efactor = Parse_Float();
		END_CASE

		CASE (FRACTAL_INTERIOR_TYPE_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			if(New->Type < FIRST_FRACTAL_PATTERN || New->Type > LAST_FRACTAL_PATTERN)
				Not_With ("fractal_interior_type","non-fractal pattern");
			New->Vals.Fractal.interior_type = (int)Parse_Float();
			Parse_Comma();
			New->Vals.Fractal.ifactor = Parse_Float();
		END_CASE

		CASE (WARP_TOKEN)
			Parse_Begin();
			Parse_Warp(&(New->Warps));
			Parse_End();
		END_CASE

		CASE (TRANSLATE_TOKEN)
			Parse_Vector (Local_Vector);
			Translate_Tpattern (New, Local_Vector);
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector (Local_Vector);
			Rotate_Tpattern (New, Local_Vector);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Scale_Vector (Local_Vector);
			Scale_Tpattern (New, Local_Vector);
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix(Local_Matrix);
			Compute_Matrix_Transform(&Local_Trans, Local_Matrix);
			Transform_Tpattern (New, &Local_Trans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
				Transform_Tpattern (New, (TRANSFORM *)Token.Data);
			#else
			{
				TRANSFORM * Trans = Parse_Transform();
				Transform_Tpattern (New, Trans);
				/*YS sept 17 2000 Memory leak*/
				POV_FREE(Trans);
			}
			#endif
		END_CASE

		#ifdef NormalBugFix
			CASE (NO_BUMP_SCALE_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				Set_Flag(New,DONT_SCALE_BUMPS_FLAG);
			END_CASE
		#endif
		
		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	if ((New->Type==AVERAGE_PATTERN) && (New->Blend_Map==NULL))
	{
		Error("Average must have map.");
	}

	if ((TPat_Type==TEXTURE_TYPE) && (New->Type!=PLAIN_PATTERN) &&	(New->Blend_Map==NULL))
	{
		Error("Patterned texture must have texture_map.");
	}
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

void Parse_Tnormal (TNORMAL **Tnormal_Ptr)
{
	EXPECT            /* Look for [tnormal_id] */
		CASE (TNORMAL_ID_TOKEN)
			Destroy_Tnormal(*Tnormal_Ptr);
			*Tnormal_Ptr = Copy_Tnormal ((TNORMAL *) Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT    /* End [tnormal_id] */

	if (*Tnormal_Ptr == NULL)
	{ /* tw */
		if ((Default_Texture->Tnormal) != NULL)
			*Tnormal_Ptr = Copy_Tnormal ((Default_Texture->Tnormal));
		else
			*Tnormal_Ptr = Create_Tnormal ();
	} /* tw */

	Parse_Pattern((TPATTERN *)*Tnormal_Ptr,NORMAL_TYPE);
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

void Parse_Finish (FINISH **Finish_Ptr)
{
	COLOUR Temp_Colour;
	FINISH *New;
	VECTOR Local_Vector;

	Parse_Begin ();

	EXPECT        /* Look for zero or one finish_id */
		CASE (FINISH_ID_TOKEN)
			Destroy_Finish(*Finish_Ptr);
			*Finish_Ptr = Copy_Finish ((FINISH *) Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT    /* End finish_id */

	New = *Finish_Ptr;

	EXPECT        /* Look for zero or more finish_body */
		/* NK conserve energy */
		CASE (CONSERVE_ENERGY_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			New->Conserve_Energy = TRUE;
		END_CASE
			/* NK ---- */

		CASE (AMBIENT_TOKEN)
			Parse_Colour(Temp_Colour);
			New->Ambient[RED]   = Temp_Colour[RED];
			New->Ambient[GREEN] = Temp_Colour[GREEN];
			New->Ambient[BLUE]  = Temp_Colour[BLUE];
		END_CASE

		CASE (BRILLIANCE_TOKEN)
			New->Brilliance = Parse_Float ();
		END_CASE

		CASE (DIFFUSE_TOKEN)
			New->Diffuse = Parse_Float ();
		END_CASE

		CASE (REFLECTION_TOKEN)
			Parse_Colour(Temp_Colour);
			New->Reflection_Max[RED]   = Temp_Colour[RED];
			New->Reflection_Max[GREEN] = Temp_Colour[GREEN];
			New->Reflection_Max[BLUE]  = Temp_Colour[BLUE];
			New->Reflection_Min[RED]   = Temp_Colour[RED];
			New->Reflection_Min[GREEN] = Temp_Colour[GREEN];
			New->Reflection_Min[BLUE]  = Temp_Colour[BLUE];
			New->Reflection_Falloff = 1;
			New->Reflection_Type = 0;
		END_CASE

		/* Tokens added by MBP for reflection-blur */
		CASE (REFLECTION_BLUR_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			New->Reflection_Blur = Parse_Float ();
		END_CASE

		CASE (REFLECTION_SAMPLES_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			New->Reflection_Samples = (int) Parse_Float ();
		END_CASE

		/* Tokens added by MBP for angle-dependent reflectivity */
		CASE (REFLECTION_MAX_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			Parse_Colour(Temp_Colour);
			New->Reflection_Max[RED]   = Temp_Colour[RED];
			New->Reflection_Max[GREEN] = Temp_Colour[GREEN];
			New->Reflection_Max[BLUE]  = Temp_Colour[BLUE];
		END_CASE

		CASE (REFLECT_METALLIC_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			New->Reflect_Metallic = 1.0;
			EXPECT
				CASE_FLOAT
					New->Reflect_Metallic = Parse_Float();
					EXIT
				END_CASE

				OTHERWISE
					UNGET
					EXIT
				END_CASE
			END_EXPECT
		END_CASE

		CASE (REFLECTION_MIN_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			Parse_Colour(Temp_Colour);
			New->Reflection_Min[RED]   = Temp_Colour[RED];
			New->Reflection_Min[GREEN] = Temp_Colour[GREEN];
			New->Reflection_Min[BLUE]  = Temp_Colour[BLUE];
		END_CASE

		CASE (REFLECTION_FALLOFF_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			New->Reflection_Falloff = Parse_Float();
		END_CASE

		CASE (REFLECTION_TYPE_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			New->Reflection_Type = (int) Parse_Float();
			if (New->Reflection_Type < 0 || New->Reflection_Type > 1)
			{
				Error("Illegal reflection_type.\n");
			}
		END_CASE
			/* End of tokens added by MBP */

		CASE (REFLECTION_EXPONENT_TOKEN)
			New->Reflect_Exp = 1.0 / Parse_Float ();
		END_CASE

		CASE (PHONG_TOKEN)
			New->Phong = Parse_Float ();
		END_CASE

		CASE (PHONG_SIZE_TOKEN)
			New->Phong_Size = Parse_Float ();
		END_CASE

		#ifdef BlinnPatch
			CASE (BLINN_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				New->Blinn = Parse_Float ();
			END_CASE

			CASE (FACETS_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				New->Facets = Parse_Float ();
			END_CASE
		#endif

		CASE (SPECULAR_TOKEN)
			New->Specular = Parse_Float ();
		END_CASE

		CASE (ROUGHNESS_TOKEN)
			New->Roughness = Parse_Float ();
			if (New->Roughness != 0.0)
				New->Roughness = 1.0/New->Roughness; /* CEY 12/92 */
			else
				Warn(0, "Zero roughness used.");
		END_CASE

		CASE (METALLIC_TOKEN)
			New->Metallic = 1.0;
			EXPECT
				CASE_FLOAT
					New->Metallic = Parse_Float();
					EXIT
					END_CASE

				OTHERWISE
					UNGET
					EXIT
				END_CASE
			END_EXPECT
		END_CASE

		CASE (CRAND_TOKEN)
			New->Crand = Parse_Float();
		END_CASE

		CASE (IRID_TOKEN)                     /* DMF */
			Parse_Begin();
			New->Irid = Parse_Float();
			EXPECT
				CASE (THICKNESS_TOKEN)           /* DMF */
					New->Irid_Film_Thickness = Parse_Float();
				END_CASE

				CASE (TURBULENCE_TOKEN)                /* DMF */
					Parse_Vector(Local_Vector);
					New->Irid_Turb = Local_Vector[X];
				END_CASE

				OTHERWISE
					UNGET
					EXIT
				END_CASE
			END_EXPECT

			Parse_End();
		END_CASE

		CASE (IOR_TOKEN)
			New->Temp_IOR = Parse_Float();
			Warn_Interior("Index of refraction value");
		END_CASE

		CASE (DISPERSION_TOKEN)
			New->Temp_Dispersion = Parse_Float();
			Warn_Interior("Index of refraction value");
		END_CASE

		CASE (DISP_NELEMS_TOKEN)
			New->Temp_NElems = (int)Parse_Float();
			Warn_Interior("Index of refraction value");
		END_CASE

		CASE (CAUSTICS_TOKEN)
			New->Temp_Caustics = Parse_Float();
			Warn_Interior("Caustics value");
		END_CASE

		CASE (REFRACTION_TOKEN)
			New->Temp_Refract = Parse_Float();
			Warn_Interior("Refraction value unnecessary to turn on refraction.\nTo attenuate, the fade_power and fade_distance keywords ");
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE

	END_EXPECT    /* End of finish_body */

	EXPECT        /* Look for finish_mods */
		/*   CASE none implemented
			END_CASE     */

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT    /* End of finish_mods */

	Parse_End ();
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

TEXTURE *Parse_Texture ()
{
	VECTOR Local_Vector;
	MATRIX Local_Matrix;
	TRANSFORM Local_Trans;
	TEXTURE *Texture;
	int Modified_Pnf;

	if (opts.Language_Version < 300)
	{
		return(Parse_Vers1_Texture());
	}

	Modified_Pnf = FALSE;

	EXPECT               /* First allow a texture identifier */
		CASE (TEXTURE_ID_TOKEN)
			Texture = Copy_Textures((TEXTURE *) Token.Data);
			Modified_Pnf = TRUE;
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			Texture = Copy_Textures (Default_Texture);
			EXIT
		END_CASE
	END_EXPECT

	/* If the texture identifer or the default texture was a PLAIN_PATTERN
	then allow its pigment, normal or finish to be overridden by
	pigment identifier, normal identifier and finish identifiers.
	This is a consession to backwards compatibility so that
	"texture{PIGMENT_IDENTIFIER}" etc. is legal even though it should
	be "texture{pigment{PIGMENT_IDENTIFIER}}"
	*/

	/* Look for [pnf_texture] */
	if (Texture->Type == PLAIN_PATTERN)
	{
		EXPECT   /* Look for [pnf_ids] */
			CASE (PIGMENT_ID_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Destroy_Pigment(Texture->Pigment);
				Texture->Pigment = Copy_Pigment ((PIGMENT *) Token.Data);
				Modified_Pnf = TRUE;
			END_CASE

			CASE (TNORMAL_ID_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				Destroy_Tnormal(Texture->Tnormal);
				Texture->Tnormal = Copy_Tnormal ((TNORMAL *) Token.Data);
				Modified_Pnf = TRUE;
			END_CASE

			CASE (FINISH_ID_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Destroy_Finish(Texture->Finish);
				Texture->Finish = Copy_Finish ((FINISH *) Token.Data);
				Modified_Pnf = TRUE;
			END_CASE

			OTHERWISE
				UNGET
				EXIT
			END_CASE
		END_EXPECT

		/* If the texture identifer or the default texture was a PLAIN_PATTERN
		then allow its pigment, normal or finish to be overridden by
		pigment, normal or finish statement.  Also allow transformations.
		*/

		EXPECT   /* Modify previous pnf */
			CASE (PIGMENT_TOKEN)
				Parse_Begin ();
				Parse_Pigment ( &(Texture->Pigment) );
				Parse_End ();
				Modified_Pnf = TRUE;
			END_CASE

			CASE (TNORMAL_TOKEN)
				Parse_Begin ();
				Parse_Tnormal ( &(Texture->Tnormal) );
				Parse_End ();
				Modified_Pnf = TRUE;
			END_CASE

			CASE (FINISH_TOKEN)
				Parse_Finish ( &(Texture->Finish) );
				Modified_Pnf = TRUE;
			END_CASE

			CASE (TRANSLATE_TOKEN)
				Parse_Vector (Local_Vector);
				Compute_Translation_Transform(&Local_Trans, Local_Vector);
				Transform_Textures (Texture, &Local_Trans);
				Modified_Pnf = TRUE;
			END_CASE

			CASE (ROTATE_TOKEN)
				Parse_Vector (Local_Vector);
				Compute_Rotation_Transform(&Local_Trans, Local_Vector);
				Transform_Textures (Texture, &Local_Trans);
				Modified_Pnf = TRUE;
			END_CASE

			CASE (SCALE_TOKEN)
				Parse_Scale_Vector (Local_Vector);
				Compute_Scaling_Transform(&Local_Trans, Local_Vector);
				Transform_Textures (Texture, &Local_Trans);
				Modified_Pnf = TRUE;
			END_CASE

			CASE (MATRIX_TOKEN)
				Parse_Matrix(Local_Matrix);
				Compute_Matrix_Transform(&Local_Trans, Local_Matrix);
				Transform_Textures (Texture, &Local_Trans);
				Modified_Pnf = TRUE;
			END_CASE

			CASE (TRANSFORM_TOKEN)
				#ifndef TransformPatch /* Chris Huff april 2000 */
					GET(TRANSFORM_ID_TOKEN)
					Transform_Textures (Texture, (TRANSFORM *)Token.Data);
				#else
				{
					TRANSFORM * Trans = Parse_Transform();
					Transform_Textures (Texture, Trans);
					/*YS sept 17 2000 Memory leak*/
					POV_FREE(Trans);
				}
				#endif
				Modified_Pnf = TRUE;
			END_CASE

			/* NK Dec 1999 added this */
			CASE (WARP_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				Parse_Begin();
				Parse_Warp(&(Texture->Warps));
				Parse_End();
			END_CASE
			/* NK ---- */

			#ifdef NormalBugFix
				CASE (NO_BUMP_SCALE_TOKEN)
					#ifdef UnofficialBlocking
						parseUnofficialFeature(30);
					#endif
					Set_Flag(Texture,DONT_SCALE_BUMPS_FLAG);
				END_CASE
			#endif

			OTHERWISE
				UNGET
				EXIT
			END_CASE
		END_EXPECT
	}
	else
	{
		/* Here it is not a PLAIN_PATTERN texture and since default textures
		must be plain then this was a texture identifier that was a special 
		texture.  Allow transforms.  The "if(!Modified_Pnf)..." below
		will always fail if we came here.  So we return after the
		transforms. */
		Parse_Texture_Transform(Texture);
	}

	/* If we've modified the default texture with a p,n, or f then this
	has to stay a PLAIN_PATTERN pnf texture.  We won't allow
	a texture_map or pattern.  Therefore quit now.
	*/

	if (!Modified_Pnf)
	{
		/* At this point we've either got a texture statement that had
		no p, n or f.  Nor any texture identifier.  Its probably 
		a patterned texture_map texture. It could be an empty
		statement such as "texture{}" */

		EXPECT
			CASE (TILES_TOKEN)
				Destroy_Textures (Texture);
				Texture = Parse_Tiles();
				Parse_Texture_Transform(Texture);
				EXIT
			END_CASE

			CASE (MATERIAL_MAP_TOKEN)
				Destroy_Textures (Texture);
				Texture = Parse_Material_Map ();
				Parse_Texture_Transform(Texture);
				EXIT
			END_CASE

			OTHERWISE
				UNGET;
				Destroy_Pigment(Texture->Pigment);
				Destroy_Tnormal(Texture->Tnormal);
				Destroy_Finish(Texture->Finish);
				Texture->Pigment = NULL;
				Texture->Tnormal = NULL;
				Texture->Finish  = NULL;
				Parse_Pattern((TPATTERN *)Texture,TEXTURE_TYPE);
				/* if following is true, parsed "texture{}" so restore
				default texture.
				*/
				if (Texture->Type <= PLAIN_PATTERN)
				{
					Destroy_Textures(Texture);
					Texture = Copy_Textures (Default_Texture);
				}
				EXIT
			END_CASE
		END_EXPECT
	}
	return (Texture);
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static TEXTURE *Parse_Tiles()
{
	TEXTURE *Texture, *Local_Texture;
	BLEND_MAP_ENTRY *Entry;

	Parse_Begin ();

	Texture = Create_Texture ();
	Destroy_Pigment(Texture->Pigment);
	Destroy_Tnormal(Texture->Tnormal);
	Destroy_Finish(Texture->Finish);
	Texture->Pigment = NULL;
	Texture->Tnormal = NULL;
	Texture->Finish  = NULL;
	Texture->Type = CHECKER_PATTERN;

	Texture->Blend_Map = Create_Blend_Map();
	Texture->Blend_Map->Number_Of_Entries = 2;
	Texture->Blend_Map->Blend_Map_Entries = Entry = Create_BMap_Entries (2);
	Texture->Blend_Map->Type = TEXTURE_TYPE;
	Entry[0].Vals.Texture=NULL;
	Entry[0].value=0.0;
	Entry[0].Same=FALSE;
	Entry[1].Vals.Texture=NULL;
	Entry[1].value=1.0;
	Entry[1].Same=FALSE;

	/* Note first tile is 1, 2nd tile is 0 to keep compatible with old tiles */

	EXPECT
		CASE (TEXTURE_TOKEN)
			Parse_Begin ();
			Local_Texture = Parse_Texture ();
			Link_Textures(&(Entry[1].Vals.Texture),Local_Texture);
			Parse_End ();
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	GET (TILE2_TOKEN);

	EXPECT
		CASE (TEXTURE_TOKEN)
			Parse_Begin ();
			Local_Texture = Parse_Texture ();
			Link_Textures(&(Entry[0].Vals.Texture),Local_Texture);
			Parse_End ();
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_End ();
	return (Texture);
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static TEXTURE *Parse_Material_Map()
{
	TEXTURE *Texture, *Local_Texture;
	Parse_Begin ();

	Texture = Create_Texture ();
	Destroy_Pigment(Texture->Pigment);
	Destroy_Tnormal(Texture->Tnormal);
	Destroy_Finish(Texture->Finish);
	Texture->Pigment = NULL;
	Texture->Tnormal = NULL;
	Texture->Finish  = NULL;
	Texture->Type = BITMAP_PATTERN;

	Texture->Vals.Image = Parse_Image(MATERIAL_FILE);
	Texture->Vals.Image->Use_Colour_Flag = FALSE;

	EXPECT
		CASE (ONCE_TOKEN)
			Texture->Vals.Image->Once_Flag=TRUE;
		END_CASE

		CASE (INTERPOLATE_TOKEN)
			Texture->Vals.Image->Interpolation_Type=(int)Parse_Float();
		END_CASE

		CASE (MAP_TYPE_TOKEN)
			Texture->Vals.Image->Map_Type = (int) Parse_Float ();
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	GET (TEXTURE_TOKEN)                /* First material */
	Parse_Begin();
	Texture->Materials = Local_Texture = Parse_Texture ();
	Parse_End();
	Texture->Num_Of_Mats++;

	EXPECT                             /* Subsequent materials */
		CASE (TEXTURE_TOKEN)
			Parse_Begin();
			Local_Texture->Next_Material = Parse_Texture ();
			Parse_End();
			Local_Texture = Local_Texture->Next_Material;
			Texture->Num_Of_Mats++;
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_End ();
	return(Texture);
}
     


/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static void SetFractalDefaults_p(PIGMENT *New)
{
	New->Vals.Fractal.interior_type = DEFAULT_FRACTAL_INTERIOR_TYPE;
	New->Vals.Fractal.exterior_type = DEFAULT_FRACTAL_EXTERIOR_TYPE;
	New->Vals.Fractal.efactor = DEFAULT_FRACTAL_EXTERIOR_FACTOR;
	New->Vals.Fractal.ifactor = DEFAULT_FRACTAL_INTERIOR_FACTOR;
}

static TEXTURE *Parse_Vers1_Texture ()
{
	VECTOR Local_Vector;
	COLOUR Local_Colour;
	MATRIX Local_Matrix;
	TRANSFORM Local_Trans;
	TURB *Local_Turb;
	TEXTURE *Texture;
	PIGMENT *Pigment;
	TNORMAL *Tnormal;
	FINISH *Finish;

	EXPECT                      /* Look for texture_body */
		CASE (TILES_TOKEN)
			Texture = Parse_Tiles();
			EXIT
		END_CASE

		CASE (MATERIAL_MAP_TOKEN)
			Texture = Parse_Material_Map ();
			EXIT
		END_CASE

		CASE (TEXTURE_ID_TOKEN)
			Texture = Copy_Textures((TEXTURE *) Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			Texture = Copy_Textures (Default_Texture);
			EXIT
		END_CASE
	END_EXPECT

	/* Look for [pnf_texture] */
	if (Texture->Type == PLAIN_PATTERN)
	{
		EXPECT   /* Look for [pnf_ids] */
			CASE (PIGMENT_ID_TOKEN)
				Destroy_Pigment(Texture->Pigment);
				Texture->Pigment = Copy_Pigment ((PIGMENT *) Token.Data);
			END_CASE

			CASE (TNORMAL_ID_TOKEN)
				Destroy_Tnormal(Texture->Tnormal);
				Texture->Tnormal = Copy_Tnormal ((TNORMAL *) Token.Data);
			END_CASE

			CASE (FINISH_ID_TOKEN)
				Destroy_Finish(Texture->Finish);
				Texture->Finish = Copy_Finish ((FINISH *) Token.Data);
			END_CASE

			OTHERWISE
				UNGET
				EXIT
			END_CASE
		END_EXPECT

		Pigment = Texture->Pigment;
		Tnormal = Texture->Tnormal;
		Finish  = Texture->Finish;

		EXPECT
			CASE (PIGMENT_TOKEN)
				Parse_Begin ();
				Parse_Pigment ( &(Texture->Pigment) );
				Parse_End ();
			END_CASE

			CASE (TNORMAL_TOKEN)
				Parse_Begin ();
				Parse_Tnormal ( &(Texture->Tnormal) );
				Parse_End ();
			END_CASE

			CASE (FINISH_TOKEN)
				Parse_Finish ( &(Texture->Finish) );
			END_CASE

			/***********************************************************************
			PIGMENT STUFF OUTSIDE PIGMENT{}
			NOTE: Do not add new keywords to this section.  Use 1.0 syntax only.
			***********************************************************************/
			CASE (AGATE_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = AGATE_PATTERN;
				Pigment->Vals.Agate_Turb_Scale = 1.0;
				Check_Turb(&(Pigment->Warps));   /* agate needs Octaves, Lambda etc. */
			END_CASE

			CASE (BOZO_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = BOZO_PATTERN;
			END_CASE

			CASE (GRANITE_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = GRANITE_PATTERN;
			END_CASE

			CASE (LEOPARD_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = LEOPARD_PATTERN;
			END_CASE

			CASE (MARBLE_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = MARBLE_PATTERN;
			END_CASE

			CASE (MAGNET1M_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = MAGNET1M_PATTERN;
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (MAGNET1J_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = MAGNET1J_PATTERN;
				Parse_UV_Vect(Pigment->Vals.Fractal.Coord);
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (MAGNET2M_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = MAGNET2M_PATTERN;
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (MAGNET2J_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = MAGNET2J_PATTERN;
				Parse_UV_Vect(Pigment->Vals.Fractal.Coord);
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (MANDEL_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = MANDEL_PATTERN;
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (MANDEL3_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);	
				Pigment->Type = MANDEL3_PATTERN;
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (MANDEL4_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = MANDEL4_PATTERN;
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (JULIA_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = JULIA_PATTERN;
				Parse_UV_Vect(Pigment->Vals.Fractal.Coord);
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (JULIA3_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = JULIA3_PATTERN;
				Parse_UV_Vect(Pigment->Vals.Fractal.Coord);
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (JULIA4_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = JULIA4_PATTERN;
				Parse_UV_Vect(Pigment->Vals.Fractal.Coord);
				Pigment->Vals.Fractal.Iterations = (int)Parse_Float();
				SetFractalDefaults_p(Pigment);
			END_CASE

			CASE (ONION_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = ONION_PATTERN;
			END_CASE

			CASE (SPOTTED_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = SPOTTED_PATTERN;
			END_CASE

			CASE (WOOD_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = WOOD_PATTERN;
			END_CASE

			CASE (GRADIENT_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = GRADIENT_PATTERN;
				Parse_Vector (Pigment->Vals.Gradient);
			END_CASE

			CASE_COLOUR
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = PLAIN_PATTERN;
				Parse_Colour (Pigment->Colour);
			END_CASE

			CASE (CHECKER_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);	
				Pigment->Type = CHECKER_PATTERN;
				Pigment->Frequency = 0.0;
				Destroy_Blend_Map(Pigment->Blend_Map);
				Pigment->Blend_Map = Parse_Blend_List(2,&Check_Default_Map,COLOUR_TYPE);
			END_CASE

			CASE (HEXAGON_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = HEXAGON_PATTERN;
				Pigment->Frequency = 0.0;
				Destroy_Blend_Map(Pigment->Blend_Map);
				Pigment->Blend_Map = Parse_Blend_List(3,&Hex_Default_Map,COLOUR_TYPE);
				END_CASE
				CASE (IMAGE_MAP_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Pigment->Type = BITMAP_PATTERN;
				Pigment->Frequency = 0.0;
				Parse_Image_Map (Pigment);
			END_CASE

			CASE (TURBULENCE_TOKEN)
				Local_Turb=Check_Turb(&(Pigment->Warps));
				Parse_Vector(Local_Turb->Turbulence);
			END_CASE

			CASE (COLOUR_MAP_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				if (Pigment->Type == CHECKER_PATTERN ||	Pigment->Type == PLAIN_PATTERN ||	Pigment->Type == BITMAP_PATTERN)
				Warn(150, "Cannot use color map with this pigment type.");
				Destroy_Blend_Map(Pigment->Blend_Map);
				Pigment->Blend_Map = Parse_Colour_Map ();
			END_CASE

			CASE (QUICK_COLOUR_TOKEN)
				Warn_State(Token.Token_Id, PIGMENT_TOKEN);
				Parse_Colour (Local_Colour);
				if (opts.Quality_Flags & Q_QUICKC)
				{
					Assign_Colour(Pigment->Colour,Local_Colour);
				}
			END_CASE

			CASE (OCTAVES_TOKEN)
				Local_Turb=Check_Turb(&(Pigment->Warps));
				Local_Turb->Octaves = (int)Parse_Float();
				if(Local_Turb->Octaves < 1)
					Local_Turb->Octaves = 1;
				if(Local_Turb->Octaves > 10)  /* Avoid DOMAIN errors */
					Local_Turb->Octaves = 10;
			END_CASE

			CASE (OMEGA_TOKEN)
				Local_Turb=Check_Turb(&(Pigment->Warps));
				Local_Turb->Omega = Parse_Float();
			END_CASE

			CASE (LAMBDA_TOKEN)
				Local_Turb=Check_Turb(&(Pigment->Warps));
				Local_Turb->Lambda = Parse_Float();
			END_CASE

			/***********************************************************************
			TNORMAL STUFF OUTSIDE NORMAL{}
			NOTE: Do not add new keywords to this section.  Use 1.0 syntax only.
			***********************************************************************/
			CASE (BUMPS_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				ADD_TNORMAL
				Tnormal->Type = BUMPS_PATTERN;
				Tnormal->Amount = Parse_Float ();
			END_CASE

			CASE (DENTS_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				ADD_TNORMAL
				Tnormal->Type = DENTS_PATTERN;
				Tnormal->Amount = Parse_Float ();
			END_CASE

			CASE (RIPPLES_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				ADD_TNORMAL
				Tnormal->Type = RIPPLES_PATTERN;
				Tnormal->Amount = Parse_Float ();
			END_CASE

			CASE (WAVES_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				ADD_TNORMAL
				Tnormal->Type = WAVES_PATTERN;
				Tnormal->Amount = Parse_Float ();
			END_CASE

			CASE (WRINKLES_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				ADD_TNORMAL
				Tnormal->Type = WRINKLES_PATTERN;
				Tnormal->Amount = Parse_Float ();
			END_CASE

			CASE (BUMP_MAP_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				ADD_TNORMAL
				Tnormal->Type = BITMAP_PATTERN;
				Tnormal->Frequency = 0.0;
				Parse_Bump_Map (Tnormal);
			END_CASE

			CASE (FREQUENCY_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				ADD_TNORMAL
				if (!(Tnormal->Type == RIPPLES_PATTERN || Tnormal->Type == WAVES_PATTERN))
					if (opts.Language_Version >= 150)
						Warn(150, "Cannot use frequency with this normal.");
				Tnormal->Frequency = Parse_Float();
			END_CASE

			CASE (PHASE_TOKEN)
				Warn_State(Token.Token_Id, TNORMAL_TOKEN);
				ADD_TNORMAL
			if (!(Tnormal->Type == RIPPLES_PATTERN || Tnormal->Type == WAVES_PATTERN))
				if (opts.Language_Version >= 150)
					Warn(150, "Cannot use phase with this normal.");
				Tnormal->Phase = Parse_Float();
			END_CASE


			/***********************************************************************
			FINISH STUFF OUTSIDE FINISH{}
			NOTE: Do not add new keywords to this section.  Use 1.0 syntax only.
			***********************************************************************/
			CASE (AMBIENT_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Ambient[RED]   =
				Finish->Ambient[GREEN] =
				Finish->Ambient[BLUE]  = Parse_Float ();
			END_CASE

			CASE (BRILLIANCE_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Brilliance = Parse_Float ();
			END_CASE

			CASE (DIFFUSE_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Diffuse = Parse_Float ();
			END_CASE

			CASE (REFLECTION_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Reflection_Max[RED]   =
				Finish->Reflection_Max[GREEN] =
				Finish->Reflection_Max[BLUE]  =
				Finish->Reflection_Min[RED]   =
				Finish->Reflection_Min[GREEN] =
				Finish->Reflection_Min[BLUE]  = Parse_Float ();
				Finish->Reflection_Falloff = 1;
			END_CASE

			CASE (PHONG_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Phong = Parse_Float ();
			END_CASE

			CASE (PHONG_SIZE_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Phong_Size = Parse_Float ();
			END_CASE

			CASE (SPECULAR_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Specular = Parse_Float ();
			END_CASE

			CASE (ROUGHNESS_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Roughness = Parse_Float ();
				if (Finish->Roughness != 0.0)
					Finish->Roughness = 1.0/Finish->Roughness; /* CEY 12/92 */
				else
					Warn(0, "Zero roughness used.");
			END_CASE

			CASE (METALLIC_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Metallic = 1.0;
			END_CASE

			CASE (CRAND_TOKEN)
				Warn_State(Token.Token_Id, FINISH_TOKEN);
				Finish->Crand = Parse_Float();
			END_CASE

			CASE_FLOAT
				Finish->Crand = Parse_Float();
				Warn(150, "Should use crand keyword in finish statement.");
			END_CASE

			CASE (IOR_TOKEN)
				Warn_State(Token.Token_Id, INTERIOR_TOKEN);
				Finish->Temp_IOR = Parse_Float();
				Warn_Interior("Index of refraction value");
			END_CASE

			CASE (DISPERSION_TOKEN)
				Warn_State(Token.Token_Id, INTERIOR_TOKEN);
				Finish->Temp_Dispersion = Parse_Float();
				Warn_Interior("Index of refraction value");
			END_CASE

			CASE (DISP_NELEMS_TOKEN)
				Warn_State(Token.Token_Id, INTERIOR_TOKEN);
				Finish->Temp_NElems = (int)Parse_Float();
				Warn_Interior("Index of refraction value");
			END_CASE

			CASE (REFRACTION_TOKEN)
				Warn_State(Token.Token_Id, INTERIOR_TOKEN);
				Finish->Temp_Refract = Parse_Float();
				Warn_Interior("Refraction value unnecessary to turn on refraction.\nTo attenuate, the fade_power and fade_distance keywords ");
			END_CASE

			CASE (TRANSLATE_TOKEN)
				Parse_Vector (Local_Vector);
				Compute_Translation_Transform(&Local_Trans, Local_Vector);
				Transform_Textures (Texture, &Local_Trans);
			END_CASE

			CASE (ROTATE_TOKEN)
				Parse_Vector (Local_Vector);
				Compute_Rotation_Transform(&Local_Trans, Local_Vector);
				Transform_Textures (Texture, &Local_Trans);
			END_CASE

			CASE (SCALE_TOKEN)
				Parse_Scale_Vector (Local_Vector);
				Compute_Scaling_Transform(&Local_Trans, Local_Vector);
				Transform_Textures (Texture, &Local_Trans);
			END_CASE

			CASE (MATRIX_TOKEN)
				Parse_Matrix(Local_Matrix);
				Compute_Matrix_Transform(&Local_Trans, Local_Matrix);
				Transform_Textures (Texture, &Local_Trans);
			END_CASE

			CASE (TRANSFORM_TOKEN)
				#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
					Transform_Textures (Texture, (TRANSFORM *)Token.Data);
				#else
				{
					TRANSFORM * Trans = Parse_Transform();
					Transform_Textures (Texture, Trans);
					/*YS sept 17 2000 Memory leak*/
					POV_FREE(Trans);
				}
				#endif
			END_CASE

			CASE (TEXTURE_ID_TOKEN)
				Warn(0, "Texture identifier overwriting previous values.");
				Destroy_Textures(Texture);
				Texture = Copy_Textures((TEXTURE *) Token.Data);
				Pigment = Texture->Pigment;
				Tnormal = Texture->Tnormal;
				Finish  = Texture->Finish;
			END_CASE

			OTHERWISE
				UNGET
				EXIT
			END_CASE

			/***********************************************************************/

		END_EXPECT

		if (Not_In_Default && (Texture->Pigment->Type == NO_PATTERN) &&	!(opts.Language_Version < 150))
			Parse_Error(PIGMENT_ID_TOKEN);
	}
	Parse_Texture_Transform(Texture);
	return (Texture);
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

static void Parse_Texture_Transform (TEXTURE *Texture)
{
	VECTOR Local_Vector;
	MATRIX Local_Matrix;
	TRANSFORM Local_Trans;

	EXPECT
		CASE (TRANSLATE_TOKEN)
			Parse_Vector (Local_Vector);
			Compute_Translation_Transform(&Local_Trans, Local_Vector);
			Transform_Textures (Texture, &Local_Trans);
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector (Local_Vector);
			Compute_Rotation_Transform(&Local_Trans, Local_Vector);
			Transform_Textures (Texture, &Local_Trans);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Scale_Vector (Local_Vector);
			Compute_Scaling_Transform(&Local_Trans, Local_Vector);
			Transform_Textures (Texture, &Local_Trans);
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix(Local_Matrix);
			Compute_Matrix_Transform(&Local_Trans, Local_Matrix);
			Transform_Textures (Texture, &Local_Trans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
				Transform_Textures (Texture, (TRANSFORM *)Token.Data);
			#else
			{
				TRANSFORM * Trans = Parse_Transform();
				Transform_Textures (Texture, Trans);
				/*YS sept 17 2000 Memory leak*/
				POV_FREE(Trans);
			}
			#endif
		END_CASE

		/* NK Dec 1999 added this */
		CASE (WARP_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			Parse_Begin();
			Parse_Warp(&(Texture->Warps));
			Parse_End();
		END_CASE
			/* NK ---- */

		#ifdef NormalBugFix
			CASE (NO_BUMP_SCALE_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				Set_Flag(Texture,DONT_SCALE_BUMPS_FLAG);
			END_CASE
		#endif

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT
}


/*****************************************************************************
*
* FUNCTION
*
*   Parse_Media
*
* INPUT
*
* OUTPUT
*
* RETURNS
*   
* AUTHOR
*
*   Dieter Bayer
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Dec 1996 : Creation.
*
******************************************************************************/

void Parse_Media(IMEDIA **Media_Ptr)
{
	IMEDIA *IMedia, *Next_Media;
	TRANSFORM Local_Trans;
	VECTOR Local_Vector;
	MATRIX Local_Matrix;
	int Tmp_Flag;

	Next_Media = *Media_Ptr;

	Parse_Begin();

	EXPECT
		CASE(MEDIA_ID_TOKEN)
			IMedia = Copy_Media((IMEDIA *)Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			IMedia = Create_Media();
			EXIT
		END_CASE
	END_EXPECT

	EXPECT
		CASE (INTERVALS_TOKEN)
			if ((IMedia->Intervals = (int)Parse_Float()) < 1)
			{
				Error("At least one interval is needed in media.\n");
			}
		END_CASE

		CASE (SAMPLES_TOKEN)
			IMedia->Min_Samples = (int)Parse_Float();
			Parse_Comma();
			IMedia->Max_Samples = (int)Allow_Float(IMedia->Min_Samples);

			if (IMedia->Min_Samples < 1)
			{
				Error("At least one sample per interval is needed in media.\n");
			}
			if (IMedia->Max_Samples < IMedia->Min_Samples)
			{
				Error("Maximum number of samples per interval smaller than minimum number.\n");
			}
		END_CASE

		/* next two courtesy of MH */
		CASE (METHOD_TOKEN)
			IMedia->Sample_Method = (int)Parse_Float();
			if (IMedia->Sample_Method != 1 && IMedia->Sample_Method!= 2 && IMedia->Sample_Method!= 3)
			{
				Error("Sample method choices are 1, 2, or 3. \n");
			}
			#ifdef UnofficialBlocking
				if (IMedia->Sample_Method > 2)
				{
					parseUnofficialFeature(30);
				}
			#endif

		END_CASE

		CASE (JITTER_TOKEN)
			IMedia->Jitter = Parse_Float();
		END_CASE

		/* NK samples */
		CASE (AA_THRESHOLD_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			IMedia->AA_Threshold = Parse_Float();
			if(IMedia->AA_Threshold<=0)
				Error("aa_threshold in media must be greater than zero.\n");
		END_CASE

		CASE (AA_LEVEL_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			IMedia->AA_Level = (int)Parse_Float();
			if(IMedia->AA_Level<1)
				Error("aa_level in media must be at least one.\n");
		END_CASE
		/* NK ---- */

		/* NK phmap */
		CASE (IGNORE_PHOTONS_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			IMedia->ignore_photons = TRUE;
		END_CASE
		/* NK ---- */

		CASE (ABSORPTION_TOKEN)
			Parse_Colour(IMedia->Absorption);
		END_CASE

		CASE (EMISSION_TOKEN)
			Parse_Colour(IMedia->Emission);
		END_CASE

		CASE (SCATTERING_TOKEN)
			Parse_Begin();
			IMedia->Type = (int)Parse_Float();
			if ((IMedia->Type < 1) || (IMedia->Type > SCATTERING_TYPES))
			{
				Warn(0, "Unknown atmospheric scattering type.");
			}
			Parse_Comma();
			Parse_Colour(IMedia->Scattering);
			EXPECT
				CASE (ECCENTRICITY_TOKEN)
					if (IMedia->Type != HENYEY_GREENSTEIN_SCATTERING)
					{
						Error("Eccentricity cannot be used with this scattering type.");
					}
					IMedia->Eccentricity = Parse_Float();
				END_CASE

				CASE (EXTINCTION_TOKEN)
					IMedia->sc_ext = Parse_Float();
				END_CASE

				OTHERWISE
					UNGET
					EXIT
				END_CASE
			END_EXPECT
			Parse_End();
		END_CASE

		CASE (CONFIDENCE_TOKEN)
			IMedia->Confidence = Parse_Float();
			if ((IMedia->Confidence <= 0.0) || (IMedia->Confidence >= 1.0))
			{
				Error("Illegal confidence value in media.\n");
			}
		END_CASE

		CASE (VARIANCE_TOKEN)
			IMedia->Variance = Parse_Float();
		END_CASE

		CASE (RATIO_TOKEN)
			IMedia->Ratio = Parse_Float();
		END_CASE

		CASE (DENSITY_TOKEN)
			Parse_Begin();
			Parse_Media_Density_Pattern(&(IMedia->Density));
			Parse_End();
		END_CASE

		CASE (TRANSLATE_TOKEN)
			Parse_Vector (Local_Vector);
			Compute_Translation_Transform(&Local_Trans, Local_Vector);
			Transform_Density (IMedia->Density, &Local_Trans);
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector (Local_Vector);
			Compute_Rotation_Transform(&Local_Trans, Local_Vector);
			Transform_Density (IMedia->Density, &Local_Trans);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Scale_Vector (Local_Vector);
			Compute_Scaling_Transform(&Local_Trans, Local_Vector);
			Transform_Density (IMedia->Density, &Local_Trans);
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix(Local_Matrix);
			Compute_Matrix_Transform(&Local_Trans, Local_Matrix);
			Transform_Density (IMedia->Density, &Local_Trans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
				Transform_Density (IMedia->Density, (TRANSFORM *)Token.Data);
			#else
			{
				TRANSFORM * Trans = Parse_Transform();
				Transform_Density (IMedia->Density, Trans);
				/*YS sept 17 2000 Memory leak*/
				POV_FREE(Trans);
			}
			#endif
		END_CASE

		CASE (LIGHT_GROUP_TOKEN)
			{
				/*YS sept 17 2000 Memory Leak*/
				char *tempstring=Parse_String();
				IMedia->Light_Group = Get_Light_Group(tempstring,&Tmp_Flag);
				POV_FREE(tempstring);
			}
			if (IMedia->Light_Group == NONE_GROUP) 
				Warn(300,"Light_Group of none is a special case, changed to All.\n");
			if (Tmp_Flag) 
				IMedia->Invert_Light_Group = TRUE;
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_End();
	IMedia->Next_Media = Next_Media;
	*Media_Ptr = IMedia;
}



/*****************************************************************************
*
* FUNCTION
*
*   Parse_Interior
*
* INPUT
*
* OUTPUT
*
* RETURNS
*   
* AUTHOR
*
*   Dieter Bayer
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jan 1997 : Creation.
*   Sep 1999 : Fade_Colour added - Edward Coffey
*
******************************************************************************/

void Parse_Interior(INTERIOR **Interior_Ptr)
{
	INTERIOR *Interior;
	Parse_Begin();

	EXPECT
		CASE(INTERIOR_ID_TOKEN)
			Destroy_Interior(*Interior_Ptr);
			*Interior_Ptr = Copy_Interior((INTERIOR *)Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	if(*Interior_Ptr == NULL)
	{
		*Interior_Ptr = Create_Interior();
	}

	Interior = *Interior_Ptr;

	EXPECT
		CASE (IOR_TOKEN)
			Interior->IOR = Parse_Float();
		END_CASE

		CASE (DISPERSION_TOKEN)
			Interior->Dispersion = Parse_Float();
		END_CASE

		CASE (DISP_NELEMS_TOKEN)
			Interior->Disp_NElems = (int)Parse_Float();
		END_CASE

		CASE (CAUSTICS_TOKEN)
			Interior->Caustics = Parse_Float() * 45.0;
		END_CASE

		CASE (FADE_DISTANCE_TOKEN)
			Interior->Fade_Distance = Parse_Float();
		END_CASE

		CASE (FADE_POWER_TOKEN)
			Interior->Fade_Power = Parse_Float();
		END_CASE

		CASE (FADE_COLOUR_TOKEN)
			Parse_Colour(Interior->Fade_Colour);
		END_CASE

		CASE (MEDIA_TOKEN)
			Parse_Media((IMEDIA **)(&Interior->IMedia));
		END_CASE

		CASE (REFRACTION_TOKEN)
			Interior->Old_Refract = Parse_Float();
			Warn_Interior("Refraction value unnecessary to turn on refraction.\nTo attenuate, the fade_power and fade_distance keywords ");
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_End();
	Init_Interior(Interior);
}



/*****************************************************************************
*
* FUNCTION
*
*   Parse_Media_Density_Pattern
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Dieter Bayer
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Dez 1996 : Creation.
*
******************************************************************************/

void Parse_Media_Density_Pattern(PIGMENT **Density_Ptr)
{
	PIGMENT *New;
	EXPECT
		CASE (DENSITY_ID_TOKEN)
			New = Copy_Pigment ((PIGMENT *) Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			New = Create_Pigment();
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_Pattern((TPATTERN *)New,DENSITY_TYPE);
	New->Next = (TPATTERN *)(*Density_Ptr);
	*Density_Ptr = New;
}

/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
* CHANGES
*
******************************************************************************/

FOG *Parse_Fog()
{
	VECTOR Vector;
	MATRIX Matrix;
	TRANSFORM Trans;
	FOG *Fog;

	Parse_Begin();

	EXPECT
		CASE(FOG_ID_TOKEN)
			Fog = Copy_Fog ((FOG *) Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			Fog = Create_Fog();
			EXIT
		END_CASE
	END_EXPECT

	EXPECT
		CASE_COLOUR
			Parse_Colour(Fog->Colour);
		END_CASE

		CASE (DISTANCE_TOKEN)
			Fog->Distance = Parse_Float();
		END_CASE

		CASE_FLOAT
			Warn(150, "Should use distance keyword.");
			Fog->Distance = Parse_Float();
		END_CASE

		CASE (FOG_TYPE_TOKEN)
			Fog->Type = (int)Parse_Float();
			if ((Fog->Type < ORIG_FOG) || (Fog->Type > FOG_TYPES))
			{
				Warn(0, "Unknown fog type.");
			}
		END_CASE

		CASE (FOG_ALT_TOKEN)
			Fog->Alt = Parse_Float();
		END_CASE

		CASE (FOG_OFFSET_TOKEN)
			Fog->Offset = Parse_Float();
		END_CASE

		CASE (TURB_DEPTH_TOKEN)
			Fog->Turb_Depth = Parse_Float();
		END_CASE

		CASE (UP_TOKEN)
			Parse_Vector(Fog->Up);
		END_CASE

		#ifdef FogPigmentPatch
			CASE (PIGMENT_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(40);
				#endif
				Parse_Begin();
				if (Fog->Pigment == NULL) 
					Fog->Pigment = Create_Pigment();
				Parse_Pigment(&(Fog->Pigment));
				Parse_End();
				Post_Pigment(Fog->Pigment);
			END_CASE
		#endif

		CASE (TURBULENCE_TOKEN)
			if (Fog->Turb == NULL)
			{
				Fog->Turb=(TURB *)Create_Warp(CLASSIC_TURB_WARP);
			}
			Parse_Vector(Fog->Turb->Turbulence);
		END_CASE

		CASE (OCTAVES_TOKEN)
			if (Fog->Turb == NULL)
			{
				Fog->Turb=(TURB *)Create_Warp(CLASSIC_TURB_WARP);
			}
			Fog->Turb->Octaves = (int)Parse_Float();
			if(Fog->Turb->Octaves < 1)
				Fog->Turb->Octaves = 1;
			if(Fog->Turb->Octaves > 10)
				Fog->Turb->Octaves = 10;
		END_CASE

		CASE (OMEGA_TOKEN)
			if (Fog->Turb == NULL)
			{
				Fog->Turb=(TURB *)Create_Warp(CLASSIC_TURB_WARP);
			}
			Fog->Turb->Omega = Parse_Float();
		END_CASE

		CASE (LAMBDA_TOKEN)
			if (Fog->Turb == NULL)
			{
				Fog->Turb=(TURB *)Create_Warp(CLASSIC_TURB_WARP);
			}
			Fog->Turb->Lambda = Parse_Float();
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector(Vector);
			Compute_Rotation_Transform(&Trans, Vector);
			MTransDirection(Fog->Up, Fog->Up, &Trans);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Vector(Vector);
			Compute_Scaling_Transform(&Trans, Vector);
			MTransDirection(Fog->Up, Fog->Up, &Trans);
		END_CASE

		CASE (TRANSLATE_TOKEN)
			Parse_Vector(Vector);
			Warn(0, "A fog's up vector can't be translated.");
			/*
			Compute_Translation_Transform(&Trans, Vector);
			MTransDirection(Fog->Up, Fog->Up, &Trans);
			*/
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix(Matrix);
			Compute_Matrix_Transform(&Trans, Matrix);
			MTransDirection(Fog->Up, Fog->Up, &Trans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
				MTransDirection(Fog->Up, Fog->Up, (TRANSFORM *)Token.Data);
			#else
			{
				TRANSFORM * Trans = Parse_Transform();
				MTransDirection(Fog->Up, Fog->Up, Trans);
				/*YS sept 17 2000 Memory leak*/
				POV_FREE(Trans);
			}
			#endif
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_End ();
	/* Make sure the up vector is normalized. */
	VNormalize(Fog->Up, Fog->Up);
	return(Fog);
}



/*****************************************************************************
*
* FUNCTION
*
*   Parse_Rainbow
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   Dieter Bayer
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jul 1994 : Creation.
*
*   Dec 1994 : Modified to work with multiple rainbows. [DB]
*
*   Apr 1995 : Added code for rainbow arcs. [DB]
*
******************************************************************************/

RAINBOW *Parse_Rainbow()
{
	int Angle1, Angle2;
	DBL dot;
	RAINBOW *Rainbow;

	Angle1 = Angle2 = FALSE;

	Parse_Begin();

	EXPECT
		CASE(RAINBOW_ID_TOKEN)
			Rainbow = Copy_Rainbow ((RAINBOW *) Token.Data);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			Rainbow = Create_Rainbow();
			EXIT
		END_CASE
	END_EXPECT

	EXPECT
		CASE (ANGLE_TOKEN)
			Rainbow->Angle = Parse_Float();
		END_CASE

		CASE (DIRECTION_TOKEN)
			Parse_Vector(Rainbow->Antisolar_Vector);
		END_CASE

		CASE (COLOUR_MAP_TOKEN)
			Rainbow->Pigment = Create_Pigment();
			Rainbow->Pigment->Blend_Map = Parse_Colour_Map();
			Rainbow->Pigment->Type = GRADIENT_PATTERN;
			Make_Vector (Rainbow->Pigment->Vals.Gradient,1.0,0.0,0.0);
		END_CASE

		CASE (DISTANCE_TOKEN)
			Rainbow->Distance = Parse_Float();
		END_CASE

		CASE (JITTER_TOKEN)
			Rainbow->Jitter = Parse_Float();
		END_CASE

		CASE (WIDTH_TOKEN)
			Rainbow->Width = Parse_Float();
		END_CASE

		CASE (UP_TOKEN)
			Parse_Vector(Rainbow->Up_Vector);
		END_CASE

		CASE (FALLOFF_ANGLE_TOKEN)
			Angle1 = TRUE;
			Rainbow->Falloff_Angle = Parse_Float();
			if ((Rainbow->Falloff_Angle < 0.0) || (Rainbow->Falloff_Angle > 360.0))
			{
				Error("Illegal falloff angle in rainbow (Use value from 0 to 360 degrees).\n");
			}
			Rainbow->Falloff_Angle *= M_PI_360;
		END_CASE

		CASE (ARC_ANGLE_TOKEN)
			Angle2 = TRUE;
			Rainbow->Arc_Angle = Parse_Float();
			if ((Rainbow->Arc_Angle < 0.0) || (Rainbow->Arc_Angle > 360.0))
			{
				Error("Illegal arc angle in rainbow (Use value from 0 to 360 degrees).\n");
			}
			Rainbow->Arc_Angle *= M_PI_360;
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_End();

	/* Setup falloff angle. */
	if (Angle2 && !Angle1)
	{
		Rainbow->Falloff_Angle = Rainbow->Arc_Angle;
	}

	/* Test if arc angle is greater or equal to falloff angle. */
	if (Rainbow->Arc_Angle < Rainbow->Falloff_Angle)
	{
		Error("Arc angle is smaller than falloff angle in rainbow.\n");
	}

	/* Get falloff region width.*/
	Rainbow->Falloff_Width = Rainbow->Arc_Angle - Rainbow->Falloff_Angle;

	/* Check for illegal vectors. */
	VDot(dot, Rainbow->Antisolar_Vector, Rainbow->Antisolar_Vector);
	if (fabs(dot) < EPSILON)
	{
		Error("Rainbow's direction vector is zero.\n");
	}
	VDot(dot, Rainbow->Up_Vector, Rainbow->Up_Vector);
	if (fabs(dot) < EPSILON)
	{
		Error("Rainbow's up vector is zero.\n");
	}
	VNormalizeEq(Rainbow->Antisolar_Vector);
	VNormalizeEq(Rainbow->Up_Vector);
	VDot(dot, Rainbow->Up_Vector, Rainbow->Antisolar_Vector);
	if (fabs(1.0 - fabs(dot)) < EPSILON)
	{
		Error("Rainbow's up and direction vector are co-linear.\n");
	}

	/* Make sure that up and antisolar vector are perpendicular. */
	VCross(Rainbow->Right_Vector, Rainbow->Up_Vector, Rainbow->Antisolar_Vector);
	VCross(Rainbow->Up_Vector, Rainbow->Antisolar_Vector, Rainbow->Right_Vector);
	VNormalizeEq(Rainbow->Up_Vector);
	VNormalizeEq(Rainbow->Right_Vector);

	/* Adjust rainbow angle and width. */
	Rainbow->Angle -= 0.5 * Rainbow->Width;
	Rainbow->Angle *= M_PI_180;
	Rainbow->Width *= M_PI_180;
	return(Rainbow);
}



/*****************************************************************************
*
* FUNCTION
*
*   Parse_Skysphere
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   Dieter Bayer
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jul 1994 : Creation.
*
*   Dec 1994 : Modified to work with multiple skyspheres. [DB]
*
******************************************************************************/

SKYSPHERE *Parse_Skysphere()
{
	VECTOR Local_Vector;
	MATRIX Local_Matrix;
	TRANSFORM Local_Trans;
	SKYSPHERE *Skysphere;

	Parse_Begin();

	EXPECT
		CASE(SKYSPHERE_ID_TOKEN)
			Skysphere = Copy_Skysphere((SKYSPHERE *)Token.Data);
			EXIT
		END_CASE
	
		OTHERWISE
			UNGET
			Skysphere = Create_Skysphere();
			EXIT
		END_CASE
	END_EXPECT

	EXPECT
		CASE (PIGMENT_TOKEN)
			Skysphere->Count++;
			Skysphere->Pigments = (PIGMENT **)POV_REALLOC(Skysphere->Pigments, Skysphere->Count*sizeof(SKYSPHERE *), "sky-sphere pigment");
			Skysphere->Pigments[Skysphere->Count-1] = Create_Pigment();
			Parse_Begin();
			Parse_Pigment(&(Skysphere->Pigments[Skysphere->Count-1]));
			Parse_End();
		END_CASE

		CASE (TRANSLATE_TOKEN)
			Parse_Vector (Local_Vector);
			Translate_Skysphere(Skysphere, Local_Vector);
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector (Local_Vector);
			Rotate_Skysphere(Skysphere, Local_Vector);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Scale_Vector (Local_Vector);
			Scale_Skysphere(Skysphere, Local_Vector);
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix(Local_Matrix);
			Compute_Matrix_Transform(&Local_Trans, Local_Matrix);
			Transform_Skysphere(Skysphere, &Local_Trans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
				Transform_Skysphere(Skysphere, (TRANSFORM *)Token.Data);
			#else
			{
				TRANSFORM * Trans = Parse_Transform();
				Transform_Skysphere(Skysphere, Trans);
				/*YS sept 17 2000 Memory leak*/
				POV_FREE(Trans);
			}
			#endif
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_End();
	if (Skysphere->Count==0)
	{
		Error("Empty sky_sphere statement.");
	}
	return(Skysphere);
}

/*****************************************************************************
*
* FUNCTION      : Check_BH_Parameters
*
* ARGUMENTS     : bh - pointer to Black_Hole
*
* AUTHOR        : CJC [7/95]
*
* DESCRIPTION   : Applies sanity checks to the parameters of a black hole.
*
* CHANGES
*
******************************************************************************/

static void Check_BH_Parameters (BLACK_HOLE *bh)
{
	if (bh->Repeat == FALSE) 
		return ;

	if (bh->Repeat_Vector [X] > 0.0)
	{
		if (bh->Center [X] < bh->Radius)
			bh->Center [X] = bh->Radius ;
		if (bh->Repeat_Vector [X] < bh->Center [X] + bh->Radius + bh->Uncertainty_Vector [X])
		{
			bh->Repeat_Vector [X] = bh->Center [X] + bh->Radius + bh->Uncertainty_Vector [X] ;
			Warning (0, "Black Hole repeat vector X too small ; increased to %g\n", bh->Repeat_Vector [X]) ;
		} 
		if (bh->Repeat_Vector [X] < Small_Tolerance)
		{
			Warning (0,"Black Hole repeat vector X is less than %f ; ignored\n", (float) Small_Tolerance) ;
			bh->Repeat_Vector [X] = 0.0 ;
		}
	}

	if (bh->Repeat_Vector [Y] > 0.0)
	{
		if (bh->Center [Y] < bh->Radius)
			bh->Center [Y] = bh->Radius ;
		if (bh->Repeat_Vector [Y] < bh->Center [Y] + bh->Radius + bh->Uncertainty_Vector [Y])
		{
			bh->Repeat_Vector [Y] = bh->Center [Y] + bh->Radius + bh->Uncertainty_Vector [Y] ;
			Warning (0, "Black Hole repeat vector Y too small ; increased to %g\n", bh->Repeat_Vector [Y]) ;
		} 
		if (bh->Repeat_Vector [Y] < Small_Tolerance)
		{
			Warning (0, "Black Hole repeat vector Y is less than %f ; ignored\n", (float) Small_Tolerance) ;
			bh->Repeat_Vector [Y] = 0.0 ;
		} 
	}

	if (bh->Repeat_Vector [Z] > 0.0)
	{
		if (bh->Center [Z] < bh->Radius)
			bh->Center [Z] = bh->Radius ;
		if (bh->Repeat_Vector [Z] < bh->Center [Z] + bh->Radius + bh->Uncertainty_Vector [Z])
		{
			bh->Repeat_Vector [Z] = bh->Center [Z] + bh->Radius + bh->Uncertainty_Vector [Z] ;
			Warning (0, "Black Hole repeat vector Z too small ; increased to %g\n", bh->Repeat_Vector [Z]) ;
		} 
		if (bh->Repeat_Vector [Z] < Small_Tolerance)
		{
			Warning (0, "Black Hole repeat vector Z is less than %f ; ignored\n", (float) Small_Tolerance) ;
			bh->Repeat_Vector [Z] = 0.0 ;
		}
	}
}

/*****************************************************************************
*
* FUNCTION
*
*   Check_Turb
*
* INPUT
*
*   Warps_Ptr : Address where the root warp of a warp list
*   is stored.
*   
* OUTPUT
*
*   Warps_Ptr : If *Warps_Ptr is NULL, a classic turb warp
*   is created and a pointer to it is stored
*   
* RETURNS
*
*   A pointer to the last warp in the chain which is guarenteed
*   to be a classic turb.
*   
* AUTHOR
*
*   CEY [2/95]
*   
* DESCRIPTION   : This routine is called when a classic outside-the-warp
*  turbulence parameter is parsed.  One and only one classic turb may exist 
*  in a warp chain.  If there is one, it must be the last.  This routine 
*  traverses the warp chain and looks at the last link.  If it is not a 
*  classic turb then it adds one to the end and returns a pointer to it.
*  If the chain is empty, it creates a single link chain consisting of a 
*  classic turb link.  Future warp links get added ahead of the chain so
*  that any classic turb link is always last.
*
* CHANGES
*
******************************************************************************/

static TURB *Check_Turb (WARP **Warps_Ptr)
{
	WARP *Temp=*Warps_Ptr;
	if (Temp == NULL)
	{
		*Warps_Ptr = Temp = Create_Warp(CLASSIC_TURB_WARP);
	}
	else
	{
		while (Temp->Next_Warp != NULL)
		{
			Temp=Temp->Next_Warp;
		}

		if (Temp->Warp_Type != CLASSIC_TURB_WARP)
		{
			Temp->Next_Warp=Create_Warp(CLASSIC_TURB_WARP);
			Temp=Temp->Next_Warp;
		}
	}
	return((TURB *)Temp);
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
* CHANGES
*  Talious 10/24/1998: Added SPherical/Cylindrical/Toroidaal warps
*
******************************************************************************/
#ifndef GlowPatch
static void Parse_Warp (WARP **Warp_Ptr)
#else
 void Parse_Warp (WARP **Warp_Ptr)
#endif
{
	WARP *New = NULL;
	TURB *Turb;
	REPEAT *Repeat;
	BLACK_HOLE *Black_Hole;
	#ifdef PatternWarpPatch
		CYLW *CylW;
		SPHEREW *SphereW;
		TOROIDAL *Toroidal;
		PLANARW *PlanarW;
	#endif
	#ifdef DisplaceWarpPatch
		DISP_WARP *DispWarp;
	#endif
	VECTOR Local_Vector;

	/*  Parse_Begin();*/

	EXPECT
		/* NK 1998 reset_children */
		CASE(RESET_CHILDREN_TOKEN)
			#ifdef UnofficialBlocking
				parseUnofficialFeature(30);
			#endif
			New=Create_Warp(RESET_CHILDREN_WARP);
			EXIT
		END_CASE
		/* NK ---- */

		CASE(TURBULENCE_TOKEN)
			New=Create_Warp(EXTRA_TURB_WARP);
			Turb=(TURB *)New;
			Parse_Vector(Turb->Turbulence);
			EXPECT
				CASE(OCTAVES_TOKEN)
					Turb->Octaves = (int)Parse_Float();
					if(Turb->Octaves < 1)
						Turb->Octaves = 1;
					if(Turb->Octaves > 10)  /* Avoid DOMAIN errors */
					Turb->Octaves = 10;
				END_CASE

				CASE (OMEGA_TOKEN)
					Turb->Omega = Parse_Float();
				END_CASE

				CASE (LAMBDA_TOKEN)
					Turb->Lambda = Parse_Float();
				END_CASE

				OTHERWISE
					UNGET
					EXIT
				END_CASE
			END_EXPECT
			EXIT
		END_CASE

		CASE(REPEAT_TOKEN)
			New=Create_Warp(REPEAT_WARP);
			Repeat=(REPEAT *)New;
			Parse_Vector(Local_Vector);
			Repeat->Axis=-1;
			if (Local_Vector[X]!=0.0) 
				Repeat->Axis=X;
			if (Local_Vector[Y]!=0.0)
			{ /* tw */
				if (Repeat->Axis < X)
					Repeat->Axis=Y;
				else 
					Error("Can only repeat along 1 axis.");
			} /* tw */
			if (Local_Vector[Z]!=0.0)
			{ /* tw */
				if (Repeat->Axis < X)
					Repeat->Axis=Z;
				else
					Error("Can only repeat along 1 axis.");
			} /* tw */
			if (Repeat->Axis < X)
					Error("No axis specified in repeat.");
			Repeat->Width=Local_Vector[Repeat->Axis];
		
			EXPECT
				CASE(OFFSET_TOKEN)
					Parse_Vector(Repeat->Offset);
				END_CASE

				CASE(FLIP_TOKEN)
					Parse_Vector(Repeat->Flip);
					if (Repeat->Flip[X]!=0.0) 
						Repeat->Flip[X]=-1.0; 
					else 
						Repeat->Flip[X]=1.0;
					if (Repeat->Flip[Y]!=0.0) 
						Repeat->Flip[Y]=-1.0; 
					else 
						Repeat->Flip[Y]=1.0;
					if (Repeat->Flip[Z]!=0.0) 
						Repeat->Flip[Z]=-1.0; 
					else 
						Repeat->Flip[Z]=1.0;
				END_CASE

				OTHERWISE
					UNGET
					EXIT
				END_CASE
			END_EXPECT
			EXIT
		END_CASE

		CASE(BLACK_HOLE_TOKEN)
			New = Create_Warp(BLACK_HOLE_WARP) ;
			Black_Hole = (BLACK_HOLE *) New ;
			Parse_Vector (Local_Vector) ;
			Assign_Vector (Black_Hole->Center, Local_Vector) ;
			Parse_Comma () ;
			Black_Hole->Radius = Parse_Float () ;
			Black_Hole->Radius_Squared = Black_Hole->Radius * Black_Hole->Radius ;
			Black_Hole->Inverse_Radius = 1.0 / Black_Hole->Radius;
			Black_Hole->Strength = 1.0 ;
			Black_Hole->Power = 2.0 ;
			Black_Hole->Inverted = FALSE ;
			Black_Hole->Type = 0 ;

			EXPECT
				CASE(STRENGTH_TOKEN)
					Black_Hole->Strength = Parse_Float () ;
				END_CASE

				CASE(FALLOFF_TOKEN)
					Black_Hole->Power = Parse_Float () ;
				END_CASE

				CASE(INVERSE_TOKEN)
					Black_Hole->Inverted = TRUE ;
				END_CASE

				CASE(TYPE_TOKEN)
					Black_Hole->Type = (int) Parse_Float () ;
				END_CASE

				CASE(REPEAT_TOKEN)
					Parse_Vector (Black_Hole->Repeat_Vector) ;
					Black_Hole->Repeat = TRUE ;
					Check_BH_Parameters (Black_Hole) ;
				END_CASE

				CASE(TURBULENCE_TOKEN)
					Parse_Vector (Black_Hole->Uncertainty_Vector) ;
					Black_Hole->Uncertain = TRUE ;
					Check_BH_Parameters (Black_Hole) ;
				END_CASE

				OTHERWISE
					UNGET
					EXIT
				END_CASE
			END_EXPECT
			EXIT
		END_CASE

		#ifdef DisplaceWarpPatch
			CASE(DISPLACE_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(60);
				#endif
				Parse_Begin();
				New = Create_Warp(DISPLACE_WARP);
				DispWarp = (DISP_WARP *) New;
				Parse_Pigment(&(DispWarp->Pig));
				Get_Token();
				if(Token.Token_Id == TYPE_TOKEN)
					DispWarp->type = (int)Parse_Float();
				else 
					Unget_Token();
				if ( DispWarp->type <0 || DispWarp->type>1)
					Error("\nOnly 0 and 1 allowed for type.\n");

				if (DispWarp->type == 1)
				{ 
					Parse_Comma(); 
					DispWarp->dist = Allow_Float(0.05);
				}
				Parse_End();
				EXIT
			END_CASE
		#endif

		#ifdef PatternWarpPatch
			CASE(CYLINDRICAL_TOKEN)
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				New = Create_Warp(CYLINDRICAL_WARP);
				CylW = (CYLW *) New ;
				EXPECT
					CASE(ORIENTATION_TOKEN)
						Parse_Vector (Local_Vector);
						VNormalizeEq(Local_Vector);
						Assign_Vector (CylW->Orientation_Vector, Local_Vector) ;
					END_CASE

					CASE(DIST_EXP_TOKEN)
						CylW->DistExp = Parse_Float () ;
					END_CASE

					OTHERWISE
						UNGET
						EXIT
					END_CASE
				END_EXPECT
				EXIT
			END_CASE

			CASE(SPHERICAL_TOKEN)
				New = Create_Warp(SPHERICAL_WARP);
				SphereW = (SPHEREW *) New ;
				EXPECT
					CASE(ORIENTATION_TOKEN)
						Parse_Vector (Local_Vector);
						VNormalizeEq(Local_Vector);
						Assign_Vector (SphereW->Orientation_Vector, Local_Vector) ;
					END_CASE

					CASE(DIST_EXP_TOKEN)
						SphereW->DistExp = Parse_Float () ;
					END_CASE

					OTHERWISE
						UNGET
						EXIT
					END_CASE
				END_EXPECT
				EXIT
			END_CASE

			CASE(PLANAR_TOKEN)
				New = Create_Warp(PLANAR_WARP);
				PlanarW = (PLANARW *) New ;
				if(Allow_Vector(Local_Vector))
				{
					VNormalizeEq(Local_Vector);
					Assign_Vector(PlanarW->Orientation_Vector,Local_Vector);
					Parse_Comma();
					PlanarW->OffSet=Parse_Float();
				}
				EXIT
			END_CASE

			CASE(TOROIDAL_TOKEN)
				New = Create_Warp(TOROIDAL_WARP);
				Toroidal = (TOROIDAL *) New ;
				EXPECT
					CASE(ORIENTATION_TOKEN)
						Parse_Vector (Local_Vector) ;
						VNormalizeEq(Local_Vector);
						Assign_Vector (Toroidal->Orientation_Vector, Local_Vector) ;
					END_CASE

					CASE(DIST_EXP_TOKEN)
						Toroidal->DistExp = Parse_Float () ;
					END_CASE

					CASE(MAJOR_RADIUS_TOKEN)
						Toroidal->MajorRadius = Parse_Float () ;
					END_CASE

					OTHERWISE
						UNGET
						EXIT
					END_CASE
				END_EXPECT
				EXIT
			END_CASE
		#endif

		OTHERWISE
			Parse_Error_Str ("warp type");
		END_CASE
	END_EXPECT

	if (New==NULL)
	{
		Error("Empty warp statement.");
	}
	New->Next_Warp=*Warp_Ptr;
	*Warp_Ptr=New;
	/*  Parse_End();*/
}


static void Warn_Interior(char *s)
{
	Warn(0,s);
	Warning(0,"should be specified in 'interior{...}' statement.\n");
	Warn_Compat(0);
}

void Parse_Material(MATERIAL *Material)
{
	MATERIAL *Temp;
	TEXTURE *Texture;
	#ifdef InteriorTexturePatch
		TEXTURE * Int_Texture;/*Chris Huff: Interior Texture patch*/
	#endif
	VECTOR Local_Vector;
	MATRIX Local_Matrix;
	TRANSFORM Local_Trans;

	Parse_Begin();

	EXPECT
		CASE(MATERIAL_ID_TOKEN)
			Temp = (MATERIAL *)Token.Data;
			Texture = Copy_Textures(Temp->Texture);
			#ifdef InteriorTexturePatch
				Int_Texture = Copy_Textures(Temp->Interior_Texture);/*Chris Huff: Interior Texture patch*/
			#endif
			Link_Textures(&(Material->Texture),Texture);
			#ifdef InteriorTexturePatch
				Link_Textures(&(Material->Interior_Texture),Int_Texture);/*Chris Huff: Interior Texture patch*/
			#endif
			Destroy_Interior(Material->Interior);
			Material->Interior = Copy_Interior(Temp->Interior);
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	EXPECT
		CASE (TEXTURE_TOKEN)
			Parse_Begin ();
			Texture = Parse_Texture ();
			Parse_End ();
			Link_Textures(&(Material->Texture),Texture);
		END_CASE

		#ifdef InteriorTexturePatch
			CASE (INTERIOR_TEXTURE_TOKEN)/*Chris Huff: Interior Texture patch*/
				#ifdef UnofficialBlocking
					parseUnofficialFeature(30);
				#endif
				Parse_Begin ();
				Int_Texture = Parse_Texture ();
				Parse_End ();
				Link_Textures(&(Material->Interior_Texture), Int_Texture);
			END_CASE
		#endif

		CASE (INTERIOR_TOKEN)
			Parse_Interior((INTERIOR **)(&(Material->Interior)));
		END_CASE

		CASE (TRANSLATE_TOKEN)
			Parse_Vector (Local_Vector);
			Compute_Translation_Transform(&Local_Trans, Local_Vector);
			Transform_Textures (Material->Texture, &Local_Trans);
			Transform_Interior (Material->Interior,&Local_Trans);
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector (Local_Vector);
			Compute_Rotation_Transform(&Local_Trans, Local_Vector);
			Transform_Textures (Material->Texture, &Local_Trans);
			Transform_Interior (Material->Interior,&Local_Trans);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Scale_Vector (Local_Vector);
			Compute_Scaling_Transform(&Local_Trans, Local_Vector);
			Transform_Textures (Material->Texture, &Local_Trans);
			Transform_Interior (Material->Interior,&Local_Trans);
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix(Local_Matrix);
			Compute_Matrix_Transform(&Local_Trans, Local_Matrix);
			Transform_Textures (Material->Texture, &Local_Trans);
			Transform_Interior (Material->Interior,&Local_Trans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
				Transform_Textures (Material->Texture, (TRANSFORM *)Token.Data);
				Transform_Interior (Material->Interior,(TRANSFORM *)Token.Data);
			#else
			{
				TRANSFORM * Trans = Parse_Transform();
				Transform_Textures (Material->Texture, Trans);
				Transform_Interior (Material->Interior, Trans);
				/*YS sept 17 2000 Memory leak*/
				POV_FREE(Trans);
			}
			#endif
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT
	Parse_End();
}
#ifdef EvalPigmentPatch
/*****************************************************************************
*
* FUNCTION Parse_eval_pat()
*
* INPUT None
*
* OUTPUT None
*
* RETURNS DBL
*
* AUTHOR Chris Huff
*
* DESCRIPTION This function parses a "eval_pat()" function which takes a pattern
*             and a point vector, and returns the value at that point.
*
* CHANGES
*
******************************************************************************/
void Parse_eval_pig(VECTOR Vect)
{
	PIGMENT *New = Create_Pigment ();
	VECTOR point;
	COLOUR result;

	GET (LEFT_PAREN_TOKEN);

	/* YS added to parse pigment{} too*/
	EXPECT
		CASE (PIGMENT_TOKEN)
			Parse_Begin ();
			Parse_Pigment (&New );
			Parse_End ();
			Post_Pigment(New);	/* YS bug fix june 11 2000 */
			EXIT
		END_CASE

		OTHERWISE
			UNGET
			Parse_Pigment(&New);
			Post_Pigment(New);	/* YS bug fix june 11 2000 */
			EXIT
		END_CASE
	END_EXPECT
	/*YS*/
	
	Parse_Comma();
	Parse_Vector(point);
	GET (RIGHT_PAREN_TOKEN);
	/*Compute_Pigment (COLOUR Colour, PIGMENT *Pigment, VECTOR EPoint)*/
	Compute_Pigment (result, New, point,NULL);
	Destroy_Pigment (New);

	Vect[0] = result[0];
	Vect[1] = result[1];
	Vect[2] = result[2];

	return;
}

#endif
#ifdef EvalPatternPatch
/*****************************************************************************
*
* FUNCTION Parse_eval_pat()
*
* INPUT None
*
* OUTPUT None
*
* RETURNS DBL
*
* AUTHOR Chris Huff
*
* DESCRIPTION This function parses a "eval_pat()" function which takes a pattern
*             and a point vector, and returns the value at that point.
*
* CHANGES
*
******************************************************************************/
DBL Parse_eval_pat()
{
	TPATTERN *New = (TPATTERN *)POV_MALLOC(sizeof (TPATTERN), "TPATTERN");
	VECTOR point;
	DBL result;
	Init_TPat_Fields((TPATTERN *)New);

	GET (LEFT_PAREN_TOKEN);
	Parse_Pattern(New, PATTERN_TYPE);
	Parse_Comma();
	Parse_Vector(point);
	GET (RIGHT_PAREN_TOKEN);

	result = Evaluate_TPat ((TPATTERN *)New, point,NULL);
	Destroy_TPat_Fields(New);
	POV_FREE(New);
	return result;
}
#endif

#ifdef VWarpPatch
/*****************************************************************************
*
* FUNCTION Parse_VWarp()
*
* INPUT None
*
* OUTPUT None
*
* RETURNS DBL
*
* AUTHOR Chris Huff
*
* DESCRIPTION 
*
* CHANGES
*
******************************************************************************/
void Parse_VWarp(VECTOR result)
{
	VECTOR EPoint;
	TPATTERN * Pat = (TPATTERN *)POV_MALLOC(sizeof(TPATTERN), "TPATTERN");
	Init_TPat_Fields(Pat);

	Pat->Type = BOZO_PATTERN;

	GET (LEFT_PAREN_TOKEN);
	Parse_Vector(EPoint);
	Parse_Comma();
	Parse_Warp(&(Pat->Warps));
	GET (RIGHT_PAREN_TOKEN);

	Warp_EPoint(result, EPoint, Pat, FALSE);

	Destroy_TPat_Fields(Pat);
	POV_FREE(Pat);
}
#endif

#ifdef BlobPatternPatch
/*Chris Huff blob pattern*/
BLOB_PATTERN_DATA * Create_Blob_Pattern_Component()
{
	BLOB_PATTERN_DATA * newComponent = (BLOB_PATTERN_DATA *)POV_MALLOC(sizeof(BLOB_PATTERN_DATA), "blob pattern component");
	newComponent->function = 0;/*defaults to standard*/
	newComponent->falloff = 2;/*defaults to standard*/
	newComponent->type = 1;/*defaults to spherical*/
	newComponent->radius = 1;
	newComponent->strength = 1;
	newComponent->inverse = FALSE;

	newComponent->transform = Create_Transform();
	newComponent->blob = NULL;
	newComponent->pigment = NULL;
	newComponent->next = NULL;

	Make_Vector(newComponent->center, 0, 0, 0);
	Make_Vector(newComponent->pointB, 0, 0, 0);
	return newComponent;
}
/*Chris Huff blob pattern*/
void Add_Blob_Pattern_Component(TPATTERN * pat, BLOB_PATTERN_DATA * comp)
{
	if(pat->Vals.Blob.blob_dat == NULL)
	{
		pat->Vals.Blob.blob_dat = comp;
	}
	else
	{
		BLOB_PATTERN_DATA * lastComponent;
		lastComponent = pat->Vals.Blob.blob_dat;
		while(lastComponent->next != NULL)
		lastComponent = lastComponent->next;
		lastComponent->next = comp;
	}
	/*
	{
	comp->next = pat->Vals.Blob.blob_dat;
	pat->Vals.Blob.blob_dat = comp;
	}*/
	comp->next = NULL;
}

void Parse_Blob_Pattern_Component_Mods(BLOB_PATTERN_DATA * comp)
{
	VECTOR transVect;
	MATRIX tempMatrix;
	TRANSFORM tempTrans;

	EXPECT
		CASE (INVERSE_TOKEN)
		#ifdef UnofficialBlocking
			parseUnofficialFeature(60);
		#endif 
			comp->inverse = TRUE;
		END_CASE

		CASE (DENSITY_FUNCTION_TOKEN)
			comp->function = (int)Parse_Float();
			if ( comp->function <0 || comp->function >4)
				Error("\nUnkonw density function.\n");
			Parse_Comma();
			comp->falloff = Parse_Float();
		END_CASE

		CASE (TRANSLATE_TOKEN)
			Parse_Vector (transVect);
			Compute_Translation_Transform(&tempTrans, transVect);
			Compose_Transforms(comp->transform, &tempTrans);
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector (transVect);
			Compute_Rotation_Transform(&tempTrans, transVect);
			Compose_Transforms(comp->transform, &tempTrans);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Vector (transVect);
			Compute_Scaling_Transform(&tempTrans, transVect);
			Compose_Transforms(comp->transform, &tempTrans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
				Compose_Transforms(comp->transform, (TRANSFORM *)Token.Data);
			#else
			{
				TRANSFORM * Trans = Parse_Transform();
				Compose_Transforms(comp->transform, Trans);
				/*YS sept 17 2000 Memory leak*/
				POV_FREE(Trans);
			}
			#endif
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix (tempMatrix);
			Compute_Matrix_Transform(&tempTrans, tempMatrix);
			Compose_Transforms(comp->transform, &tempTrans);
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT
}

void Parse_Blob_Pigment_Component_Mods(BLOB_PATTERN_DATA * comp)
{
	VECTOR transVect;
	MATRIX tempMatrix;
	TRANSFORM tempTrans;

	EXPECT
		CASE(INVERSE_TOKEN)
			comp->inverse = TRUE;
		END_CASE

		CASE(PIGMENT_TOKEN)
			if(comp->type == 3)
			{
				Error("Can't use pigment in blob_pigment pigment components.");
			}
			else
			{
				Parse_Begin();
				if(comp->pigment == NULL)
					comp->pigment = Create_Pigment();
				Parse_Pigment(&comp->pigment);
				Post_Pigment(comp->pigment);
				Parse_End();
			}
		END_CASE

		CASE (DENSITY_FUNCTION_TOKEN)
			comp->function = (int)Parse_Float();
			if ( comp->function <0 || comp->function >4)
				Error("\nOnly 0,1,2,3 and 4 allowed for density function.\n");
			Parse_Comma();
			comp->falloff = Parse_Float();
		END_CASE

		CASE (TRANSLATE_TOKEN)
			Parse_Vector (transVect);
			Compute_Translation_Transform(&tempTrans, transVect);
			Compose_Transforms(comp->transform, &tempTrans);
			if(comp->pigment!=NULL)
				Transform_Tpattern((TPATTERN *)comp->pigment, & tempTrans);
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector (transVect);
			Compute_Rotation_Transform(&tempTrans, transVect);
			Compose_Transforms(comp->transform, &tempTrans);
			if(comp->pigment!=NULL)
				Transform_Tpattern((TPATTERN *)comp->pigment, & tempTrans);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Vector (transVect);
			Compute_Scaling_Transform(&tempTrans, transVect);
			Compose_Transforms(comp->transform, &tempTrans);
			if(comp->pigment!=NULL)
				Transform_Tpattern((TPATTERN *)comp->pigment, & tempTrans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			#ifndef TransformPatch /* Chris Huff april 2000 */
				GET(TRANSFORM_ID_TOKEN)
				Compose_Transforms(comp->transform, (TRANSFORM *)Token.Data);
				if(comp->pigment!=NULL)
					Transform_Tpattern((TPATTERN *)comp->pigment, (TRANSFORM *)Token.Data);
			#else
			{
				TRANSFORM * Trans = Parse_Transform();
				Compose_Transforms(comp->transform, Trans);	
				if(comp->pigment!=NULL)
				Transform_Tpattern((TPATTERN *)comp->pigment, Trans);
				/*YS sept 17 2000 Memory leak*/
				POV_FREE(Trans);
			}
		#endif
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix (tempMatrix);
			Compute_Matrix_Transform(&tempTrans, tempMatrix);
			Compose_Transforms(comp->transform, &tempTrans);
			if(comp->pigment!=NULL)
				Transform_Tpattern((TPATTERN *)comp->pigment, & tempTrans);
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT
}

#endif

