/****************************************************************************
*                df3file.c
*
*  This module contains the code to write df3 format density files.
*
*  from Persistence of Vision(tm) Ray Tracer
*  Copyright 1996,1998 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file.
*  If POVLEGAL.DOC is not available or for more info please contact the POV-Ray
*  Team Coordinator by leaving a message in CompuServe's GO POVRAY Forum or visit
*  http://www.povray.org. The latest version of POV-Ray may be found at these sites.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
* By Phil Carrig (PoD)
* Used the file targa.c as a framework for this one. 
*
*****************************************************************************/

/****************************************************************************
*
*  Explanation:
*
*  Write a frame of an animation sequence into an existing density file.
*  If this is the first frame of a complete sequence, create the file and 
*    write the df3 header and a complete file full of zeros.
*  If the dimensions of the existing file differ from the supplied parameters,
*    (width,height,number of frames) then create new file as above.
*
*
*****************************************************************************/

#include "frame.h"
#include "povproto.h"
#include "povray.h"
#include "df3file.h"



/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

#define boolean  int



/*****************************************************************************
* Local typedefs
******************************************************************************/


/*****************************************************************************
* Local variables
******************************************************************************/
/*YS added these variables */
static long Framesize;
static long Height;
static long Width;


/*****************************************************************************
* Static functions
******************************************************************************/

static int Open_DF3_File (FILE_HANDLE *handle, char *name, int *width, int *height, int buffer_size, int mode);
static void Write_DF3_Line (FILE_HANDLE *handle, COLOUR *line_data, int line_number);
static int Read_DF3_Line (FILE_HANDLE *handle, COLOUR *line_data, int *line_number);
static void Close_DF3_File (FILE_HANDLE *handle);
static void Write_DF3_Pixel (FILE_HANDLE *handle, DBL b, DBL g, DBL r);
static void Read_DF3_Image(IMAGE *Image, char *name);



/*****************************************************************************
*
* FUNCTION
*
*   Get_DF3_File_Handle
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POD
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

FILE_HANDLE *Get_DF3_File_Handle()
{
  FILE_HANDLE *handle;

  handle = (FILE_HANDLE *)POV_MALLOC(sizeof(FILE_HANDLE), "DF3 file handle");

  handle->Open_File_p         = Open_DF3_File;
  handle->Write_Line_p        = Write_DF3_Line;
  handle->Read_Line_p         = Read_DF3_Line;
  handle->Read_Image_p        = Read_DF3_Image;
  handle->Close_File_p        = Close_DF3_File;

  handle->file = NULL;
  handle->buffer = NULL;
  handle->buffer_size = 0;

  return(handle);
}


/*****************************************************************************
*
* FUNCTION
*
*   Open_DF3_File
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   PoD
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*
******************************************************************************/

static int Open_DF3_File (FILE_HANDLE *handle, char *name, int *width, int *height, int buffer_size, int mode)
{
    char *linebuff;
    int line,frames,currframe;
    char Mode[4] = "";
    boolean newfile;
    int w,h,d;
    unsigned char header[6];
	Height=(long)*height;	
    Width=(long)*width;
    handle->mode = mode;
    handle->filename = name;
    
    frames = opts.FrameSeq.ActualFinalFrame - opts.FrameSeq.ActualInitialFrame + 1;
    Framesize = (long)*width * (long)*height;
    
    newfile = 1;
    strcpy( Mode, "wb");
	
    if( !(opts.Options & TO_STDOUT) && (handle->file = fopen (name, "rb")) != NULL)
    {
        if (fread(header, 6, 1, handle->file) == 1)
	{
	    w = (header[0] << 8) + header[1];
	    h = (header[2] << 8) + header[3];
	    d = (header[4] << 8) + header[5];
		
	    if( w == *width && h == *height && d == frames && opts.FrameSeq.FrameNumber != opts.FrameSeq.InitialFrame)
	    {
		newfile = 0;
		strcpy(Mode,"rb+");
	    }
	}
	fclose( handle->file );
    }
    if (opts.Options & TO_STDOUT)
    {
	buffer_size = 0;
	handle->file = stdout;
    }
    else if ((handle->file = fopen (name, Mode)) == NULL)
    {
	return(0);
    }
    
    if (buffer_size != 0)
    {
	handle->buffer = (char *)POV_MALLOC((size_t)buffer_size, "DF3 file buffer");
	setvbuf (handle->file, handle->buffer, _IOFBF, buffer_size);
    }

    handle->width = *width;
    handle->height = *height;
    if( newfile )
    {
		/* Output df3 file header info */
		if( opts.FrameSeq.FrameNumber == opts.FrameSeq.InitialFrame )
		{
		    putc(*width / 256, handle->file);
		    putc(*width % 256, handle->file);
		    putc(*height / 256, handle->file);
		    putc(*height % 256, handle->file);
		    putc(frames / 256, handle->file);
		    putc(frames % 256, handle->file);
		}
	
		/* write out whole file */
		if(!(opts.Options & TO_STDOUT))
		{
			linebuff = (char *)POV_MALLOC((size_t)*width, "DF3 line buffer");
			memset( linebuff,0,*width);
			for( currframe = 0; currframe < frames; currframe++ )
			{
				for( line = 0; line < *height; line++ )
					fwrite( linebuff, (size_t)*width, 1, handle->file);
			}
			POV_FREE( linebuff );
		}
	}

    if(!(opts.Options & TO_STDOUT))
	{
		currframe = opts.FrameSeq.FrameNumber-opts.FrameSeq.ActualInitialFrame;
		fseek( handle->file, 6L+(long)Framesize*currframe, SEEK_SET );
    }
    handle->buffer_size = buffer_size;
    
    Debug_Info( "ActualInitFrame = %d\n", opts.FrameSeq.ActualInitialFrame);
    Debug_Info( "InitFrame = %d\n", opts.FrameSeq.InitialFrame);
    Debug_Info( "SubsetStartFrame = %d\n", opts.FrameSeq.SubsetStartFrame);
    Debug_Info( "SubsetEndFrame = %d\n", opts.FrameSeq.SubsetEndFrame);
    Debug_Info( "ActualFinalFrame = %d\n", opts.FrameSeq.ActualFinalFrame);
    Debug_Info( "FinalFrame = %d\n", opts.FrameSeq.FinalFrame);
    Debug_Info( "Frames = %d\n", frames);
    
    return(1);
}



/*****************************************************************************
*
* FUNCTION
*
*   Write_DF3_Pixel
*
* INPUT
*
*   handle     - Current file handel
*   r, g, b    - Color values to write
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   PoD
*   
* DESCRIPTION   :
*
*
* CHANGES
*
*
******************************************************************************/

static void Write_DF3_Pixel (FILE_HANDLE *handle, DBL  b, DBL  g, DBL r)
{
    unsigned int grey;

    grey = ((0.30 * r) + (0.59 * g) + (0.11 * b)) * 255;

    if ((putc(grey , handle->file) == EOF))
    {
	Error("Error writing DF3 output data to %s.\n",handle->filename);
    }
}


/*****************************************************************************
*
* FUNCTION
*
*   Write_DF3_Line
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   PoD
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*
******************************************************************************/

static void Write_DF3_Line (FILE_HANDLE *handle, COLOUR *line_data, int line_number)
{
   register int x;
	/*YS changed things so that each frame is written with line 0 at the bottom
		Have to do this because the density file is used that way.
		df3 goes from 0 to 1 (bottom to top) so write the file that way too*/

	long AddLines=(Height-(long)line_number)*Width;	
	
    if(!(opts.Options & TO_STDOUT))
	{
		long currframe = opts.FrameSeq.FrameNumber-opts.FrameSeq.ActualInitialFrame;
		fseek( handle->file, (6L+(long)Framesize*currframe)+AddLines, SEEK_SET );
    }
    for (x = 0; x < handle->width; x++)
    {
	Write_DF3_Pixel (handle, line_data[x][BLUE], line_data[x][GREEN], line_data[x][RED]);
    }
}



/*****************************************************************************
*
* FUNCTION
*
*   Read_DF3_Line
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   PoD
*   
* DESCRIPTION
*
*   Hmm.. don't know about reading df3 images.
*
* CHANGES
*
*
******************************************************************************/

static int Read_DF3_Line (FILE_HANDLE *handle, COLOUR *line_data, int *line_number)
{

  return(-1);
}



/*****************************************************************************
*
* FUNCTION
*
*   Close_DF3_File
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   PoD
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Close_DF3_File (FILE_HANDLE *handle)
{
  if (handle->file)
  {
    fflush(handle->file);

    if (!(opts.Options & TO_STDOUT))
      fclose (handle->file);
  }

  if (handle->buffer != NULL)
  {
    POV_FREE (handle->buffer);
  }

  handle->file = NULL;
  handle->buffer = NULL;
}



/*****************************************************************************
*
* FUNCTION
*
*   Read_DF3_Image
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   PoD
*   
* DESCRIPTION
*
*   Don't know about this yet, returns error.
*
* CHANGES
*
*
******************************************************************************/

void Read_DF3_Image(IMAGE *Image, char *name)
{
    Error ("Can't read DF3 image.\n");
    return;
}
