/****************************************************************************
*                   isoblob.h
*
*  This module contains all defines, typedefs, and prototypes for ISOBLOB.C.
*
*  By Lummox JR, July 1999
*
*  Almost all of the functions and structures were based on blob
*  functions by Dieter Bayer and Alexander Enzmann, and some on
*  isosurface functions created by R. Suzuki, D. Skarda, and T. Bily.
*  Almost all have been modified somewhat from their original versions.
*
*****************************************************************************/


#ifndef ISOBLOB_H
#define ISOBLOB_H

#include "bsphere.h"

#ifndef ISOSURFACE_OBJECT
#include "isosrf.h"
#endif


/*****************************************************************************
* Global preprocessor defines
******************************************************************************/

#define ISOBLOB_OBJECT (HIERARCHY_OK_OBJECT)

/* Bounding object types */

#define ISOBLOB_SPHERE            2
#define ISOBLOB_CYLINDER          4

/* Define max. number of isoblob components. */

#define MAX_ISOBLOB_COMPONENTS 1000000

/* Generate additional isoblob statistics. */

#define ISOBLOB_EXTRA_STATS 1



/*****************************************************************************
* Global typedefs
******************************************************************************/

typedef struct Isoblob_Struct ISOBLOB;
typedef struct Isoblob_Element_Struct ISOBLOB_ELEMENT;
typedef struct Isoblob_Data_Struct ISOBLOB_DATA;
typedef struct Isoblob_List_Struct ISOBLOB_LIST;
typedef struct Isoblob_Interval_Struct ISOBLOB_INTERVAL;
typedef struct Isoblob_Function_List_Struct ISOBLOB_FUNC_LIST;

struct Isoblob_Element_Struct
{
  short Type;       /* Type of bounding object                         */
  int index;
  DBL len;          /* Cylinder's length or box length along Z axis    */
  DBL rad2;         /* Sphere's/Cylinder's radius^2                    */
  VECTOR BC;        /* Bounding object center                          */
  DBL str;          /* Strength setting                                */
  FUNCTION *Func;   /* Density function                                */
  TEXTURE *Texture; /* Component's texture                             */
  TRANSFORM *Trans; /* Component's transformation                      */
  VECTOR PP;        /* Temporary storage for ray origin, transformed   */
  VECTOR DD;        /* Temporary storage for ray direction, transformed */
  ISOBLOB_ELEMENT *nextelement; /* Pointer to next element to evaluate  */
};

struct Isoblob_Data_Struct
{
  int References;               /* Number of references     */
  int Number_Of_Components;     /* Number of components     */
  DBL Threshold;                /* Isoblob threshold        */
  DBL accuracy;                 /* Isoblob function accuracy    */
  int max_trace;                /* Isoblob function maximum number of traces */
  char normal_type;             /* Isoblob function normal method */
  ISOBLOB_ELEMENT *Entry;       /* Array of isoblob components  */
  ISOBLOB_INTERVAL* Intervals;  /* Intervals used during intersection testing */
  BSPHERE_TREE *Tree;           /* Bounding hierarchy       */
  ISOBLOB_FUNC_LIST *firstfunc; /* First function in list   */
};

struct Isoblob_Struct
{
  OBJECT_FIELDS
  TRANSFORM *Trans;
  ISOBLOB_DATA *Data;       /* Pointer to isoblob data  */
  TEXTURE **Element_Texture;
};

struct Isoblob_List_Struct
{
  ISOBLOB_ELEMENT elem;     /* Current element          */
  ISOBLOB_LIST *next;       /* Pointer to next element  */
};

struct Isoblob_Interval_Struct
{
  int type;
  DBL bound;
  ISOBLOB_ELEMENT *Element;
  VECTOR normal;
};

/* Allows functions to be listed and then referred to by number */
struct Isoblob_Function_List_Struct
{
  FUNCTION *func;           /* Current function          */
  ISOBLOB_FUNC_LIST *next;  /* Pointer to next function  */
};



/*****************************************************************************
* Global variables
******************************************************************************/

extern METHODS Isoblob_Methods;



/*****************************************************************************
* Global functions
******************************************************************************/

void Init_Isoblob_Queue (void);
ISOBLOB *Create_Isoblob (void);
void Make_Isoblob (ISOBLOB *isoblob, DBL threshold, DBL accuracy, int max_trace, char normal_type, ISOBLOB_FUNC_LIST *isoblobfunclist, ISOBLOB_LIST *isobloblist, int npoints);
ISOBLOB_LIST *Create_Isoblob_List_Element (void);
ISOBLOB_FUNC_LIST *Create_Isoblob_Func_Element (void);
void Create_Isoblob_Element_Texture_List (ISOBLOB *Isoblob, ISOBLOB_LIST *IsoblobList, int npoints);
void Determine_Isoblob_Textures (ISOBLOB *Isoblob, VECTOR P, int *Count, TEXTURE **Textures, DBL *Weights);
void Test_Isoblob_Opacity (ISOBLOB *Isoblob);

void Translate_Isoblob_Element (ISOBLOB_ELEMENT *Element, VECTOR Vector);
void Rotate_Isoblob_Element (ISOBLOB_ELEMENT *Element, VECTOR Vector);
void Scale_Isoblob_Element (ISOBLOB_ELEMENT *Element, VECTOR Vector);
void Invert_Isoblob_Element (ISOBLOB_ELEMENT *Element);
void Transform_Isoblob_Element (ISOBLOB_ELEMENT *Element, TRANSFORM *Trans);
void Destroy_Isoblob_Queue (void);

#endif