/****************************************************************************
*                f_func.c
*
*  This module implements the isosurface shapetype.
*  This module was written by D.Skarda&T.Bily and modified by R.Suzuki 
*  for POV3.0.
*
*  Further modifications made by Lummox JR, June and July 1999:
*
*    - new functions added:
*	   ceil(a), floor(a), pi, radians(a), degrees(a), if(a,b,c),
*      atan2(a,b), clock
*    - int_func3d() completely re-written
*    - norm_ functions written for function-normal calculations
*    - evaluate_normal() added, evaluate_interval() changed
*
*****************************************************************************/
#include "frame.h"
#include "mem.h"
#include "isosrf.h"
#include "f_expr.h"
#include "texture.h"
#ifdef IsoPigmentPatch
#include "pigment.h" /* Added MCB 12-28-98 */
#endif
/* R. Suzuki  '96 */
#ifdef IsoBlobPatch
#include "povray.h"  /* Added by Lummox JR, June 1999, to support "clock" */
#include "vector.h"  /* Added by Lummox JR, July 1999, for int_func3d() */
#endif

#ifdef POVISO


#ifndef PI
#define PI  3.14159265359
#endif
static DBL ftemp_sign;
FUNCTION *First_Func= NULL, *Last_Func = NULL;
/*--------------------*/

static DBL ftemp1, ftemp2, ftemp3, ftemp4, ftemp5, ftemp6;
VECTOR fvectemp;
static int fint;


#define NOISE3D_TEST_SQR    8.0
#define NOISE3D_LIPSCH      ( 3.25 / 2.0 )

#ifdef IsoBlobPatch
/* Function-normal modifications by Lummox JR, July 1999 */
/*
 *  Description of function-normal calculation:
 *
 *  To find the normal for any 3D function, it is necessary to find
 *  its partial derivatives for x, y, and z. Knowing the value of
 *  any expression and of its derivative, it is possible to derive
 *  a more complex expression. A simple example is x*y, where:
 *
 *  d(x*y) = y*dx + x*dy
 *
 *  This patch extends the concept to 8 variables, including r, s,
 *  t, u, and v. The mathematics are based on fairly simple
 *  calculus rules.
 *
 *  Exceptions: Some functions use the old isosurface method of
 *  close approximation to find a normal. noise3d() is fudged
 *  because its normal is not known. Any pre-defined functions such
 *  as Sphere, Helix, etc. are also calculated this way, to avoid
 *  having to re-write every single one.
 *
 *  Note that this method can be slow because of the eight variables
 *  involved, yet in some cases it has a slight speed advantage,
 *  depending on the type of shape. Using fudged functions such as
 *  pigment and Sphere make matters worse. In any case, however, this
 *  will deilver the most accurate normal possible.
 */
static ISOFUNC_NORMALVEC *vtemp1,*vtemp2,*vtemp3;
static DBL nvt0,nvt1,nvt2;
#define NORMAL_Clear(vec) { vtemp1=(vec); vtemp1->x=vtemp1->y=vtemp1->z=vtemp1->r=vtemp1->s=vtemp1->t=vtemp1->u=vtemp1->v=0.0; }
#define NORMAL_Assign(result,a) memcpy((result),(a),sizeof(ISOFUNC_NORMALVEC))
#define NORMAL_Add(result,a,b) { vtemp1=(result); vtemp2=(a); vtemp3=(b); \
  vtemp1->x=vtemp2->x+vtemp3->x; \
  vtemp1->y=vtemp2->y+vtemp3->y; \
  vtemp1->z=vtemp2->z+vtemp3->z; \
  vtemp1->r=vtemp2->r+vtemp3->r; \
  vtemp1->s=vtemp2->s+vtemp3->s; \
  vtemp1->t=vtemp2->t+vtemp3->t; \
  vtemp1->u=vtemp2->u+vtemp3->u; \
  vtemp1->v=vtemp2->v+vtemp3->v; }
#define NORMAL_AddEq(result,a) { vtemp1=(result); vtemp2=(a); \
  vtemp1->x+=vtemp2->x; \
  vtemp1->y+=vtemp2->y; \
  vtemp1->z+=vtemp2->z; \
  vtemp1->r+=vtemp2->r; \
  vtemp1->s+=vtemp2->s; \
  vtemp1->t+=vtemp2->t; \
  vtemp1->u+=vtemp2->u; \
  vtemp1->v+=vtemp2->v; }
#define NORMAL_SubEq(result,a) { vtemp1=(result); vtemp2=(a); \
  vtemp1->x-=vtemp2->x; \
  vtemp1->y-=vtemp2->y; \
  vtemp1->z-=vtemp2->z; \
  vtemp1->r-=vtemp2->r; \
  vtemp1->s-=vtemp2->s; \
  vtemp1->t-=vtemp2->t; \
  vtemp1->u-=vtemp2->u; \
  vtemp1->v-=vtemp2->v; }
#define NORMAL_Scale(result,a,b) { vtemp1=(result); vtemp2=(a); nvt0=(b); \
  vtemp1->x=vtemp2->x*nvt0; \
  vtemp1->y=vtemp2->y*nvt0; \
  vtemp1->z=vtemp2->z*nvt0; \
  vtemp1->r=vtemp2->r*nvt0; \
  vtemp1->s=vtemp2->s*nvt0; \
  vtemp1->t=vtemp2->t*nvt0; \
  vtemp1->u=vtemp2->u*nvt0; \
  vtemp1->v=vtemp2->v*nvt0; }
#define NORMAL_ScaleEq(result,a) { vtemp1=(result); nvt0=(a); \
  vtemp1->x*=nvt0; \
  vtemp1->y*=nvt0; \
  vtemp1->z*=nvt0; \
  vtemp1->r*=nvt0; \
  vtemp1->s*=nvt0; \
  vtemp1->t*=nvt0; \
  vtemp1->u*=nvt0; \
  vtemp1->v*=nvt0; }
#define NORMAL_AddScaledEq(result,a,b) { vtemp1=(result); vtemp2=(a); nvt0=(b); \
  vtemp1->x+=vtemp2->x*nvt0; \
  vtemp1->y+=vtemp2->y*nvt0; \
  vtemp1->z+=vtemp2->z*nvt0; \
  vtemp1->r+=vtemp2->r*nvt0; \
  vtemp1->s+=vtemp2->s*nvt0; \
  vtemp1->t+=vtemp2->t*nvt0; \
  vtemp1->u+=vtemp2->u*nvt0; \
  vtemp1->v+=vtemp2->v*nvt0; }
#define NORMAL_Combine(result,a,s1,b,s2) { vtemp1=(result); vtemp2=(a); vtemp3=(b); nvt0=(s1); nvt1=(s2); \
  vtemp1->x=vtemp2->x*nvt0+vtemp3->x*nvt1; \
  vtemp1->y=vtemp2->y*nvt0+vtemp3->y*nvt1; \
  vtemp1->z=vtemp2->z*nvt0+vtemp3->z*nvt1; \
  vtemp1->r=vtemp2->r*nvt0+vtemp3->r*nvt1; \
  vtemp1->s=vtemp2->s*nvt0+vtemp3->s*nvt1; \
  vtemp1->t=vtemp2->t*nvt0+vtemp3->t*nvt1; \
  vtemp1->u=vtemp2->u*nvt0+vtemp3->u*nvt1; \
  vtemp1->v=vtemp2->v*nvt0+vtemp3->v*nvt1; }
#define NORMAL_CombineScale(result,s0,a,s1,b,s2) { vtemp1=(result); vtemp2=(a); vtemp3=(b); nvt2=(s0); nvt0=(s1); nvt1=(s2); \
  vtemp1->x=(vtemp2->x*nvt0+vtemp3->x*nvt1)*nvt2; \
  vtemp1->y=(vtemp2->y*nvt0+vtemp3->y*nvt1)*nvt2; \
  vtemp1->z=(vtemp2->z*nvt0+vtemp3->z*nvt1)*nvt2; \
  vtemp1->r=(vtemp2->r*nvt0+vtemp3->r*nvt1)*nvt2; \
  vtemp1->s=(vtemp2->s*nvt0+vtemp3->s*nvt1)*nvt2; \
  vtemp1->t=(vtemp2->t*nvt0+vtemp3->t*nvt1)*nvt2; \
  vtemp1->u=(vtemp2->u*nvt0+vtemp3->u*nvt1)*nvt2; \
  vtemp1->v=(vtemp2->v*nvt0+vtemp3->v*nvt1)*nvt2; }

ISOFUNC_NORMALVEC norm_stack_d[FUNC_MAX_STACK_LEN];
ISOFUNC_NORMALVEC *norm_stack=norm_stack_d;
ISOFUNC_NORMALVEC *norm_stack_top;
DBL norm_smallnudge;
/* End of Lummox JR's normal-calculation modifications */
#endif
                         /* stack and variables for computing func. values */

DBL    func_U, func_V, func_R, func_S, func_T;
VECTOR func_XYZ;  /* for implicit functions                         */
DBL calc_stack_d[FUNC_MAX_STACK_LEN];
DBL *calc_stack=calc_stack_d;
DBL *calc_stack_top;   /* pointer to top of the stack               */

FUNCTION Imp_Func;


                  /* ... and for interval arithmetix                */

DBL func_U_hi, func_V_hi, func_R_hi, func_S_hi, func_T_hi;
DBL func_U_low, func_V_low, func_R_low, func_S_low, func_T_low;
VECTOR func_XYZ_hi, func_XYZ_low;


DBL cs_int_hi[FUNC_MAX_STACK_LEN], cs_int_low[FUNC_MAX_STACK_LEN];        
DBL *cs_hi_top, *cs_low_top;
#ifdef IsoBlobPatch
/* Added by Lummox JR, July 1999, to support int_func3d() */
DBL *cs_hi_stack=cs_int_hi,*cs_low_stack=cs_int_low;
/* End Lummox JR's addition */
#endif

char *cur_fun_queue;
DBL  *cur_num_queue;



void func_x(void)
{
  *(++calc_stack_top)= func_XYZ[X];
}

void func_y(void)
{
  *(++calc_stack_top)= func_XYZ[Y];
}

void func_z(void)
{
  *(++calc_stack_top)= func_XYZ[Z];
}

void func_r(void)
{
  *(++calc_stack_top)= func_R;
}

void func_s(void)
{
  *(++calc_stack_top)= func_S;
}

void func_t(void)
{
  *(++calc_stack_top)= func_T;
}

void func_u(void)
{
  *(++calc_stack_top)= func_U;
}

void func_v(void)
{
  *(++calc_stack_top)= func_V;
}

void func_const(void)
{
  *(++calc_stack_top)= *(cur_num_queue++);
}

void func_uminus(void)
{
  *(calc_stack_top)= - *(calc_stack_top);
}

void func_sin(void)
{
  *(calc_stack_top)= SIN( *calc_stack_top);
}

void func_asin(void)
{
  *(calc_stack_top)= ASIN( *calc_stack_top);
}

void func_cos(void)
{
  *(calc_stack_top)= COS( *calc_stack_top);
}

void func_acos(void)
{
  *(calc_stack_top)= ACOS( *calc_stack_top);
}

void func_tan(void)
{
  *(calc_stack_top)= TAN( *calc_stack_top);
}

void func_atan(void)
{
  *(calc_stack_top)= ATAN( *calc_stack_top);
}

void func_sinh(void)
{
  *(calc_stack_top)= SINH( *calc_stack_top);
}

void func_asinh(void)
{
#ifndef _WIN32
  *(calc_stack_top)= ASINH( *calc_stack_top);   
#endif
}

void func_cosh(void)
{
  *(calc_stack_top)= COSH( *calc_stack_top);
}

void func_acosh(void)
{
#ifndef _WIN32
  *(calc_stack_top)= ACOSH( *calc_stack_top);
#endif
}

void func_tanh(void)
{
  *(calc_stack_top)= TANH( *calc_stack_top);
}

void func_atanh(void)
{
#ifndef _WIN32
  *(calc_stack_top)= ATANH( *calc_stack_top);
#endif
}

void func_ln(void)
{
  *(calc_stack_top)= LOG( ABS( *calc_stack_top ));
}

void func_exp(void)
{
  *(calc_stack_top)= EXP( *calc_stack_top);
}

void func_exp_xy(void)
{
  if (((*calc_stack_top)>0) && ((*calc_stack_top)<16) &&
    (FFLOOR(*calc_stack_top)== *calc_stack_top))
  {                    
    fint= *(calc_stack_top--);
    ftemp1= (*calc_stack_top);
    for (; fint>1; fint--) 
      (*calc_stack_top)*= ftemp1;
  }
  else
  {
    calc_stack_top--;
    *(calc_stack_top)= pow( ABS(*(calc_stack_top)),*(calc_stack_top+1));
  }
}

void func_sqrt(void)
{
  *(calc_stack_top)= SQRT( ABS( *calc_stack_top ));
}

void func_sqr(void)
{
  *(calc_stack_top)*= (*calc_stack_top);
}

void func_cub(void)
{
  *(calc_stack_top)*= (*calc_stack_top)*(*calc_stack_top);
}

void func_noise3d(void)
{
  fvectemp[Z]= *(calc_stack_top--);
  fvectemp[Y]= *(calc_stack_top--);
  fvectemp[X]= *(calc_stack_top);
  *calc_stack_top= Noise(fvectemp);
}

void func_func3d(void)
{
  FUNCTION *func;
  DBL   *temp_calc_stack_top;
  DBL   *temp_calc_stack;
  char  *temp_cur_fun_queue;
  DBL   *temp_cur_num_queue;
  DBL   temp_ftemp_sign;
  DBL   temp_result;
  VECTOR f3dV;
  temp_cur_num_queue= cur_num_queue;
  temp_cur_fun_queue= cur_fun_queue;
  temp_ftemp_sign =  ftemp_sign;

  Assign_Vector(f3dV,func_XYZ);
  func_XYZ[Z]= *(calc_stack_top--);
  func_XYZ[Y]= *(calc_stack_top--);
  func_XYZ[X]= *(calc_stack_top--);
  memcpy( &func, calc_stack_top, sizeof(char *));
  temp_calc_stack_top= calc_stack_top;  
  temp_calc_stack= calc_stack;  
  calc_stack=calc_stack_top;

  evaluate_function(func);

  temp_result= *(calc_stack);
  calc_stack= temp_calc_stack;  

  Assign_Vector(func_XYZ,f3dV);
  calc_stack_top=temp_calc_stack_top;
  cur_num_queue= temp_cur_num_queue;
  cur_fun_queue= temp_cur_fun_queue;
  ftemp_sign =  temp_ftemp_sign;
  *(calc_stack_top) = temp_result;
}

void func_abs(void)
{
  *(calc_stack_top)= ABS( *calc_stack_top);
}

/* Fri 09-27-1996 0. */
void func_b_min(void)
{
  calc_stack_top--;
  if (*calc_stack_top > *(calc_stack_top+1))  
    *calc_stack_top = *(calc_stack_top+1);
}

void func_b_max(void)
{
  calc_stack_top--;
  if (*calc_stack_top < *(calc_stack_top+1))  
      *calc_stack_top = *(calc_stack_top+1);
}
/* 0 */

/* R.S. July 8, 96  */
void func_b_or(void)
{
  calc_stack_top--;
  if (ftemp_sign>0.) 
    *(calc_stack_top)= min(*(calc_stack_top), *(calc_stack_top+1));
  else
    *(calc_stack_top)= max(*(calc_stack_top), *(calc_stack_top+1));
}

void func_b_and(void)
{
  calc_stack_top--;
  if (ftemp_sign>0.) 
    *(calc_stack_top)= max(*(calc_stack_top), *(calc_stack_top+1));
  else
    *(calc_stack_top)= min(*(calc_stack_top), *(calc_stack_top+1));
}

void func_b_mod(void)
{
  calc_stack_top--;
  *(calc_stack_top) -= ((int) (*(calc_stack_top)/ *(calc_stack_top+1)))* *(calc_stack_top+1);
}
/*--------------*/

void func_b_plus(void)
{
  calc_stack_top--;
  *(calc_stack_top)+= *(calc_stack_top+1);
}

void func_b_minus(void)
{
  calc_stack_top--;
  *(calc_stack_top)-= *(calc_stack_top+1);
}

void func_b_mult(void)
{
  calc_stack_top--;
  *(calc_stack_top)*= *(calc_stack_top+1);
}

void func_b_div(void)
{
  calc_stack_top--;
  *(calc_stack_top)/= *(calc_stack_top+1);
}

#ifdef IsoBlobPatch
/* Functions added by Lummox JR, June 1999 */

void func_ceil(void)
{
  *(calc_stack_top)= ceil(*calc_stack_top);
}

void func_floor(void)
{
  *(calc_stack_top)= floor(*calc_stack_top);
}

void func_pi(void)
{
  *(++calc_stack_top)= PI;
}

void func_radians(void)
{
  *(calc_stack_top)= (*calc_stack_top)*PI/180.0;
}

void func_degrees(void)
{
  *(calc_stack_top)= (*calc_stack_top)*180.0/PI;
}

/*
if(arg1,arg2,arg3)
If arg1 > 0, return arg2
Else return arg3

Example:
have value 1 everywhere inside the sphere, 0.5 outside
When x^2+y^2+z^2-25>0, point is outside the sphere
if(x^2+y^2+z^2-25,0.5,1)
*/
void func_if(void)
{
  calc_stack_top--;
  calc_stack_top--;
  *(calc_stack_top)=*(((*calc_stack_top)>0.0)?(calc_stack_top+1):(calc_stack_top+2));
}

void func_atan2(void)
{
  calc_stack_top--;
  *(calc_stack_top)=
  (fabs(*(calc_stack_top))>EPSILON || fabs(*(calc_stack_top+1))>EPSILON)?
  atan2(*(calc_stack_top),*(calc_stack_top+1)):0.0;
}

void func_clock(void)
{
  *(++calc_stack_top)= opts.FrameSeq.Clock_Value;
}

/* End Lummox JR's additions */
#endif

/* same rules as for func_*, but these         */
/*      compute with interval arithmetic       */

void int_x(void)
{
  *(++cs_hi_top)= func_XYZ_hi[X];
  *(++cs_low_top)= func_XYZ_low[X];
}

void int_y(void)
{
  *(++cs_hi_top)= func_XYZ_hi[Y];
  *(++cs_low_top)= func_XYZ_low[Y];
}

void int_z(void)
{
  *(++cs_hi_top)= func_XYZ_hi[Z];
  *(++cs_low_top)= func_XYZ_low[Z];
}

void int_r(void)
{
  *(++cs_hi_top)= func_R_hi;
  *(++cs_low_top)= func_R_low;
}

void int_s(void)
{
  *(++cs_hi_top)= func_S_hi;
  *(++cs_low_top)= func_S_low;
}

void int_t(void)
{
  *(++cs_hi_top)= func_T_hi;
  *(++cs_low_top)= func_T_low;
}

void int_u(void)
{
  *(++cs_hi_top)= func_U_hi;
  *(++cs_low_top)= func_U_low;
}

void int_v(void)
{
  *(++cs_hi_top)= func_V_hi;
  *(++cs_low_top)= func_V_low;
}

void int_const(void)
{
  *(++cs_hi_top)= *(cur_num_queue);
  *(++cs_low_top)= *(cur_num_queue);
  cur_num_queue++;
}


void int_uminus(void)
{
  ftemp1= *cs_hi_top;
  *cs_hi_top= -*cs_low_top;
  *cs_low_top= -ftemp1;
}

void int_sin(void)
{
  /* Fri 09-27-1996 0.  - mhmmm ...

  *cs_hi_top+=  PI/2.0;
  *cs_low_top+= PI/2.0;
  */
  *cs_hi_top-=  PI/2.0;
  *cs_low_top-= PI/2.0;
  int_cos();
}

void int_asin(void)
{
}


void int_cos(void)
{
  ftemp1= FCEIL( (*cs_low_top)/PI );
  ftemp2= (*cs_hi_top) / PI;

  if (ftemp1 > ftemp2)
  {
    if ( (ftemp3= COS(*cs_hi_top)) >= (ftemp4= COS(*cs_low_top)) )
    { 
      *cs_hi_top= ftemp3; 
      *cs_low_top= ftemp4;
     }
    else
    {
      *cs_hi_top= ftemp4; 
      *cs_low_top= ftemp3; 
    }
  }
  else
    if (ftemp1+1.0<=ftemp2)
    { 
      *cs_hi_top=1.0; 
      *cs_low_top= -1.0; 
    }
    else   /* ftemp1 < ftemp2 */
    {
      if ( (ftemp2= ftemp3= COS(*cs_hi_top)) < (ftemp4= COS(*cs_low_top)) )
      { 
        ftemp3= ftemp4;  
        ftemp4= ftemp2; 
      }
      if ((((int)ftemp1) % 2) == 0)
      { 
        *cs_low_top= ftemp4; 
        *cs_hi_top=1.0;
      } 
      else 
      {
        *cs_low_top= -1.0; 
        *cs_hi_top= ftemp3; 
      }
  }
}

void int_acos(void)
{
}

void int_tan(void)
{
}

void int_atan(void)
{
  *cs_hi_top= ATAN(*cs_hi_top);
  *cs_low_top= ATAN(*cs_low_top);
}

void int_sinh(void)
{
  *cs_hi_top= SINH( *cs_hi_top );
  *cs_low_top= SINH( *cs_low_top );
}


void int_asinh(void)
{
  *cs_hi_top= SINH( *cs_hi_top );
  *cs_low_top= SINH( *cs_low_top );
}

void int_cosh(void)
{
  int_abs();
  *cs_hi_top= COSH( *cs_hi_top );
  *cs_low_top= COSH( *cs_low_top );
}

void int_acosh(void)
{
}

void int_tanh(void)
{
  *cs_hi_top= TANH( *cs_hi_top );
  *cs_low_top= TANH( *cs_low_top );
}

void int_atanh(void)
{
}

void int_ln(void)
{
  int_abs();
  if ((*cs_low_top)==0.0) 
    *cs_low_top= - 1000;
  else 
    *cs_low_top= log(*cs_low_top);
  if ((*cs_hi_top)==0.0) 
    *cs_hi_top= - 1000;
  else 
    *cs_hi_top= log(*cs_hi_top);
}

void int_exp(void)
{
  *cs_low_top= exp(*cs_low_top);
  *cs_hi_top=  exp(*cs_hi_top);
}

void int_exp_xy(void)
{
  if (((*cs_low_top)>0) && ((*cs_low_top)<16) &&  
  (FFLOOR(*cs_low_top)== *cs_low_top) &&((*cs_low_top)==(*cs_hi_top)))
  {      
    fint= (int) *(cs_hi_top--);
    cs_low_top--;

    if ((fint & 1) == 0) 
      int_abs();
    ftemp1= (*cs_hi_top);
    ftemp2= (*cs_low_top);
    for (; fint>1; fint--)
    {
      (*cs_hi_top)*= ftemp1;
      (*cs_low_top)*= ftemp2;
    }
  }
  else
  {
    ftemp5= (*cs_low_top--);
    ftemp6= (*cs_hi_top--);
    int_abs();

    ftemp1= pow( (*cs_hi_top),ftemp5);
    ftemp2= pow( (*cs_low_top),ftemp5);
    ftemp3= pow( (*cs_hi_top),ftemp6);
    ftemp4= pow( (*cs_low_top),ftemp6);

    if (ftemp1 > ftemp3)
    { 
      *cs_hi_top= ftemp1; 
      *cs_low_top= ftemp3; 
    }
    else
    { 
      *cs_hi_top= ftemp3; 
      *cs_low_top= ftemp1;
    }
    if (ftemp2 > ftemp4)
    {
      if (ftemp2 > (*cs_hi_top)) 
        *cs_hi_top= ftemp2;
      if (ftemp4 < (*cs_low_top)) 
        *cs_low_top= ftemp4;
    }
    else 
    {
      if (ftemp4 > (*cs_hi_top)) 
        *cs_hi_top= ftemp4;
      if (ftemp2 < (*cs_low_top)) 
        *cs_low_top= ftemp2;
    }
  }
}

void int_sqrt(void)
{
  int_abs();
  *cs_low_top= SQRT( *cs_low_top );
  *cs_hi_top= SQRT( *cs_hi_top );
}

void int_sqr(void)
{
  ftemp2= *cs_low_top;

  if ((ftemp1= *cs_hi_top) < 0)
  {
    *cs_hi_top= ftemp2*ftemp2;
    *cs_low_top= ftemp1*ftemp1;
  }
  else
  {
    if (ftemp2 > 0)
    {
      *cs_hi_top*= ftemp1;
      *cs_low_top*= ftemp2;
    }
    else
    {
      *cs_low_top= 0;
      ftemp1*= ftemp1;
      ftemp2*= ftemp2;
      *cs_hi_top=  (ftemp1>ftemp2)?ftemp1:ftemp2;
    }
  }
}

void int_cub(void)
{
  *cs_hi_top*= (*cs_hi_top)*(*cs_hi_top);
  *cs_low_top*= (*cs_low_top)*(*cs_low_top);
}

void int_noise3d(void)
{
  DBL ftemp1, ftemp2, ftemp3, ftemp4;
  VECTOR fvectemp;

  fvectemp[Z]= ((ftemp1= *(cs_low_top--))+(ftemp2= *(cs_hi_top--))) / 2.0;
  ftemp4= ftemp2-ftemp1;
  ftemp3= ftemp4*ftemp4;    

  fvectemp[Y]= ((ftemp1= *(cs_low_top--))+(ftemp2= *(cs_hi_top--))) / 2.0;
  ftemp4= ftemp2-ftemp1;
  ftemp3+= ftemp4*ftemp4;    

  fvectemp[X]= ((ftemp1= *(cs_low_top))+(ftemp2= *(cs_hi_top))) / 2.0;
  ftemp4= ftemp2-ftemp1;
  ftemp3+= ftemp4*ftemp4;    

  *cs_low_top=0.0;
  *cs_hi_top=1.0;

  if (ftemp3 >= NOISE3D_TEST_SQR) 
    return;

  ftemp4= Noise(fvectemp);
  ftemp3= NOISE3D_LIPSCH * SQRT(ftemp3) / 2.0;

  if ((ftemp1= ftemp4 + ftemp3) < 1.0) 
    *cs_hi_top= ftemp1;
  if ((ftemp1= ftemp4 - ftemp3) > 0.0) 
    *cs_low_top= ftemp1;
}


void int_func3d(void)
{
#ifdef IsoBlobPatch
  /*
  *  This function has been re-written by Lummox JR to actually
  *  work correctly. The old code below was buggy and wrong, and
  *  how it survived as long as it did I'll never know.
  */

  FUNCTION *func;
  DBL   *temp_cs_hi_top;
  DBL   *temp_cs_int_hi;
  DBL   *temp_cs_low_top;
  DBL   *temp_cs_int_low;
  char  *temp_cur_fun_queue;
  DBL   *temp_cur_num_queue;
  DBL   temp_ftemp_sign;
  DBL   temp_result_low,temp_result_hi;
  DBL   len;
  VECTOR f3dV1,f3dV2,lvec;
  temp_cur_num_queue= cur_num_queue;
  temp_cur_fun_queue= cur_fun_queue;
  temp_ftemp_sign =  ftemp_sign;

  Assign_Vector(f3dV1,func_XYZ_low);
  Assign_Vector(f3dV2,func_XYZ_hi);
  func_XYZ_low[Z]= *(cs_low_top--);
  func_XYZ_low[Y]= *(cs_low_top--);
  func_XYZ_low[X]= *(cs_low_top--);
  func_XYZ_hi[Z]= *(cs_hi_top--);
  func_XYZ_hi[Y]= *(cs_hi_top--);
  func_XYZ_hi[X]= *(cs_hi_top--);
  memcpy( &func, cs_low_top, sizeof(char *));
  temp_cs_hi_top= cs_hi_top;
  temp_cs_int_hi= cs_int_hi;
  temp_cs_low_top= cs_low_top;
  temp_cs_int_low= cs_int_low;
  cs_low_stack=cs_low_top;
  cs_hi_stack=cs_hi_top;

  /* Length shouldn't matter much here, but just in case... */
  VSub(lvec,func_XYZ_hi,func_XYZ_low);
  VLength(len,lvec);
  evaluate_interval(func,len);

  temp_result_low= *(cs_low_stack);
  temp_result_hi = *(cs_hi_stack);
  cs_hi_stack = temp_cs_int_hi;
  cs_low_stack= temp_cs_int_low;

  Assign_Vector(func_XYZ_low,f3dV1);
  Assign_Vector(func_XYZ_hi,f3dV2);
  cs_hi_top=temp_cs_hi_top;
  cs_low_top=temp_cs_low_top;
  cur_num_queue= temp_cur_num_queue;
  cur_fun_queue= temp_cur_fun_queue;
  ftemp_sign =  temp_ftemp_sign;
  *(cs_low_top) = temp_result_low;
  *(cs_hi_top)  = temp_result_hi;
#else
  cs_hi_top--;
  cs_low_top--;
  cs_hi_top--;
  cs_low_top--;
#endif

}

void int_abs(void)
{
  if ((ftemp1= *cs_hi_top) < 0)
  {
    *cs_hi_top= ABS( *cs_low_top );
    *cs_low_top= ABS(ftemp1);
  }
  else  if ((ftemp2= *cs_low_top) < 0)       /* and *cs_hi_top > 0 */
  {
    /* if ftemp2 > 0  => [a,b]>0 - no op. required */
    *cs_low_top= 0;
    ftemp1= ABS(ftemp1);
    ftemp2= ABS(ftemp2);
    *cs_hi_top=  (ftemp1>ftemp2)?ftemp1:ftemp2;
  }
}

/* Fri 09-27-1996 0. */
void int_b_min(void)
{
  cs_hi_top--;
  if (*cs_hi_top > *(cs_hi_top+1))  
    *cs_hi_top = *(cs_hi_top+1);
  cs_low_top--;
  if (*cs_low_top > *(cs_low_top+1))  
    *cs_low_top = *(cs_low_top+1);
}

void int_b_max(void)
{
  cs_hi_top--;
  if (*cs_hi_top < *(cs_hi_top+1))  
    *cs_hi_top = *(cs_hi_top+1);
  cs_low_top--;
  if (*cs_low_top < *(cs_low_top+1))  
    *cs_low_top = *(cs_low_top+1);
}
/* 0 */

/*--   R.S.   July 8, 96  --*/
void int_b_or(void)
{
  cs_hi_top--;
  cs_low_top--;
  if (ftemp_sign>0)
  {
    *(cs_hi_top)=min(*(cs_hi_top), *(cs_hi_top+1));
    *(cs_low_top) = min(*(cs_low_top), *(cs_low_top+1));
  }
  else
  {
    *(cs_hi_top)=min(*(cs_hi_top), *(cs_hi_top+1));
    *(cs_low_top) = min(*(cs_low_top), *(cs_low_top+1));
  }
}

void int_b_and(void)
{
  cs_hi_top--;
  cs_low_top--;
  if (ftemp_sign>0)
  {
    *(cs_hi_top)=max(*(cs_hi_top), *(cs_hi_top+1));
    *(cs_low_top) = max(*(cs_low_top), *(cs_low_top+1));
  }
  else
  {
    *(cs_hi_top)=min(*(cs_hi_top), *(cs_hi_top+1));
    *(cs_low_top) = min(*(cs_low_top), *(cs_low_top+1));
  }
}

void int_b_mod(void)
{
  if (((ftemp1= *cs_hi_top) * (ftemp2= *cs_low_top)) <=0 )
  {                                                /* [a,b] includes 0 */
    *(--cs_hi_top)= HUGE_VAL;
    *(--cs_low_top)= -HUGE_VAL;
    return;
  }
  *cs_hi_top=  1.0 / ftemp2;
  *cs_low_top= 1.0 / ftemp1;
  int_b_mult();
}

/*-------*/


void int_b_plus(void)
{
  cs_hi_top--;
  *(cs_hi_top)+= *(cs_hi_top+1);
  cs_low_top--;
  *(cs_low_top)+= *(cs_low_top+1);
}

void int_b_minus(void)
{
  cs_hi_top--;
  *(cs_hi_top)-= *(cs_low_top);
  cs_low_top--;
  *(cs_low_top)-= *(cs_hi_top+1);
}

void int_b_mult(void)      /* [a,b]*[c,d]=[min(ac,ad,bc,bd),max(ac,ad,bc,bd)] */
{
  ftemp1=ftemp2= *(cs_hi_top--);
  ftemp3=ftemp4= *(cs_low_top--);

  ftemp2*= (*cs_low_top); 
  ftemp4*= (*cs_low_top); 
  if ((ftemp1*= *cs_hi_top) > (ftemp3*= *cs_hi_top))
  { 
    *cs_hi_top= ftemp1; 
    *cs_low_top= ftemp3; 
  }
  else
  { 
    *cs_hi_top= ftemp3; 
    *cs_low_top= ftemp1; 
  }
  if (ftemp2 > ftemp4)
  {
    if (ftemp2 > (*cs_hi_top)) 
      *cs_hi_top= ftemp2;
    if (ftemp4 < (*cs_low_top)) 
      *cs_low_top= ftemp4;
  }
  else 
  {
    if (ftemp4 > (*cs_hi_top)) 
      *cs_hi_top= ftemp4;
    if (ftemp2 < (*cs_low_top)) 
      *cs_low_top= ftemp2;
  }
}

void int_b_div(void)
{
  if (((ftemp1= *cs_hi_top) * (ftemp2= *cs_low_top)) <=0 )
  {                                                /* [a,b] includes 0 */
    *(--cs_hi_top)= HUGE_VAL;
    *(--cs_low_top)= -HUGE_VAL;
    return;
  }
  *cs_hi_top=  1.0 / ftemp2;
  *cs_low_top= 1.0 / ftemp1;
  int_b_mult();
}

#ifdef IsoBlobPatch
/* Functions added by Lummox JR, June 1999 */

void int_ceil(void)
{
  *(cs_hi_top)= ceil(*cs_hi_top);
  *(cs_low_top)=ceil(*cs_low_top);
}

void int_floor(void)
{
  *(cs_hi_top)= floor(*cs_hi_top);
  *(cs_low_top)=floor(*cs_low_top);
}

void int_pi(void)
{
  *(++cs_hi_top)= PI;
  *(++cs_low_top)=PI;
}

void int_radians(void)
{
  *(cs_hi_top)= (*cs_hi_top)*PI/180.0;
  *(cs_low_top)=(*cs_low_top)*PI/180.0;
}

void int_degrees(void)
{
  *(cs_hi_top)= (*cs_hi_top)*180.0/PI;
  *(cs_low_top)=(*cs_low_top)*180.0/PI;
}

/*
if(arg1,arg2,arg3)
If arg1 > 0, return arg2
Else return arg3
*/
void int_if(void)
{
  cs_hi_top--;
  cs_hi_top--;
  cs_low_top--;
  cs_low_top--;
  /* if() is always true if the interval >=0 */
  if(*cs_low_top>=0.0) 
  {
    *(cs_hi_top)= *(cs_hi_top+1);
    *(cs_low_top)=*(cs_low_top+1);
  }
  /* if() is always false if the interval <0 */
  else if(*cs_hi_top<0.0) 
  {
    *(cs_hi_top)= *(cs_hi_top+2);
    *(cs_low_top)=*(cs_low_top+2);
  }
  /* Mixed case: Interval crosses the if() condition boundary */
  else 
  {
    *(cs_hi_top)= max(*(cs_hi_top+1),*(cs_hi_top+2));
    *(cs_low_top)=min(*(cs_low_top+1),*(cs_low_top+2));
  }
}

/* atan2([a,b],[c,d])
If atan2() is continuous along [a,b],[c,d]:
atan2 = [min(atan2(a,c),atan2(a,d),atan2(b,c),atan2(b,d)),max(...)]
Otherwise:
atan2 = full range of atan2()--either [0,2*pi) or (-pi,pi]
*/
void int_atan2(void)
{
  --cs_hi_top;
  --cs_low_top;
  /* because atan2() may be [0,2*pi) or (-pi,pi], check for either case */
  /* case 1: [a,b] and [c,d] don't cross the boundary where atan2() repeats */
  if((*(cs_low_top+1))>=0.0 || (*(cs_hi_top+1))<=0.0 ||
    ((ftemp5=atan2(0,-1))<0.0 && (*(cs_low_top))>0.0) || (ftemp5>0.0 && (*(cs_hi_top))<0.0))
  {
    /* find min and max of four possible extreme atan2() values */
    ftemp1=
    (fabs(*(cs_low_top))>EPSILON || fabs(*(cs_low_top+1))>EPSILON)?
    atan2(*(cs_low_top),*(cs_low_top+1)):0.0;
    ftemp2=
    (fabs(*(cs_hi_top))>EPSILON || fabs(*(cs_hi_top+1))>EPSILON)?
    atan2(*(cs_hi_top),*(cs_hi_top+1)):0.0;
    ftemp3=
    (fabs(*(cs_hi_top))>EPSILON || fabs(*(cs_low_top+1))>EPSILON)?
    atan2(*(cs_hi_top),*(cs_low_top+1)):0.0;
    ftemp4=
    (fabs(*(cs_low_top))>EPSILON || fabs(*(cs_hi_top+1))>EPSILON)?
    atan2(*(cs_low_top),*(cs_hi_top+1)):0.0;
    /* a quick bubble sort */
    if(ftemp1>ftemp2)
    {
      ftemp5=ftemp1; 
      ftemp1=ftemp2; 
      ftemp2=ftemp5;
    }
    if(ftemp2>ftemp3) 
    {
      ftemp5=ftemp2; 
      ftemp2=ftemp3;
      ftemp3=ftemp5;
    }
    if(ftemp3>ftemp4)
    {
      ftemp5=ftemp3;
      ftemp3=ftemp4;
      ftemp4=ftemp5;
    }
    if(ftemp2>ftemp3) 
    {
      ftemp5=ftemp2; 
      ftemp2=ftemp3; 
      ftemp3=ftemp5;
    }
    if(ftemp1>ftemp2) 
    {
      ftemp5=ftemp1; 
      ftemp1=ftemp2; 
      ftemp2=ftemp5;
    }
    *cs_low_top=ftemp1; 
    *cs_hi_top=ftemp4;
  }
  /* case 2: atan2([a,b],[c,d]) is not continuous, range of atan2() = [0,2*pi) */
  else if(ftemp5>0.0) 
  {
    *cs_low_top=0.0; 
    *cs_hi_top=2.0*PI;
  }
  /* case 3: atan2([a,b],[c,d]) is not continuous, range of atan2() = (-pi,pi] */
  else 
  {
    *cs_low_top=-PI; 
    *cs_hi_top=PI;
  }
}

void int_clock(void)
{
  *(++cs_hi_top)= opts.FrameSeq.Clock_Value;
  *(++cs_low_top)= opts.FrameSeq.Clock_Value;
}

/* End Lummox JR's additions */


/* Normal calculation functions, added by Lummox JR, July 1999 */

void norm_x(void)
{
  NORMAL_Clear(++norm_stack_top);
  norm_stack_top->x=1.0;
  *(++calc_stack_top)= func_XYZ[X];
}

void norm_y(void)
{
  NORMAL_Clear(++norm_stack_top);
  norm_stack_top->y=1.0;
  *(++calc_stack_top)= func_XYZ[Y];
}

void norm_z(void)
{
  NORMAL_Clear(++norm_stack_top);
  norm_stack_top->z=1.0;
  *(++calc_stack_top)= func_XYZ[Z];
}

void norm_r(void)
{
  NORMAL_Clear(++norm_stack_top);
  norm_stack_top->r=1.0;
  *(++calc_stack_top)= func_R;
}

void norm_s(void)
{
  NORMAL_Clear(++norm_stack_top);
  norm_stack_top->s=1.0;
  *(++calc_stack_top)= func_S;
}

void norm_t(void)
{
  NORMAL_Clear(++norm_stack_top);
  norm_stack_top->t=1.0;
  *(++calc_stack_top)= func_T;
}

void norm_u(void)
{
  NORMAL_Clear(++norm_stack_top);
  norm_stack_top->u=1.0;
  *(++calc_stack_top)= func_U;
}

void norm_v(void)
{
  NORMAL_Clear(++norm_stack_top);
  norm_stack_top->v=1.0;
  *(++calc_stack_top)= func_V;
}

void norm_const(void)
{
  NORMAL_Clear(++norm_stack_top);
  *(++calc_stack_top)= *(cur_num_queue++);
}

void norm_uminus(void)
{
  NORMAL_ScaleEq(norm_stack_top,-1);
  *(calc_stack_top)= - *(calc_stack_top);
}

void norm_sin(void)
{
  NORMAL_ScaleEq(norm_stack_top,COS(*calc_stack_top));
  *(calc_stack_top)= SIN( *calc_stack_top);
}

void norm_asin(void)
{
  NORMAL_ScaleEq(norm_stack_top,1.0/SQRT(ABS(1-(*calc_stack_top)*(*calc_stack_top))));
  *(calc_stack_top)= ASIN( *calc_stack_top);
}

void norm_cos(void)
{
  NORMAL_ScaleEq(norm_stack_top,-SIN(*calc_stack_top));
  *(calc_stack_top)= COS( *calc_stack_top);
}

void norm_acos(void)
{
  NORMAL_ScaleEq(norm_stack_top,-1.0/SQRT(ABS(1-(*calc_stack_top)*(*calc_stack_top))));
  *(calc_stack_top)= ACOS( *calc_stack_top);
}

void norm_tan(void)
{
  NORMAL_ScaleEq(norm_stack_top,1.0/(COS(*calc_stack_top)*COS(*calc_stack_top)));
  *(calc_stack_top)= TAN( *calc_stack_top);
}

void norm_atan(void)
{
  NORMAL_ScaleEq(norm_stack_top,1.0/(1.0+(*calc_stack_top)*(*calc_stack_top)));
  *(calc_stack_top)= ATAN( *calc_stack_top);
}

void norm_sinh(void)
{
  NORMAL_ScaleEq(norm_stack_top,COSH(*calc_stack_top));
  *(calc_stack_top)= SINH( *calc_stack_top);
}

void norm_asinh(void)
{
  NORMAL_ScaleEq(norm_stack_top,1.0/SQRT(1+(*calc_stack_top)*(*calc_stack_top)));
  #ifndef _WIN32
  *(calc_stack_top)= ASINH( *calc_stack_top);
  #endif
}

void norm_cosh(void)
{
  NORMAL_ScaleEq(norm_stack_top,SINH(*calc_stack_top));
  *(calc_stack_top)= COSH( *calc_stack_top);
}

void norm_acosh(void)
{
  NORMAL_ScaleEq(norm_stack_top,1.0/SQRT(ABS((*calc_stack_top)*(*calc_stack_top)-1)));
  #ifndef _WIN32
  *(calc_stack_top)= ACOSH( *calc_stack_top);
  #endif
}

void norm_tanh(void)
{
  NORMAL_ScaleEq(norm_stack_top,1.0/(COSH(*calc_stack_top)*COSH(*calc_stack_top)));
  *(calc_stack_top)= TANH( *calc_stack_top);
}

void norm_atanh(void)
{
  NORMAL_ScaleEq(norm_stack_top,1.0/(1-(*calc_stack_top)*(*calc_stack_top)));
  #ifndef _WIN32
  *(calc_stack_top)= ATANH( *calc_stack_top);
  #endif
}

void norm_ln(void)
{
  NORMAL_ScaleEq(norm_stack_top,1.0/(*calc_stack_top));
  *(calc_stack_top)= LOG( ABS( *calc_stack_top ));
}

void norm_exp(void)
{
  *(calc_stack_top)= EXP( *calc_stack_top);
  NORMAL_ScaleEq(norm_stack_top,(*calc_stack_top));
}

void norm_exp_xy(void)
{
  ftemp5=*(calc_stack_top-1);
  ftemp6=*(calc_stack_top);
  if (((*calc_stack_top)>0) && ((*calc_stack_top)<16) &&
    (FFLOOR(*calc_stack_top)== *calc_stack_top))
  {
    fint= *(calc_stack_top--);
    ftemp1= (*calc_stack_top);
    for (; fint>1; fint--) 
      (*calc_stack_top)*= ftemp1;
  }
  else
  {
    calc_stack_top--;
    /* *(calc_stack_top)= pow( ABS(*(calc_stack_top)),*(calc_stack_top+1)); */ /* -hdf- 27-May-1999 */
    *(calc_stack_top)= pow( *(calc_stack_top),*(calc_stack_top+1));
  }
  norm_stack_top--;
  if(fabs(ftemp5)>=EPSILON) 
  {
    NORMAL_CombineScale(norm_stack_top,*calc_stack_top,norm_stack_top,ftemp6/ftemp5,norm_stack_top+1,LOG(ABS(ftemp5)));
  }
  else 
  {
    NORMAL_Clear(norm_stack_top);
  }
}

void norm_sqrt(void)
{
  /* Because this is really sqrt(abs(x)), adjust normal */
  if((*calc_stack_top)<0) 
  {
    NORMAL_ScaleEq(norm_stack_top,-1.0); *calc_stack_top=-(*calc_stack_top);
  }
  *(calc_stack_top)= SQRT( *calc_stack_top );
  NORMAL_ScaleEq(norm_stack_top,0.5/(*calc_stack_top));
}

void norm_sqr(void)
{
  NORMAL_ScaleEq(norm_stack_top,2.0*(*calc_stack_top));
  *(calc_stack_top)*= (*calc_stack_top);
}

void norm_cub(void)
{
  ftemp1=(*calc_stack_top)*(*calc_stack_top);
  NORMAL_ScaleEq(norm_stack_top,3.0*ftemp1);
  *(calc_stack_top)*=ftemp1;
}

/* The normal of the Noise() function is unknown, so fudge it */
void norm_noise3d(void)
{
  norm_stack_top--;
  norm_stack_top--;
  fvectemp[Z]= *(calc_stack_top--);
  fvectemp[Y]= *(calc_stack_top--);
  fvectemp[X]= *(calc_stack_top);
  *calc_stack_top= Noise(fvectemp);
  fvectemp[X]+=norm_smallnudge;
  ftemp1= Noise(fvectemp);
  fvectemp[X]-=norm_smallnudge*2;
  ftemp2= Noise(fvectemp);
  NORMAL_ScaleEq(norm_stack_top,(ftemp1-ftemp2)/(norm_smallnudge*2));
  fvectemp[X]+=norm_smallnudge;
  fvectemp[Y]+=norm_smallnudge;
  ftemp1= Noise(fvectemp);
  fvectemp[Y]-=norm_smallnudge*2;
  ftemp2= Noise(fvectemp);
  NORMAL_AddScaledEq(norm_stack_top,norm_stack_top+1,(ftemp1-ftemp2)/(norm_smallnudge*2));
  fvectemp[Y]+=norm_smallnudge;
  fvectemp[Z]+=norm_smallnudge;
  ftemp1= Noise(fvectemp);
  fvectemp[Z]-=norm_smallnudge*2;
  ftemp2= Noise(fvectemp);
  NORMAL_AddScaledEq(norm_stack_top,norm_stack_top+2,(ftemp1-ftemp2)/(norm_smallnudge*2));
  /*NORMAL_ScaleEq(norm_stack_top,norm_smallnudge); */
}

void norm_func3d(void)
{
  FUNCTION *func;
  DBL   *temp_calc_stack_top;
  DBL   *temp_calc_stack;
  ISOFUNC_NORMALVEC *temp_norm_stack_top;
  ISOFUNC_NORMALVEC *temp_norm_stack;
  char  *temp_cur_fun_queue;
  DBL   *temp_cur_num_queue;
  DBL   temp_ftemp_sign;
  DBL   temp_result;
  ISOFUNC_NORMALVEC temp_normal;
  ISOFUNC_NORMALVEC nx,ny,nz;
  VECTOR f3dV;
  temp_cur_num_queue= cur_num_queue;
  temp_cur_fun_queue= cur_fun_queue;
  temp_ftemp_sign =  ftemp_sign;

  Assign_Vector(f3dV,func_XYZ);
  func_XYZ[Z]= *(calc_stack_top--); NORMAL_Assign(&nz,norm_stack_top--);
  func_XYZ[Y]= *(calc_stack_top--); NORMAL_Assign(&ny,norm_stack_top--);
  func_XYZ[X]= *(calc_stack_top--); NORMAL_Assign(&nx,norm_stack_top--);
  memcpy( &func, calc_stack_top, sizeof(char *));
  temp_calc_stack_top= calc_stack_top;
  temp_calc_stack= calc_stack;
  temp_norm_stack_top= norm_stack_top;
  temp_norm_stack= norm_stack;
  calc_stack=calc_stack_top;
  norm_stack=norm_stack_top;

  evaluate_normal(func,norm_smallnudge);

  temp_result= *(calc_stack);
  NORMAL_Assign(&temp_normal,norm_stack);
  calc_stack= temp_calc_stack;
  norm_stack= temp_norm_stack;

  Assign_Vector(func_XYZ,f3dV);
  calc_stack_top=temp_calc_stack_top;
  norm_stack_top=temp_norm_stack_top;
  cur_num_queue= temp_cur_num_queue;
  cur_fun_queue= temp_cur_fun_queue;
  ftemp_sign =  temp_ftemp_sign;
  NORMAL_Scale(norm_stack_top,&nx,temp_normal.x);
  NORMAL_AddScaledEq(norm_stack_top,&ny,temp_normal.y);
  NORMAL_AddScaledEq(norm_stack_top,&nz,temp_normal.z);
  norm_stack_top->r+=temp_normal.r;
  norm_stack_top->s+=temp_normal.s;
  norm_stack_top->t+=temp_normal.t;
  norm_stack_top->u+=temp_normal.u;
  norm_stack_top->v+=temp_normal.v;
  *(calc_stack_top) = temp_result;
}

void norm_abs(void)
{
  if((*calc_stack_top)<0.0) 
    NORMAL_ScaleEq(norm_stack_top,-1.0);
  *(calc_stack_top)= ABS( *calc_stack_top);
}

void norm_b_min(void)
{
  norm_stack_top--;
  calc_stack_top--;
  if (*calc_stack_top > *(calc_stack_top+1))
  {
    *calc_stack_top = *(calc_stack_top+1);
    NORMAL_Assign(norm_stack_top,norm_stack_top+1);
  }
}

void norm_b_max(void)
{
  norm_stack_top--;
  calc_stack_top--;
  if (*calc_stack_top < *(calc_stack_top+1))
  {
    *calc_stack_top = *(calc_stack_top+1);
    NORMAL_Assign(norm_stack_top,norm_stack_top+1);
  }
}

void norm_b_or(void)
{
  if (ftemp_sign>0.)
    func_b_min();
  else
    func_b_max();
}

void norm_b_and(void)
{
  if (ftemp_sign>0.)
    func_b_max();
  else
    func_b_min();
}

void norm_b_mod(void)
{
  calc_stack_top--;
  norm_stack_top--;
  NORMAL_CombineScale(norm_stack_top,1.0/((*(calc_stack_top+1))*(*(calc_stack_top+1))),norm_stack_top,*(calc_stack_top+1),norm_stack_top+1,-(*calc_stack_top));
  *(calc_stack_top) -= ((int) (*(calc_stack_top)/ *(calc_stack_top+1)))* *(calc_stack_top+1);
}

void norm_b_plus(void)
{
  calc_stack_top--;
  norm_stack_top--;
  NORMAL_AddEq(norm_stack_top,norm_stack_top+1);
  *(calc_stack_top)+= *(calc_stack_top+1);
}

void norm_b_minus(void)
{
  calc_stack_top--;
  norm_stack_top--;
  NORMAL_SubEq(norm_stack_top,norm_stack_top+1);
  *(calc_stack_top)-= *(calc_stack_top+1);
}

void norm_b_mult(void)
{
  calc_stack_top--;
  norm_stack_top--;
  NORMAL_Combine(norm_stack_top,norm_stack_top,*(calc_stack_top+1),norm_stack_top+1,*calc_stack_top);
  *(calc_stack_top)*= *(calc_stack_top+1);
}

void norm_b_div(void)
{
  calc_stack_top--;
  norm_stack_top--;
  NORMAL_CombineScale(norm_stack_top,1.0/((*(calc_stack_top+1))*(*(calc_stack_top+1))),norm_stack_top,*(calc_stack_top+1),norm_stack_top+1,-(*calc_stack_top));
  *(calc_stack_top)/= *(calc_stack_top+1);
}

void norm_ceil(void)
{
  *(calc_stack_top)= ceil(*calc_stack_top);
  NORMAL_Clear(norm_stack_top);
}

void norm_floor(void)
{
  *(calc_stack_top)= floor(*calc_stack_top);
  NORMAL_Clear(norm_stack_top);
}

void norm_pi(void)
{
  *(++calc_stack_top)= PI;
  NORMAL_Clear(++norm_stack_top);
}

void norm_radians(void)
{
  *(calc_stack_top)= (*calc_stack_top)*PI/180.0;
  NORMAL_ScaleEq(norm_stack_top,PI/180.0);
}

void norm_degrees(void)
{
  *(calc_stack_top)= (*calc_stack_top)*180.0/PI;
  NORMAL_ScaleEq(norm_stack_top,180.0/PI);
}

void norm_if(void)
{
  norm_stack_top--;
  norm_stack_top--;
  calc_stack_top--;
  calc_stack_top--;
  if((*calc_stack_top)>0.0)
  {
    *(calc_stack_top)=*(calc_stack_top+1);
    NORMAL_Assign(norm_stack_top,norm_stack_top+1);
  }
  else
  {
    *(calc_stack_top)=*(calc_stack_top+2);
    NORMAL_Assign(norm_stack_top,norm_stack_top+2);
  }
}

/*
*  norm_atan2()
*
*  This function absolutely does not work as it should. It is useless
*  for all practical purposes, but included for completeness. The
*  normal appears to be completely wrong in certain spots, for reasons
*  completely unknown. Try this isosurface with the function-normal
*  patch as an example:
*
*  sqrt(sqr(sqrt(sqr(x)+sqr(z))-1.75)+sqr(y))
*    -(0.1+sqrt(sin(atan2(x,z)*5))*0.3)
*
*  The result should be something like a string of beads (rocks, if
*  you add noise), but there are bizarre "gaps" in the ring regardless
*  of whether method 1 or 2 is used. Only the standard means of
*  finding an isosurface normal (i.e., not using the norm_ funcs at
*  all) works. The bug is definitely in this routine, but it cannot be
*  isolated; the mathematics, equating d(atan2(x,y)) to d(atan(y/x)),
*  are entirely correct. (Replacing atan2(x,z) with atan(z/x), or
*  acos(x/sqrt(sqr(x)+sqr(z))), etc. in the example function solves
*  the problem; however, atan(z/x) leaves an unsightly seam at x=0.)
*  Workarounds do not work to calculate the normal of atan2(). Trying
*  to divide parameter 2 by parameter 1, and then finding the
*  arctangent, does not work at all. Other attempts to make this
*  routine work by fudging the calculations have all failed miserably
*  and even more inexplicably.
*
*  Programmers with a good grasp of calculus and even a hint at how
*  the norm_ functions work are invited to try to fix this. At least
*  everything else works.              - Lummox JR, July 1999
*/
void norm_atan2(void)
{
  calc_stack_top--;
  norm_stack_top--;
  if(fabs(*(calc_stack_top))>EPSILON || fabs(*(calc_stack_top+1))>EPSILON)
  {
    NORMAL_CombineScale(norm_stack_top,1.0/((*calc_stack_top)*(*calc_stack_top)+(*(calc_stack_top+1))*(*(calc_stack_top+1))),norm_stack_top,-(*(calc_stack_top+1)),norm_stack_top+1,(*calc_stack_top));
    *(calc_stack_top)=atan2(*(calc_stack_top),*(calc_stack_top+1));
  }
  else
  {
    NORMAL_Clear(norm_stack_top);
    *(calc_stack_top)=0.0;
  }
}

void norm_clock(void)
{
  *(++calc_stack_top)= opts.FrameSeq.Clock_Value;
  NORMAL_Clear(++norm_stack_top);
}

/* End Lummox JR's normal calculation additions */
#endif


void evaluate_function( FUNCTION *func )         /* result is in *calc_stack_top      */
{
  int i;
  /* R.S.  July 9, 96 */
  VECTOR VT;
  /*  if ((func->isosf !=NULL)&&(func->iso_func !=NULL)) */
  if ((func->iso_func !=NULL)&&(func->iso_func!=Imp_Func.iso_func)) 
  { 
    Assign_Vector(VT,func_XYZ);
    *calc_stack=func->iso_func(func, VT)*func->sign; 
    return; 
  }
  /*---------*/
  calc_stack_top= calc_stack;
  calc_stack_top--;
  cur_num_queue= func-> number_stack;
  cur_fun_queue= func-> ops_stack;
  ftemp_sign = func->sign;

  for (i= func->opslen; i>0; i--)
  FuncInfo[ (int) *( cur_fun_queue++ ) ].function();
}

void evaluate_interval( FUNCTION *func, DBL length )  /* result is in *calc_stack_top      */
{
  int i;
  /* R.S.  July 9, 96 */
  DBL a1,a2;
  VECTOR VT;
  /*-----*/
  #ifdef IsoBlobPatch
  /* Change made by Lummox JR, July 1999 */
  /* cs_hi_top= cs_int_hi;
  cs_low_top= cs_int_low; */
  cs_hi_top= cs_hi_stack;
  cs_low_top= cs_low_stack;
  /* End Lummox JR's change */
  #else
  cs_hi_top= cs_int_hi;
  cs_low_top= cs_int_low;
  #endif
  ftemp_sign= func->sign;

  /* R.S.  
  if ((func->isosf !=NULL)&&(func->iso_func !=NULL)) */
  if ((func->iso_func !=NULL)&&(func->iso_func!=Imp_Func.iso_func)) 
  {
    Assign_Vector(VT,func_XYZ_low);    
    a1=func->iso_func(func, VT);
    Assign_Vector(VT,func_XYZ_hi);     
    a2=func->iso_func(func, VT);
    *cs_hi_top=(a1+a2+length*func->Max_gradient)*.5;
    *cs_low_top=(a1+a2-length*func->Max_gradient)*.5;
    return;
  }
  /*-----*/
  cs_hi_top--;
  cs_low_top--;
  cur_num_queue= func-> number_stack;
  cur_fun_queue= func-> ops_stack;

  for (i= func->opslen; i>0; i--)
    FuncInfo[ (int) *( cur_fun_queue++ ) ].interval();   
  /*R.S. July 96*/
  *cs_hi_top = (*cs_hi_top - func->threshold);
  *cs_low_top = (*cs_low_top - func->threshold);
}


#ifdef IsoBlobPatch
/* Added by Lummox JR, July 1999 */

void evaluate_normal( FUNCTION *func, DBL accuracy )
{
  int i;
  VECTOR VT;
  DBL t1,t2;
  norm_smallnudge=accuracy;
  Assign_Vector(VT,func_XYZ);
  if ((func->iso_func !=NULL)&&(func->iso_func!=Imp_Func.iso_func))
  {
    *calc_stack=func->iso_func(func, VT)*func->sign;
    /* Instead of adding normal calculation capability to all iso_func()'s,
    just fudge it */
    Assign_Vector(VT,func_XYZ);
    VT[X]+=norm_smallnudge; t1=func->iso_func(func, VT);
    Assign_Vector(VT,func_XYZ);
    VT[X]-=norm_smallnudge; t2=func->iso_func(func, VT);
    norm_stack->x=(t1-t2)*func->sign/(norm_smallnudge*2);
    Assign_Vector(VT,func_XYZ);
    VT[Y]+=norm_smallnudge; t1=func->iso_func(func, VT);
    Assign_Vector(VT,func_XYZ);
    VT[Y]-=norm_smallnudge; t2=func->iso_func(func, VT);
    norm_stack->y=(t1-t2)*func->sign/(norm_smallnudge*2);
    Assign_Vector(VT,func_XYZ);
    VT[Z]+=norm_smallnudge; t1=func->iso_func(func, VT);
    Assign_Vector(VT,func_XYZ);
    VT[Z]-=norm_smallnudge; t2=func->iso_func(func, VT);
    norm_stack->z=(t1-t2)*func->sign/(norm_smallnudge*2);
    norm_stack->r=
    norm_stack->s=
    norm_stack->t=
    norm_stack->u=
    norm_stack->v=0;
    return;
  }
  calc_stack_top= calc_stack;
  calc_stack_top--;
  norm_stack_top= norm_stack;
  norm_stack_top--;
  cur_num_queue= func-> number_stack;
  cur_fun_queue= func-> ops_stack;
  ftemp_sign = func->sign;

  for (i= func->opslen; i>0; i--)
  FuncInfo[ (int) *( cur_fun_queue++ ) ].normal();
}

/* End Lummox JR's additions */

#endif
void Destroy_Function( FUNCTION *Func )
{
  /* R.S. */
  if (Func->prev) 
    Func->prev->next=Func->next;
  else 
    First_Func=Func->next;
  if (Func->next) 
    Func->next->prev=Func->prev;
  else 
    Last_Func=Func->prev;

  POV_FREE(Func->func_name);
  if (Func->pnum >0) 
    POV_FREE (Func->parm);
  /*----------*/
  if (Func->opslen >0) 
    POV_FREE(Func->ops_stack);
  if (Func->numlen >0) 
    POV_FREE(Func->number_stack);
  #ifdef IsoPigmentPatch
  Destroy_Pigment(Func->Pigment);    /* Added MCB 12-28-98 */
  #endif
  if (Func->cache != NULL) 
    POV_FREE(Func->cache); /* Added MCB 01-06-00 */
  POV_FREE(Func);
}


/* RS July 22 */
void Destroy_All_Functions( )
{
  FUNCTION *TFunc, *NFunc;
  if (First_Func)
  {
    TFunc=First_Func;
    while (TFunc)
    {
      NFunc=TFunc->next;
      Destroy_Function (TFunc);
      TFunc=NFunc;
    }
  }
}
/*----*/



FUNCTION *Copy_Function( FUNCTION *Func )
{
  FUNCTION *loc_func;

  loc_func = (FUNCTION *) POV_MALLOC (sizeof(FUNCTION), "function");

  /* R.S. July '96 */
  *loc_func=*Func;
  loc_func->gradient = 0.;
  loc_func->prev = Last_Func;
  loc_func->next = NULL;
  Last_Func->next = loc_func;
  /*----*/

  /* R.S.   July 7, '96  */
  if (loc_func->opslen>0) 
    loc_func->ops_stack= (char *)POV_MALLOC((Func->opslen), "function-ops");
  if (loc_func->numlen>0) 
    loc_func->number_stack= (DBL *)POV_MALLOC(sizeof(DBL)*(Func->numlen), "function-number");
  loc_func->func_name = (char *) POV_MALLOC (strlen(Func->func_name)+1, "function-name");
  if (loc_func->pnum>0) 
    loc_func->parm = (DBL *) POV_MALLOC (sizeof(DBL)*(Func->pnum), "function-parm");

  memcpy( loc_func->func_name, Func->func_name, strlen (Func->func_name)+1);
  memcpy( loc_func->parm, Func->parm, sizeof(DBL)*(Func->pnum));
  /*------*/
  memcpy( loc_func->ops_stack, Func->ops_stack, Func->opslen );
  memcpy( loc_func->number_stack, Func->number_stack, 
                    sizeof(DBL) * Func->numlen );
  #ifdef IsoPigmentPatch
  loc_func->Pigment=Copy_Pigment(Func->Pigment); /* Added MCB 12-28-98 */
  #endif
  return (loc_func);
}

void Imp2XY (FUNCTION *Func)
{
  char *tmp;
  /* R.S.   July 14, '96  */
  tmp = (char *) POV_MALLOC ((Func->opslen)+2, "function-XY");
  /*------*/
  memcpy ( tmp, Func->ops_stack, Func->opslen);
  tmp[(Func->opslen)++]= FE_Z;
  tmp[(Func->opslen)++]= FE_B_MINUS;
  POV_FREE(Func->ops_stack);
  Func->ops_stack= tmp;
}


void Destroy_Library( LIBRARY *Lib )
{
  if (Lib->prev) 
    Lib->prev->next=Lib->next;
  else 
    First_Lib=Lib->next;
  if (Lib->next) 
    Lib->next->prev=Lib->prev;
  else 
    Last_Lib=Lib->prev;
  Finish_Lib(Lib);
  if (Lib->lib_name)    
    POV_FREE(Lib->lib_name);
  if (Lib->pnum >0 )
    POV_FREE(Lib->parm);
  POV_FREE(Lib);
}  

/* RS July 22 '96*/
void Destroy_All_Library( )
{
  LIBRARY *TLib, *NLib;
  if (First_Lib)
  {
    TLib=First_Lib;
    while (TLib)
    {
      NLib=TLib->next;
      Destroy_Library (TLib);
      TLib=NLib;
    }
  }
}

#else
	static char dmm[2];
#endif

