/*
 * ttf2tfm.c
 *
 * This file is part of the ttf2pk package.
 *
 * Copyright 1997-1998 by
 *    Frederic Loyer <loyer@ensta.fr>
 *    Werner Lemberg <a7971428@unet.univie.ac.at>
 */

/*
 *   This program converts TTF files to TeX TFM files, and optionally
 *   to TeX VPL files that retain all kerning and ligature information.
 *   Both files make the characters not normally encoded by TeX available
 *   by character codes greater than 0x7F.
 */

/*
 *   Adapted from afm2tfm by F. Loyer <loyer@ensta.fr>
 *   Following comments are from afm2tfm.
 */

/*   (Modified by Don Knuth from Tom Rokicki's pre-VPL version.) */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ttf2tfm.h"
#include "newobj.h"
#include "ttfenc.h"
#include "ligkern.h"
#include "texenc.h"
#include "ttfaux.h"
#include "tfmaux.h"
#include "vplaux.h"
#include "errormsg.h"
#include "filesrch.h"


#if (defined(MSDOS) && defined(__TURBOC__)) || \
    (defined(OS2) && defined(_MSC_VER))
#define SMALLMALLOC
#endif


char ident[] = "ttf2tfm version 1.0";
char progname[] = "ttf2tfm";            /* for error/warning messages */

int ignoreligkern;          /* do we look at ligkern info in the encoding? */
int sawligkern;

FILE *encin;                /* encoding file handles */

#define BUF_SIZE 255

char buffer[BUF_SIZE];          /* input buffer (modified while parsing) */
char obuffer[BUF_SIZE];         /* unmodified copy of input buffer */

char *cur_pos;                  /* current position in input buffer */

char smbuffer[100];             /* for tokens */

char makevpl;                   /* command line options */
char pedantic;
char quiet;
char forceoctal;


struct Font font;               /* the main structure */



static void
init_font_structure(struct Font *fnt)
{
  int i;


  fnt->boundarychar = -1;
  fnt->fontname = default_fontname;
  fnt->codingscheme = default_codingscheme;
  fnt->xheight = 400;
  fnt->efactor = 1.0;
  fnt->capheight = 0.8;
  fnt->pid = 3;
  fnt->eid = 1;

  for (i = 0; i < 256; i++)
    fnt->nextout[i] = -1;           /* encoding chains have length 0 */
}


/* getline() will truncate input lines after BUF_SIZE characters */

static int
getline(void)
{
  register char *p;
  register int c;
  register int count;


  cur_pos = buffer;
  for (p = buffer, count = 0;
       (c = getc(encin)) != EOF && c != '\n' && count < BUF_SIZE - 1; count++)
    *p++ = c;
  *p = '\0';

  (void)strcpy(obuffer, buffer);

  if (p == buffer && c == EOF)
    return 0;
  else
    return 1;
}


static struct encoding *readencoding(char *enc, struct Font *fnt);


/*
 *   Re-encode the TTF font.
 */

static void
handlereencoding(struct Font *fnt)
{
  int i;
  struct ttfinfo *ti;
  char *p;


  if (fnt->inencname)
  {
    ignoreligkern = 1;
    fnt->inencoding = readencoding(fnt->inencname, fnt);

    /* reset all pointers in the mapping table */
    for (i = 0; i <= 0xFF; i++)
      if (NULL != (ti = fnt->inencptrs[i]))
      {
        ti->incode = -1;
        fnt->inencptrs[i] = NULL;
      }

    /*
     *   reencode TTF <--> raw TeX.  Only these code points will be used
     *   for the output encoding.
     */

    for (i = 0; i <= 0xFF; i++)
    {
      p = fnt->inencoding->vec[i];
      if (p && *p)
        if((ti = findadobe(p, fnt->charlist)))
        {
          if (ti->incode >= 0)
            oops("Character `%s' encoded twice in input encoding\n"
                 "        (positions %x and %x)", p, ti->incode, i);
          if (ti->charcode >= 0)
          {
            ti->incode = i;
            fnt->inencptrs[i] = ti;
          }
        }
        else
          warning("Cannot find character `%s'\n"
                  "         specified in input encoding", p);
    }
    fnt->codingscheme = fnt->inencoding->name;
  }

  if (!quiet)
  {
    if (fnt->inencname)
      printf("\nUsing %s as input encoding\n", fnt->inencname);
    else
    {
      printf(
        "\nUsing the first 256 glyphs in the following input encoding:\n\n");
      for (i = 0; i <= 0xFF; i++)
      {
        if ((ti = fnt->inencptrs[i]))
          printf("  0x%02x    %s\n", i, ti->adobename);
      }
      printf("\n");
    }
  }

  ignoreligkern = 0;
  if (fnt->outencname)
    fnt->outencoding = readencoding(fnt->outencname, fnt);
  else
    fnt->outencoding = readencoding(NULL, fnt);
}


static void
assignchars(struct Font *fnt)
{
  register char **p;
  register int i, j;
  register struct ttfinfo *ti;
  int nextfree = 0x80;


  /*
   *   First, we assign all those that match perfectly.
   */

  for (i = 0, p = fnt->outencoding->vec; i <= 0xFF; i++, p++)
    if ((ti = findmappedadobe(*p, fnt->inencptrs)))
    {
      if (ti->outcode >= 0)
        fnt->nextout[i] = ti->outcode;       /* linked list */
      ti->outcode = i;
      fnt->outencptrs[i] = ti;
    }
    else if (strcmp(*p, ".notdef") != 0)
      warning("Cannot map character `%s'\n"
              "         specified in output encoding", *p);

  if (pedantic)
    goto end;

  /*
   *   Next, we assign all the others, retaining the TTF code positions,
   *   possibly multiplying assigning characters, unless the output encoding
   *   was precisely specified.
   */

  for (i = 0; i <= 0xFF; i++)
    if ((ti = fnt->inencptrs[i]) &&
         ti->charcode >= 0 && ti->charcode <= 0xFF &&
         ti->outcode < 0 && fnt->outencptrs[ti->charcode] == NULL)
    {
      ti->outcode = ti->charcode;
      fnt->outencptrs[ti->charcode] = ti;
    }

  /*
   *   Finally, we map all remaining characters into free locations beginning
   *   with 0x80.
   */

  for (i = 0; i <= 0xFF; i++)
    if ((ti = fnt->inencptrs[i]) && ti->outcode < 0)
    {
      while (fnt->outencptrs[nextfree])
      {
        nextfree = (nextfree + 1) & 0xFF;
        if (nextfree == 0x80)
          goto finishup;        /* all slots full */
      }
      ti->outcode = nextfree;
      fnt->outencptrs[nextfree] = ti;
    }

finishup:

  /*
   *   Now, if any of the characters are encoded multiple times, we want
   *   ti->outcode to be the first one assigned, since that is most likely
   *   to be the most important one.  So we reverse the above lists.
   */

  for (i = 0; i <= 0xFF; i++)
    if ((ti = fnt->inencptrs[i]) && ti->outcode >= 0)
    {
      j = -1;
      while (fnt->nextout[ti->outcode] >= 0)
      {
        i = fnt->nextout[ti->outcode];
        fnt->nextout[ti->outcode] = j;
        j = ti->outcode;
        ti->outcode = i;
      }
      fnt->nextout[ti->outcode] = j;
    }

end:

  if (!quiet)
  {
    printf("\nUsing the following output encoding:\n\n");
    for (i = 0; i <= 0xFF; i++)
    {
      if ((ti = fnt->outencptrs[i]))
        printf("  0x%02x    %s\n", i, ti->adobename);
    }
    printf("\n");
  }
}


#define VERSION "\
Copyright (C) 1997-1998 Frederic Loyer and Werner Lemberg.\n\
There is NO warranty.  You may redistribute this software\n\
under the terms of the GNU General Public License\n\
and the Dvips copyright.\n\
\n\
For more information about these matters, see the files\n\
named COPYING and ttf2tfm.c.\n\
\n\
Primary authors of ttf2tfm: F. Loyer and W. Lemberg.\n\
\n\
ttf2tfm is based on afm2tfm from T. Rokicki\n\
and the FreeType project from\n\
David Turner, Robert Wilhelm, and Werner Lemberg\n\
"

static void
version(void)
{
  fputs(ident, stdout);
  fprintf(stdout, " (%s)\n", TeX_search_version());
  fputs(VERSION, stdout);
  exit(0);
}


#define USAGE "\
  Convert a TrueType font table to TeX's font metric format.\n\
\n\
-c REAL             use REAL for height of small caps made with -V [0.8]\n\
-e REAL             widen (extend) characters by a factor of REAL [1.0]\n\
-E INT              select INT as the TTF encoding ID [1]\n\
-O                  use octal for all character codes in the vpl file\n\
-p ENCFILE          read ENCFILE for the TTF->raw TeX mapping\n\
-P INT              select INT as the TTF platform ID [3]\n\
-q                  suppress informational output\n\
-s REAL             oblique (slant) characters by REAL, usually <<1 [0.0]\n\
-t ENCFILE          read ENCFILE for the encoding of the vpl file\n\
-T ENCFILE          equivalent to -p ENCFILE -t ENCFILE\n\
-u                  output only characters from encodings, nothing extra\n\
-v FILE[.vpl]       make a VPL file for conversion to VF\n\
-V SCFILE[.vpl]     like -v, but synthesize smallcaps as lowercase\n\
--help              print this message and exit\n\
--version           print version number and exit\n\
"

static void
usage(void)
{
  fputs("Usage: ttf2tfm FILE[.ttf] [OPTION]... [FILE[.tfm]]\n", stdout);
  fputs(USAGE, stdout);
  exit(0);
}


static void
openfiles(int argc, char *argv[], struct Font *fnt)
{
  register int lastext;
  register int i;
  int arginc;


#if defined(MSDOS) || defined(OS2) || defined(ATARIST)
  /* Make VPL file identical to that created under Unix */
  (void)sprintf(fnt->titlebuf, "ttf2tfm %s", argv[1]);
#else
  (void)sprintf(fnt->titlebuf, "%s %s", argv[0], argv[1]);
#endif

  (void)strcpy(fnt->ttfname, argv[1]);
  lastext = -1;
  for (i = 0; fnt->ttfname[i]; i++)
    if (fnt->ttfname[i] == '.')
      lastext = i;
    else if (fnt->ttfname[i] == '/' ||
             fnt->ttfname[i] == ':' ||
             fnt->ttfname[i] == '\\')
      lastext = -1;

  if (lastext == -1)
    (void)strcat(fnt->ttfname, ".ttf");

  while (argc > 2 && *argv[2] == '-')
  {
    arginc = 2;
    i = argv[2][1];
    if (i == '/')
      i = argv[2][2] - 32; /* a ==> A for VMS */

    switch (i)
    {
    case 'V':
      makevpl++;
    case 'v':
      makevpl++;
      (void)strcpy(fnt->outname, argv[3]);
      lastext = -1;
      for (i = 0; fnt->outname[i]; i++)
        if (fnt->outname[i] == '.')
          lastext = i;
        else if (fnt->outname[i] == '/' ||
                 fnt->outname[i] == ':' ||
                 fnt->outname[i] == '\\')
          lastext = -1;

      if (lastext == -1)
        (void)strcat(fnt->outname, ".vpl");

      if ((fnt->vplout = fopen(fnt->outname, "wb")) == NULL)
        oops("Cannot open vpl output file");
      break;

    case 'E':
      if (sscanf(argv[3], "%d", &(fnt->eid)) == 0 || fnt->eid < 0)
        oops("Invalid encoding ID");
      fnt->eidparam = argv[3];
      break;

    case 'P':
      if (sscanf(argv[3], "%d", &(fnt->pid)) == 0 || fnt->pid < 0)
        oops("Invalid platform ID");
      fnt->pidparam = argv[3];
      break;

    case 'e':
      if (sscanf(argv[3], "%lf", &(fnt->efactor)) == 0 || fnt->efactor < 0.01)
        oops("Bad extension factor");
      fnt->efactorparam = argv[3];
      break;

    case 'c':
      if (sscanf(argv[3], "%lf", &(fnt->capheight)) == 0 ||
          fnt->capheight < 0.01)
        oops("Bad small caps height");
      break;

    case 's':
      if (sscanf(argv[3], "%lf", &(fnt->slant)) == 0)
        oops("Bad slant parameter");
      fnt->slantparam = argv[3];
      break;

    case 'p':
      fnt->inencname = argv[3];
      break;

    case 'T':
      fnt->inencname = fnt->outencname = argv[3];
      break;

    case 't':
      fnt->outencname = argv[3];
      break;

    case 'O':
      forceoctal = 1;
      arginc = 1;
      break;

    case 'u':
      pedantic = 1;
      arginc = 1;
      break;

    case 'q':
      quiet = 1;
      arginc = 1;
      break;

    default:
      (void)fprintf(stderr, "Unknown option %s %s will be ignored.\n",
                    argv[2], argv[3]);
    }

    for (i = 0; i < arginc; i++)
    {
      (void)sprintf(fnt->titlebuf + strlen(fnt->titlebuf), " %s", argv[2]);
      argv++;
      argc--;
    }
  }

  fnt->real_ttfname = TeX_search_ttf_file(fnt->ttfname);
  if (!fnt->real_ttfname)
    oops("Cannot find %s", fnt->ttfname);

  if (argc > 3 || (argc == 3 && *argv[2] == '-'))
    oops("Need at most two non-option arguments");

  if (argc == 2)
    (void)strcpy(fnt->outname, fnt->ttfname);
  else
    (void)strcpy(fnt->outname, argv[2]);

  lastext = -1;
  for (i = 0; fnt->outname[i]; i++)
    if (fnt->outname[i] == '.')
      lastext = i;
    else if (fnt->outname[i] == '/' ||
             fnt->outname[i] == ':' ||
             fnt->outname[i] == '\\')
      lastext = -1;

  if (argc == 2)
  {
    fnt->outname[lastext] = '\0';
    lastext = -1;
  }
  if (lastext == -1)
  {
    lastext = strlen(fnt->outname);
    (void)strcat(fnt->outname, ".tfm");
  }
  if ((fnt->tfmout = fopen(fnt->outname, "wb")) == NULL)
    oops("Cannot open tfm output file");
  fnt->outname[lastext] = '\0';     /* strip off the extension */

  /*
   *   Now we strip off any directory information, so we only use the
   *   base name in the vf file.  We accept any of /, :, or \ as directory
   *   delimiters, so none of these are available for use inside the
   *   base name; this shouldn't be a problem.
   */

  for (i = 0, lastext = 0; fnt->outname[i]; i++)
    if (fnt->outname[i] == '/' ||
        fnt->outname[i] == ':' ||
        fnt->outname[i] == '\\')
      lastext = i + 1;

  if (lastext)
    strcpy(fnt->outname, fnt->outname + lastext);
}


/*
 *   Here we get a token from the encoding file.  We parse just as much
 *   PostScript as we expect to find in an encoding file.  We allow commented
 *   lines and names like 0, .notdef, _foo_.  We do not allow //abc.
 */

static char *
gettoken(struct Font *fnt)
{
  char *p, *q;


  while (1)
  {
    while (cur_pos == NULL || *cur_pos == '\0')
    {
      if (getline() == 0)
        oops("Premature end in encoding file");
      for (p = buffer; *p; p++)
        if (*p == '%')
        {
          if (ignoreligkern == 0)
            checkligkern(p, fnt, &sawligkern, &cur_pos);
          *p = '\0';
          break;
        }
    }

    while (*cur_pos && *cur_pos <= ' ')
      cur_pos++;

    if (*cur_pos)
    {
      if (*cur_pos == '[' || *cur_pos == ']' ||
          *cur_pos == '{' || *cur_pos == '}')
      {
        smbuffer[0] = *cur_pos++;
        smbuffer[1] = '\0';
        return smbuffer;
      }
      else if (*cur_pos == '/' || *cur_pos == '-' || *cur_pos == '_' ||
               *cur_pos == '.' ||
               ('0' <= *cur_pos && *cur_pos <= '9') ||
               ('a' <= *cur_pos && *cur_pos <= 'z') ||
               ('A' <= *cur_pos && *cur_pos <= 'Z'))
      {
        smbuffer[0] = *cur_pos;
        for (p = cur_pos+1, q = smbuffer+1;
             *p == '-' || *p == '_' || *p == '.' ||
             ('0' <= *p && *p <= '9') ||
             ('a' <= *p && *p <= 'z') ||
             ('A' <= *p && *p <= 'Z'); p++, q++)
          *q = *p;

        *q = '\0';
        cur_pos = p;
        return smbuffer;
      }
    }
  }
}


/*
 *   This routine reads in an encoding file, given the name.  It returns
 *   the final total structure.  It performs a number of consistency checks.
 */

static struct encoding *
readencoding(char *enc, struct Font *fnt)
{
  char *real_encname;
  char *p, c;
  int i;
  long l;
  struct encoding *e =
    (struct encoding *)mymalloc((unsigned long)sizeof (struct encoding));


  sawligkern = 0;

  if (enc)
  {
    real_encname = TeX_search_encoding_file(enc);
    if (!real_encname)
      oops("Cannot find encoding file `%s'", enc);

    encin = fopen(real_encname, "r");
    if (encin == NULL)
      oops("Cannot open encoding file `%s'", enc);

    cur_pos = NULL;
    p = gettoken(fnt);
    if (*p != '/' || p[1] == '\0')
      oops("First token in encoding must be literal encoding name");
    e->name = newstring(p+1);
    p = gettoken(fnt);
    if (strcmp(p, "["))
      oops("Second token in encoding must be mark ([) token");

    for (i = 0; i < 256; i++)
    {
      p = gettoken(fnt);
      if (*p != '/' || p[1] == '\0')
        oops("Tokens 3 to 257 in encoding must be literal names");

      /* now we test for a generic code point resp. glyph index value */

      c = p[2];
      if (p[1] == '.' && (c == 'c' || c == 'g') && '0' <= p[3] && p[3] <= '9')
      {
        l = strtol(p + 3, &p, 0);
        if (*p != '\0' || l < 0 || l > 0xFFFF)
          oops("Encoding token %d is invalid", i + 2);
        sprintf(p, ".%c0x%x", c, (int)l);
        e->vec[i] = newstring(p);
      }
      else
        e->vec[i] = newstring(p+1);
    }

    p = gettoken(fnt);
    if (strcmp(p, "]"))
      oops("Token 258 in encoding must be make-array (])");

    while (getline())
    {
      for (p = buffer; *p; p++)
        if (*p == '%')
        {
          if (ignoreligkern == 0)
            checkligkern(p, fnt, &sawligkern, &cur_pos);
          *p = '\0';
          break;
        }
    }

    fclose(encin);
    encin = NULL;
    if (ignoreligkern == 0 && sawligkern == 0)
      getligkerndefaults(fnt, &sawligkern, buffer, obuffer, &cur_pos);
  }
  else
  {
    e = &staticencoding;
    getligkerndefaults(fnt, &sawligkern, buffer, obuffer, &cur_pos);
  }
  cur_pos = NULL;
  return e;
}


/*
 *   This routine prints out the line that needs to be added to ttfonts.map.
 */

static void
consttfonts(struct Font *fnt)
{
  (void)printf("\n");
  (void)printf("%s   %s", fnt->outname, fnt->ttfname);

  if (fnt->slantparam || fnt->efactorparam ||
      fnt->inencname ||
      fnt->pidparam || fnt->eidparam)
  {
    if (fnt->slantparam)
      (void)printf(" Slant=%s", fnt->slantparam);
    if (fnt->efactorparam)
      (void)printf(" Extend=%s", fnt->efactorparam);
    if (fnt->inencname)
      (void)printf(" Encoding=%s", fnt->inencname);
    if (fnt->pidparam)
      (void)printf(" Pid=%s", fnt->pidparam);
    if (fnt->eidparam)
      (void)printf(" Eid=%s", fnt->eidparam);
  }
  (void)printf("\n");
}



void
main(int argc, char *argv[])
{
  init_font_structure(&font);

  TeX_search_init(argv[0], "TTF2TFM");

  if (argc == 1)
  {
    fputs("ttf2tfm: Need at least one file argument.\n", stderr);
    fputs("Try `ttf2tfm --help' for more information.\n", stderr);
    exit(1);
  }
  if (argc == 2)
  {
    if (strcmp(argv[1], "--help") == 0)
      usage();
    else if (strcmp(argv[1], "--version") == 0)
      version();
  }

  openfiles(argc, argv, &font);

  readttf(&font, quiet);

  if (font.fontspace == 0)
  {
    struct ttfinfo *ti;


    if (NULL != (ti = findadobe("space", font.charlist)))
      font.fontspace = ti->width;
    else if (NULL != (ti = findadobe(".0x20", font.charlist)))
      font.fontspace = ti->width;
    else
      font.fontspace = transform(500, 0, font.efactor, font.slant);
  }

  handlereencoding(&font);

  buildtfm(&font);
  writetfm(font.tfmout);

  if (makevpl)
  {
    assignchars(&font);
    if (makevpl > 1)
      upmap(&font);
    writevpl(&font, makevpl, forceoctal);
  }

  consttfonts(&font);

  exit(0);
}


/* end */
