/*
 * Copyright (c) 2011-2012, Centre National d'Etudes Spatiales (CNES), France 
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS `AS IS'
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * compare_dump_files.c
 *
 *  Created on: 25 juil. 2011
 *      Author: mickael
 * BASELINE MUST BE GENERATED BY UNIX PLATFORM REGARDING TO THE CRLF PROBLEM
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "opj_getopt.h"

typedef struct test_cmp_parameters
{
  /**  */
  char* base_filename;
  /**  */
  char* test_filename;
} test_cmp_parameters;

/*******************************************************************************
 * Command line help function
 *******************************************************************************/
static void compare_dump_files_help_display(void) {
  fprintf(stdout,"\nList of parameters for the compare_dump_files function  \n");
  fprintf(stdout,"\n");
  fprintf(stdout,"  -b \t REQUIRED \t filename to the reference/baseline dump file \n");
  fprintf(stdout,"  -t \t REQUIRED \t filename to the test dump file image\n");
  fprintf(stdout,"\n");
}
/*******************************************************************************
 * Parse command line
 *******************************************************************************/
static int parse_cmdline_cmp(int argc, char **argv, test_cmp_parameters* param)
{
  int sizemembasefile, sizememtestfile;
  int index;
  const char optlist[] = "b:t:";
  int c;

  /* Init parameters */
  param->base_filename = NULL;
  param->test_filename = NULL;

  opj_opterr = 0;

  while ((c = opj_getopt(argc, argv, optlist)) != -1)
    switch (c)
      {
      case 'b':
        sizemembasefile = (int)strlen(opj_optarg)+1;
        param->base_filename = (char*) malloc(sizemembasefile);
        param->base_filename[0] = '\0';
        strncpy(param->base_filename, opj_optarg, strlen(opj_optarg));
        param->base_filename[strlen(opj_optarg)] = '\0';
        /*printf("param->base_filename = %s [%d / %d]\n", param->base_filename, strlen(param->base_filename), sizemembasefile );*/
        break;
      case 't':
        sizememtestfile = (int) strlen(opj_optarg) + 1;
        param->test_filename = (char*) malloc(sizememtestfile);
        param->test_filename[0] = '\0';
        strncpy(param->test_filename, opj_optarg, strlen(opj_optarg));
        param->test_filename[strlen(opj_optarg)] = '\0';
        /*printf("param->test_filename = %s [%d / %d]\n", param->test_filename, strlen(param->test_filename), sizememtestfile);*/
       break;
      case '?':
        if ( (opj_optopt == 'b') || (opj_optopt == 't') )
          fprintf(stderr, "Option -%c requires an argument.\n", opj_optopt);
        else
          if (isprint(opj_optopt)) fprintf(stderr, "Unknown option `-%c'.\n", opj_optopt);
          else fprintf(stderr, "Unknown option character `\\x%x'.\n", opj_optopt);
        return 1;
      default:
        fprintf(stderr, "WARNING -> this option is not valid \"-%c %s\"\n", c, opj_optarg);
        break;
      }

  if (opj_optind != argc)
    {
    for (index = opj_optind; index < argc; index++)
      fprintf(stderr,"Non-option argument %s\n", argv[index]);
    return EXIT_FAILURE;
    }

  return EXIT_SUCCESS;
}
/*******************************************************************************
 * MAIN
 *******************************************************************************/
int main(int argc, char **argv)
{
  test_cmp_parameters inParam;
  FILE *fbase=NULL, *ftest=NULL;
  int chbase, chtest;
  int same = 1;
  unsigned long l=1, pos;

  if( parse_cmdline_cmp(argc, argv, &inParam) == EXIT_FAILURE )
    {
    compare_dump_files_help_display();
    if (!inParam.base_filename) free(inParam.base_filename);
    if (!inParam.test_filename) free(inParam.test_filename);
    return EXIT_FAILURE;
    }

  /* Display Parameters*/
  printf("******Parameters********* \n");
  printf(" base_filename = %s\n"
          " test_filename = %s\n",
          inParam.base_filename, inParam.test_filename);
  printf("************************* \n");

  /* open base file */
  printf("Try to open: %s for reading ... ", inParam.base_filename);
  if((fbase = fopen(inParam.base_filename, "rb"))==NULL)
    {
    printf("Failed.\n");
    free(inParam.base_filename);
    free(inParam.test_filename);
    return EXIT_FAILURE;
    }
  printf("Ok.\n");

  /* open test file */
  printf("Try to open: %s for reading ... ", inParam.test_filename);
  if((ftest = fopen(inParam.test_filename, "rb"))==NULL)
    {
    printf("Failed.\n");
    fclose(fbase);
    free(inParam.base_filename);
    free(inParam.test_filename);
    return EXIT_FAILURE;
    }
  printf("Ok.\n");

  pos=ftell(fbase);

  while(!feof(fbase))
    {
    chbase = fgetc(fbase);
    if(ferror(fbase))
      {
      printf("Error reading base file.\n");
      return EXIT_FAILURE;
      }

    chtest = fgetc(ftest);
    if(ferror(ftest))
      {
      printf("Error reading test file.\n");
      return EXIT_FAILURE;
      }

    /* CRLF problem (Baseline must be always generated by unix platform)*/
    if (chbase == '\n' && chtest == '\r')
      if (fgetc(ftest) == '\n')
        chtest = '\n';

    if(chbase != chtest)
      {
      size_t nbytes = 2048;
      int CRLF_shift=1;
      char *strbase, *strtest, *strbase_d, *strtest_d;

      printf("Files differ at line %lu:\n", l);
      fseek(fbase,pos,SEEK_SET);

      /* Take into account CRLF characters when we write \n into
      // dump file when we used WIN platform*/
#ifdef _WIN32
      CRLF_shift = 2;
      fseek(ftest,pos + l - 1,SEEK_SET);
#else
      fseek(ftest,pos,SEEK_SET);
#endif

      strbase = (char *) malloc(nbytes + 1);
      strtest = (char *) malloc(nbytes + 1);

	if (fgets(strbase, nbytes, fbase) == NULL)
		fprintf(stderr,"\nWARNING: fgets return a NULL value");
	else
	{
		if (fgets(strtest, nbytes, ftest) == NULL)
			fprintf(stderr,"\nWARNING: fgets return a NULL value");
		else
		{
			strbase_d = (char *) malloc(strlen(strbase)+1);
			strtest_d = (char *) malloc(strlen(strtest)+1);
			strncpy(strbase_d, strbase, strlen(strbase)-1);
			strncpy(strtest_d, strtest, strlen(strtest)-CRLF_shift);
			strbase_d[strlen(strbase)-1] = '\0';
			strtest_d[strlen(strtest)-CRLF_shift] = '\0';
			printf("<%s> vs. <%s>\n", strbase_d, strtest_d);
			free(strbase_d);free(strtest_d);
		}
	}

	free(strbase);free(strtest);
	
	same = 0;

      break;
      }
    else
      {
      if (chbase == '\n')
        {
        l++;
        pos = ftell(fbase);
        }
      }
    }

  /*Close File*/
  fclose(fbase);
  fclose(ftest);

  /* Free memory*/
  free(inParam.base_filename);
  free(inParam.test_filename);

  if(same)
    {
      printf("\n***** TEST SUCCEED: Files are the same. *****\n");
      return EXIT_SUCCESS;
    }
  else return EXIT_FAILURE;
}
