/* makeskkdb.c
 * Convert SKK dictionary file into a Berkeley-DBnized dictionary file
 * Version 0.1
 * Copyright (c) 1996 by Kenji Rikitake <kenji@reseau.toyonaka.osaka.jp>
 * $Id: makeskkdb.c,v 1.2 1996/10/17 06:15:20 kenji Exp $
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <limits.h>
#include <db.h>
#include <err.h>

#define DB_SUFFIX ".db"
#define DEFAULT_SKKDIC "./SKK-JISYO.L"
#define BUFSIZE (2048)

/*
 *	Global Variables
 */

char	*pgmnm;			/* program name */
char	*skk_dicname;		/* name of skk dictionary */
char	db_dicname[BUFSIZE];	/* name of db dictionary */
DB	*dbp;			/* DB structure for the db dictionary */
HASHINFO db_hash;		/* HASHINFO for the db dictionary */
FILE	*skkdic;		/* SKK jisho */

#ifdef PROGNAME_REQUIRED
char *__progname;		/* program name for err.c */
#else /* PROGNAME_REQUIRED */
extern char     *__progname;    /* program name for err.c */
#endif /* PROGNAME_REQUIRED */

main(argc, argv)
int argc;
char *argv[];
{
  char	buf[BUFSIZE];
  char	*p;
  DBT	key, data;
  unsigned long	linecount;

#ifdef PROGNAME_REQUIRED
  __progname = argv[0];		/* program name for err.c */
#endif /* PROGNAME_REQUIRED */
    
  if (argc > 2) 
	showusage();
  else if (argc == 2) skk_dicname = argv[1];
  else skk_dicname = DEFAULT_SKKDIC;
  strncpy(db_dicname, skk_dicname, BUFSIZE - 4);
  strcat(db_dicname, DB_SUFFIX);
 
  /* open dictionary files */
  if ((skkdic = fopen(skk_dicname, "r")) == NULL) {
    err(1, "cannot fopen() %s", skk_dicname);
  }
  /* values set from ndbm-compatible default */
  db_hash.bsize = 4096;
  db_hash.ffactor = 40;
  db_hash.nelem = 1;
  db_hash.cachesize = (1024 * 1024); 
  db_hash.hash = NULL;
  db_hash.lorder = 0;
  if ((dbp = dbopen(db_dicname, (O_RDWR | O_CREAT), 
		    (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH), 
		     DB_HASH, &db_hash)) == NULL) {
    err(1, "cannot dbopen() %s", db_dicname);
  }

  linecount = 0;
  while (fgets(buf, BUFSIZE, skkdic) != NULL) {
    linecount++;
#ifdef DEBUG
    fprintf(stderr, "linecount = %lu\n", linecount);
#endif /* DEBUG */
    /* skip if comments */
    if (buf[0] != ';') {
      key.data = &buf[0];
      if ((p = strchr(buf, ' ')) == NULL)
        err(1, "cannot find space in file %s line %lu", 
                skk_dicname, linecount);
      key.size = p - &buf[0];
      data.data = ++p;
      data.size = (strlen(buf) - 1) - (key.size + 1);

#ifdef DEBUG
      fprintf(stderr, "key = '");
      fflush(stderr);
      (void)write(fileno(stderr), key.data, key.size);
      fprintf(stderr, "', size = %d\n", key.size);
      fprintf(stderr, "data = '");
      fflush(stderr);
      (void)write(fileno(stderr), data.data, data.size);
      fprintf(stderr, "', size = %d\n", data.size);
      fflush(stderr);
#endif /* DEBUG */

      /* put data to db file */
      switch (dbp->put(dbp, &key, &data, 0)) {
      case 0:
	break;
      case -1:
	err(1, "dbp->put() failed at line %lu", linecount);
	/* NOTREACHED */
      case 1:
	warn("dbp->put() did not overwrite at line %lu", linecount);
	break;
      default:
	break;
      }

      if (linecount % 1000 == 0) {
	fprintf(stderr, "%lu ", linecount);
	if (dbp->sync(dbp, 0) != 0) {
	  err(1, "dbp->sync() failed at line %lu", linecount);
	}
      }

    }
  }
  /* close dictionary db file */
  if (fclose(skkdic) < 0)
    err(1, "cannot fclose() %s", skk_dicname);
  if (dbp->close(dbp)) 
    err(1, "cannot dbp->close() %s", db_dicname);
}

showusage()
{
  fprintf(stderr, "Usage: %s skk-dicfile\n", __progname); 
  exit(1);
}

/* end of program */
/* comment follows */

/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either versions 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SKK, see the file COPYING.  If not, write to the Free
 * Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * Description:
 *	An SKK dictionary is a sorted sequence of lines in 
 *	the form: 
 *		YOMI /TANGO1/TANGO2/.../TANGOn/
 *	and lines started with ";;" as comments/directives:
 *		;; okuri-ari entries.
 *
 *	This program converts a existing SKK dictionary file into
 *	a Berkeley DB-format file.
 */

 /* end of source file */
