;; SKK (Simple Kana to Kanji conversion program)
;; version 3.47 of April 29, 1990
;; Copyright (C) 1990 Masahiko Sato (masahiko@sato.riec.tohoku.ac.jp)

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either versions 1, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SKK, see the file COPYING.  If not, write to the Free
;; Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; version 3.47 released 1990.4.29
;; version 3.46 released 1990.4.22
;; version 3.45 released 1990.4.15
;; modified by Masakazu Takahashi (masaka-t@ascii.co.jp) 1989.8.13, 1989.11.6
;; version 2.25 released 1989.4.15
;; version 1.13 released 1988.4.22

(defun skk-version ()
  (interactive)
  (if (interactive-p)
      (message "SKK version %s of April 29, 1990" (skk-version))
    "3.47"))

;; variable declarations

(setq enable-recursive-minibuffers t)

(defvar j-count-touroku 0)
(defvar j-count-kakutei 0)

(defvar skk-mode nil
  "The value is t if skk-mode has been invoked on the current buffer.")
(make-variable-buffer-local 'skk-mode)

(defvar skk-map (make-keymap)
  "Keymap for kana mode.")
(make-variable-buffer-local 'skk-map)

(defvar skk-zenkaku-map (make-keymap)
  "Keymap for zenkaku mode.")
(make-variable-buffer-local 'skk-zenkaku-map)

(defvar skk-abbrev-map (make-keymap)
  "Keymap for abbrev input.")
(make-variable-buffer-local 'skk-abbrev-map)

(defvar j-emacs-local-map nil
  "Original keymap for emacs mode.")
(make-variable-buffer-local 'j-emacs-local-map)

(defvar j-mode-line-skk nil)
(make-variable-buffer-local 'j-mode-line-skk)

(defvar j-prefix "")
(make-variable-buffer-local 'j-prefix)

(defvar j-prefix-o nil
  "If T, it means that the character o is part of kana-prefix")
(make-variable-buffer-local 'j-prefix-o)

(defvar j-mode nil)
(make-variable-buffer-local 'j-mode)

(defvar j-henkan-on nil)
(make-variable-buffer-local 'j-henkan-on)

(defvar j-henkan-start-point (make-marker)
  "Henkan start point as a marker.")
(make-variable-buffer-local 'j-henkan-start-point)

(defvar j-henkan-end-point (make-marker)
  "Henkan end point as a marker.")
(make-variable-buffer-local 'j-henkan-end-point)

(defvar j-kana-start-point (make-marker)
  "Start point of a kana character as a marker.")
(make-variable-buffer-local 'j-kana-start-point)

(defvar j-okurigana nil
  "If t, it means that okurigana part of henkan word is being input.")
(make-variable-buffer-local 'j-okurigana)

(defvar j-okurigana-start-point (point-marker)
  "Start point of okurigana")
(make-variable-buffer-local 'j-okurigana-start-point)

(defvar j-henkan-count 0)
(make-variable-buffer-local 'j-henkan-count)

(defvar j-henkan-active nil)
(make-variable-buffer-local 'j-henkan-active)

(defvar j-katakana nil
  "t means katakana and nil means hirakana input.")
(make-variable-buffer-local 'j-katakana)

(defvar j-henkan-key)
(make-variable-buffer-local 'j-henkan-key)

(defvar j-henkan-key2 nil)
(make-variable-buffer-local 'j-henkan-key2)

(defvar j-henkan-okurigana nil)
(make-variable-buffer-local 'j-henkan-okurigana)

(defvar j-henkan-vector (vector))
(make-variable-buffer-local 'j-henkan-vector)

(defvar j-henkan-vector1 (vector))
(make-variable-buffer-local 'j-henkan-vector1)

(defvar j-henkan-vector2 (vector))
(make-variable-buffer-local 'j-henkan-vector2)

(defvar j-henkan-vector3 (vector))
(make-variable-buffer-local 'j-henkan-vector3)

(defvar j-henkan-vector4 (vector))
(make-variable-buffer-local 'j-henkan-vector4)

(defvar j-mode-line-buffer-identification)
(make-variable-buffer-local 'j-mode-line-buffer-identification)

(defvar j-zenkaku)
(make-variable-buffer-local 'j-zenkaku)

(defvar j-abbrev nil)
(make-variable-buffer-local 'j-abbrev)

(defvar j-auto-fill nil)
(make-variable-buffer-local 'j-auto-fill)

(defvar j-completion-word "")

;; skk server related variables

(defvar j-local-length 0)
(make-variable-buffer-local 'j-local-length)
(defvar j-server-called nil)
(make-variable-buffer-local 'j-server-called)

;; constants

;; Cutomization of these constants cannot be done in .emacs file.  This
;; should be done in .skk file or by using skk-mode-hook.

;; We are using `defconst' rather than `defvar' here for the following
;; reason:  The SKK tutorial program, which is intended for the beginners,
;; must be run on a vanilla SKK.  To achieve this, the tutorial program
;; loads SKK (even if it has been loaded already) and undoes the customization
;; which minght have been done by the user.  (Note that `defconst' always sets
;; a new value, while `defvar' does that only when the value is unbound when
;; `defvar' is executed.)

(defconst skk-jisyo "~/.skk-jisyo"
  "*The name of the skk jisyo file.")

(defconst skk-keep-record t
  "*If t, skk will keep record of the success rate of kana to kanji conversion
etc. in the file ~/.skk-record.")

(defconst skk-use-vip nil
  "*If t, skk will assume that the use will use vip-mode at the same time.")

(defconst skk-process-okuri-early nil
  "*If t, skk will start kana to kanji conversion when it reads the first
roman character for okurigana, if nil then kana to kanji conversion will
start only when it reads a complete roman representation for a kana
character.")

(defconst skk-mode-string "--SKK:"
  "*The value of this constant is used in the mode line when skk is in ascii
mode.")

(defconst skk-echo t
  "*If T, echoes prefix for kana characters.")

(defconst skk-server-host (getenv "SKKSERVER")
  "*Name of the host running SKK Jisyo Server")

(defconst skk-serv (getenv "SKKSERV")
  "*Name of the SKK Jisyo Server program")

(defconst skk-portnum nil
  "*If the value is non-nil, it must be the PORTNUM defined in skkserv.h")

(defconst skk-report-server-response nil
  "*When SKK sends a request to SKK SERVER, SKK waits for the process output
by repeatedly executing the accept-process-output function.  If this constant
is t, then SKK will report the number of times the function is called.")

(defconst skk-prefix-list
  '("b" "by" "c" "ch" "cy" "d" "dh" "dy" "f" "fy" "g" "gy" "h" "hy" "j" "jy"
    "k" "ky" "m" "my" "n" "ny" "p" "py" "r" "ry" "s" "sh" "sy"
    "t" "th" "ts" "ty" "v" "w" "x" "xk" "xt" "xts" "xw" "xy" "y" "z" "zy"))

(defconst skk-roma-kana-a
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . "")) 
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("դ" . "ե")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("" . ""))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xk" . ("" . ""))
    ("xw" . ("" . "")) ("xy" . ("" . ""))
    ("y" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-roma-kana-i
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . ""))
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("դ" . "ե")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("ꤣ" . "ꥣ"))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xw" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-roma-kana-u
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . ""))
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("" . "")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("" . ""))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ts" . ("" . "")) ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xt" . ("" . ""))
    ("xts" . ("" . "")) ("xy" . ("" . ""))
    ("y" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-roma-kana-e
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . ""))
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("դ" . "ե")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("ꤧ" . "ꥧ"))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xk" . ("" . ""))
    ("xw" . ("" . ""))
    ("y" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-roma-kana-o
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . ""))
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("դ" . "ե")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("" . ""))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xy" . ("" . ""))
    ("y" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-input-vector
  [nil  nil  nil  nil  nil  nil  nil  nil
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  "" nil  nil  nil  nil  nil  nil
   nil  nil  nil  nil  "" "" "" nil
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  "" "" nil  nil  nil  ""
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  "" nil  "" nil  nil
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil])

(defconst skk-zenkaku-vector
  [nil  nil  nil  nil  nil  nil  nil  nil
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" nil])

;; utilities

(defun j-change-mode-line (string)
  (setq j-mode-line-skk string)
  (set-buffer-modified-p (buffer-modified-p)))

(defun j-today ()
  "Insert current date at point in Kanji."
  (interactive)
  (insert (j-date)))

(defun j-date ()
  (let* ((str (current-time-string))
	 (y (- (string-to-int (substring str 22 24)) 88))
	 (year (if (= y 1) "" (j-num (int-to-string y))))
	 (mon (substring str 4 7))
	 (month
	  (cdr (assoc mon
		      '(("Jan" . "") ("Feb" . "") ("Mar" . "")
			("Apr" . "") ("May" . "") ("Jun" . "")
			("Jul" . "") ("Aug" . "") ("Sep" . "")
			("Oct" . "") ("Nov" . "") ("Dec" . "")))))
	 (d1 (j-num (substring str 8 9)))
	 (d2 (j-num (substring str 9 10)))
	 (dw (substring str 0 3))
	 (day-of-week
	  (cdr (assoc dw
		      '(("Sun" . "") ("Mon" . "") ("Tue" . "")
			("Wed" . "") ("Thu" . "") ("Fri" . "")
			("Sat" . "")))))
	 )
    (concat "ʿ" year "ǯ" month "" d1 d2 "" "\(" day-of-week "\)")))

(defun j-num (str)
  (cdr (assoc str
	      '((" " . "") ("0" . "") ("1" . "") ("2" . "") ("3" . "")
		("4" . "") ("5" . "") ("6" . "") ("7" . "")
		("8" . "") ("9" . "")))))

(defun j-convert-to-vector (map)
  "convert a map in list form to a map in vector form."
  (if (listp map)
      (let ((new-map (make-keymap)) (l (cdr map)))
	(while l
	  (aset new-map (car (car l)) (cdr (car l)))
	  (setq l (cdr l)))
	new-map)
    map))

(defun j-get-editor-command (l-map g-map &optional init-char)
  "Read characters from keyboard until an editor command is formed, using
local keymap L-MAP and global keymap G-MAP."
  (let (char l-bind g-bind)
    (setq char (or init-char (read-char)))
    (setq last-command-char char)
    (setq l-bind (j-binding-of char l-map))
    (if (null l-bind)
	;; since local binding is empty, we concentrate on global one.
	(progn
	  (setq g-bind (j-binding-of char g-map))
	  (if (null g-bind)
	      nil ;; return nil, since both bindings are void.
	    (if (keymapp g-bind)
		(j-get-editor-command nil g-bind)
	      g-bind)))
      ;; local binding is nonvoid
      (if (keymapp l-bind)
	  ;; since l-bind is a keymap, we consider g-bind as well.
	  (progn
	    (setq g-bind (j-binding-of char g-map))
	    (if (null g-bind)
		(j-get-editor-command l-bind nil)
	      (if (keymapp g-bind)
		  ;; both bindings are keymap
		  (j-get-editor-command l-bind g-bind)
		;; l-bind is a keymap, so we neglect g-bind
		(j-get-editor-command l-bind nil))))
	;; l-bind is a command
	l-bind))))

(defun j-binding-of (char map)
  "Return key-binding of CHAR under keymap MAP.  It is nil if the binding
is void, or a command, or a keymap"
  (let ((val (if (listp map)
		 (cdr (assq char map))
	       (aref map char))))
    (cond ((null val) nil)
	  ((keymapp val)
	   (if (symbolp val) (symbol-function val) val))
	  (t
	   ;; otherwise, it is a function which is either a real function or
	   ;; a keymap fset to val.
	   (let ((fun (symbol-function val)))
	     (if (or (null fun) (keymapp fun)) fun val))))))

(defun j-string-tail (str)
  (if (or (null str) (string= str "")) nil
    (substring str 1)))

;; auto fill

;; modified by Masahiko Sato, 1990.3.20
;; modified by Masakazu Takahashi, 1989.8.13, 1989.11.6
(defun j-kanji-fill ()
  "Auto fill fuction for Japanese text. This function returns t if a new
line is inserted, and returns nil otherwise."
  (interactive)
  (save-excursion
    (let ((val nil))
      (if (>= (current-column) fill-column)
	  (let ((p (point)))
	    ;; move cursor to newline point
	    (move-to-column (1- fill-column))
	    (if (and (< (current-column) fill-column)
		     (not (eolp)))
		(forward-char 1))
	    (if (and (< (following-char) ?\200)
		     (not (looking-at " \\|\t")))
		(while (and (< (preceding-char) ?\200)
			    (not (and (/= (preceding-char) ? )
				      (/= (preceding-char) ?\t)
				      (looking-at " \\|\t")))
			    (< 0 (current-column)))
		  (backward-char 1)))
	    (kinsoku-shori)
	    ;; newline
	    (if (< (point) p)
		(progn
		  (delete-horizontal-space)
		  (insert ?\n)
		  (setq val t)))))
      val)))

(defun skk-auto-fill-mode (arg)
  (interactive "P")
  (setq j-auto-fill
	(cond ((null arg) (not j-auto-fill))
	      ((> (prefix-numeric-value arg) 0) t)
	      (t nil)))
  (auto-fill-mode (if j-auto-fill 1 -1))
  (skk-mode))

;; kana inserting functions

(defun j-insert (table)
  (let ((char (assoc j-prefix table)))
    (j-erase-prefix (length j-prefix))
    (if (null char)
	;; j-prefix not found in the table
	(setq j-prefix ""
	      unread-command-char last-command-char)
      (insert (if j-katakana (cdr (cdr char)) (car (cdr char))))
      (if j-okurigana
	  (j-set-okurigana)
	(setq j-prefix ""
	      j-prefix-o nil)))))

(defun j-set-okurigana ()
  (let (w-head)
    (move-marker j-henkan-end-point j-okurigana-start-point)
    (save-excursion
      ;; just in case
      (goto-char j-okurigana-start-point)
      (or (looking-at "*") (insert "*")))
    (setq w-head (buffer-substring
		  j-henkan-start-point
		  j-henkan-end-point))
    (setq j-henkan-key2
	  (concat w-head
		  "*"
		  (setq j-henkan-okurigana
			(buffer-substring
			 (1+ j-okurigana-start-point)
			 (point)))))
    (setq j-henkan-key
	  (concat w-head
		  (if (string= j-prefix "")
		      (char-to-string last-command-char)
		    (char-to-string (aref j-prefix 0)))))
    (setq j-prefix ""
	  j-prefix-o nil)
    (delete-region
     j-okurigana-start-point (1+ j-okurigana-start-point))
    (j-henkan)
    (setq j-okurigana nil)))

(defun j-insert-a ()
  (interactive)
  (j-insert skk-roma-kana-a))

(defun j-insert-i ()
  (interactive)
  (j-insert skk-roma-kana-i))

(defun j-insert-u ()
  (interactive)
  (j-insert skk-roma-kana-u))

(defun j-insert-e ()
  (interactive)
  (j-insert skk-roma-kana-e))

(defun j-insert-o ()
  (interactive)
  (let ((null-prefix (string= j-prefix "")))
    (j-insert skk-roma-kana-o)
    (if null-prefix (setq j-prefix-o t))))

(defun j-insert-n ()
  (interactive)
  (cond ((string= j-prefix "")
	 (move-marker j-kana-start-point (point-marker))
	 (setq j-prefix "n"
	       j-prefix-o nil)
	 (j-insert-prefix))
	((string= j-prefix "n")
	 (j-erase-prefix 1)
	 (insert (if j-katakana "" ""))
	 (if j-okurigana (j-set-okurigana)
	   (setq j-prefix "")))
	(t
	 (j-erase-prefix (length j-prefix))
	 (move-marker j-kana-start-point (point-marker))
	 (setq j-prefix "n")
	 (j-insert-prefix))))

(defun j-erase-prefix (len)
  "erase j-prefix characters if skk-echo is on"
  (if (and skk-echo (> len 0))
      (delete-region j-kana-start-point
		     (min (point-max) (+ j-kana-start-point len)))))

(defun j-insert-prefix (&optional char)
  "insert j-prefix if skk-echo is on"
  (if skk-echo (insert (or char j-prefix))))

(defun j-consonant ()
  "handle consonant characters properly."
  (interactive)
  (if j-prefix-o
      (cond ((string= j-prefix "")
	     (if (= last-command-char ?h)
		 (progn
		   (move-marker j-kana-start-point (point-marker))
		   (setq j-prefix "h")
		   (j-insert-prefix))
	       (setq j-prefix-o nil
		     unread-command-char last-command-char)))
	    ((string= j-prefix "h")
	     (j-erase-prefix 1)
	     (insert (if j-katakana "" ""))
	     (setq j-prefix ""
		   j-prefix-o nil
		   unread-command-char last-command-char))
	    (t
	     (setq j-prefix ""
		   j-prefix-o nil
		   unread-command-char last-command-char)))
    (if (and (string= j-prefix "n") (not (= last-command-char ?y)))
	(progn
	 (j-erase-prefix 1)
	 (insert (if j-katakana "" ""))
	 (if j-okurigana (j-set-okurigana)
	   (setq j-prefix ""))
	 (setq unread-command-char last-command-char))
      (let* ((char (char-to-string last-command-char))
	     (prefix (concat j-prefix char)))
	(if (string= j-prefix char)
	    (progn
	      (j-erase-prefix 1)
	      (insert (if j-katakana "" ""))
	      (move-marker j-kana-start-point (point-marker))
	      (j-insert-prefix))
	  (if (j-member prefix skk-prefix-list)
	      (progn
		(if (string= j-prefix "")
		    (move-marker j-kana-start-point (point-marker)))
		(j-insert-prefix char)
		(setq j-prefix prefix))
	    (j-erase-prefix (length j-prefix))
	    (setq j-prefix (char-to-string last-command-char))
	    (j-insert-prefix)))))))

(defun j-member (x l)
  "check if X is a member of L, where string= is used to compare elements
of L against X"
  (cond ((null l) nil)
	((string= x (car l)) t)
	(t (j-member x (cdr l)))))

;; other inserting functions

(defun j-insert-comma (count)
  (interactive "p")
  (if (equal last-command 'j-completion)
      (j-previous-completion)
    (j-self-insert count)))

(defun j-insert-minus (count)
  (interactive "p")
  (j-process-pending-n)
  (let ((str (aref skk-input-vector last-command-char)))
    (if str
	(while (> count 0)
	  (insert str)
	  (setq count (1- count)))
      (j-emulate-original-map last-command-char arg)
      (j-kakutei))))

(defun j-insert-period (count)
  (interactive "p")
  (if (equal last-command 'j-completion)
      (progn
	(setq this-command 'j-completion)
	(j-completion nil))
    (j-self-insert count)))

(defun j-insert-quote (count)
  (interactive "p")
  (if (string= j-prefix "n")
      (j-process-pending-n)
    (j-self-insert count)))

(defun j-abbrev-period (arg)
  (interactive "P")
  (if (equal last-command 'j-completion)
      (progn
	(setq this-command 'j-completion)
	(j-completion nil))
    (j-emulate-original-map last-command-char arg)))

(defun j-abbrev-comma (arg)
  (interactive "P")
  (if (equal last-command 'j-completion)
      (j-previous-completion)
    (j-emulate-original-map last-command-char arg)))

;; mode setup

(defun skk-mode ()
  "SKK mode enables input of Japanese texts without affecting the
functionality of the original mode."
  (interactive)
  (if skk-mode
      (progn
	(if (and skk-use-vip (= (minibuffer-depth) 0))
	    (vip-change-mode 'insert-mode))
	(j-change-mode-line "--")
	(setq j-mode t)
	(use-local-map skk-map))
    (if (not (featurep 'skk))
	(progn
	  (provide 'skk)
	  (if (file-exists-p "~/.skk") (load "~/.skk"))
	  (if skk-server-host (j-open-server))
	  (let ((inhibit-quit t))
	    (save-window-excursion
	      (set-buffer (get-buffer-create " *jisyo*"))
	      (erase-buffer)
	      (insert-file-contents skk-jisyo)))))
    (if (and skk-use-vip (not (boundp 'vip-current-mode))) (load "vip"))
    (if (and skk-use-vip (= (minibuffer-depth) 0))
	(vip-change-mode 'insert-mode))
    (or j-mode-line-skk
	(setq mode-line-format
	      (cons "" (cons 'j-mode-line-skk
			     (cdr (default-value 'mode-line-format))))))
    (j-change-mode-line "--")
    (setq skk-mode t
	  j-mode t
	  j-katakana nil)
    (local-set-key "\C-j" 'j-kakutei)
    ;(local-set-key "\C-m" 'j-newline)
    (setq j-emacs-local-map
	  (j-convert-to-vector (copy-keymap (current-local-map))))
    (define-key minibuffer-local-map "\C-j" 'skk-mode)
    (define-key minibuffer-local-map "\C-m" 'j-newline)
    (j-setup-skk-map)
    (j-setup-skk-zenkaku-map)
    (j-setup-skk-abbrev-map)
    (use-local-map skk-map)
    (run-hooks 'skk-mode-hook)))

(defun j-setup-skk-map ()
  (setq skk-map (copy-keymap j-emacs-local-map))
  (define-key skk-map "\C-j" 'j-kakutei)
  (define-key skk-map "\C-m" 'j-newline)
  (define-key skk-map "\t" 'j-try-completion)
  (define-key skk-map "\ " 'j-start-henkan)
  (define-key skk-map "!" 'j-self-insert)
  (define-key skk-map "\"" 'j-self-insert)
  (define-key skk-map "#" 'j-self-insert)
  (define-key skk-map "$" 'j-self-insert)
  (define-key skk-map "%" 'j-self-insert)
  (define-key skk-map "&" 'j-self-insert)
  (define-key skk-map "'" 'j-insert-quote)
  (define-key skk-map "\(" 'j-self-insert)
  (define-key skk-map "\)" 'j-self-insert)
  (define-key skk-map "*" 'j-self-insert)
  (define-key skk-map "+" 'j-self-insert)
  (define-key skk-map "," 'j-insert-comma)
  (define-key skk-map "-" 'j-insert-minus)
  (define-key skk-map "." 'j-insert-period)
  (define-key skk-map "/" 'j-abbrev-input)
  (define-key skk-map "0" 'j-self-insert)
  (define-key skk-map "1" 'j-self-insert)
  (define-key skk-map "2" 'j-self-insert)
  (define-key skk-map "3" 'j-self-insert)
  (define-key skk-map "4" 'j-self-insert)
  (define-key skk-map "5" 'j-self-insert)
  (define-key skk-map "6" 'j-self-insert)
  (define-key skk-map "7" 'j-self-insert)
  (define-key skk-map "8" 'j-self-insert)
  (define-key skk-map "9" 'j-self-insert)
  (define-key skk-map "@" 'j-today)
  (define-key skk-map "?" 'j-self-insert)
  (define-key skk-map "A" 'j-set-henkan-point)
  (define-key skk-map "B" 'j-set-henkan-point)
  (define-key skk-map "C" 'j-set-henkan-point)
  (define-key skk-map "D" 'j-set-henkan-point)
  (define-key skk-map "E" 'j-set-henkan-point)
  (define-key skk-map "F" 'j-set-henkan-point)
  (define-key skk-map "G" 'j-set-henkan-point)
  (define-key skk-map "H" 'j-set-henkan-point)
  (define-key skk-map "I" 'j-set-henkan-point)
  (define-key skk-map "J" 'j-set-henkan-point)
  (define-key skk-map "K" 'j-set-henkan-point)
  (define-key skk-map "L" 'j-zenkaku-eiji)
  (define-key skk-map "M" 'j-set-henkan-point)
  (define-key skk-map "N" 'j-set-henkan-point)
  (define-key skk-map "O" 'j-set-henkan-point)
  (define-key skk-map "P" 'j-set-henkan-point)
  (define-key skk-map "Q" 'j-set-henkan-point-subr)
  (define-key skk-map "R" 'j-set-henkan-point)
  (define-key skk-map "S" 'j-set-henkan-point)
  (define-key skk-map "T" 'j-set-henkan-point)
  (define-key skk-map "U" 'j-set-henkan-point)
  (define-key skk-map "V" 'j-set-henkan-point)
  (define-key skk-map "W" 'j-set-henkan-point)
  (define-key skk-map "X" 'j-purge-from-jisyo)
  (define-key skk-map "Y" 'j-set-henkan-point)
  (define-key skk-map "Z" 'j-set-henkan-point)
  (define-key skk-map "\[" 'j-self-insert) 
  (define-key skk-map "\\" 'j-input-by-code)
  (define-key skk-map "\]" 'j-self-insert)
  (define-key skk-map "^" 'j-self-insert)
  (define-key skk-map "_" 'j-self-insert)
  (define-key skk-map "`" 'j-self-insert)
  (define-key skk-map "a" 'j-insert-a)
  (define-key skk-map "b" 'j-consonant)
  (define-key skk-map "c" 'j-consonant)
  (define-key skk-map "d" 'j-consonant)
  (define-key skk-map "e" 'j-insert-e)
  (define-key skk-map "f" 'j-consonant)
  (define-key skk-map "g" 'j-consonant)
  (define-key skk-map "h" 'j-consonant)
  (define-key skk-map "i" 'j-insert-i)
  (define-key skk-map "j" 'j-consonant)
  (define-key skk-map "k" 'j-consonant)
  (define-key skk-map "l" 'j-mode-off)
  (define-key skk-map "m" 'j-consonant)
  (define-key skk-map "n" 'j-insert-n)
  (define-key skk-map "o" 'j-insert-o)
  (define-key skk-map "p" 'j-consonant)
  (define-key skk-map "q" 'j-toggle-kana)
  (define-key skk-map "r" 'j-consonant)
  (define-key skk-map "s" 'j-consonant)
  (define-key skk-map "t" 'j-consonant)
  (define-key skk-map "u" 'j-insert-u)
  (define-key skk-map "v" 'j-consonant)
  (define-key skk-map "w" 'j-consonant)
  (define-key skk-map "x" 'j-previous-candidate)
  (define-key skk-map "y" 'j-consonant)
  (define-key skk-map "z" 'j-consonant)
  (define-key skk-map "\{" 'j-self-insert)
  (define-key skk-map "|" 'j-self-insert)
  (define-key skk-map "\}" 'j-self-insert)
  (define-key skk-map "~" 'j-self-insert)
  (define-key skk-map "\177" 'j-delete-backward-char))

(defun j-setup-skk-zenkaku-map ()
  (setq skk-zenkaku-map (copy-keymap j-emacs-local-map))
  (define-key skk-zenkaku-map "\C-j" 'skk-mode)
  (let ((i 33))
    (while (< i 127)
      (define-key skk-zenkaku-map (char-to-string i) 'j-zenkaku-insert)
      (setq i (1+ i)))))

(defun j-setup-skk-abbrev-map ()
  (setq skk-abbrev-map (copy-keymap j-emacs-local-map))
  (define-key skk-abbrev-map "\C-j" 'j-kakutei)
  (define-key skk-abbrev-map "\t" 'j-try-completion)
  (define-key skk-abbrev-map "\C-m" 'j-newline)
  (define-key skk-abbrev-map "." 'j-abbrev-period)
  (define-key skk-abbrev-map "," 'j-abbrev-comma)
  (define-key skk-abbrev-map "\ " 'j-start-henkan)
  (define-key skk-abbrev-map "\177" 'j-delete-backward-char))

(defun j-process-pending-n ()
  (if (string= j-prefix "n")
      ;; process pending j-prefix
      (progn
	(j-erase-prefix 1)
	(setq j-prefix "")
	(move-marker j-kana-start-point nil)
	(insert (if j-katakana "" "")))))

(defun j-self-insert (arg)
  (interactive "p")
  (j-process-pending-n)
  (let ((str (aref skk-input-vector last-command-char)))
    (if str
	(while (> arg 0)
	  (insert str)
	  (setq arg (1- arg)))
      (j-emulate-original-map last-command-char arg))
    (j-kakutei)))

(defun j-zenkaku-insert (count)
  (interactive "p")
  (while (> count 0)
    (insert (aref skk-zenkaku-vector last-command-char))
    (setq count (1- count))))

(defun j-zenkaku-self-insert (count)
  (interactive "p")
  (while (> count 0)
    (insert (concat (char-to-string 163)
		    (char-to-string (+ last-command-char 128))))
    (setq count (1- count))))

(defun j-mode-off ()
  "return to old ascii mode."
  (interactive)
  (j-kakutei)
  (setq j-mode nil
	j-prefix ""
	j-zenkaku nil)
  (use-local-map j-emacs-local-map)
  (j-change-mode-line skk-mode-string))

(defun j-zenkaku-eiji ()
  "zenkaku-eiji input"
  (interactive)
  (j-kakutei)
  (setq j-mode nil
	j-prefix ""
	j-zenkaku t)
  (use-local-map skk-zenkaku-map)
  (j-change-mode-line "--"))

(defun j-abbrev-input ()
  (interactive)
  (if (and j-henkan-on (not j-henkan-active)) (error ""))
  (j-kakutei)
  (j-set-henkan-point-subr)
  (use-local-map skk-abbrev-map)
  (setq j-abbrev t))

;; vip related functions

(defun vip-change-mode (new-mode)
  "(NEW-MODE)  Change mode to NEW-MODE.  NEW-MODE is either emacs-mode,
vi-mode, or insert-mode."
  (if (> (minibuffer-depth) 0)
      (error "Can't change mode in minibuffer!")
    (or (eq new-mode vip-current-mode)
	(progn
	  (cond ((eq new-mode 'vi-mode)
		 (if (eq vip-current-mode 'insert-mode)
		     (progn
		       (if (and skk-mode j-mode) (j-kakutei))
		       (vip-copy-region-as-kill (point) vip-insert-point)
		       (vip-repeat-insert-command))
		   (setq vip-emacs-local-map (current-local-map)
			 vip-emacs-mode-line-buffer-identification
			 mode-line-buffer-identification
			 vip-insert-local-map (vip-copy-keymap
					       (current-local-map))))
		 (vip-change-mode-line "Vi:   ")
		 (if skk-mode (j-change-mode-line skk-mode-string))
		 (use-local-map vip-mode-map))
		((eq new-mode 'insert-mode)
		 (move-marker vip-insert-point (point))
		 (if (eq vip-current-mode 'emacs-mode)
		     (setq vip-emacs-local-map (current-local-map)
			   vip-emacs-mode-line-buffer-identification
			   mode-line-buffer-identification
			   vip-insert-local-map (vip-copy-keymap
						 (current-local-map))))
		 (vip-change-mode-line "Insert")
		 (if skk-mode
		     (j-change-mode-line
		      (if j-mode (if j-katakana "--" "--")
			(if j-zenkaku "--" skk-mode-string))))
		 (if (and skk-mode (not j-mode) (not j-zenkaku))
		     (setq j-mode-line-buffer-identification
			   mode-line-buffer-identification))
		 (if skk-mode
		     (if j-mode
			 (use-local-map skk-map)
		       (if j-zenkaku
			   (use-local-map skk-zenkaku-map)
			 (use-local-map vip-insert-local-map)))
		   (use-local-map vip-insert-local-map))
		 (local-set-key "\C-j" 'j-kakutei)
		 (local-set-key "\e" 'vip-change-mode-to-vi)
		 (local-set-key "\C-z" 'vip-ESC)
		 (local-set-key "\C-h"
				(if vip-help-in-insert-mode 'help-command
				  'j-delete-backward-char))
		 (local-set-key "\C-w"
				'vip-delete-backward-word))
		((eq new-mode 'emacs-mode) 
		 (vip-change-mode-line "Emacs:")
		 (if skk-mode (j-change-mode-line skk-mode-string))
		 (use-local-map vip-emacs-local-map)))
	  (setq vip-current-mode new-mode)
	  (vip-refresh-mode-line)))))

(if (not (fboundp 'skk-kill-emacs))
    (fset 'skk-kill-emacs (symbol-function 'kill-emacs)))

(defun kill-emacs (&optional query)
  (interactive "P")
  (skk-save-jisyo)
  (if (process-status "skkservd")
      (process-send-string "skkservd" "0")) ;; disconnect server
  (skk-kill-emacs query))

(defun skk-kill-emacs-without-saving-jisyo ()
  "Kill Emacs without saving jisyo."
  (interactive)
  (if (yes-or-no-p "Do you really wish to kill emacs without saving Jisyo? ")
      (progn
	(setq j-count-kakutei 0)
	(kill-emacs))))

(defun skk-save-jisyo ()
  "Save jisyo buffer into jisyo file."
  (interactive)
  (if (= j-count-kakutei 0)
      (message "No need to save Jisyo.")
    (message "Saving Jisyo...")
    (get-buffer-create " *record*")
    (let ((inhibit-quit t) lines (skk-jisyo (expand-file-name skk-jisyo)))
      (save-window-excursion
	(call-process "/bin/mv" nil nil nil skk-jisyo
		      (concat skk-jisyo ".BAK"))
	(set-buffer " *jisyo*")
	(setq lines (count-lines (point-min) (point-max)))
	(let ((go-on t))
	  (goto-char (point-min))
	  (while go-on
	    (set-mark (point))
	    (goto-char (+ 50000 (point)))
	    (end-of-line)
	    (write-region (mark) (point) skk-jisyo t 'nomsg)
	    (if (eobp) (setq go-on nil))))
	(message "Saving Jisyo...done.")
	(if skk-keep-record
	    (progn
	      (set-buffer " *record*")
	      (erase-buffer)
	      (insert-file "~/.skk-record")
	      (goto-char (point-min))
	      (insert
	       (format "%s  Ͽ: %3d  : %4d  Ψ: %3d%%  : %6d\n"
		       (current-time-string)
		       j-count-touroku
		       j-count-kakutei
		       (/ (* 100 (- j-count-kakutei j-count-touroku))
			  j-count-kakutei)
		       lines))
	      (write-region (point-min) (point-max) "~/.skk-record" nil 'nomsg)
	      (set-buffer-modified-p nil)
	      (setq j-count-touroku 0
		    j-count-kakutei 0)))))))

;; input functions

;; modified by Masahiko Sato, 1990.3.19
;; modified by Masakazu Takahashi, 1989.8.10
(defun j-newline (arg)
  (interactive "p")
  (if (> (minibuffer-depth) 0)
      (progn
	(j-kakutei)
	(setq j-mode nil)
	(exit-minibuffer))
    (if (j-kakutei) (setq arg (1- arg)))
    (j-emulate-original-map last-command-char arg)))

(defun j-delete-backward-char (count)
  (interactive "p")
  (if j-okurigana
      (save-excursion
	(goto-char j-okurigana-start-point)
	(if (looking-at "*") (delete-region (point) (1+ (point))))
	(setq j-okurigana nil)))
  (if (and j-henkan-on (= (point) j-henkan-start-point))
      (progn
	(setq j-henkan-count 0)
	(j-kakutei))
    (if (string= j-prefix "")
	(let ((p (if j-henkan-active (+ j-henkan-end-point 0) 0)))
	  (delete-backward-char count)
	  (if (and j-henkan-active (< (point) p)) (j-kakutei)))
      (if skk-echo
	  (j-erase-prefix (length j-prefix))
	(message "Deleted prefix \"%s\"." j-prefix))
      (setq j-prefix "")
      (move-marker j-kana-start-point nil))))

;; input by EUC/JIS code

(defun j-input-by-code ()
  (interactive)
  (let* ((str (read-string "JIS or EUC code: "))
	 (n1 (+ (* 16 (j-jis-string-to-hex (substring str 0 1)))
		(j-string-to-hex (substring str 1 2))))
	 (n2 (+ (* 16 (j-jis-string-to-hex (substring str 2 3)))
		(j-string-to-hex (substring str 3 4)))))
    (insert (concat (char-to-string n1) (char-to-string n2)))))

(defun j-string-to-hex (str)
  (let ((num (string-to-char str)))
  (cond ((> num 96) (- num 87)) ;; a-f
	((> num 64) (- num 55)) ;; A-F
	((> num 47) (- num 48)) ;; 0-9
	(t (error "")))))

(defun j-jis-string-to-hex (str)
  (let ((num (string-to-char str)))
  (cond ((> num 96) (- num 87)) ;; a-f
	((> num 64) (- num 55)) ;; A-F
	((> num 47) (- num 40)) ;; 0-9
	(t (error "")))))

;; henkan routines

(defun j-toggle-kana ()
  (interactive)
  (j-kakutei)
  ;(if j-henkan-active (j-kakutei))
  (setq j-katakana (not j-katakana))
  (j-change-mode-line (if j-katakana "--" "--")))

(defun j-set-henkan-point ()
  (interactive)
  (let* ((last-char (+ 32 last-command-char))
	 (sokuon (and (string= j-prefix (char-to-string last-char))
		      (not j-prefix-o))))
    (if (or (not j-henkan-on) j-henkan-active)
	(j-set-henkan-point-subr)
      (if (= j-henkan-start-point
	     (if skk-echo (- (point) (length j-prefix)) (point)))
	  ;; this case takes care of the rare case where one types
	  ;; two characters in upper case consequtively.  For example,
	  ;; one sometimes types "TE" when one should type "Te"
	  t ; do nothing
	;; prepare for the processing of okurigana if not j-okurigana
	(if (not j-okurigana)
	    (progn
	      ;; process pending "n" first
	      (if (string= j-prefix "n")
		  (apply (aref skk-map last-char) nil))
	      (if skk-process-okuri-early
		  (progn
		    (move-marker j-henkan-end-point (point))
		    (setq j-henkan-key
			  (concat (buffer-substring j-henkan-start-point
						    (if sokuon
							j-kana-start-point
						      (point)))
				  (if sokuon
				      (if j-katakana "" "")
				    "")
				  (char-to-string last-char)))
		    (if sokuon (j-erase-prefix 1))
		    (insert (if sokuon
				(if j-katakana "" "")
			      "")
			    " ")
		    (setq j-prefix "")
		    (j-henkan)
		    (delete-backward-char (if sokuon 2 1)))
		(if sokuon
		    (progn
		      (j-erase-prefix 1)
		      (insert (if j-katakana "" ""))
		      (setq j-prefix "")))
		(move-marker j-okurigana-start-point (point))
		(insert "*")
		(setq j-okurigana t))))))
    (setq unread-command-char last-char)))

(defun j-start-henkan (arg)
  (interactive "P")
  (if j-henkan-on
      (progn
	(setq j-mode t)
	(if j-henkan-active (j-next-candidate)
	  (j-process-pending-n)
	  (or (string= j-prefix "") (error ""))
	  (move-marker j-henkan-end-point (point))
	  (setq j-henkan-key (buffer-substring j-henkan-start-point
						j-henkan-end-point)
		j-henkan-vector (vector)
		j-henkan-vector1 (vector)
		j-henkan-vector2 (vector)
		j-henkan-vector3 (vector)
		j-henkan-vector4 (vector))
	  (insert " ")
	  (j-henkan)
	  (delete-backward-char 1)
	  (if (and j-abbrev j-henkan-active) (use-local-map skk-map))))
    (j-emulate-original-map last-command-char arg)))

(defun j-set-henkan-point-subr ()
  (interactive)
  (if j-henkan-active (j-kakutei))
  (insert "")
  (setq j-henkan-on t)
  (move-marker j-henkan-start-point (point)))

(defun j-change-marker ()
  (save-excursion
    (goto-char (- j-henkan-start-point 2))
    (if (looking-at "")
	(progn
	  (insert "")
	  (delete-char 1))
      (goto-char j-henkan-start-point)
      (insert "")
      (move-marker j-henkan-start-point (point))
      (message "It seems that you have deleted ."))))

(defun j-change-marker-to-white ()
  (save-excursion
    (goto-char (- j-henkan-start-point 2))
    (if (looking-at "")
	(progn
	  (insert "")
	  (delete-char 1))
      (goto-char j-henkan-start-point)
      (insert "")
      (move-marker j-henkan-start-point (point))
      (message "It seems that you have deleted ."))))

(defun j-search ()
  "search for KEY in *jisyo* buffer and create the vectors
of candidates."
  (let ((key j-henkan-key) (okuri-ari j-okurigana)
	(okurigana j-henkan-okurigana) (inhibit-quit t) l)
    (setq l
	  (save-excursion
	    (set-buffer " *jisyo*")
	    (goto-char (point-min))
	    (let ((w (concat key " /")) (kanji-flag nil)
		  (case-fold-search nil))
	      ;; kanji-flag must be set nil for efficiency.
	      (if (or (and (looking-at w) (search-forward w))
		      (search-forward (concat "\n" w) (point-max) t))
		  (j-compute-henkan-vectors)
		'([] [] [] [])))))
    (setq j-henkan-vector1 (nth 0 l)
	  j-henkan-vector2 (nth 1 l)
	  j-henkan-vector3 (nth 2 l)
	  j-henkan-vector4 (nth 3 l)
	  j-henkan-vector (j-vector-add j-henkan-vector3 j-henkan-vector1))))

(defun j-compute-henkan-vectors ()
  (let ((cont t) (stage 1)
	(v1 (vector)) (v2 (vector)) (v3 (vector)) (v4 (vector)))
    (while (and cont (not (eolp)))
      (let ((item (buffer-substring
		   (point)
		   (progn
		     (search-forward "/")
		     (1- (point))))))
	(cond ((not okuri-ari)
	       (if (and skk-process-okuri-early (= (aref item 0) ?\[))
		   (setq cont nil)
		 (setq v1 (vconcat v1 (vector item)))))
	      ((and (= (aref item 0) ?\[) (<= stage 2))
	       (if (string= item
			    (concat "\[" okurigana))
		   (progn
		     (setq stage 3)
		     (setq v2 (vconcat v2 (vector item))))
		 (setq stage 2)
		 (setq v2 (vconcat v2 (vector item)))))
	      ((and (= (aref item 0) ?\]) (= stage 3))
	       (setq stage 4)
	       (setq v4 (vconcat v4 (vector item))))
	      ((= stage 1)
	       (setq v1 (vconcat v1 (vector item))))
	      ((= stage 2)
	       (setq v2 (vconcat v2 (vector item))))
	      ((= stage 3)
	       (setq v3 (vconcat v3 (vector item))))
	      ((= stage 4)
	       (setq v4 (vconcat v4 (vector item)))))))
    (list v1 v2 v3 v4)))

(defun j-update-jisyo (word &optional purge)
  "update jisyo buffer so that WORD will become the first candidate
next time.  If PURGE is t, the word is  removed from the jisyo entry."
  (let ((vector j-henkan-vector)
	(vector1 j-henkan-vector1)
	(vector2 j-henkan-vector2)
	(vector3 j-henkan-vector3)
	(vector4 j-henkan-vector4)
	(okuri-ari (and (> (aref j-henkan-key 0) 127)
			(< (aref j-henkan-key
				 (1- (length j-henkan-key))) 128)))
	(okurigana j-henkan-okurigana)
	(key (concat j-henkan-key " /"))
	(inhibit-quit t))
    (save-excursion
      (set-buffer " *jisyo*")
      (goto-char (point-min))
      ;; delete old entry
      (if (or (and (looking-at key) (search-forward key))
	      (search-forward (concat "\n" key) (point-max) t))
	  (delete-region
	   (progn (beginning-of-line) (point))
	   (progn (end-of-line) (forward-char 1) (point))))
      ;; insert new entry
      (goto-char (point-min))
      (insert key (if (and purge (not okuri-ari)) "" (concat word "/")))
      (let ((index 0) (len (length vector1)))
	(while (< index len)
	  (let ((w (aref vector1 index)))
	    (or (string= word w)
		(insert w "/"))
	    (setq index (1+ index)))))
      (if (and okuri-ari
	       (not (memq (aref key (- (length key) 3)) '(?a ?i ?u ?e ?o))))
	  (progn
	    (let ((index 0) (len (length vector2)))
	      (while (< index len)
		(insert (aref vector2 index) "/")
		(setq index (1+ index))))
	    (if (equal vector3 [])
		(or purge (insert "\[" okurigana "/" word "/\]/"))
	      (let ((index 0) (len (length vector3)))
		(or purge (insert word "/"))
		(while (< index len)
		  (let ((w (aref vector3 index)))
		    (or (string= word w)
			(insert w "/"))
		    (setq index (1+ index))))))
	    (let ((index 0) (len (length vector4)))
	      (if (> len 0)
		  (let ((p (point)))
		    (save-excursion
		      (search-backward "/" nil nil 2)
		      (if (looking-at  "/\\[")
			  (progn
			    (delete-region (1+ (point)) p)
			    (setq index 1))))))
	      (while (< index len)
		(insert (aref vector4 index) "/")
		(setq index (1+ index))))))
      (if purge
	  (let ((uniq t) (index 0) (l2 (length vector2))
		(l4 (length vector4)))
	    (while (and uniq (< index l2))
	      (if (string= word (aref vector2 index))
		  (setq uniq nil)
		(setq index (1+ index))))
	    (while (and uniq (< index l4))
	      (if (string= word (aref vector4 index))
		  (setq uniq nil)
		(setq index (1+ index))))
	    (if uniq
		;; the WORD is not in vector3 nor in vector4, so purge it
		;; also from vector1 (if any)
		(save-excursion (let ((q (point)))
				  (goto-char (point-min))
				  (if (search-forward word q t)
				      (delete-region (1+ (point))
						     (progn
						       (search-backward word)
						       (point)))))))))
      (insert "\n")
      (goto-char (point-min))
      (search-forward " /")
      (if (eolp)
	  ;; the key has no words for it so delete everything
	  (delete-region (point-min) (1+ (point)))))))

(defun j-henkan ()
  "The main kana-to-kanji conversion routine."
  (save-excursion
    (or j-henkan-active
	(progn
	  (j-change-marker)
	  (setq j-henkan-active t)))
    (if (string= j-henkan-key "")
	(progn
	  (setq j-henkan-active nil)
	  (j-kakutei))
      (let (new-word (block nil) (found nil) (kakutei nil) (quit nil))
	(if (= j-henkan-count 0)
	    (progn
	      (j-search)
	      (if (and skk-server-host (= (length j-henkan-vector) 0))
		  (progn
		    (setq j-local-length 0)
		    (setq j-henkan-vector (j-search-server))
		    (setq j-server-called t)))
	      (setq j-search-called j-henkan-key)
	      (if (> (length j-henkan-vector) 0)
		  (setq found t
			j-henkan-count 1
			new-word (aref j-henkan-vector 0))))
	  (if (< j-henkan-count 4)
	      (if (< j-henkan-count (length j-henkan-vector))
		  (setq new-word (aref j-henkan-vector j-henkan-count)
			found t
			j-henkan-count (1+ j-henkan-count)))
	    ;; show candidates in minibuffer
	    (j-henkan-show-candidates)))
	(if (not quit)
	    (if (and skk-server-host (not found) (not j-server-called))
		(progn
		  ;; call skk-server
		  (setq j-local-length (length j-henkan-vector))
		  (setq j-henkan-vector
			(j-vector-add j-henkan-vector (j-search-server)))
		  (setq j-server-called t)
		  (j-henkan))
	      (if (or found
		      ;; use recursive minibuffer
		      (let ((save-start (copy-marker j-henkan-start-point))
			    (save-end (copy-marker j-henkan-end-point))
			    t-or-f)
			(setq t-or-f (j-henkan-in-minibuff))
			(move-marker j-henkan-start-point save-start)
			(move-marker j-henkan-end-point save-end)
			t-or-f))
		  (progn
		    (delete-region j-henkan-start-point j-henkan-end-point)
		    (goto-char j-henkan-start-point)
		    (insert new-word)
		    (if (equal j-kana-start-point j-henkan-end-point)
			(move-marker j-kana-start-point (point)))
		    (move-marker j-henkan-end-point (point))
		    (if kakutei
			(let ((save-prefix j-prefix))
			  ;; save and restore j-prefix after j-kakutei
			  (j-kakutei new-word)
			  (setq j-prefix save-prefix)))))))))))

(defun j-henkan-show-candidates ()
  "Show candidates in the minibuffer."
  (let ((cont t) (count 1) (l (length j-henkan-vector)))
    (setq block t found t)
    (delete-region j-henkan-start-point j-henkan-end-point)
    (let ((h-count j-henkan-count) (unread nil))
      (save-excursion
	(j-call-server)
	(if (= h-count l)
	    (setq cont nil found nil))
	(while cont
	  (j-call-server)
	  (if (= h-count l)
	      (setq cont nil)
	    (message "%s%s%s%s%s%s%s%s"
		     (concat "A:" (aref j-henkan-vector h-count))
		     (if (< (1+ h-count) l)
			 (concat "  S:" (aref j-henkan-vector (1+ h-count)))
		       "")
		     (if (< (+ h-count 2) l)
			 (concat "  D:" (aref j-henkan-vector (+ h-count 2)))
		       "")
		     (if (< (+ h-count 3) l)
			 (concat "  F:" (aref j-henkan-vector (+ h-count 3)))
		       "")
		     (if (< (+ h-count 4) l)
			 (concat "  J:" (aref j-henkan-vector (+ h-count 4)))
		       "")
		     (if (< (+ h-count 5) l)
			 (concat "  K:" (aref j-henkan-vector (+ h-count 5)))
		       "")
		     (if (< (+ h-count 6) l)
			 (concat "  L:" (aref j-henkan-vector (+ h-count 6)))
		       "")
		     (format "  [Ĥ %d%s]"
			     (1+ (- l (+ h-count
					 (setq count
					       (min 8 (1+ (- l h-count)))))))
			     (if (or j-server-called (not skk-server-host))
				 ""
			       "+")))
	    (setq h-count (+ h-count (1- count)))
	    (condition-case conditions
		(let ((char (read-char)))
		  (cond ((memq char '(?a ?s ?d ?f ?j ?k ?l))
			 (let ((c
				(cdr
				 (assoc char
					'((?a . 0) (?s . 1) (?d . 2) (?f . 3)
					  (?j . 4) (?k . 5) (?l . 6))))))
			   (if (> (+ c 2) count)
			       (progn
				 (message "`%c' is not valid here!" char)
				 (sit-for 1)
				 (message "")
				 (setq h-count
				       (- h-count (1- count))))
			     (setq h-count
				   (-
				    (+ c
				       (* 7
					  (/ (+ h-count 2) 7)))
				    2))
			     (setq new-word
				   (aref j-henkan-vector (1- h-count)))
			     (setq kakutei t cont nil))))
			((= char ?x)
			 (let* ((a (/ (- h-count 5) 7))
				(b (- h-count (+ 4 (* 7 a)))))
			   (if (= a 0)
			       (setq h-count (- h-count (1- b))
				     cont nil unread t)
			     (setq h-count (- (- h-count b) 7)))))
			((= char 32)
			 (if (= h-count l)
			     (setq cont nil found nil)))
			(t
			 (message "`%c' is not valid here!" char)
			 (sit-for 1)
			 (message "")
			 (setq h-count (- h-count (1- count))))))
	      (quit
	       (setq h-count 1 cont nil unread t))))))
      (if unread (setq unread-command-char ?x quit t))
      (setq j-henkan-count h-count))))

(defun j-call-server ()
  "This function is called from j-henkan-show-candidates."
  (if (and skk-server-host
	   (< (- l h-count) 7)
	   (not j-server-called))
      (progn
	(setq j-local-length (length j-henkan-vector))
	(setq j-henkan-vector
	      (j-vector-add j-henkan-vector (j-search-server)))
	(setq j-server-called t)
	(setq l (length j-henkan-vector)))))

(defun j-henkan-in-minibuff ()
  (setq new-word
	(let ((mode j-mode) r-word
	      (h-count j-henkan-count))
	  (condition-case conditions
	      (progn
		(save-excursion
		  (set-buffer
		   (get-buffer-create
		    (concat
		     " *Minibuf-"
		     (int-to-string (minibuffer-depth))
		     "*")))
		  (setq j-mode t))
		(setq r-word
		      (let ((unread-command-char ?\C-j)
			    (skk-mode t))
			(read-string
			 (concat (or j-henkan-key2 j-henkan-key) " ")))
		      j-mode mode))
	    (quit
	     (setq r-word "")))
	  (setq j-henkan-count h-count)
	  r-word))
  (if (string= new-word "")
      (cond ((> j-henkan-count 4)
	     (let* ((a (/ (- j-henkan-count 5) 7))
		    (b (- j-henkan-count (+ 4 (* 7 a)))))
	       (setq j-henkan-count (- j-henkan-count b))
	       (j-henkan)))
	    ((= j-henkan-count 0)
	     (setq j-henkan-active nil
		   j-henkan-key2 nil)
	     (let ((j-henkan-start-point save-start))
	       (j-change-marker-to-white)))
	    (t nil))
    (setq kakutei t)
    (setq j-count-touroku (1+ j-count-touroku))
    (setq j-henkan-count (if (= (length j-henkan-vector) 1)
			     1
			   (1+ j-henkan-count))))
  (not (string= new-word "")))

(defun j-next-candidate ()
  "Show next henkan candidate."
  (interactive)
  (if j-henkan-active
      (progn
	(insert " ")
	(j-henkan)
	(delete-backward-char 1))))

(defun j-previous-candidate ()
  "Show previous henkan candidate."
  (interactive)
  (if (and j-henkan-active (not (string= j-henkan-key "")))
      (progn
	(insert " ")
	(let ((word (concat j-henkan-key " ")) new-word
	      (found nil) p)
	  (save-excursion
	    (cond ((= j-henkan-count 1)
		   (setq j-henkan-active nil
			 j-henkan-count 0
			 j-okurigana nil
			 j-henkan-okurigana nil
			 j-henkan-key2 nil
			 j-server-called nil)
		   (delete-region j-henkan-start-point j-henkan-end-point)
		   (goto-char j-henkan-end-point)
		   (insert j-henkan-key)
		   (if (> (length j-prefix) 0)
		       (move-marker j-kana-start-point (point-marker)))
		   (setq p (point-marker))
		   (goto-char j-henkan-start-point)
		   (if (> (following-char) 127)
		       (progn
			 ;; roman prefix for okurigana is removed
			 (while (and (< (point) p) (> (following-char) 127))
			   (forward-char 1))
			 (delete-region (point) p))
		     (goto-char p))
		   (j-change-marker-to-white))
		  ((<= j-henkan-count 7)
		   (setq word (aref j-henkan-vector (- j-henkan-count 2)))
		   (delete-region j-henkan-start-point j-henkan-end-point)
		   (goto-char j-henkan-end-point)
		   (insert word)
		   (move-marker j-henkan-end-point (point))
		   (setq j-henkan-count (1- j-henkan-count)))
		  ((< 7 j-henkan-count)
		   (setq j-henkan-count (- j-henkan-count 8))
		   (setq unread-command-char 32) ;; 32 is space char
		   )))
	  (if (and (> (point) 1) (= (char-after (1- (point))) 32))
	      (delete-backward-char 1))
	  (if (= j-henkan-count 0)
	      (progn
		(goto-char p)
		(if j-abbrev (use-local-map skk-abbrev-map))))))
    (j-consonant)))

(defun j-kakutei (&optional word)
  "This function does what it says.  If WORD is given, it is used as the new
word for jisyo update."
  (interactive)
  (if (string= j-prefix "n") (j-insert-n))
  (let ((inhibit-quit t))
    (if skk-mode
	(progn
	  (j-change-mode-line (if j-katakana "--" "--"))
	  (use-local-map skk-map))
      ;; if skk-mode has not been called in this buffer, call it now.
      (skk-mode))
    (j-erase-prefix (length j-prefix))
    ;(set-marker j-kana-start-point nil)
    (setq j-mode t
	  j-prefix ""
	  j-prefix-o nil)
    (if j-henkan-on
	(progn
	  (if (and j-henkan-active (> j-henkan-count 0))
	      (progn
		(j-update-jisyo
		 (or word (aref j-henkan-vector (1- j-henkan-count))))
		(setq j-count-kakutei (1+ j-count-kakutei))))
	  (save-excursion
	    (if skk-process-okuri-early
		(progn
		  (goto-char j-henkan-end-point)
		  (if (looking-at "*") (delete-char 1))))
	    (goto-char (- j-henkan-start-point 2))
	    (if (or (looking-at "") (looking-at ""))
		(delete-char 1)
	      (message "It seems that you have deleted  or ."))
	    (if j-okurigana
		(progn
		  (goto-char j-okurigana-start-point)
		  (if (looking-at "*") (delete-char 1))
		  (setq j-okurigana nil))))
	  (setq j-abbrev nil)))
    (setq j-henkan-active nil
	  j-henkan-on nil
	  j-okurigana nil
	  j-henkan-key2 nil
	  j-henkan-okurigana nil
	  j-henkan-count 0
	  j-server-called nil)
;modified by Masakazu Takahashi, 1989.8.9
    (if j-auto-fill (j-kanji-fill))))

;; completion

(defun j-completion (first)
  (let (c-word (cont t))
    (if first
	(setq j-completion-word (buffer-substring j-henkan-start-point (point))
	      j-completion-list (list j-completion-word)))
    (let ((inhibit-quit t))
      (save-excursion
	(set-buffer " *jisyo*")
	(if first (goto-char (point-min)))
	(let ((kanji-flag nil) (case-fold-search nil))
	  ;; kanji-flag must be set nil for efficiency.
	  (while (and cont
		      (or (and (looking-at j-completion-word)
			       (search-forward j-completion-word))
			  (search-forward (concat "\n" j-completion-word)
					  (point-max) t)))
	    (if (not (looking-at " /"))
		(progn
		  (setq p (point))
		  (search-forward " /")
		  (setq c-word (concat j-completion-word
				       (buffer-substring p (- (point) 2))))
		  (setq cont nil)))))))
    (if cont
	(progn
	  (message "No %scompletion for \"%s\""
		   (if first "" "more ") j-completion-word)
	  (ding)))
    (if c-word
	(progn
	  (delete-region j-henkan-start-point (point))
	  (insert c-word)))))

(defun j-try-completion (arg)
  (interactive "P")
  (if (and j-henkan-on (not j-henkan-active))
      (progn
	(setq this-command 'j-completion)
	(j-completion t))
    (j-emulate-original-map last-command-char arg)))

(defun j-emulate-original-map (char arg)
  (let ((prefix-arg arg))
    (command-execute
     (j-get-editor-command j-emacs-local-map global-map char))))

(defun j-previous-completion ()
  (let ((inhibit-quit t) (found nil))
    (save-excursion
      (set-buffer " *jisyo*")
      (beginning-of-line)
      (let ((kanji-flag nil) (case-fold-search nil) (cont t))
	;; kanji-flag must be set nil for efficiency.
	(while (and cont
		    (not (bobp))
		    (or (search-backward (concat "\n" j-completion-word)
					 (point-min) t)
			(progn
			  (goto-char (point-min))
			  (looking-at j-completion-word))))
	  (search-forward j-completion-word)
	  (if (not (looking-at " /"))
	      (progn
		(setq p (point))
		(search-forward " /")
		(setq c-word (concat j-completion-word
				     (buffer-substring p (- (point) 2))))
		(setq cont nil))
	    (beginning-of-line)))
	(if (not cont) (setq found t))))
    (delete-region j-henkan-start-point (point))
    (if found
	(insert c-word)
      (insert j-completion-word)
      (message "No more previous completion.")
      (ding))
    (setq this-command 'j-completion)))

(defun j-purge-from-jisyo ()
  "Purge wrong words from jisyo buffer."
  (interactive)
  (if (and j-henkan-active (not (string= j-henkan-key ""))
	   (yes-or-no-p "Really purge? "))
      (progn
	(j-update-jisyo (aref j-henkan-vector (1- j-henkan-count)) t)
	(delete-region j-henkan-start-point j-henkan-end-point)
	(setq j-henkan-active nil)
	(j-kakutei))
    (j-set-henkan-point)))

;; skk server related functions

(defun j-open-server ()
  "Establish connection with skk server."
  (or (get-process "skkservd")
      (let ((succ nil))
	(get-buffer-create " *skkwork*")
	(if (j-open-server-subr)
	    (setq succ t)
	  (progn
	    (let ((count 5))
	      (while (> count 0)
		(message "SKK SERVER is not active, I will activate it%s"
			 (make-string count ?.))
		(if (string= skk-server-host (system-name))
		    ;; server host is local machine
		    (call-process skk-serv nil 0 nil)
		  (call-process
		   "/usr/ucb/rsh" nil 0 nil skk-server-host skk-serv))
		(sit-for 3)
		(if (j-open-server-subr)
		    (setq succ t
			  count 0)
		  (setq count (1- count)))))
	    (if succ
		(progn
		  (message "SKK SERVER is active now.")
		  (sit-for 1)))))
	(if succ
	    (save-excursion
	      ;; If your Nemacs is an old one, it may not have the function
	      ;; skk-process-kanji-code.
	      (if (fboundp 'set-process-kanji-code)
		  (set-process-kanji-code (get-process "skkservd") 0))
	      (set-buffer " *skkwork*")
	      (setq kanji-fileio-code 0) ;; for old Nemacs
	      )
	  (message "Could not activate SKK SERVER.")
	  (sit-for 1)
	  ;; reset skk-server-host so as not to use server in this session
	  (setq skk-server-host nil)
	  (ding)))))

(defun j-open-server-subr ()
  (condition-case nil
      (open-network-stream
       "skkservd" " *skkwork*" skk-server-host (or skk-portnum "skkserv"))
    (error
     nil)))

(defun j-search-server ()
  "search for KEY through skkserver and create the vector of candidates."
  (let ((key j-henkan-key) (okuri-ari j-okurigana)
	(okurigana j-henkan-okurigana) (v (vector)))
    (save-excursion
      (set-buffer " *skkwork*")
      (erase-buffer)
      (or (process-status "skkservd") (j-open-server))
      (if (process-status "skkservd")
	  (let ((cont t) (count 0))
	    (process-send-string "skkservd" (concat "1" key " "))
	    (while (and cont (memq (process-status "skkservd") '(open run)))
	      (accept-process-output)
	      (setq count (1+ count))
	      (if (> (buffer-size) 0)
		  (progn
		    (if (= ?1 (char-after 1))
			;; found key successfully, so check if a whole line
			;; is received.
			(if (= ?\n (char-after (1- (point-max))))
			    (setq cont nil))
		      ;; not found or error, so exit
		      (setq cont nil)))))
	    (goto-char (point-min))
	    (if skk-report-server-response
		(message "waited for server response %d times." count))
	    (if (looking-at "1")
		(progn
		  (forward-char 2)
		  (let ((l (j-compute-henkan-vectors)))
		    (j-vector-add (nth 0 l) (nth 2 l))))
	      ;; not found or error
	      []))
	[]))))

(defun j-vector-add (u v)
  "Concatenate U and elements of V not in U."
    (let ((w u) (lv (length v)) (i 0))
      (while (< i lv)
	(let ((e (aref v i)))
	  (or (j-vector-mem e u)
	      (setq w (vconcat w (vector e))))
	  (setq i (1+ i))))
      w))

(defun j-vector-mem (e v)
  "Test if the string E is a member of the vector V."
  (let ((l (length v)) (i 0) (cont t))
    (while (and cont (< i l))
      (if (string= e (aref v i)) (setq cont nil) (setq i (1+ i))))
    (not cont)))
