;; x-mycar.el --- insert X-MyCar field

;; Copyright (C) 1997 Yuuichi Teranishi

;; Author: Yuuichi Teranishi <teranisi@isl.ntt.co.jp>
;; Maintainer: Yuuichi Teranishi <teranisi@isl.ntt.co.jp>
;; Version: 0.08
;; Created: 28 Oct 1997
;; Date: 4 Nov 1997

;; X-MyCar is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or
;; any later version.

;; X-MyCar is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;
;; Example Setup:
;;
;; (require 'x-mycar)
;; (setq x-mycar-name "EUNOS 500")
;; (setq x-mycar-since "Nov.1997")
;; 
;; Distance editor:
;; (define-key global-map "\C-x\C-m" 'x-mycar-edit)
;; For used car:
;; (setq x-mycar-start-distance 15000)
;; For Mew: 
;; (add-hook 'mew-draft-mode-hook 'x-mycar)
;; For tm-edit:
;; (add-hook 'mime/editor-mode-hook 'x-mycar)
;; (setq x-mycar-field-insert-before "X-Mailer:")

(defconst x-mycar-appname "x-mycar")
(defconst x-mycar-version-number "v0.08")
(defconst x-mycar-codename
;;;; 80's Pops series ;;;;
;; "Africa"
;; "Beat It"
;; "Careless Wisper"
;; "Dirty Diana"
;; "Ebony And Ivory"
;; "Freedom"
;; "Girls On Film"
"Human Nature"
;; "I'm Your Man"
;; "Joanna"
;; "Kyrie"
;; "Last Christmas"
;; "Material Girl"
;; "Notorious"
;; "On My Own"
;; "Physical"
;; "Rock Me Amadeus"
;; "Set On You"
;; "Thriller"
;; "Up Town Girl"
)

(defconst x-mycar-version 
  (concat x-mycar-appname " "
	  x-mycar-version-number " - \""
	  x-mycar-codename "\""))

(provide 'x-mycar)
(require 'time-stamp)
(if (featurep 'xemacs)
    (require 'highlight-headers))

(defvar x-mycar-name "RAV4J"
  "*your car name.")
(defvar x-mycar-since "Nov.1995"
  "*when you start to use your car.")
(defvar x-mycar-distance-file "~/.mycar"
  "*distance filename.")
(defvar x-mycar-start-distance 0
  "*for used car.")
(defvar x-mycar-field-insert-before "Mime-Version:"
  "*where X-MyCar field inserted")
(defvar x-mycar-oil-exchange-threshold 5000
  "*the distance user should exchange oil.")
(defvar x-mycar-field-name "X-MyCar"
  "*field name for X-MyCar. 
If you are a biker, you can set this as X-Bike, X-MyBike...")
(defvar x-mycar-show-total nil
  "*print total distance too.")

(defvar x-mycar-last-oil-exchanged-distance 0)
(defvar x-mycar-edit-height 6)
(defconst x-mycar-edit-buffer "X-Mycar-Edit")
(defvar x-mycar-add-version nil)

(defvar x-mycar-edit-mode-map nil)
(defvar x-mycar-orig-window-config nil)

(if x-mycar-edit-mode-map
    nil
  (setq x-mycar-edit-mode-map (make-sparse-keymap))
  (define-key x-mycar-edit-mode-map [up]    'x-mycar-edit-up)
  (define-key x-mycar-edit-mode-map "p"     'x-mycar-edit-up)
  (define-key x-mycar-edit-mode-map "\C-p"  'x-mycar-edit-up)
  (define-key x-mycar-edit-mode-map [down]  'x-mycar-edit-down)
  (define-key x-mycar-edit-mode-map "n"     'x-mycar-edit-down)
  (define-key x-mycar-edit-mode-map "\C-n"  'x-mycar-edit-down)
  (define-key x-mycar-edit-mode-map [right] 'x-mycar-edit-forward)
  (define-key x-mycar-edit-mode-map "f"     'x-mycar-edit-forward)
  (define-key x-mycar-edit-mode-map "\C-f"  'x-mycar-edit-forward)
  (define-key x-mycar-edit-mode-map [left]  'x-mycar-edit-backward)
  (define-key x-mycar-edit-mode-map "b"     'x-mycar-edit-backward)
  (define-key x-mycar-edit-mode-map "\C-b"  'x-mycar-edit-backward)
  (define-key x-mycar-edit-mode-map "\C-m"  'x-mycar-edit-save-exit)
  (define-key x-mycar-edit-mode-map "q"     'x-mycar-edit-exit)
  (define-key x-mycar-edit-mode-map "o"     'x-mycar-edit-toggle-oil)
  (define-key x-mycar-edit-mode-map "\C-g"  'x-mycar-edit-exit)
  (define-key x-mycar-edit-mode-map "\e<"   'x-mycar-edit-top)
  (define-key x-mycar-edit-mode-map "\e>"   'x-mycar-edit-bottom)
  )

(defun x-mycar-edit-toggle-oil ()
  "toggle oil"
  (interactive)
  (save-excursion
    (let ((buffer-read-only nil)
	  (new-oil-distance 
	   (string-to-int (read-from-minibuffer 
			   "Oil last exchanged(km): " 
			   (int-to-string 
			    (x-mycar-edit-get-distance)))))
	  )
      (x-mycar-edit-save-oil-distance new-oil-distance)
      (goto-char (point-min))
      (re-search-forward "Oil: .*$" nil t)
      (delete-region (match-beginning 0) (match-end 0))
      (insert (format "Oil: last exchanged %d km"
		      new-oil-distance
		      ))
      (if (> (- (x-mycar-edit-get-distance)
		new-oil-distance)
	     x-mycar-oil-exchange-threshold)
	  (insert " ... YOU'D BETTER EXCHANGE!!")))))

(defun x-mycar-edit-get-distance ()
  (save-excursion
  (goto-char (point-min))
  (re-search-forward "Distance: \\([0-9]+\\)")
  (string-to-int 
   (buffer-substring (match-beginning 1) (match-end 1)))))

(defun x-mycar-edit-save-oil-distance (oil-distance)
  (interactive)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create "*x-mycar-save*")))
      (set-buffer tmp-buffer)
      (if (file-exists-p x-mycar-distance-file)
	  (insert-file-contents x-mycar-distance-file))
      (goto-char (point-min))
      (insert (int-to-string oil-distance))
      (insert " Oil\n")
      (write-region (point-min) (point-max) x-mycar-distance-file)
      (kill-buffer tmp-buffer))))

(defun x-mycar-edit-insert ()
  "Insert x-mycar field. Replace if already exists."
  (save-window-excursion
    (delete-window)
    (x-mycar 'replace)))

(defun x-mycar-edit-save-exit ()
  (interactive)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create "*x-mycar-save*"))
	  (distance 
	   (int-to-string (x-mycar-edit-get-distance)))
	  (oil-exchanged (re-search-forward "Oil: exchanged" nil t))
	  (last-distance "0")
	  )
      (set-buffer tmp-buffer)
      (if (file-exists-p x-mycar-distance-file)
	  (insert-file-contents x-mycar-distance-file))
      (goto-char (point-min))
      (if (looking-at "\\([0-9]+\\).*$")
	  (setq last-distance 
		(buffer-substring (match-beginning 1) (match-end 1))))
      (if (and (string= last-distance distance)
	       (not oil-exchanged))
	  (message "Meter distance was not modified.")
	(insert distance)
	(insert (concat " " (time-stamp-yyyy/mm/dd) 
			"\n"))
	(write-region (point-min) (point-max) x-mycar-distance-file))
      (kill-buffer tmp-buffer)
      (x-mycar-edit-insert)
      (x-mycar-edit-exit)
      )
    )
  )

(defun x-mycar-edit-exit ()
  (interactive)
  (kill-buffer x-mycar-edit-buffer)
  (set-window-configuration x-mycar-orig-window-config))

(defun x-mycar-edit-up ()
  (interactive)
  (x-mycar-edit-forward)
  (x-mycar-edit-backward)
  (save-excursion
    (let ((val (string-to-int (buffer-substring (point) (+ 1 (point)))))
	  (buffer-read-only nil)
	  )
      (delete-char 1)
      (if (eq val 9)
 	  (insert-char ?0 1)
	(insert-string (int-to-string (+ val 1)))
	)
      )
    ))

(defun x-mycar-edit-down ()
  (interactive)
  (x-mycar-edit-forward)
  (x-mycar-edit-backward)
  (save-excursion
    (let ((val (string-to-int (buffer-substring (point) (+ 1 (point)))))
	  (buffer-read-only nil)
	  )
      (delete-char 1)
      (if (eq val 0)
	  (insert-char ?9 1)
	(insert-string (int-to-string (- val 1)))
	)
      )
    ))

(defun x-mycar-edit-top ()
  (interactive)
  (goto-char (point-min))
  (re-search-forward "\\(Distance: \\)")
  )

(defun x-mycar-edit-bottom ()
  (interactive)
  (goto-char (point-min))
  (re-search-forward "\\(Distance: \\)")
  (goto-char (+ (point) 5))
  )

(defun x-mycar-edit-forward ()
  (interactive)
  (let ((cur (point))
	start-point)
    (goto-char (point-min))
    (re-search-forward "\\(Distance: \\)")
    (setq start-point (match-end 1))
    (if (or
	 (< cur start-point) 
	 (>= (- cur start-point) 5))
	(goto-char start-point)
      (goto-char (+ cur 1)))))

(defun x-mycar-edit-backward ()
  (interactive)
  (let ((cur (point))
	start-point)
    (goto-char (point-min))
    (re-search-forward "\\(Distance: \\)")
    (setq start-point (match-end 1))
    (if (or 
	 (<= cur start-point)
	 (> (- cur start-point) 5))
	(goto-char (+ start-point 5))
      (goto-char (- cur 1)))))

(defun x-mycar-edit-mode ()
  "Major mode for X-MyCar Information."
  (interactive)
  (setq major-mode 'x-mycar-edit-mode)
  (setq mode-name "X-MyCar")
  (use-local-map x-mycar-edit-mode-map)
  )

(defun x-mycar-edit ()
  (interactive)
  (setq x-mycar-orig-window-config (current-window-configuration))
  (if (get-buffer x-mycar-edit-buffer)
      (pop-to-buffer x-mycar-edit-buffer)
    (let* (meter) 
      (pop-to-buffer x-mycar-edit-buffer)
      (enlarge-window (- x-mycar-edit-height (window-height)))
      (insert (concat " " (user-full-name) "'s my car information\n"))
      (insert (concat "       Car Name: " x-mycar-name "\n"))
      (insert (concat "          Since: " x-mycar-since))
      (if (eq x-mycar-start-distance 0)
	  nil
	(insert (format " (%skm)" 
			x-mycar-start-distance)))
      (insert "\n")
      (setq meter (string-to-int
		   (if (file-exists-p x-mycar-distance-file)
		       (x-mycar-get-distance t)
		     "0")))
      (insert (format " Meter Distance: %06d (km)" meter))
      (insert (format "\n            Oil: last exchanged %d km"
		      x-mycar-last-oil-exchanged-distance
		      ))
      (if (> (- meter x-mycar-last-oil-exchanged-distance) 
	     x-mycar-oil-exchange-threshold)
	  (insert " ... YOU'D BETTER EXCHANGE!!"))
      (x-mycar-edit-bottom)
      (toggle-read-only)))
  (x-mycar-edit-mode))

(defun x-mycar-get-last-oil-exchanged-distance ()
  "get last oil exchanged distance from current buffer."
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (if (re-search-forward "Oil" nil t)
	(progn
	  (beginning-of-line)
	  (if (looking-at "\\([0-9]+\\).*$")
	      (string-to-int 
	       (buffer-substring (match-beginning 1) (match-end 1)))
	    0)
	  )
      0)))

(defun x-mycar-make-number-string (number)
  "create number string with \",\"."
  (if (>= number 1000)
      (format "%d,%03d" (/ number 1000) (% number 1000))
    (int-to-string number)))

(defun x-mycar-get-distance (meter)
  "guess distance"
  (save-excursion
    (let ((tmp-buffer (get-buffer-create "*x-mycar*"))
	  distance meter-distance
	  )
      (set-buffer tmp-buffer)
      (erase-buffer)
      (insert-file-contents x-mycar-distance-file)
      (setq x-mycar-last-oil-exchanged-distance
	    (x-mycar-get-last-oil-exchanged-distance))
      (goto-char (point-min))
      (while (looking-at "\\([0-9]+\\).* Oil$")
	(forward-line 1))
      (if (looking-at "\\([0-9]+\\).*$")
	  (progn 
	    (setq meter-distance 
		  (buffer-substring (match-beginning 1) (match-end 1)))
	    (kill-buffer tmp-buffer)
	    (setq distance (- (string-to-int meter-distance) 
			      x-mycar-start-distance))
	    (if meter
		meter-distance 
	      (cons 
	       (x-mycar-make-number-string distance)
	       (x-mycar-make-number-string (string-to-int meter-distance)))
	      ))
	(kill-buffer tmp-buffer)
	(if meter "0" (cons "0" "0"))
	))))

(defun x-mycar-delete-x-mycar-field ()
  "delete X-MyCar: field. returns t if X-MyCar: field already exists."
  (save-excursion
    (let ((ret-val nil))
      (goto-char (point-min))
      (while (re-search-forward 
	      (format "^\\(%s:\\) *\\(.*\\(\n[ \t].*\\)*\\)\n" 
		      x-mycar-field-name) nil t)
	(progn 
	  (delete-region (match-beginning 0) (match-end 0))
	  (setq ret-val t);; found!
	  ))
      ret-val)))

(defun x-mycar (&optional replace)
  "insert X-MyCar: field."
  (interactive)
  (if (and (not (x-mycar-delete-x-mycar-field)) ;; no X-MyCar field.
	   replace)
      ()
    (save-excursion
      (let ((distance-on (file-exists-p x-mycar-distance-file))
	    data)
	(goto-char (point-min))
	(re-search-forward x-mycar-field-insert-before nil t)
	(beginning-of-line)
	(insert "\n")
	(previous-line 1)
	(if distance-on 
	    (progn 
	      (setq data (x-mycar-get-distance nil))
	      (insert (car data))))
	(beginning-of-line)
	(let ((beg (point)))
	  (insert (format "%s: " x-mycar-field-name))
	  (insert x-mycar-name)
	  (insert " (")
	  (end-of-line)
	  (insert (concat (if distance-on "km ") 
			  "since " x-mycar-since))
	  (if (and distance-on 
		   x-mycar-show-total)
	      (insert (format "; %skm total" (cdr data))))
	  (insert ")")
	  (if x-mycar-add-version
	      (progn
		(insert "/")
		(insert x-mycar-version))))))))
    
