;;; tbl-mg.el --- Table Manager

;; Copyright (C) 1997 Miyashita Hisashi

;; Keywords: mule, multilingual, table, CCL

;; This file is part of MULE-UCS

;; MULE-UCS is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; MULE-UCS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; This module manages tables for translation.
;;; This combines some tables to a table set that a unit of translation.

(require 'mucs)
(require 'trans-util)

(defun get-table-set-table-name (table-set number &optional decodep)
  "Return a name of a symbol used to register a ccl translation table,
which is managed by TABLE-SET."
  (let ((nam (if decodep "decode" "encode")))
    (format "%s-table-%s-%d" (symbol-name table-set) nam number)))

(defvar tbl-mg-temporary-table-postfix "tmp-table-"
  "Use this postfix to make a new symbol
for specifying a temporary table.")

(defvar tbl-mg-temporary-table-set-postfix "tmp-table-set-"
  "Use this postfix to make a new symbol
for specifying a temporary table set.")

(defvar default-max-codepoint-table-gap 256
  "Default max length of gap in a code point table.")

;;;
;;; Table package registration manager.
;;;

(defun mucs-table-registered-p (table)
  (memq table (get mucs-current-package
		   'mucs-registered-tables)))

(defun mucs-add-table-registration (table)
  (if (and mucs-current-package
	   (symbolp mucs-current-package))
      (or
       (mucs-table-registered-p table)
       (put mucs-current-package
	    'mucs-registered-tables
	    (cons table
		  (get mucs-current-package
		       'mucs-registered-tables))))))

(defun mucs-delete-table-registration (table)
  (if (and mucs-current-package
	   (symbolp mucs-current-package))
      (put mucs-current-package
	   'mucs-registered-tables
	   (delq
	    (get mucs-current-package
		 'mucs-registered-tables)))))
  
;;;
;;; Table set manager.
;;;
;; tableset is used for combining some tables that means certain unit
;; for translation.
;; the entity of tableset is a symbol that have 'table-set key,
;; whose property is a list of tables.
;;

(defun put-table-set-encode-tables (table-set tables)
  (put table-set 'table-set-encode-tables tables))

(defun put-table-set-encode-method (table-set func)
  "Put a translation method to modify table set encoding.
FUNC is called, when encode table is generated, by passing
(CHARACTER . RESULT). "
  (put table-set 'table-set-encode-method func))

(defun put-table-set-decode-tables (table-set tables)
  (put table-set 'table-set-decode-tables tables))

(defun put-table-set-decode-method (table-set func)
  "Put a translation method to modify table set decoding.
FUNC is called, when decode table is generated, by passing
(CHARACTER . RESULT). "
  (put table-set 'table-set-encode-method func))

(defun get-table-set-encode-method (table-set)
  (get table-set 'table-set-encode-method))

(defun get-table-set-decode-method (table-set)
  (get table-set 'table-set-decode-method))

(defun get-table-set-modified-alist (table-set decodep)
  (let ((alist
	 (eval (get-table-set-alist table-set)))
	(map-func
	 (if decodep
	     (get-table-set-decode-method table-set)
	   (get-table-set-encode-method table-set))))
    (if (null map-func)
	(copy-sequence alist)
      (mapcar map-func alist))))

(defun get-table-set-encode-tables (table-set)
  (or (get table-set 'table-set-encode-tables)
      (put-table-set-encode-tables
       table-set
       (make-translation-tables
	(get-table-set-modified-alist table-set nil)
	nil nil (get table-set 'table-set-max-gap)))))

(defun get-table-set-decode-tables (table-set)
  (or (get table-set 'table-set-decode-tables)
      (put-table-set-decode-tables
       table-set
       (make-translation-tables
	(get-table-set-modified-alist table-set t)
	t nil (get table-set 'table-set-max-gap)))))

(defun get-table-set-tables (table-set decodep)
  (let ((i 1) table-sym)
    (mapcar
     (lambda (x)
       (setq table-sym
	     (intern (get-table-set-table-name table-set i decodep)))
       (register-ccl-translation-table table-sym x)
       (setq i (1+ i))
       table-sym)
     (if decodep
	 (get-table-set-decode-tables table-set)
       (get-table-set-encode-tables table-set)))))

(defun put-table-set-alist (table-set alist-val-symbol)
  (put table-set 'table-set-alist alist-val-symbol))

(defun get-table-set-alist (table-set)
  (get table-set 'table-set-alist))

(defun table-set-p (table-set)
  (get table-set 'table-set-alist))

;;;
;;; Table creater
;;;



(defmacro define-ccl-codepoint-translation-table (symbol &rest args)
  `(let ((vector ,(apply 'make-codepoint-vector args)))
     (register-ccl-translation-table ,symbol vector)
     vector))

(defmacro define-ccl-identity-translation-table (symbol start len)
  `(let ((vector ,(make-identity-translation-vector start len)))
     (register-ccl-translation-table ,symbol vector)
     vector))

(defmacro define-ccl-slide-translation-table (symbol start-s start-d len)
  `(let ((vector ,(make-slide-translation-vector start-s start-d len)))
     (register-ccl-translation-table ,symbol vector)
     vector))

(defmacro define-ccl-constant-translation-table (symbol start-s constant len)
  `(let ((vector ,(make-constant-translation-vector start-s constant len)))
     (register-ccl-translation-table ,symbol vector)
     vector))

(defun make-codepoint-vector (&rest args)
  "Return a vector of codepoints of given characters.
Each argument is a character or t or nil or lambda or string.
String must be an expression that is evaled into number."
  (let ((arg args) elem elem2
	table len vector)
    (while arg
      (setq elem (car arg))
      (cond ((numberp elem)
	     (setq table (cons (char-codepoint elem) table)))
	    ((or (eq elem t)
		 (eq elem 'lambda)
		 (null elem))
	     (setq table (cons elem table)))
	    ((stringp elem)
	     (setq elem2 (read elem))
	     (if (numberp elem2)
		 (setq table (cons elem2 table))
	       (error "Invalid argument %s" elem)))
	    (t
	     (error "Invalid argument %s" elem)))
      (setq arg (cdr arg)))
    (setq len (length table)
	  vector (make-vector len nil)
	  arg table)
    (while (> len 0)
      (setq len (1- len))
      (aset vector len (car arg))
      (setq arg (cdr arg)))
    vector))

(defun make-identity-translation-vector (start len)
  (vector t t start (+ start len -1)))

(defun make-slide-translation-vector (start-s start-d len)
  (setq len (1+ len))
  (let ((vector (make-vector len 0))
	(i 1))
    (aset vector 0 start-s)
    (while (< i len)
      (aset vector i start-d)
      (setq start-s (1+ start-s)
	    start-d (1+ start-d)
	    i (1+ i)))
    vector))

(defun make-constant-translation-vector (start-s constant len)
  (vector t constant start-s (+ start-s len -1)))

(defsubst get-table-key (cell decodep)
  (if decodep (cdr cell) (car cell)))

(defsubst get-table-val (cell decodep)
  (if decodep (car cell) (cdr cell)))

(defun make-translation-tables (alist decodep copyp &optional max)
  "Make translation tables.
When DECODEP is non-nil, make tables to decode.
Wehn COPYP is non-nil, copy alist not to destroy ALIST."
  (if (null max) (setq max default-max-codepoint-table-gap))
  (let* ((alist-copy (sort
		      (if copyp
			  (copy-sequence alist)
			alist)
		      (if decodep
			  (lambda (x y) (< (cdr x) (cdr y)))
			(lambda (x y) (< (car x) (car y))))))
	 (curll alist-copy)
	 (stll alist-copy)
	 (ctll alist-copy)
	 (stp (get-table-key (car stll) decodep))
	 (ctp stp)
	 veclist
	 vec
	 curp
	 curle)
    (while ctll
      (setq curle (car curll)
	    curp (get-table-key curle decodep))
      (if (and curll
	       (or
		(eq max t)
		(<= (- curp ctp) max)))
	  (setq ctp curp
		ctll curll)
	(setq vec (make-vector (- ctp stp -2) nil))
	(aset vec 0 stp)
	(while 
	    (prog2
		(aset vec 
		      (- (get-table-key (car stll) decodep) stp -1)
		      (get-table-val (car stll) decodep))
		(not (eq stll ctll))
	      (setq stll (cdr stll))))
	(setq veclist (cons vec veclist)
	      ctll curll
	      stll curll
	      stp (get-table-key (car stll) decodep)
	      ctp stp))
      (setq curll (cdr curll)))
    (nreverse veclist)))

(defun compile-codepoint-alist-to-vector (table &optional cont func)
)

(defun merge-codepoint-vector (table11 table2)
)

;;;
;;; Table registration programs
;;;

(defun generate-table-registration-program (tables)
  (let (table-alist table)
    (while (setq table (car tables))
      (if (mucs-table-registered-p table)
	  nil
	(setq table-alist
	      (cons
	       (cons table (get table 'ccl-translation-table))
	       table-alist))
	(mucs-add-table-registration table))
      (setq tables (cdr tables)))
    (if (null table-alist)
	nil
      `(let ((tbls (quote ,table-alist))
	     tbel)
	 (while (setq tbel (car tbls))
	   (register-ccl-translation-table
	    (car tbel)
	    (cdr tbel))
	   (setq tbls (cdr tbls)))))))

(defun generate-meta-table-registration-program (symbol table)
  (register-ccl-translation-table symbol table)
  (if (mucs-table-registered-p symbol)
      nil
    (mucs-add-table-registration symbol)
    `(let ((tbl ,table)
	   (i 1)
	   (j (length tbl))
	   elem id)
       (while (< i j)
	 (setq elem (aref tbl i))
	 (if (setq id (get 'ccl-translation-table-id elem))
	     (aset tbl i id))
	 (setq i (1+ i)))
       (register-ccl-translation-table ,symbol tbl))))

(provide 'tbl-mg)
