;;; mew-scan.el --- Scanning messages for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Oct  2, 1996

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Scan info
;;;

(defvar mew-scan-info-list '("folder" "message"))
;; See mew-scan-fields. 0th is fld, 1st is msg (ie num).

(mew-info-defun "mew-scan-" mew-scan-info-list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Variables
;;;

(defvar mew-summary-form-size-unit '("" "k" "M" "G" "T"))
(defvar mew-vec [0 1 2 3 4 5 6 8 9 10 11 12 13 14 15 16 17 18 19 20]
  "Just for test of (MEW-FOO).")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Setup
;;;

(defun mew-scan-setup ()
  "Define functions (MEW-FOO) according 'mew-scan-fields-alias'."
  (let ((n (length mew-scan-fields-alias))
	(i 0))
    (while (< i n)
      (fset (intern (concat "MEW-" (nth i mew-scan-fields-alias)))
	    `(lambda () (aref mew-vec ,i)))
      (setq i (1+ i)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Pre-defined functions for mew-summary-form
;;;

(defun mew-summary-form-mark ()
  "A function to return a mark.
'mew-summary-form-mark-delete' and 'mew-summary-form-mark-review'
effect to this function."
  (let ((mark-delete mew-mark-delete)
	(mark-review mew-mark-review)
	(unread (and mew-use-unread-mark (mew-sinfo-get-unread-mark)))
	duplicated review id)
    (when mew-summary-form-mark-delete
      (if (mew-characterp mew-summary-form-mark-delete)
	  (setq mark-delete mew-summary-form-mark-delete))
      (when (setq id (mew-idstr-get-first-id (MEW-ID)))
	(if (member id (mew-sinfo-get-scan-id))
	    (unless (mew-scan-message-invalidp)
	      (setq duplicated t))
	  (mew-sinfo-set-scan-id (cons id (mew-sinfo-get-scan-id))))))
    (when mew-summary-form-mark-review
      (if (mew-characterp mew-summary-form-mark-review)
	  (setq mark-review mew-summary-form-mark-review))
      (let* ((mew-header-max-depth nil)
	     (to (mew-addrstr-parse-address-list (MEW-TO)))
	     (cc (mew-addrstr-parse-address-list (MEW-CC))))
	(setq to (nconc to cc))
	(catch 'loop
	  (while to
	    (if (mew-is-my-address mew-regex-my-address-list (car to))
		(throw 'loop (setq review t)))
	    (setq to (cdr to))))))
    (cond
     (duplicated (char-to-string mark-delete))
     (review     (char-to-string mark-review))
     (unread     (char-to-string mew-mark-unread))
     (t          (char-to-string mew-mark-read)))))

(defun mew-summary-form-type ()
  "A function to return a mark of content type."
  (let ((ct (MEW-CT))
	(case-fold-search t))
    (cond
     ((mew-scan-message-invalidp)                  "#")
     ((mew-scan-message-truncatedp)                "T")
     ((string-match "Multipart/Signed"    ct)      "S")
     ((string-match "Multipart/Encrypted" ct)      "E")
     ((string-match "Application/X-Pkcs7-Mime" ct) "E")
     ((mew-ct-multipartp ct)                       "M")
     ((string-match "Message/Partial"     ct)      "P")
     (t " "))))

(defun mew-summary-form-time ()
  "A function to return a message time, HH:MM"
  (let ((s (MEW-DATE)))
    (if (or (string= s "")
	    (not (string-match mew-time-rfc-regex s)))
	(setq s (mew-time-ctz-to-rfc
		 (mew-file-get-time (mew-expand-folder (MEW-FLD) (MEW-NUM))))))
    (if (string-match mew-time-rfc-regex s)
	(format "%02d:%02d"
		(or (mew-time-rfc-hour) 0)
		(or (mew-time-rfc-min)  0))
      "00:00")))

(defun mew-summary-form-date ()
  "A function to return a date, MM/DD."
  (let ((s (MEW-DATE)))
    (when (or (string= s "")
	      (not (string-match mew-time-rfc-regex s)))
      (setq s (mew-time-ctz-to-rfc
	       (mew-file-get-time (mew-expand-folder (MEW-FLD) (MEW-NUM))))))
    (if (string-match mew-time-rfc-regex s)
	(format "%02d/%02d"
		(mew-time-mon-str-to-int (mew-time-rfc-mon))
		(mew-time-rfc-day))
      "")))

(defun mew-summary-form-year ()
  "A function to return a message year, YYYY"
  (let ((s (MEW-DATE)) year)
    (when (or (string= s "")
	      (not (string-match mew-time-rfc-regex s)))
      (setq s (mew-time-ctz-to-rfc
	       (mew-file-get-time (mew-expand-folder (MEW-FLD) (MEW-NUM))))))
    (if (not (string-match mew-time-rfc-regex s))
	"0000"
      (setq year (mew-time-rfc-year))
      (cond
       ((< year 50)
	(setq year (+ year 2000)))
       ((< year 100)
	(setq year (+ year 1900))))
      (int-to-string year))))

(defun mew-summary-form-size ()
  "A function to return the size of the message. Should be used
with -4. See also 'mew-summary-form-size-0k' and 'mew-summary-form-size-huge'."
  (let ((len-1 (1- (length mew-summary-form-size-unit)))
	(SIZE (mew-scan-uid-size (MEW-UID)))
	(i 0) size unit)
    (if (and SIZE (string-match "^[0-9]+$" SIZE))
	(setq size (string-to-int SIZE))
      (setq size (mew-file-get-size (mew-expand-folder (MEW-FLD) (MEW-NUM)))))
    (while (and (< i len-1) (>= size 1000))
      (setq size (/ size 1000))
      (setq i (1+ i)))
    (if (and mew-summary-form-size-huge (>= size 1000))
	"HUGE"
      (setq unit (nth i mew-summary-form-size-unit))
      (if (and mew-summary-form-size-0k (string= unit ""))
	  "0k"
	(concat
	 (if (integerp size)
	     (int-to-string size)
	   (format "%.0f" size))
	 unit)))))

(defun mew-summary-form-extract-addr (addr)
  "Extract addr according to 'mew-summary-form-extract-rule'."
  (let* ((func (if mew-addrbook-for-summary
		   (mew-addrbook-func mew-addrbook-for-summary)))
	 (raw (or (mew-addrstr-parse-address addr) ""))
	 (rules mew-summary-form-extract-rule)
	 rule ret nickname)
    (catch 'matched
      (while rules
	(setq rule (car rules))
	(setq rules (cdr rules))
	(cond
	 ((and (eq rule 'name)
	       (or (string-match "^\"\\([^\"]+\\)\"[ \t]*<[^>]+>" addr)
		   (string-match "^\\([^<]+\\)<[^>]+>" addr)))
	  (throw 'matched (setq ret (mew-chop (match-string 1 addr)))))
	 ((and (eq rule 'comment)
	       (string-match "^[^(]+(\\(.+\\))" addr))
	  (throw 'matched (setq ret (mew-chop (match-string 1 addr)))))
	 ((eq rule 'address)
	  (throw 'matched (setq ret raw)))
	 ((and (eq rule 'nickname)
	       ;; set nickname here for efficiency
	       (or nickname
		   (setq nickname (if func (funcall func raw)))))
	  (throw 'matched (setq ret nickname)))
	 ((and (stringp rule)
	       (string-match rule addr))
	  (throw 'matched (setq ret (mew-chop (match-string 1 addr))))))))
    (or ret addr)))

(defun mew-summary-form-from ()
  "A function to return an address.
If the message is destined to me AND 'mew-summary-form-from-me-prefix'
is a string, an address on To:, is returned. In this
case, 'mew-summary-form-from-me-prefix' is prepended to the address. 

Otherwise, an address on From: is returned.

Address is converted by 'mew-summary-form-extract-addr'. See also
'mew-summary-form-extract-rule'."
  (let* ((FROM (MEW-FROM)) (TO (MEW-TO))
	 (from (or (mew-addrstr-parse-address FROM) "")))
    (cond
     ((string= FROM "")
      "")
     ((and (stringp mew-summary-form-from-me-prefix)
	   (not (string= TO ""))
	   (mew-is-my-address mew-regex-my-address-list from))
      (mew-replace-white-space
       (concat mew-summary-form-from-me-prefix (mew-summary-form-extract-addr TO))))
     (t
      (mew-replace-white-space (mew-summary-form-extract-addr FROM))))))

(defun mew-summary-form-subj ()
  "A function to return Subject:. Unnecessary white spaces are removed."
  ;; The beginning white spaces have been removed in mew-scan-header
  ;; (mew-keyval).
  (let ((subj (MEW-SUBJ)))
    (if (string= subj "") (setq subj mew-error-no-subject))
    (if mew-decode-broken
	subj
      ;; already well-formatted
      (mew-replace-white-space subj))))

(defun mew-summary-form-body ()
  (mew-header-sanity-check-string (MEW-BODY)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Scan from
;;;

(defun mew-get-summary-form (folder &optional column)
  "Get summary-form from 'mew-summary-form-list', 
'mew-summary-form-list-string-type, and 'mew-summary-form-list-list-type'.
'mew-summary-form-header' is prepended. If COLUMN is non-nil, the number
of thread indent position is returned."
  (let (ret form col)
    (setq ret (mew-folder-spec folder
			       mew-summary-form-list
			       mew-summary-form-list-string-type
			       mew-summary-form-list-list-type))
    (if ret
	(mew-set '(form col) ret)
      (setq form mew-summary-form))
    (setq form (append mew-summary-form-header form))
    (if column
	(if (numberp col) col (mew-thread-column form))
      form)))

(defun mew-thread-column (form)
  (let ((col 0) ret ent)
    (catch 'loop
      (while form
	(setq ent (car form))
	(setq form (cdr form))
	(cond
	 ((consp ent)
	  (setq col (+ col (abs (car ent)))))
	 ((stringp ent)
	  (setq col (+ col (string-width ent))))
	 ((eq ent t)
	  (throw 'loop (setq ret col)))
	 (t
	  (setq col (1+ col))))))
    (or ret mew-thread-column)))

(defun mew-get-unread-mark (folder)
  (car (mew-folder-spec folder
			mew-unread-mark-list
			mew-unread-mark-list-string-type
			mew-unread-mark-list-list-type)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The engine function to call mew-summary-form-*
;;;

(defun mew-scan-get-piece (spec)
  (let (func width str rightp nopad)
    (if (stringp spec)
	(progn
	  (setq TOTAL (+ TOTAL (string-width spec)))
	  spec)
      (if (symbolp spec)
	  (setq width 1 func spec)
	(mew-set '(width func) spec))
      (when (and (integerp width) (symbolp func))
	(when (= width 0)
	  (setq width (- WIDTH TOTAL 1))
	  (unless mew-use-spc-padding
	    (setq nopad t)))
	(if (< width 0) (setq width (abs width) rightp t))
	(setq TOTAL (+ TOTAL width))
	(setq func (intern-soft
		    (concat mew-summary-form-func-prefix (symbol-name func))))
	(when (fboundp func)
	  (setq str (funcall func))
	  (if rightp
	      (if (<= (string-width str) width)
		  (format (format "%%%ds" width) str)
		(setq TOTAL (+ (- TOTAL width) (string-width str)))
		str) ;; width may exceed.
	    (mew-substring str width nil nopad)))))))

(defun mew-scan-get-line (mew-vec WIDTH)
  (let* ((TOTAL 0) (fld "")
	 (line (mapconcat 'mew-scan-get-piece (mew-sinfo-get-summary-form) ""))
	 par-id my-id msg ld uid siz irt-list)
    (setq my-id (or (mew-idstr-get-first-id (MEW-ID)) ""))
    ;; RFC 2822 says: the "In-Reply-To:" field may be used to identify
    ;; the message (or messages) to which the new message is a reply,
    ;; while the "References:" field may be used to identify a
    ;; "thread" of conversation.
    ;;
    ;; However, even if the References field exists, it may not contain
    ;; a parent's ID. So, if the In-Reply-To field contain one ID, 
    ;; we use it for thread.
    ;;
    ;; (1) The In-Reply-To contains one ID, use it.
    ;; (2) The References contains one or more IDs, use the last one.
    ;; (3) The In-Reply-To contains two or more IDs, use the first one.
    (setq par-id (or (mew-idstr-get-first-id (MEW-XREF)) ""))
    (when (string= par-id "")
      (setq irt-list (mew-idstr-to-id-list (MEW-IRT)))
      (if (= (length irt-list) 1)
	  (setq par-id (car irt-list))
	(setq par-id (or (mew-idstr-get-last-id (MEW-REF))
			 (car irt-list)
			 ""))))
    (when (mew-virtual-p)
      (setq fld (cdr (assoc (mew-scan-get-folder mew-vec)
			    (mew-vinfo-get-lra)))))
    (setq msg (mew-scan-get-message mew-vec))
    (setq uid (or (mew-scan-uid-uid (MEW-UID)) ""))
    (setq siz (or (mew-scan-uid-size (MEW-UID)) ""))
    (setq ld (format "\r %s %s %s %s %s %s\n" fld msg my-id par-id uid siz))
    (cons line ld)))

;; See also mew-summary-cook-region
(defun mew-scan-insert-line (folder vec width lmsg &optional mark-or-dst)
  (when (get-buffer folder)
    (save-excursion
      (set-buffer folder)
      (let* ((line (mew-scan-get-line vec width))
	     (opos (point))
	     (omax (point-max))
	     beg med face olen nlen mark msg)
	(mew-elet
	 (if (null lmsg)
	     (goto-char (point-max))
	   ;; a message marked with 'T'.
	   (goto-char (point-min))
	   (if (not (re-search-forward (mew-regex-sumsyn-msg lmsg) nil t))
	       (goto-char (point-max)) ;; xxx
	     (setq mark (mew-summary-get-mark))
	     (beginning-of-line)
	     (setq beg (point))
	     (forward-line)
	     ;; To avoid inserting a line AFTER the cursor underline,
	     ;; keep this line and make it invisible.
	     (put-text-property beg (point) 'invisible t)
	     (mew-front-nonsticky beg (1+ beg)) ;; for XEmacs
	     (forward-line -1)))
	 (setq beg (point))
	 ;; To "insert" just after mew-marker-decode-syntax-end.
	 (insert (car line))
	 (setq med (point))
	 (insert (cdr line))
	 (goto-char beg)
	 (mew-front-nonsticky beg med) ;; for XEmacs
	 (cond
	  ((stringp mark-or-dst) ;; xxx
	   (setq msg (mew-scan-get-message vec))
	   (mew-refile-reset msg)
	   (mew-refile-set msg (mew-split mark-or-dst ?,))
	   (mew-summary-refile-log folder mark-or-dst)
	   (mew-summary-refile-override-body mark-or-dst)
	   (mew-mark-put mew-mark-refile))
	  ((mew-characterp mark-or-dst) ;; mew-inbox-action-alist
	   (mew-mark-put mark-or-dst))
	  (mark
	   (mew-summary-mark-as mark))
	  ((and mew-use-highlight-mark ;; mew-summary-form-mark
		(setq mark (mew-summary-get-mark)) ;; duplicated, etc
		(setq face (mew-highlight-mark-get-face mark)))
	   (put-text-property beg med 'face face)))
	 (if mew-use-highlight-mouse-line
	     (put-text-property
	      beg med 'mouse-face mew-highlight-mouse-line-face))
	 (forward-line)
	 (put-text-property med (1- (point)) 'invisible t)
	 ;; Removing the invisible line.
	 (when lmsg
	   ;; UID information will be removed. So, we need to adjust
	   ;; the position.
	   (setq nlen (- (point) beg))
	   (setq beg (point))
	   (forward-line)
	   (when (> opos beg)
	     (setq olen (- (point) beg))
	     (setq opos (- opos (- olen nlen))))
	   (delete-region beg (point))))
	(if (or (eq opos (mew-sinfo-get-start-point))
		(/= opos omax))
	    ;; move the cursor to the original position.
	    (goto-char opos))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sub-functions for Scan
;;;

(defun mew-scan-header (&optional draftp)
  (let ((vec (make-vector (length mew-scan-fields) ""))
	(lim (1- mew-scan-max-field-length))
	i key med str n)
    (goto-char (point-min))
    (unless (re-search-forward mew-eoh nil t)
      (goto-char (point-max)))
    (save-restriction
      (narrow-to-region (point-min) (point))
      (goto-char (point-min))
      (while (not (eobp))
	(if (not (looking-at mew-keyval))
	    (forward-line)
	  (setq key (mew-capitalize (mew-match-string 1)))
	  (setq med (match-end 0))
	  ;; Three lines should be enough for Summary mode.
	  (forward-line)
          (setq i 0)
	  (while (and (< i lim) (looking-at mew-lwsp))
	    (forward-line)
	    (setq i (1+ i)))
	  (when (and (setq n (mew-member-case-equal key mew-scan-fields))
		     (string= (aref vec n) "")) ;; avoid multiple times
	    (when (member key mew-scan-decode-fields)
	      (mew-header-decode-region key med (point) draftp))
	    ;; We need to keep composite properties of charset.
	    ;; This must be "buffer-substring".
	    (setq str (buffer-substring med (1- (point))))
	    (aset vec n str))
	  (mew-header-goto-next))))
    vec))

(defun mew-scan-field-to-func (key)
  (let ((n (mew-member-case-equal key mew-scan-fields))
	(len (length mew-scan-fields-alias))
	ali)
    (if (and n (< n len)) (setq ali (nth n mew-scan-fields-alias)))
    (if (stringp ali) (symbol-function (intern-soft (concat "MEW-" ali))))))

(defun mew-scan-inbox-action (mew-vec)
  (let ((alist mew-inbox-action-alist)
	ent key act val val-func ret mark-or-dst regex-list)
    (catch 'loop
      (while alist
	(setq ent (car alist))
	(setq alist (cdr alist))
	(setq key (car ent))
	(setq val-func (mew-scan-field-to-func key))
	(if (and val-func (functionp val-func))
	    (setq val (funcall val-func))
	  (setq val nil))
	(when val
	  (setq ent (cdr ent))
	  (while ent
	    (setq act (car ent))
	    (setq ent (cdr ent))
	    (if (symbolp act)
		(when (fboundp act)
		  (setq ret (funcall act val))
		  (if ret (throw 'loop nil)))
	      (when (listp act)
		(setq mark-or-dst (car act))
		(setq regex-list (cdr act))
		(while regex-list
		  (if (string-match (car regex-list) val)
		      (throw 'loop (setq ret mark-or-dst)))
		  (setq regex-list (cdr regex-list)))))))))
    ret))

(defvar mew-regex-ignore-scan-body-list
  '("^[ \t]*$"
    "^[ \t]*[-a-zA-Z0-9]+: "
    "^[ \t]*[>:|#;/_}]"
    "^[ \t]*\\w+\\(['._-]+\\w+\\)*>"
    "^[ \t]*[[</(.]+ *snip"
    "^   "
    "^--"
    "^- --"
    "^=2D"
    "^.*\\(:\\|;\\|/\\)[ \t]*$"
    "^.*\\(wrote\\|writes?\\|said\\|says?\\)[^.!\n]?[ \t]*$"
    "^[ \t]*\\(On\\|At\\) .*[^.! \t\n][ \t]*$"
    "^[ \t]*In \\(message\\|article\\|mail\\|news\\|<\\|\"\\|\\[\\|(\\)"))

(defun mew-scan-body (mew-vec &optional draftp)
  (forward-line)
  (let* ((i 0) (I mew-scan-max-body-length)
	 (j 0) (J 3)
	 (ctr (MEW-CT))
	 (cte (MEW-CTE))
	 (body "")
	 (case-fold-search t)
	 textp charset cs beg skip boundary found regex)
    (catch 'break
      (cond
       (draftp
	(setq textp t)
	(setq cs mew-cs-m17n))
       ((string= ctr "")
	(if (mew-case-equal cte mew-b64) (throw 'break nil))
	(setq textp t)
	(setq cs mew-cs-autoconv))
       (t
	;; The following code is generic but too slow.
	;; (setq ctl (mew-param-decode ctr))
	;; (setq ct (mew-syntax-get-value ctl 'cap))
	;; So, this hard coding is used.
	(when (and (string-match "^Multipart/" ctr)
		   (string-match "boundary=\"?\\([^\"\n\t;]+\\)\"?" ctr))
	  (setq boundary (mew-match-string 1 ctr))
	  (setq boundary (concat "^--" (regexp-quote boundary)))
	  (catch 'loop
	    (while (< i I) 
	      (if (looking-at boundary) (throw 'loop (setq found t)))
	      (forward-line)
	      (setq i (1+ i))))
	  (if (not found)
	      (throw 'break nil)
	    (forward-line)
	    (save-restriction
	      (narrow-to-region (point) (point-max))
	      (setq ctr (mew-header-get-value mew-ct:))
	      (setq cte (mew-addrstr-parse-value
			 (mew-header-get-value mew-cte:)))
	      (mew-header-goto-end)) ;; should be in the narrowed region
	    (unless ctr ;; not ""
	      (setq textp t)
	      (setq cs mew-cs-autoconv)
	      (throw 'break nil))))
	(if (and cte (mew-case-equal cte mew-b64)) (throw 'break nil))
	(when (string-match "^Text/Plain" ctr)
	  (when (string-match "charset=\"?\\([^\"\n\t;]+\\)\"?" ctr)
	    (setq charset (mew-match-string 1 ctr))) 
	  ;; xxx quoted-printable. not enough DB in mew-mule3.el.
	  (setq textp t)
	  (setq cs (mew-charset-to-cs charset))
	  (if (null cs) (setq cs mew-cs-autoconv)))))) ;; end of 'break
    (when (and textp (mew-coding-system-p cs))
      (setq i 0)
      (while (and (not (eobp)) (< i I) (< j J))
	(setq regex mew-regex-ignore-scan-body-list)
	(setq skip nil)
	(catch 'matched
	  (while regex
	    (if (looking-at (car regex))
		(throw 'matched (setq skip t)))
	    (setq regex (cdr regex))))
	(if skip
	    (forward-line)
	  (when (looking-at "^[ \t]+")
	    (goto-char (match-end 0)))
	  (setq beg (point))
	  (forward-line)
	  (setq body (concat body (mew-buffer-substring beg (1- (point))) " "))
	  (setq j (1+ j)))
	(setq i (1+ i)))
      (setq body (mew-cs-decode-string (mew-replace-white-space body) cs))
      (aset mew-vec (1- (length mew-vec)) body))))

(defun mew-scan-width ()
  (if (and (integerp mew-summary-scan-width)
	   (> mew-summary-scan-width 40)) ;; xxx
      mew-summary-scan-width
    (max mew-window-magic (window-width))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; X-Mew-Uidl:
;;;

(defsubst mew-scan-uid-uid (uid)
  (nth 0 (mew-split uid mew-sp)))

(defsubst mew-scan-uid-size (uid)
  (nth 1 (mew-split uid mew-sp)))

(defun mew-header-insert-xmu (uid siz truncated)
  (when (and (stringp uid) (stringp siz))
    (setq siz (int-to-string (string-to-int siz))) ;; removing 0
    (let (fields)
      (if truncated
	  (setq fields (concat uid " 0" siz)) ;; e.g. 0500 == truncated
	(setq fields (concat uid " " siz)))
      (save-excursion
	(mew-header-delete-lines (list mew-x-mew-uidl:)))
      (mew-header-insert mew-x-mew-uidl: fields 'no-fold))))

(defun mew-scan-message-truncatedp ()
  (mew-msg-truncatedp (mew-scan-uid-size (MEW-UID))))

(defun mew-scan-message-invalidp ()
  (mew-msg-invalidp (MEW-NUM)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Scanning a folder
;;;

(defun mew-summary-ls (&optional header-only goend)
  "List this folder asynchronously.

In a LOCAL folder: messages in the local folder are scanned according
to the range which you specify.

In a REMOTE folder: messages in the server's folder are cached
according to the range which you specify. If
'mew-pop-header-only'/'mew-imap-header-only'/'mew-nntp-header-only' is
non-nil, only headers of messages are cacched. If executed with
'\\[universal-argument]', these variables are considered reversed."
  (interactive "P")
  (mew-summary-only
   (when (mew-summary-exclusive-p)
     (let* ((bnm (mew-summary-folder-name 'ext))
	    (case (mew-sinfo-get-case))
	    (fld (mew-sinfo-get-folder))
	    (askp mew-ask-range)
	    (directive 'scan)
	    (get-body (not header-only))
	    scanp range dir-newp)
       (mew-summary-folder-cache-load)
       (cond
	((interactive-p) ;; "s"
	 (setq askp t)
	 (setq scanp t))
	((mew-summary-folder-dir-newp) ;; "g"
	 (setq askp nil)
	 (setq scanp t)))
       (if (mew-summary-folder-dir-newp) (setq dir-newp t))
       (if (or (interactive-p) goend) (goto-char (point-max)))
       (set-buffer-modified-p nil)
       (if (not scanp)
	   (progn
	     (run-hooks 'mew-summary-ls-no-scan-hook)
	     t) ;; return value (not scanned)
	 (mew-summary-reset)
	 ;;
	 (mew-sinfo-set-direction 'down)
	 (cond
	  ((and (mew-folder-remotep fld)
		(not (mew-folder-imap-queuep)))
	   (if (and dir-newp (mew-folder-imapp fld))
	       (mew-local-retrieve 'scan bnm (mew-range-update bnm))
	     (setq range (mew-input-range-remote bnm))
	     (when (eq range 'sync)
	       (setq range nil)
	       (setq directive 'sync))
	     (cond
	      ((mew-folder-popp fld)
	       (if (mew-pop-header-only case)
		   (setq get-body (not get-body)))
	       (mew-pop-retrieve case directive bnm range get-body))
	      ((mew-folder-imapp fld)
	       (if (mew-imap-header-only case)
		   (setq get-body (not get-body)))
	       (mew-imap-retrieve case directive bnm range get-body))
	      ((mew-folder-nntpp fld)
	       (if (mew-nntp-header-only case)
		   (setq get-body (not get-body)))
	       (mew-nntp-retrieve case directive bnm range get-body)))))
	  (t ;; local
	   (setq range (mew-input-range bnm askp))
	   (if range
	       (mew-local-retrieve 'scan bnm (nth 0 range) (nth 1 range))
	     (message "range is wrong"))))
	 nil))))) ;; return value (scanned)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Summary file cache
;;;

(defun mew-compare-times (t1 t2)
  ;; Is t1 newer than t2?
  (cond
   ((null t1) nil)
   ((null t2) t) ;; do update
   ((> (nth 0 t1) (nth 0 t2)) t)
   ((= (nth 0 t1) (nth 0 t2))
    (if (> (nth 1 t1) (nth 1 t2)) t nil)) ;; nil if equal
   (t nil)))

(defsubst mew-summary-folder-dir-newp ()
  (let* ((folder (mew-summary-folder-name 'ext))
	 (dir (file-chase-links (mew-expand-folder folder)))
	 (mfile (expand-file-name mew-summary-touch-file dir))
	 (t1 (mew-file-get-time mfile))
	 (cache (expand-file-name mew-summary-cache-file dir))
	 (t2 (mew-file-get-time cache)))
    (if (and (null t1)
	     (file-directory-p dir)
	     (mew-dir-messages (mew-expand-folder dir)))
	t
      (mew-compare-times t1 t2))))

(defsubst mew-summary-folder-cache-newp ()
  (let* ((folder (mew-summary-folder-name 'ext))
	 (cache (mew-expand-folder folder mew-summary-cache-file))
	 (t1 (mew-file-get-time cache))
	 (t2 (mew-sinfo-get-cache-time)))
    (mew-compare-times t1 t2)))

(defun mew-summary-set-count-line ()
  (let* ((ttl-line (mew-count-lines (point-min) (point-max)))
	 (mid-point (/ (buffer-size) 2))
	 (mid-marker (mew-sinfo-get-mid-marker))
	 mid-line)
    (save-excursion
      (goto-char mid-point)
      (beginning-of-line)
      (if (and (mew-thread-p) mew-use-thread-separator
	       (looking-at mew-regex-thread-separator))
	  (forward-line))
      (setq mid-point (point))
      (setq mid-line (mew-count-lines (point-min) (point))))
    (mew-sinfo-set-ttl-line ttl-line)
    (mew-sinfo-set-mid-line mid-line)
    (unless (markerp mid-marker)
      (setq mid-marker (make-marker))
      (mew-sinfo-set-mid-marker mid-marker))
    (set-marker mid-marker mid-point)))

(defun mew-summary-folder-cache-load ()
  (let* ((folder (mew-summary-folder-name 'ext))
	 (cache (mew-expand-folder folder mew-summary-cache-file))
	 refile)
    (when (and (file-readable-p cache)
	       (mew-summary-folder-cache-newp))
      (mew-elet
       (mew-erase-buffer)
       (mew-frwlet
	mew-cs-m17n mew-cs-dummy
	(mew-insert-file-contents cache))
       (mew-sinfo-set-cache-time (mew-file-get-time cache))
       (if (= (point-max) 1)
	   (setq mew-summary-buffer-raw nil)
	 (setq mew-summary-buffer-raw t))
       (mew-sinfo-load)
       (setq refile (mew-summary-mark-recover
		     (mew-sinfo-get-mark-hist) (mew-sinfo-get-refile)))
       (mew-sinfo-set-refile refile)
       (mew-summary-set-count-line)
       (set-buffer-modified-p nil)))))

(defun mew-summary-folder-cache-save ()
  (let* ((folder (mew-summary-folder-name 'ext))
	 (cache (mew-expand-folder folder mew-summary-cache-file)))
    (when (file-writable-p cache)
      (mew-touch-folder folder)
      (save-restriction
	(widen)
	(if (mew-decode-syntax-p)
	    (let ((cbuf (current-buffer))
		  (min (point-min))
		  (max (point-max))
		  (beg (mew-decode-syntax-begin))
		  (end (mew-decode-syntax-end)))
	      (with-temp-buffer
		(mew-insert-buffer-substring cbuf min beg)
		(mew-insert-buffer-substring cbuf end max)
		(mew-frwlet
		 mew-cs-dummy mew-cs-m17n
		 (write-region (point-min) (point-max) cache nil 'no-msg))))
	  ;; (write-region 1 1 ...) does not update the file timestamp
	  ;; but does the directory timestamp. So, we need to delete
	  ;; the file to update the file timestamp.
	  (if (= (point-min) (point-max)) (mew-delete-file cache))
	  (mew-frwlet
	   mew-cs-dummy mew-cs-m17n
	   (write-region (point-min) (point-max) cache nil 'no-msg))
	  (mew-set-file-modes cache))
	(mew-summary-set-count-line)
	(mew-sinfo-set-cache-time (mew-file-get-time cache))
	(mew-sinfo-save)
	(mew-sinfo-set-mark-hist nil)))))

;; See also mew-net-folder-clean.
(defun mew-summary-folder-cache-clean (folder)
  "Erase Summary mode then remove and touch the cache file."
  (if (get-buffer folder)
      (save-excursion
	(set-buffer folder)
	(mew-erase-buffer)
	(set-buffer-modified-p nil)))
  (let ((cfile (mew-expand-folder folder mew-summary-cache-file)))
    (if (file-exists-p cfile)
	(write-region "" nil cfile nil 'no-msg))))

(provide 'mew-scan)

;;; Copyright Notice:

;; Copyright (C) 1996-2003 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-scan.el ends here
