;;; mew-pop.el

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Jun 28, 2000
;; Revised: Jul  8, 2001

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; POP info
;;;

(defvar mew-pop-info-list
  '("status" "cnt" "inbox" "messages" "auth" "server" "ssh-process"
    "try" "key" "user" "size" "delete" "top" "lmsg" "uidl"
    "port" "sul" "body-lines" "case" "directive" "diag"
    "string" "keep-str" "passwd"))

(mew-info-defun "mew-pop-" mew-pop-info-list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Process name
;;;

(defconst mew-pop-info-prefix "mew-pop-info-")

(defsubst mew-pop-info-name (server &optional sshsrv)
  (if sshsrv
      (concat mew-pop-info-prefix sshsrv "-" server)
    (concat mew-pop-info-prefix server)))

(defsubst mew-pop-buffer-name (pnm)
  (concat mew-buffer-prefix pnm))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; FSM
;;;

(defvar mew-pop-fsm
  '(("grtg"         nil ("+OK" . "capa"))
    ("capa"         t   ("+OK" . "auth") ("-ERR" . "pswd"))
    ("cram-md5"     nil ("+OK" . "pwd-cram-md5") ("-ERR" . "wpwd"))
    ("pwd-cram-md5" nil ("+OK" . "list") ("-ERR" . "wpwd"))
    ("apop"         nil ("+OK" . "list") ("-ERR" . "wpwd"))
    ("user"         nil ("+OK" . "pass") ("-ERR" . "wpwd2"))
    ("pass"         nil ("+OK" . "list") ("-ERR" . "wpwd"))
    ("list"         t   ("+OK" . "uidl"))
    ("uidl"         t   ("+OK" . "umsg") ("-ERR" . "retr"))
    ("retr"         t   ("+OK" . "dele"))
    ("dele"         nil ("+OK" . "retr"))
    ("quit"         nil ("+OK" . "noop"))))

(defsubst mew-pop-fsm-by-status (status)
  (assoc status mew-pop-fsm))

(defsubst mew-pop-fsm-next (status code)
  (cdr (mew-assoc-match code (nthcdr 2 (mew-pop-fsm-by-status status)) 0)))

(defsubst mew-pop-fsm-reply (status)
  (nth 1 (mew-pop-fsm-by-status status)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Commands
;;;

(defsubst mew-pop-message (pnm &rest args)
  (if (mew-pop-get-diag pnm) (apply (function message) args)))

(defun mew-pop-retrieve (case &optional sul rev-del fld directive passwd)
  ;; in +inbox
  (let* ((server (mew-pop-server case))
	 (port (mew-pop-port case))
	 (sshsrv (mew-pop-ssh-server case))
	 (pnm (mew-pop-info-name server sshsrv))
	 (ret t)
	 (buf (get-buffer-create (mew-pop-buffer-name pnm)))
	 process sshpro sshname lport)
    (mew-info-clean-up pnm)
    (mew-pop-set-diag pnm (not (eq directive 'biff)))
    (if (null sshsrv)
	(progn
	  (mew-pop-set-port pnm port)
	  (setq process (mew-pop-open pnm server port)))
      (mew-pop-set-port pnm port)
      (setq sshpro (mew-open-ssh-stream server port sshsrv))
      (mew-pop-set-ssh-process pnm sshpro)
      (when sshpro
	(setq sshname (process-name sshpro))
	(setq lport (mew-ssh-pnm-to-lport sshname))
	(if lport (setq process (mew-pop-open pnm "localhost" lport)))))
    (cond
     (process
      (mew-pop-message pnm "Communicating with the POP server ... ")
      (setq mew-summary-buffer-process process)
      (mew-pop-set-status pnm "grtg")
      (mew-pop-set-cnt pnm 0)
      (mew-pop-set-try pnm 0)
      (mew-pop-set-keep-str pnm t)
      (mew-pop-set-server pnm server)
      (mew-pop-set-sul pnm sul)
      (mew-pop-set-directive pnm directive)
      (mew-pop-set-passwd pnm passwd)
      (mew-pop-set-case pnm case)
      (mew-pop-set-inbox pnm (or fld (mew-inbox-folder case)))
      (mew-pop-set-auth pnm (mew-pop-auth case))
      (mew-pop-set-user pnm (mew-pop-user case))
      (mew-pop-set-size pnm (mew-pop-size case))
      (mew-pop-set-body-lines pnm (mew-pop-body-lines case))
      (if rev-del
	  (mew-pop-set-delete pnm (not (mew-pop-delete case)))
	(mew-pop-set-delete pnm (mew-pop-delete case)))
      ;;
      (set-process-sentinel process 'mew-pop-sentinel)
      (set-process-filter process 'mew-pop-filter)
      (set-process-buffer process buf)
      (mew-sinfo-set-scan-id nil))
     (t
      (mew-pop-tear-down pnm)))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Biff
;;;

(defvar mew-biff-string nil)
(defvar mew-pop-biff-timer-id nil)

(defun mew-pop-biff ()
  (let* ((case mew-case-input)
	 (inbox (mew-inbox-folder case))
	 (tag (mew-pop-passtag2 case))
	 passwd)
    (save-excursion
      (set-buffer inbox)
      (when (and (mew-summary-exclusive-p)
		 (and mew-use-cached-passwd
		      (setq passwd (mew-passwd-get-passwd tag))))
	(mew-sinfo-set-scan-form (mew-summary-scan-form inbox))
	(mew-pop-retrieve case nil nil nil 'biff passwd)))))

(defun mew-pop-check ()
  (interactive)
  (let* ((case mew-case-input)
	 (inbox (mew-inbox-folder case)))
    (save-excursion
      (set-buffer inbox)
      (when (mew-summary-exclusive-p)
	(mew-sinfo-set-scan-form (mew-summary-scan-form inbox))
	(mew-pop-retrieve case nil nil nil 'biff)))))

(defun mew-pop-biff-setup ()
  (if (not mew-use-biff)
      (mew-pop-biff-clean-up)
    (if mew-pop-biff-timer-id (cancel-timer mew-pop-biff-timer-id))
    (setq mew-pop-biff-timer-id
	  (mew-timer (* 60 mew-pop-biff-interval) (function mew-pop-biff))))
  (let ((ent '(mew-biff-string mew-biff-string)))
    (if (member ent global-mode-string)
	()
      (if global-mode-string
	  (setq global-mode-string
		(append global-mode-string (list " " ent)))
	(setq global-mode-string (list ent))))))

(defun mew-pop-biff-clean-up ()
  (if mew-pop-biff-timer-id (cancel-timer mew-pop-biff-timer-id))
  (setq mew-pop-biff-timer-id nil))

(defun mew-pop-biff-bark (n)
  (if (= n 0)
      (setq mew-biff-string nil)
    (if (and mew-use-biff-bell (eq mew-biff-string nil))
	(beep))
    (setq mew-biff-string (format "Mail(%d)" n))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Opening POP
;;;

(defun mew-pop-open (pnm server port)
  (let ((sprt (mew-port-sanity-check port))
	pro tm)
    (condition-case emsg
	(progn
	  (setq tm (mew-timer mew-pop-timeout-time 'mew-pop-timeout))
	  (mew-pop-message pnm "Connecting to the POP server ... ")
	  (setq pro (open-network-stream pnm nil server sprt))
	  (process-kill-without-query pro)
	  (mew-set-process-cs pro mew-cs-text-for-net mew-cs-text-for-net)
	  (mew-pop-message pnm "Connecting to the POP server ... done"))
      (quit
       (mew-pop-message pnm "Can't connect to the POP server due to time out")
       (setq pro nil))
      (error
       (mew-pop-message pnm "%s, %s" (nth 1 emsg) (nth 2 emsg))
       (setq pro nil)))
    (if tm (cancel-timer tm))
    pro))

(defun mew-pop-timeout ()
  (signal 'quit nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Closing POP
;;;

(defun mew-pop-close (pnm)
  (interactive)
  (let* ((process (get-process pnm))
	 (buf (process-buffer process)))
    (when (and (processp process) (memq (process-status process) '(open)))
      (set-process-buffer process nil)
      (mew-remove-buffer buf)
      (set-process-filter process nil)
      (process-send-string process (format "QUIT%s" mew-cs-eol))))
  (mew-pop-tear-down pnm))

(defun mew-pop-tear-down (pnm)
  (let ((process (get-process pnm))
	(sshpro (mew-pop-get-ssh-process pnm))
	(inbox (mew-pop-get-inbox pnm)))
    (mew-info-clean-up pnm)
    (if (processp process) (delete-process process))
    (if (processp sshpro) (delete-process sshpro))
    (when inbox
      (mew-set-summary-buffer-process-status inbox nil)
      (save-excursion
	(set-buffer inbox)
	(setq mew-summary-buffer-process nil)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Filter and sentinel
;;;

(defun mew-pop-debug (label string)
  (when (mew-debug 'net)
    (save-excursion
      (set-buffer (get-buffer-create mew-buffer-debug))
      (insert (format "\n<%s>\n%s\n" label string)))))

(defun mew-set-summary-buffer-process-status (inbox msg)
  "Add informational string MSG to summary buffer process status of Mew.
Usually displayed as \" Running\" in minor mode indicator area.
If string argument MSG is specified, change as \" Running:...\".
If MSG is nil, restore usual format."
  (save-excursion
    (set-buffer inbox)
    (setq mew-summary-buffer-process-status
	  (if (and msg (stringp msg))
	      (concat " Running:" msg)
	    " Running"))))

(defun mew-pop-filter (process string)
  (let* ((pnm (process-name process))
	 (status (mew-pop-get-status pnm))
	 (cnt (mew-pop-get-cnt pnm))
	 (msgs (mew-pop-get-messages pnm))
	 (msg (car msgs))
	 (siz (nth 1 msg))
	 (inbox (mew-pop-get-inbox pnm))
	 (mulrep (mew-pop-fsm-reply status))
	 stay next func progress total)
    (mew-pop-debug (upcase status) string)
    (mew-filter
     ;; This code may create a long string. So, take care.
     (if (mew-pop-get-keep-str pnm)
	 (mew-pop-set-string pnm (concat (mew-pop-get-string pnm) string)))
     (when (string= status "retr")
       (setq total (+ (length msgs) cnt))
       (if (>= cnt total)
 	   (setq progress nil)
  	 (setq siz (string-to-int siz))
	 (if (= siz 0) (setq siz 1)) ;; xxx
 	 (setq progress (format "%d/%d:%02d%%" (1+ cnt) total
  				(if (< 10000 siz)
  				    (/ (buffer-size) (/ siz 100))
  				  (/ (* (buffer-size) 100) siz)))))
       (mew-set-summary-buffer-process-status inbox progress))
     ;; Process's buffer
     (goto-char (point-max))
     (mew-set-buffer-multibyte nil)
     (insert string)
     (cond
      ((and (and (goto-char (point-min)) (looking-at "-ERR"))
	    (and (goto-char (1- (point-max))) (looking-at "\n$")))
       (setq next (mew-pop-fsm-next status "-ERR")))
      ((and (and (goto-char (point-min)) (looking-at "\\+")) ;; +OK
	    (or (and mulrep
		     (goto-char (point-max))
		     (forward-line -1)
		     (looking-at "^\\.\r?$"))
		(and (not mulrep)
		     (goto-char (1- (point-max)))
		     (looking-at "\n$"))))
       (setq next (mew-pop-fsm-next status "+OK")))
      (t
       (setq stay t)))
     (unless stay
       (unless next (setq next "quit"))
       (mew-pop-set-status pnm next)
       (setq func (intern-soft (concat "mew-pop-command-" next)))
       (goto-char (point-min))
       (if (fboundp func)
	   (and func (funcall func process pnm))
	 (error "No function called %s" (symbol-name func)))
       (mew-pop-set-string pnm nil)
       (mew-erase-buffer)))))

(defun mew-pop-sentinel (process event)
  (let* ((pnm (process-name process))
	 (cnt (mew-pop-get-cnt pnm))
	 (inbox (mew-pop-get-inbox pnm))
	 (directive (mew-pop-get-directive pnm))
	 (n (length (mew-pop-get-messages pnm)))
	 savep)
    (mew-pop-debug "POP SENTINEL" event)
    (mew-filter
     (if (eq directive 'biff)
	 (funcall mew-pop-biff-function n)
       (setq mew-biff-string nil)) ;; received
     (cond
      ((eq cnt nil)
       ())
      ((= cnt 0)
       (mew-pop-message pnm "No new message"))
      ((= cnt 1)
       (mew-pop-message pnm "1 message has been received.")
       (setq savep t))
      (t
       (mew-pop-message pnm "%d messages have been received." cnt)
       (setq savep t)))
     (set-buffer inbox)
     (if savep (mew-summary-folder-cache-save))
     (set-buffer-modified-p nil)
     (mew-pop-tear-down pnm)
     (unless (eq directive 'biff)
       (run-hooks 'mew-pop-sentinel-non-biff-hook))
     (run-hooks 'mew-pop-sentinel-hook)
     (if (and mew-auto-flush-queue
	      (not (or (eq directive 'biff) (eq directive 'no-flush))))
	 (mew-smtp-flush-queue
	  (mew-queue-folder mew-case-output) mew-case-output)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; POP AUTH
;;;

(defun mew-pop-passtag (pnm)
  (let ((server (mew-pop-get-server pnm))
	(port (mew-pop-get-port pnm))
	(user (mew-pop-get-user pnm)))
    (concat user "@" server ":" port)))

(defun mew-pop-passtag2 (case)
  (let ((server (mew-pop-server case))
	(port (mew-pop-port case))
	(user (mew-pop-user case)))
    (concat user "@" server ":" port)))

(defun mew-pop-input-passwd (prompt pnm)
  (let ((tag (mew-pop-passtag pnm))
	(directive (mew-pop-get-directive pnm)))
    (if (eq directive 'biff)
	(or (mew-pop-get-passwd pnm)       ;; mew-pop-biff
	    (mew-input-passwd prompt tag)) ;; mew-pop-check
      (mew-input-passwd prompt tag))))

(defvar mew-pop-auth-alist
  '(("CRAM-MD5" mew-pop-command-auth-cram-md5)))

(defsubst mew-pop-auth-get-func (auth)
  (nth 1 (mew-assoc-case-equal auth mew-pop-auth-alist 0)))

(defun mew-pop-command-auth-cram-md5 (pro pnm)
  (process-send-string
   pro (format "AUTH CRAM-MD5%s" mew-cs-eol))
  (mew-pop-set-status pnm "cram-md5"))

(defun mew-pop-command-pwd-cram-md5 (pro pnm)
  (let ((str (mew-pop-get-string pnm))
	(user (mew-pop-get-user pnm))
	challenge passwd cram-md5)
    (if (string-match " \\([A-Za-z0-9+/]+=*\\)" str) ;; xxx
	(setq challenge (mew-match 1 str)))
    (setq passwd (mew-pop-input-passwd "CRAM-MD5: " pnm))
    (setq cram-md5 (mew-cram-md5 user passwd challenge))
    (process-send-string pro (format "%s%s" cram-md5 mew-cs-eol))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; UIDL
;;;

(defvar mew-pop-uidl-db nil)

(defsubst mew-pop-uidl-db-get (pnm)
  (cdr (assoc (mew-pop-passtag pnm) mew-pop-uidl-db)))

(defsubst mew-pop-uidl-db-set (pnm uidl)
  (let* ((tag (mew-pop-passtag pnm))
	 (ent (assoc tag mew-pop-uidl-db)))
    (if ent
	(setcdr ent uidl)
      (setq mew-pop-uidl-db (cons (cons tag uidl) mew-pop-uidl-db)))))

(defvar mew-pop-uidl-file ".mew-uidl")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;

(defun mew-pop-setup ()
  (setq mew-pop-uidl-db (mew-lisp-load mew-pop-uidl-file))
  (add-hook 'kill-emacs-hook (function mew-pop-clean-up)))

(defun mew-pop-clean-up ()
  (remove-hook 'kill-emacs-hook (function mew-pop-clean-up))
  (mew-lisp-save mew-pop-uidl-file mew-pop-uidl-db))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Filters
;;;

(defun mew-pop-command-capa (pro pnm)
  (if (re-search-forward "<[^>]+>" nil t)
      (mew-pop-set-key pnm (mew-match 0)))
  (process-send-string pro (format "capa%s" mew-cs-eol)))

(defun mew-pop-command-auth (pro pnm)
  (cond
   ((eq (mew-pop-get-auth pnm) t) ;; t means SASL
    (let ((str (mew-pop-get-string pnm)) auth func)
      (if (and (string-match "SASL \\([^\n\r]+\\)\r?\n" str)
	       (setq auth (mew-auth-select
			   (mew-match 1 str) mew-pop-auth-list))
	       (setq func (mew-pop-auth-get-func auth))
	       (fboundp func))
	  (progn
	    (mew-pop-set-auth pnm auth)
	    (funcall func pro pnm))
	(mew-pop-debug "<AUTH>" "No preferred POP AUTH.\n"))))
   (t
    (mew-pop-command-pswd pro pnm))))

(defun mew-pop-command-pswd (pro pnm)
  (let ((auth (mew-pop-get-auth pnm)))
    (cond
     ((or (eq auth 'pass) (eq auth 'user))
      (mew-pop-set-status pnm "user")
      (mew-pop-command-user pro pnm))
     (t
      (mew-pop-set-status pnm "apop")
      (mew-pop-command-apop pro pnm)))))

(defun mew-pop-command-user (pro pnm)
  (mew-pop-set-try pnm 0)
  (process-send-string
   pro (format "user %s%s" (mew-pop-get-user pnm) mew-cs-eol)))

(defun mew-pop-command-pass (pro pnm)
  (let (passwd)
    (setq passwd (mew-pop-input-passwd "POP password: " pnm))
    (mew-pop-message pnm "Sending your POP password to the POP server ... ")
    (process-send-string
     pro (format "pass %s%s" passwd mew-cs-eol))))

(defun mew-pop-command-apop (pro pnm)
  (let ((try (mew-pop-get-try pnm))
	passwd key kmd5)
    (catch 'passwd-try
      (cond
       ((= try 0)
	(setq key (mew-pop-get-key pnm)))
       ((< try 3)
	(mew-pop-message pnm "APOP password is wrong!")
	(mew-passwd-set-passwd (mew-pop-passtag pnm) nil)
	(setq key (mew-pop-get-key pnm)))
       (t
	(mew-passwd-set-passwd (mew-pop-passtag pnm) nil)
	(mew-pop-set-status pnm "quit")
	(mew-pop-command-quit pro pnm)
	(throw 'passwd-try nil)))
      (setq passwd (mew-pop-input-passwd "APOP password: " pnm))
      (setq kmd5 (mew-keyed-md5 key passwd))
      (mew-pop-set-try pnm (1+ try))
      (mew-pop-message pnm "Sending your APOP password to the POP server ... ")
      (process-send-string
       pro (format "apop %s %s%s" (mew-pop-get-user pnm) kmd5 mew-cs-eol)))))

(defun mew-pop-command-wpwd (pro pnm)
  (let ((auth (mew-pop-get-auth pnm))
	(str (mew-pop-get-string pnm))
	(clear-pass t))
    (cond
     ((and (stringp str) (string-match " lock" str)) ;; very ad hoc
      (mew-pop-message pnm "The mailbox is locked!")
      (setq clear-pass nil))
     ((or (eq auth 'pass) (eq auth 'user))
      (mew-pop-message pnm "POP password is wrong!"))
     ((eq auth 'apop)
      (mew-pop-message pnm "APOP password is wrong!"))
     ((stringp auth)
      (mew-pop-message pnm "%s password is wrong!" (upcase auth)))
     (t
      ;; pnm may be cleared already
      (mew-pop-message pnm "Password is wrong!")))
    (if clear-pass (mew-passwd-set-passwd (mew-pop-passtag pnm) nil))
    (mew-pop-tear-down pnm))) ;; Typical servers disconnect, so no quit

(defun mew-pop-command-wpwd2 (pro pnm)
  (mew-pop-message pnm "Stronger password scheme should be used!")
  (mew-passwd-set-passwd (mew-pop-passtag pnm) nil)
  (mew-pop-tear-down pnm)) ;; Typical servers disconnect, so no quit

(defun mew-pop-command-list (pro pnm)
  (mew-pop-message pnm "Communicating with the POP server ... ")
  (mew-pop-set-keep-str pnm nil)
  (cond
   ((mew-pop-get-sul pnm)
    (mew-pop-set-status pnm "uidl")
    (process-send-string pro (format "uidl%s" mew-cs-eol)))
   (t
    (process-send-string pro (format "list%s" mew-cs-eol)))))

(defun mew-pop-command-uidl (pro pnm)
  (let (msgs num siz)
    (while (re-search-forward "^\\([0-9]+\\) \\([0-9]+\\)" nil t)
      (setq num (mew-match 1))
      (setq siz (mew-match 2))
      (setq msgs (cons (list num siz) msgs)))
    (if msgs
	(progn
	  (setq msgs (nreverse msgs))
	  (mew-pop-set-messages pnm msgs)
	  (process-send-string pro (format "uidl%s" mew-cs-eol)))
      (mew-pop-set-status pnm "quit")
      (mew-pop-command-quit pro pnm))))

(defun mew-pop-command-umsg (pro pnm)
  (let ((old-msgs (mew-pop-get-messages pnm)) ;; (num siz)
	(sul (mew-pop-get-sul pnm))           ;; (siz uid lmsg)
	(old-uidl (mew-pop-uidl-db-get pnm))
	num siz uid msgs uidl n siz-uid-lmsg)
    (while (re-search-forward "^\\([0-9]+\\) +\\([!-~]+\\)" nil t)
      (setq num (mew-match 1))
      (setq uid (mew-match 2))
      (cond
       (sul ;; (num siz uidl lmsg)
	(if (setq siz-uid-lmsg (mew-assoc-equal uid sul 1))
	    (setq msgs (cons (cons num siz-uid-lmsg) msgs))))
       (t ;; (num siz uidl)
	(setq uidl (cons uid uidl))
	(if (member uid old-uidl)
	    ()
	  (setq siz (nth 1 (assoc num old-msgs)))
	  (setq msgs (cons (list num siz uid) msgs))))))
    (mew-pop-set-uidl pnm uidl)
    (setq msgs (nreverse msgs))
    (mew-pop-set-messages pnm msgs)
    (setq n (length msgs))
    (if (= n 0)
	(progn
	  (mew-pop-message pnm "No new message")
	  (mew-pop-set-status pnm "quit")
	  (mew-pop-command-quit pro pnm))
      (if (= n 1)
	  (mew-pop-message pnm "Receiving 1 message in background ... " )
	(mew-pop-message pnm "Receiving %d messages in background ... " n))
      (mew-pop-set-status pnm "retr")
      (mew-pop-command-retr pro pnm))))

(defun mew-pop-command-retr (pro pnm)
  (let* ((directive (mew-pop-get-directive pnm))
	 (msgs (mew-pop-get-messages pnm))
	 (msg  (car msgs))
	 (num  (nth 0 msg))
	 (siz  (nth 1 msg))
	 (lmsg (nth 3 msg))
	 (lim (mew-pop-get-size pnm))
	 (lines (mew-pop-get-body-lines pnm)))
    (cond
     ((eq directive 'biff)
      (mew-pop-set-status pnm "quit")
      (mew-pop-command-quit pro pnm))
     ((null num)
      (unless (mew-pop-get-sul pnm)
	(mew-pop-uidl-db-set pnm (mew-pop-get-uidl pnm)))
      (mew-pop-set-status pnm "quit")
      (mew-pop-command-quit pro pnm 'save))
     (lmsg
      (mew-pop-set-lmsg pnm lmsg)
      (mew-pop-set-top pnm nil)
      (process-send-string pro (format "retr %s%s" num mew-cs-eol)))
     ((or (= lim 0) (< (string-to-int siz) lim))
      (mew-pop-set-top pnm nil)
      (process-send-string pro (format "retr %s%s" num mew-cs-eol)))
     (t
      (mew-pop-set-top pnm t)
      (process-send-string
       pro (format "top %s %d%s" num lines mew-cs-eol))))))

(defun mew-pop-command-dele (pro pnm)
  (let* ((inbox (mew-pop-get-inbox pnm))
	 (case (mew-pop-get-case pnm))
	 (width (1- (mew-scan-width)))
	 (msgs (mew-pop-get-messages pnm))
	 (msg  (car msgs))
	 (num  (nth 0 msg))
	 (siz  (nth 1 msg))
	 (uid  (nth 2 msg))
	 (lmsg (nth 3 msg))
	 (truncated (mew-pop-get-top pnm))
	 vec file)
    ;; deleting +OK
    (goto-char (point-min))
    (forward-line)
    (delete-region (point-min) (point))
    ;; line delimiters
    (when (string= mew-cs-eol "\r\n")
      (goto-char (point-min))
      (while (search-forward "\r\n" nil t) (replace-match "\n" nil t)))
    ;; deleting \n.\n
    (goto-char (point-max))
    (forward-line -2)
    (end-of-line)
    (delete-region (point) (point-max))
    ;; unescape ^.
    (goto-char (point-min))
    (while (re-search-forward "^\\." nil t)
      (delete-char -1)
      (forward-line))
    (if lmsg
	(setq file (mew-expand-folder inbox lmsg))
      (setq lmsg (mew-pop-get-lmsg pnm))
      (if lmsg 
	  (setq file (mew-expand-folder inbox lmsg))
	(setq file (mew-folder-new-message inbox))
	(setq lmsg (file-name-nondirectory file)))
      (mew-pop-set-lmsg pnm (int-to-string (1+ (string-to-int lmsg)))))
    (when truncated
      (goto-char (point-min))
      (unless (re-search-forward mew-eoh nil t)
	(goto-char (point-max)))
      (when uid
	(if case
	    (mew-header-insert mew-x-mew-uidl: (concat uid " " siz " " case))
	  (mew-header-insert mew-x-mew-uidl: (concat uid " " siz))))
      (forward-line)
      (insert
       (format mew-pop-size-over-message
	       (substitute-command-keys 
		"\\<mew-summary-mode-map>\\[mew-summary-retrieve-message]"))))
    (catch 'write-error
      (condition-case nil
	  (mew-frwlet 
	   mew-cs-dummy mew-cs-text-for-write
	   (write-region (point-min) (point-max) file nil 'no-msg))
	(error
	 (mew-pop-set-status pnm "quit")
	 (mew-pop-command-quit pro pnm)
	 (throw 'write-error nil)))
      ;;
      (set-file-modes file mew-file-mode)
      (mew-pop-set-cnt pnm (1+ (mew-pop-get-cnt pnm)))
      ;;
      (mew-set-buffer-multibyte t)
      (setq vec (mew-pop-scan-header))
      (mew-scan-set-folder vec inbox)
      (mew-scan-set-message vec lmsg)
      (mew-set-buffer-multibyte nil)
      (mew-scan-insert-line inbox vec width lmsg nil)
      (mew-pop-set-messages pnm (cdr msgs))
      (if (and (mew-pop-get-delete pnm) (not truncated))
	  (process-send-string
	   pro (format "dele %s%s" num mew-cs-eol))
	(mew-pop-set-status pnm "retr")
	(mew-pop-command-retr pro pnm)))))

(defun mew-pop-command-quit (pro pnm &optional save)
  (if save (mew-lisp-save mew-pop-uidl-file mew-pop-uidl-db))
  (process-send-string
   pro (format "quit%s" mew-cs-eol)))

(defun mew-pop-command-noop (pro pnm)
  ())

(defun mew-pop-scan-header ()
  (goto-char (point-min))
  (if (re-search-forward mew-eoh nil t)
      ()
    (goto-char (point-max)))
  (mew-scan-header))

(provide 'mew-pop)

;;; Copyright Notice:

;; Copyright (C) 1999-2001 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-pop.el ends here
