/*
** TextExport.c
**
**	Functions to convert internal represetantion of the Text into
**	external export format(s).
**
**	Currently:
**		- STRING (plain 8bit ISO 8859-1)
**		- COMPOUND_TEXT	(extended version ISO 2022 + ISO 6429)
**
** Copyright 1993-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <string.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xew/TextP.h>
#include "TextImport.h"
#include "TextExport.h"

#define ExportFormatted(f) ((f)&1)
#define ExportOdif(f) ((f)&2)
#define ExportOdifFP(f) ((f)&4)

#define ESC 0x1b

#define PLD_Fe 0x4b	/* Fe for PLD in 7bit ESC Fe */
#define PLU_Fe 0x4c	/* Fe for PLU in 7bit ESC Fe */
#define PLD 0x8b	/* 8bit PLD (C1) */
#define PLU 0x8c	/* 8bit PLU (C1) */

typedef struct Rendition
    {

	unsigned int mask, value;
	int set;
    } Rendition;

#define NO_VALUE (~0)

static Rendition renditions[] =
    {
	{Weight_MASK, Weight_BOLD, 1},
	{Weight_MASK, Weight_FAINT, 2},
	{Weight_MASK, Weight_NORMAL, 22},
	{Italicized_MASK, Italicized_MASK, 3},
	{Italicized_MASK, 0, 23},
	{Underline_MASK, Underline_SINGLE, 4},
	{Underline_MASK, Underline_DOUBLE, 21},
	{Underline_MASK, Underline_NONE, 24},
	{Blinking_MASK, Blinking_SLOWLY, 5},
	{Blinking_MASK, Blinking_RAPIDLY, 6},
	{Blinking_MASK, Blinking_STEADY, 25},
	{ImageInversion_MASK, ImageInversion_MASK, 7},
	{ImageInversion_MASK, 0, 27},
	{CrossedOut_MASK, CrossedOut_MASK, 9},
	{CrossedOut_MASK, 0, 29},
	{Framing_MASK, Framing_FRAMED, 51},
	{Framing_MASK, Framing_ENCIRCLED, 52},
	{Framing_MASK, Framing_NONE, 54},
	{Overlined_MASK, Overlined_MASK, 53},
	{Overlined_MASK, 0, 55},
	{Foreground_MASK, 0, 39},	/* Use default foreground */
	{Background_MASK, 0, 49},	/* Use default background */
	{Foreground_MASK, NO_VALUE, -1},
	{Background_MASK, NO_VALUE, -1},
	{Font_MASK, NO_VALUE, -1},
    };

/*
** Change_ISO6429
**	Generate an ESC sequence to match the change in the state
**	defined by ISO 6429
**
**	NOTE:	The use of HEX character values is intentional.
**		These codes must not depend on C compiler characterset.
*/
#if NeedFunctionPrototypes
static void Change_ISO6429
	(SnipMode *, SnipMode *, XeTextExtractFeed, XeTextTag, XtPointer);
#endif
static void Change_ISO6429(old, new, feed, tag, client_data)
SnipMode *old, *new;
XeTextExtractFeed feed;
XtPointer client_data;
XeTextTag tag;
    {
	unsigned int change = old->bits ^ new->bits;
	char buf[3+XtNumber(renditions)*4+	/* Max use by SGR */
		 4+				/* Max use by PLD/PLU */
		 5+2*6+				/* Max use by GSM */
		 1];				/* NUL */
	Rendition *r = &renditions[0];
	char *s = buf, *mark;
	int delim;
	if (change == 0 && old->size_modification == new->size_modification)
		return;
	mark = s;
	*s++ = ESC;
	delim = 0x5b;	/* [   */
	for ( ;r < &renditions[XtNumber(renditions)]; r += 1)
		if (change & r->mask)
		    {
			int par;

			if (r->value == (r->mask & new->bits))
				par = r->set;
			else if (r->value != NO_VALUE)
				continue;
			else if (r->mask == Background_MASK)
				par = 39 + Background_COLOR(new->bits);
			else if (r->mask == Foreground_MASK)
				par = 29 + Foreground_COLOR(new->bits);
			else if (r->mask == Font_MASK)
				par = 10 + Font_NUMBER(new->bits);
			else
				continue;
			sprintf(s, "%c%d", delim, par);
			s += strlen(s);
			delim = 0x3b;		/* ; */
			change &= ~r->mask;	/* This change processed */
		    }
	if (s > mark+1) /* Anything really generated? */
		*s++ = 0x6d;	/* m  */
	else
		s = mark; /* Reset to empty buffer */
	if (change & Justify_MASK)
	    {
		/* Generate JFY sequence (ISO 6429) */

		XeAlignment align = Justify_MODE(new->bits);

		*s++ = ESC;
		*s++ = 0x5b;	/* [ */
		switch (align)
		    {
		    case XeAlignment_START_FILL:
			*s++ = 0x31;	/* Word Fill */
			*s++ = 0x3b;
		    case XeAlignment_START:
			*s++ = 0x35;
			break;

		    case XeAlignment_CENTER_FILL:
			*s++ = 0x31;	/* Word Fill */
			*s++ = 0x3b;
		    case XeAlignment_CENTER:
			*s++ = 0x36;
			break;

		    case XeAlignment_END_FILL:
			*s++ = 0x31;	/* Word Fill */
			*s++ = 0x3b;
		    case XeAlignment_END:
			*s++ = 0x37;
			break;

		    case XeAlignment_JUSTIFIED:
			*s++ = 0x32;	/* Word Space */
			break;

		    default:
			break;
		    }
		*s++ = 0x20;
		*s++ = 0x46;
	    }
	if (change & PartialLine_MASK)
	    {
		/*
		** Generate PLD/PLU with ESC Fe, not C1 control
		*/
		int i = 0, Fe;

		if (old->bits & PartialLine_UP)
			i -= 1;
		else if (old->bits & PartialLine_DOWN)
			i += 1;
		if (new->bits & PartialLine_UP)
			i += 1;
		else if (new->bits & PartialLine_DOWN)
			i -= 1;
		if (i < 0)
		    {
			Fe = PLD_Fe;
			i = -i;
		    }
		else
			Fe = PLU_Fe;
		while (--i >= 0)
		    {
			*s++ = ESC;
			*s++ = Fe;
		    }
	    }
	if (old->size_modification != new->size_modification)
	    {
		/*
		** Graphic Size Modification (ISO 6429)
		** (Only vertical size change supported)
		*/
		sprintf(s,"\033[%d B", (int)new->size_modification);
		s += strlen(s);
	    }
	if (s > buf)
		(*feed)(buf, s - buf, tag, client_data);
    }

typedef struct CodingState
    {
	int I[2];			/* Current invocations */
	XeCharsetDesignation G[4];	/* Current set of designations */
	XeCharsetIdentifier charset;	/* Current active charset */
	XeCharsetDesignation *assumed;	/* Assumed Default (GL,GR) State */
    } CodingState;

static CodingState iso8859_state =
    {
	{0, 1},		/* GL <- G0, GR <- G1 */
	{0x42|XeCharset_G0, 0x41|XeCharset_96, 0, 0},
    };

static char lockshift[2][4][3] =
    {
	{
	    {0x0f, 0x00, 0x00},	/* G0 to GL */
	    {0x0e, 0x00, 0x00},	/* G1 to GL */
	    {0x1b, 0x6e, 0x00},	/* G2 to GL */
	    {0x1b, 0x6f, 0x00},	/* G3 to GL */
	},
	{
	    {0x00, 0x00, 0x00},	/* G0 to GR (invalid combination *) */
	    {0x1b, 0x7e, 0x00},	/* G1 to GR */
	    {0x1b, 0x7d, 0x00},	/* G2 to GR */
	    {0x1b, 0x7c, 0x00},	/* G3 to GR */
	},

    };

/*
** CodeSwitch
**	Generate an ISO 2022 ESC sequences to change the current state
**	to fit the required (GL,GR) configuration. Returns a pointer to
**	to a staticly allocated buffer that will contain the required
**	state change escapes if any. The current state is updated to
**	match the state that will be in effect after the escape sequences
**	have been executed.
**
**	state	describes the current state
**
**	gx[0]	what should be invoked and designated into GL
**	gx[1]	what should be invoked and designated into GR 
*/
#if NeedFunctionPrototypes
static char *CodeSwitch (CodingState *, XeCharsetDesignation *);
#endif
static char *CodeSwitch(cur, gx)
CodingState *cur;
XeCharsetDesignation *gx;
    {
	static char buffer[30];

	int i, j, g;
	char *s, *buf = buffer;

	for (g = 0; g < 2; g++)
	    {
		if (gx[g] == 0)
			continue;	/* Don't care */
		/*
		** Search the G0 to G4 if the designation already exists
		** and if not, designate it into one of them (if there is
		** no free G slot, use G0 for GL and G1 for GR by default).
		*/
		for (i = 0, j = g; ; i++)
			if (i == 4)
			    {
				/* Always use G0? */
				if (XeCharset_G0 & gx[g])
					j = 0;
				/* Invoke character set into G[j] */
				cur->G[j] = gx[g];
				*buf++ = 0x1b;	/* ESC */
				if (gx[g] & XeCharset_M)
					*buf++ = 0x24;	/* $ */
				i = j +	((gx[g] & XeCharset_96) ? 0x2c : 0x28);
				if (j > 0 || !(gx[g] & XeCharset_M))
					*buf++ = i;
				*buf++ = (gx[g] & XeCharset_F);
				i = j;
				break;
			    }
			else if (gx[g] == cur->G[i])
				break;
			else if (!cur->G[i] && j == g)
				j = i;
		/*
		** Invoke G[i] into GL (g == 0) or GR (g == 1), if not
		** already invoked.
		*/
		if (cur->I[g] != i)
		    {
			for (s = &lockshift[g][i][0]; *s; *buf++ = *s++);
			cur->I[g] = i;
		    }
	    }
	*buf = 0;
	return buffer;
    }


#if NeedFunctionPrototypes
static void FeedFromSnip
	(CodingState *,Snip *,int,int,XeTextExtractFeed,XeTextTag,XtPointer);
#endif
static void FeedFromSnip(state, s, offset, length, feed, tag, client_data)
CodingState *state;
Snip *s;
int offset;
int length;
XeTextExtractFeed feed;
XeTextTag tag;
XtPointer client_data;
    {
	if (s->data == NULL)
		return;
	if (s->length - offset < length)
		length = s->length - offset;
	if (state->charset != s->content.head->character_set)
	    {
		/*
		** A very temporary kludge for generating at least
		** some ISO 2022 character set switching to the output.
		*/
		char *str = CodeSwitch(state,
			  XeTextCharsetSelect(s->content.head->character_set));
		if (*str)
			(*feed)(str, strlen(str), tag, client_data);
		state->charset = s->content.head->character_set;
	    }
	/*
	** Never call "feed" with ZERO length for normal data! ZERO length
	** is reserved for indication of insets..
	*/
	if (length)
		(*feed)((char *)(&s->data[s->content.head->bytes * offset]),
			s->content.head->bytes * length, tag, client_data);
    }

#if NeedFunctionPrototypes
static void FeedEndSeq
	(CodingState *, Snip *, XeTextExport, XeTextExtractFeed,
	 XeTextTag, XtPointer);
#endif
static void FeedEndSeq(state, s, format, feed, tag, client_data)
CodingState *state;
Snip *s;
XeTextExport format;
XeTextExtractFeed feed;
XeTextTag tag;
XtPointer client_data;
    {
	static char CR[] = { 0x0d};
	static char LF[] = { 0x0a};
	static char HT[] = { 0x09};
	static char RS[] = { 0x1e};

	char *str;

	/*
	** Return to the assumed default state at end of each sequence
	*/
	str = CodeSwitch(state, state->assumed);
	if (*str)
	    {
		(*feed)(str, strlen(str), tag, client_data);
		state->charset = NULL;
	    }
	/*
	** Currently, insets are always marked with EndSeq and it is
	** sufficient to check them here only. If an inset appears, just
	** call feed with ZERO length and pass the widget as data. It is
	** up to the application to hook up something to this (by default
	** insets will get skipped due to length being ZERO).
	*/
	if (s->widget)
		(*feed)((char *)s->content.widget, 0, tag, client_data);
	/*
	** Output ISO 6429 QUAD sequence
	*/
	if (s->quad && ExportOdif(format))
	    {
		char *b;

		if (s->quad == XeAlignment_START)
			b = "\033[0 H";
		else if (s->quad == XeAlignment_CENTER)
			b = "\033[2 H";
		else if (s->quad == XeAlignment_END)
			b = "\033[4 H";
		else if (s->quad == XeAlignment_JUSTIFIED)
			b = "\033[6 H";
		else
			b = NULL;
		if (b)
			(*feed)(b, strlen(b), tag, client_data);
	    }
	switch (s->endseq)
	    {
	    case Snip_End:
		if (s->tab && s->tabref == Snip_NEXTTABSTOP)
			(*feed)(HT, sizeof(HT), tag, client_data);
		else if (s->tab)
		    {
			char b[4+sizeof(int)*3+1];
			/* ..should not use ASCII string! */
			sprintf(b,"\33[%d ^", (int)s->tabref);
			(*feed)(b, strlen(b), tag, client_data);
		    }
		else if (s->widget)
			break; /* -- already done */
		else if (!s->quad || !ExportOdif(format))
			(*feed)(CR, sizeof(CR), tag, client_data);
		break;
	    case Snip_EndParagraph:
		/*
		** This is not really ODIF compatible. Use
		** RS (Record Separator) as a temporary
		** solution for Paragraph End marker.
		*/
		if (ExportOdif(format))
			(*feed)(RS, sizeof(RS), tag, client_data);
		else
		    {
			(*feed)(LF,sizeof(LF), tag, client_data);
			(*feed)(LF,sizeof(LF),tag,client_data);
		    }
		break;
	    default:
		(*feed)(LF, sizeof(LF), tag, client_data);
		break;
	    }
    }

void XeTextExtractContent(w, start, end, format, feed, client_data)
XeTextWidget w;		/* Text Widget */
XeTextLocation *start;	/* Start position */
XeTextLocation *end;	/* End position */
XeTextExport format;	/* Export format */
XeTextExtractFeed feed;	/* Function to process the extract */
XtPointer client_data;	/* client_data for ExtractFeed function */
    {
	static SnipMode init_mode = {0,0,100};

	Snip *s = start->snip;
	int offset = start->offset;
	SnipMode m;
	XeTextTag tag = 0;
	CodingState state;
	char *str;

	static char SOS[] = { 0x98};
	static char ST[] = { 0x9c};

	m = init_mode;
	state = iso8859_state;
	state.assumed = XeTextCharsetSelect(XeTextCharsetDefault());
	if (s == NULL)
	    {
		offset = 0;
		s = w->text.first;
	    }
	for ( ;s; s = s->next)
	    {
		int source = IsEditableContent(s->mode.bits);
		if (source)
			tag = s->mode.tag;
		if (!ExportFormatted(format) && !source)
			continue;
		if (ExportOdifFP(format) && !source)
			(*feed)(SOS, sizeof(SOS), tag, client_data);
		if (ExportOdif(format))
			Change_ISO6429(&m, &s->mode, feed, tag, client_data);
		if (s == end->snip)
		    {
			FeedFromSnip(&state, s, offset, end->offset - offset,
				     feed, tag, client_data);
			if (end->offset > s->length && s->endseq)
				FeedEndSeq(&state, s, format, feed, tag,
					   client_data);
			break;
		    }
		FeedFromSnip(&state, s, offset, s->length - offset, feed,
			     tag, client_data);
		if (s->endseq)
			FeedEndSeq(&state, s, format, feed, tag, client_data);
		if (ExportOdifFP(format) && !source)
			(*feed)(ST, sizeof(ST), tag, client_data);
		offset = 0;
		m = s->mode;
	    }
	/*
	** Return to the assumed default state at end of extract
	*/
	str = CodeSwitch(&state, state.assumed);
	if (*str)
		(*feed)(str, strlen(str), tag, client_data);
    }

