/*
** Copyright 1994-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/CharSet.h>
#include <X11/Xew/XewInit.h>
#include <X11/Xew/BasicP.h>
#include <X11/Xew/FrameP.h>

#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define MIN(a,b) ((a) < (b) ? (a) : (b))

/*
** FRAME_NO_PIXEL
**	is used in constraint border_pixel[i] to indicate that
**	core.border_pixel value should be used. This value should be such
**	that XAllocColor never returns it as pixel. [If this value happens
**	to be a valid pixel, the color will not be shown, and it is never
**	released, if allocated]
*/
#define FRAME_NO_PIXEL	~((unsigned long)0)

#if NeedFunctionPrototypes
static void Notify(Widget, XEvent *, String *, Cardinal *);
#else
static void Notify();
#endif

static XtActionsRec actions[] =
    {
	    {"notify",		Notify},
    };

#define offset(field) XtOffsetOf(XeFrameRec, frame.field)

static XeFramePosition init_FramePosition;
static XeFrameDimension init_FrameDimension;
static XeFrameBorder init_FrameBorder;
static XeFrameAlignment init_FrameAlignment = XeFrameAlignment_RIGHT;

static XtResource resources[] =
    {
	{XtNbottomShadowContrast, XtCBottomShadowContrast, XtRInt, sizeof(int),
		 offset(shadow_contrast[FRAME_BOTTOM]),
		 XtRImmediate, (XtPointer)40},
	{XtNinvertShadow, XtCInvertShadow, XtRBoolean, sizeof(Boolean),
		 offset(invert_shadow), XtRImmediate, (XtPointer)False},
	{XtNlayoutPath, XtCLayoutPath, XtRInt, sizeof(int),
		 offset(layout_path), XtRImmediate, (XtPointer)270},
	{XtNleadingBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), offset(border[FRAME_LEADING]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNleftBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), offset(border[FRAME_LEFT]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNoverflowCallback,XtCOverflowCallback,XtRCallback,sizeof(XtPointer),
		 offset(overflow_callbacks), XtRCallback, (XtPointer)NULL},
	{XtNrightBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), offset(border[FRAME_RIGHT]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNtopShadowContrast, XtCTopShadowContrast, XtRInt, sizeof(int),
		 offset(shadow_contrast[FRAME_TOP]),
		 XtRImmediate,(XtPointer)20},
	{XtNtrailingBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), offset(border[FRAME_TRAILING]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
    };

#undef offset


/* using 'OFFSET' instead of 'offset', because offset is a field also! */

#define OFFSET(field) XtOffsetOf(XeFrameConstraintsRec, frame.field)

static XtResource subresources[] =
    {
	{XtNbottomShadowContrast, XtCBottomShadowContrast, XtRInt, sizeof(int),
		 OFFSET(shadow_contrast[FRAME_BOTTOM]),
		 XtRImmediate,(XtPointer)40},
	{XtNframeAlignment, XtCFrameAlignment, XeRFrameAlignment,
		 sizeof(XeFrameAlignment), OFFSET(alignment),
		 XeRFrameAlignment, (XtPointer)&init_FrameAlignment},
	{XtNframePosition, XtCFramePosition, XeRFramePosition,
		 sizeof(XeFramePosition), OFFSET(position),
		 XeRFramePosition, (XtPointer)&init_FramePosition},
	{XtNcenterSeparation, XtCFrameSeparation, XtRInt,
		 sizeof(int),OFFSET(separation[FRAME_CENTER]),XtRImmediate,0},
	{XtNhorizontal, XtCFrameDimension, XeRFrameDimension,
		 sizeof(XeFrameDimension), OFFSET(dimension[FRAME_HORIZONTAL]),
		 XeRFrameDimension, (XtPointer)&init_FrameDimension},
	{XtNinvertShadow, XtCInvertShadow, XtRBoolean, sizeof(Boolean),
		 OFFSET(invert_shadow), XtRImmediate, (XtPointer)False},
	{XtNleadingBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), OFFSET(border[FRAME_LEADING]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNleadingOffset, XtCFrameOffset, XtRInt,
		 sizeof(int), OFFSET(offset[FRAME_LEADING]), XtRImmediate, 0},
	{XtNleadingSeparation, XtCFrameSeparation, XtRInt,
		 sizeof(int),OFFSET(separation[FRAME_LEADING]),XtRImmediate,0},
	{XtNleftBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), OFFSET(border[FRAME_LEFT]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNleftOffset, XtCFrameOffset, XtRInt,
		 sizeof(int), OFFSET(offset[FRAME_LEFT]), XtRImmediate, 0},
	{XtNrightBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), OFFSET(border[FRAME_RIGHT]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNrightOffset, XtCFrameOffset, XtRInt,
		 sizeof(int), OFFSET(offset[FRAME_RIGHT]), XtRImmediate, 0},
	{XtNtopShadowContrast, XtCTopShadowContrast, XtRInt, sizeof(int),
		 OFFSET(shadow_contrast[FRAME_TOP]),
		 XtRImmediate,(XtPointer)20},
	{XtNtrailingBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), OFFSET(border[FRAME_TRAILING]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNtrailingOffset, XtCFrameOffset, XtRInt,
		 sizeof(int), OFFSET(offset[FRAME_TRAILING]), XtRImmediate, 0},
	{XtNtrailingSeparation, XtCFrameSeparation, XtRInt, sizeof(int),
		 OFFSET(separation[FRAME_TRAILING]),XtRImmediate,0},
	{XtNvertical, XtCFrameDimension, XeRFrameDimension,
		 sizeof(XeFrameDimension), OFFSET(dimension[FRAME_VERTICAL]),
		 XeRFrameDimension, (XtPointer)&init_FrameDimension},

	/* Just to silence purify, initialize the 4th element too! */
	{"unusedSeparation", XtCFrameSeparation, XtRInt,
		 sizeof(int),OFFSET(separation[FRAME_LEFT]),XtRImmediate,0},
    };

#undef OFFSET

static void ClassInitialize(), Initialize();
static void Realize(), Resize(), Redisplay(), Destroy();
static XtGeometryResult GeometryManager();
static void ChangeManaged(), InsertChild(), DeleteChild();
static Boolean SetValues(), FrameSetValues();
static XtGeometryResult QueryGeometry();

static void PrintInitialize();
static void PrintCancel(), PrintDestroy();
static XtPointer Print();
static void PrintConstraintInitialize(), PrintConstraintDestroy();

#ifdef USING_MOTIF_122
static XmNavigability WidgetNavigable() ;

/*******************************************/
/*  Declaration of class extension records */

XmBaseClassExtRec _XeFramebaseClassExtRec = {
    NULL,
    NULLQUARK,
    XmBaseClassExtVersion,
    sizeof(XmBaseClassExtRec),
    NULL,				/* InitializePrehook	*/
    NULL,				/* SetValuesPrehook	*/
    NULL,				/* InitializePosthook	*/
    NULL,				/* SetValuesPosthook	*/
    NULL,				/* secondaryObjectClass	*/
    NULL,				/* secondaryCreate	*/
    NULL,		                /* getSecRes data	*/
    { 0 },				/* fastSubclass flags	*/
    NULL,				/* get_values_prehook	*/
    NULL,				/* get_values_posthook	*/
    NULL,                               /* classPartInitPrehook */
    NULL,                               /* classPartInitPosthook*/
    NULL,                               /* ext_resources        */
    NULL,                               /* compiled_ext_resources*/
    0,                                  /* num_ext_resources    */
    FALSE,                              /* use_sub_resources    */
    WidgetNavigable,                    /* widgetNavigable      */
    NULL,                               /* focusChange          */
};
#endif

#define SuperClass (&xeBasicClassRec)

XeFrameClassRec xeFrameClassRec = {
  { /* core class fields */
    /* superclass		*/	(WidgetClass) SuperClass,
    /* class name		*/	"XeFrame",
    /* size			*/	sizeof(XeFrameRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part init		*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* resource_count		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	XtInheritTranslations,
    /* query_geometry		*/	QueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
#ifdef USING_MOTIF_122
    /* extension		*/	(XtPointer)&_XeFramebaseClassExtRec,
#else
    /* extension		*/	NULL,
#endif
  },
  { /* composite class fields */
    /* geometry_manager		*/	GeometryManager,
    /* change_managed		*/	ChangeManaged,
    /* insert_child		*/	InsertChild,
    /* delete_child		*/	DeleteChild,
    /* extension		*/	NULL
  },
  { /* constraint class fields */
    /* subresources		*/	subresources,
    /* subresource_count	*/	XtNumber(subresources),
    /* constraint_size		*/	sizeof(XeFrameConstraintsRec),
    /* initialize		*/	_XeFrameInitialize,
    /* destroy			*/	_XeFrameDestroy,
    /* set_values		*/	FrameSetValues,
    /* extension		*/	NULL
  },
#ifdef USING_MOTIF_122
  { /* manager class record	*/
    /* translations		*/	XtInheritTranslations,
    /* syn_resources		*/	NULL,
    /* num_syn_resources	*/	0,
    /* syn_constraint_resources */	NULL,
    /* num_syn_constraint_resources */	0,
    /* parent_process		*/	XmInheritParentProcess,
    /* extension		*/	NULL  
  },
#endif
  { /* basic fields */
    /* print_initialize		*/	PrintInitialize,
    /* print			*/	Print,
    /* print_cancel		*/	PrintCancel,
    /* print_destroy		*/	PrintDestroy,
    /* print_constraint_initialize */	PrintConstraintInitialize,
    /* print_constraint_destroy */	PrintConstraintDestroy
  },
  { /* XeFrame fields */
    /* not used			*/	0
  }
};

WidgetClass xeFrameWidgetClass = (WidgetClass)&xeFrameClassRec;


static void ClassInitialize()
    {
	XewInitializeWidgetSet();
	_XeFrameConverters();	/* Setup Converters for frame resources */
#ifdef USING_MOTIF_122
        _XeFramebaseClassExtRec.record_type = XmQmotif;
#endif
    }


typedef struct LayoutChild
    {
	struct LayoutChild *next;/* Previous item in the stack */
	struct LayoutStack *head;/* Back link to the stack structure */
	Widget child;		/* Child Widget */
	int position[2];	/* (x,y) position */
	int dimension[2];	/* (horizontal, vertical) dimensions */
	int b;			/* Core Border Width */
	XeFrameAlignment align;	/* Horizontal alignment of the child */
	int left, right;	/* Offsets from left, right */
	int trailing, leading;	/* Total leading/trailing border area */
	unsigned int no_query:1;/* True == No QueryGeometry calls for this! */
	unsigned int sized:1;	/* True == Layout size determined */
	unsigned int located:1;	/* True == Layout location determined */
    } LayoutChild;

typedef struct LayoutStack
    {
	struct XeFrameLayoutInfoRec *info; /* Back link to InfoRec */
	LayoutChild *top;	/* Topmost child in the stack */
	int allocation;		/* Allocation in layout path direction */
	int max_dimension;	/* Max dimension orthogonal to layout path */
	int start, end;		/* Offsets from the first/last child */
	int separation;		/* Center separation of the first/last child */
	int min_separation;	/* Minimum separation from the next child */
    } LayoutStack;

typedef struct XeFrameLayoutInfoRec
    {
	int path;		/* Encoded layout path mask */
	int redo;		/* Set True if layout should be redone */
	XeFrameDimension dimension[2];/* Frame Dimensions */
	int b_left;		/* Frame left border allocation */
	int b_right;		/* Frame right border allocation */
	int b_trailing;		/* Frame trailing border allocation */
	int b_leading;		/* Frame leading border allocation */
	LayoutStack normal;	/* Children with normal layout positioning */
	LayoutStack reverse;	/* Children with reverse layout positioning */
	LayoutStack fixed;	/* Children with fixed layout positioning */
	int maximize;		/* "Maximize" children in layout direction */
	int num_children;	/* Number of children in this structure */
	LayoutChild list[1];	/* Computed layout information */
    } LayoutInfo;

typedef struct FrameSize
    {
	XeFrameDimension width;
	XeFrameDimension height;
	int border_width;
    } FrameSize;

/*
** Alignment
**	Compute alignment offset orthogonal to the layouout path
**	direction. Returns the the coordinate value (x or y) which
**	is directly usable for geometry (all adjustments for all
**	borders have been computed).
*/
#if NeedFunctionPrototypes
static int Alignment(LayoutInfo *, LayoutChild *, int, int);
#endif

static int Alignment(info, child, path, invert)
LayoutInfo *info;
LayoutChild *child;
int path, invert;
    {
	int offset;
	int free_space = info->dimension[1-path].value
		- child->left - child->right - child->dimension[1-path]
		- info->b_left - info->b_right;

	if (invert ^ (path == FRAME_HORIZONTAL))
		offset = info->b_left + child->left;
	else
		offset = info->b_right + child->right;
	offset -= child->b;
	if (child->align ==  XeFrameAlignment_CENTER)
		offset += free_space / 2;
	else if (((child->align == XeFrameAlignment_LEFT) == (!invert))
		 ^ (path == FRAME_HORIZONTAL))
		offset += free_space;
	return offset;
    }

/*
** ComputeStack
**	compute positions of the children on stack
*/
#if NeedFunctionPrototypes
static void ComputeStack(LayoutInfo *, LayoutChild *, int, int, int);
#endif
static void ComputeStack(info, child, top, path, invert)
LayoutInfo *info;
LayoutChild *child;
int top, path, invert;
    {
	int space;

	for (; child; child = child->next)
	    {
		space = child->trailing + child->dimension[path]
			+ child->leading;
		if (invert)
		    {
			child->position[path] = top + child->leading;
			top += space;
		    }
		else
		    {
			top -= space;
			child->position[path] = top + child->trailing;
		    }
		child->position[path] -= child->b;
		child->position[1-path] =
			Alignment(info, child, path, invert);
		child->located = True;
	    }
    }

static void LimitDimension(dim)
XeFrameDimension *dim;
    {
	if (dim->min > 0 && dim->value < dim->min)
		dim->value = dim->min;
	if (dim->max > 0 && dim->value > dim->max)
		dim->value = dim->max;
    }

static void GetGeometry(child, w, h, w_out, h_out)
LayoutChild *child;
int w, h;
int *w_out, *h_out;
    {
	XtWidgetGeometry intended, preferred;

	if (h <= 0 || w <= 0)
	    {
		intended.request_mode = 0;
		if (h > 0)
		    {
			intended.request_mode |= CWHeight;
			intended.height = h;
		    }
		if (w > 0)
		    {
			intended.request_mode |= CWWidth;
			intended.width = w;
		    }
		if (child->no_query)
		    {
			preferred.width = child->dimension[0];
			preferred.height = child->dimension[1];
		    }
		else
			XtQueryGeometry(child->child, &intended, &preferred);
		if (h <= 0)
			h = preferred.height;
		if (w <= 0)
			w = preferred.width;
	    }
	*w_out = w;
	*h_out = h;
    }

#if NeedFunctionPrototypes
static void MaximizeChildren(LayoutInfo *, int, int, int);
#endif
static void MaximizeChildren(info, num, spare, path)
LayoutInfo *info;
int num, spare, path;
    {
	int adjust = spare / num;
	int extra = spare - adjust * num;
	int i;

	for (i = 0; i < info->num_children; i++)
	    {
		XeFrameConstraints fc = (XeFrameConstraints)
			info->list[i].child->core.constraints;

		if (fc->frame.dimension[path].value == XeFrameDimension_MAX)
		    {
			int twiddle = (extra > 0);

			info->list[i].dimension[path] += adjust + twiddle;
			info->list[i].head->allocation += adjust + twiddle;
			extra -= twiddle;
		    }
	    }
    }

#if NeedFunctionPrototypes
static void AddToStack(LayoutChild *,XeFrameConstraints,LayoutStack *,int,int);
#endif
static void AddToStack(child, fc, stack, path, reverse)
LayoutChild *child;
XeFrameConstraints fc;
LayoutStack *stack;
int path, reverse;
    {
	int separation, room;
	int frame_size[2], padding[2];
	int b_left, b_right, b_leading, b_trailing;
	XeFrameDimension dimension[2];

	frame_size[path] = stack->info->dimension[path].value;
	frame_size[1-path] = stack->info->dimension[1-path].value;
	/*
	** If the frame size in layout path direction is fixed, then
	** take off the space already allocated.
	**
	** (Unfortunately, this is a moving target. The final allocation
	** depends also on the children and their required separation...)
	*/
	if (frame_size[path] > 0)
		frame_size[path] -=
			stack->info->reverse.allocation +
			stack->info->normal.allocation +
			stack->info->b_leading +
			stack->info->b_trailing;

	dimension[0] = fc->frame.dimension[0];
	dimension[1] = fc->frame.dimension[1];
	/*
	** trailing will indicate how much distance there is to be from
	** the leading edge border of the previous item (or from the frame
	** border itself, if this is the first item.
	*/
	if (stack->top)
		separation = MAX(stack->min_separation,
				 fc->frame.separation[FRAME_TRAILING]);
	else
	    {
		if (reverse)
		    {
			stack->end = fc->frame.offset[FRAME_TRAILING];
			stack->separation = fc->frame.separation[FRAME_CENTER];
		    }
		else
			stack->start = fc->frame.offset[FRAME_TRAILING];
		separation = 0;
	    }
	if (reverse)
		stack->start = fc->frame.offset[FRAME_LEADING];
	else
	    {
		stack->end = fc->frame.offset[FRAME_LEADING];
		stack->separation = fc->frame.separation[FRAME_CENTER];
	    }
	/*
	** Note: If the child is XeFrame, it will draw own borders inside
	*/
	if (!XtIsSubclass(child->child, xeFrameWidgetClass))
	    {
		b_left = fc->frame.border[FRAME_LEFT].value.space
			+ fc->frame.border[FRAME_LEFT].value.width;
		b_right = fc->frame.border[FRAME_RIGHT].value.space
			+ fc->frame.border[FRAME_RIGHT].value.width;
		b_leading = fc->frame.border[FRAME_LEADING].value.space
			+ fc->frame.border[FRAME_LEADING].value.width;
		b_trailing = fc->frame.border[FRAME_TRAILING].value.space
			+ fc->frame.border[FRAME_TRAILING].value.width;
		/*
		** Negative "space" values are possible, for example to
		** enable "shadow like" borders.
		*/
		if (b_left < 0)
			b_left = 0;
		if (b_right < 0)
			b_right = 0;
		if (b_leading < 0)
			b_leading = 0;
		if (b_trailing < 0)
			b_trailing = 0;
	    }
	else
		b_left = b_right = b_leading = b_trailing = 0;
	/*
	** Initialize total required border areas around the child
	** (include borders from Frame and also, the window borders)
	** Note, that only the largest value of Offset/Border pair
	** on each edge is used).
	*/
	child->left = MAX(fc->frame.offset[FRAME_LEFT], b_left) + child->b;
	child->right = MAX(fc->frame.offset[FRAME_RIGHT], b_right) + child->b;
	child->trailing = MAX(separation, b_trailing) + child->b;
	child->leading = b_leading + child->b;
	/*
	** padding[1-path] will contain all the extra stuff that must be
	** accounted in direction orthogonal to the layout path
	*/
	padding[1-path] = stack->info->b_left	/* Frame left border */
		+ stack->info->b_right	/* Frame right border */
		+ child->left + child->right;	/* Child left/right offsets */
	room = frame_size[1-path] - padding[1-path];
	if (room > 0 &&
	    (dimension[1-path].value == 0 || dimension[1-path].value > room))
		dimension[1-path].value = room;
	/*
	** If child requests Max in layout path direction, bump the counter.
	** This child will receive more space later...
	*/
	if (dimension[path].value == XeFrameDimension_MAX)
		stack->info->maximize += 1;
	GetGeometry(child, dimension[0].value, dimension[1].value,
		    &dimension[0].value, &dimension[1].value);
	/*
	** ugh.. again? The code below is very suspicious. It attempts to
	** force child size into hard limits, if the frame has fixed
	** dimension. Works reasonable for "width", but the height portion
	** is only checked individually for each child (e.g. none of childs
	** alone can exceed the fixed dimension in layout path direction).
	** --msa
	*/
	room = frame_size[1-path] - padding[1-path];
	if (room > 0 && dimension[1-path].value > room)
		dimension[1-path].value = room;
	padding[path] = child->trailing + child->leading;
	room = frame_size[path] - padding[path];
	if (room > 0 && dimension[path].value > room)
		dimension[path].value = room;
	/*
	** Further limit dimensions between (min,max) if dimension is not
	** explicitly  specified, and if min,max have been specified.
	*/
	if (fc->frame.dimension[0].value <= 0)
		LimitDimension(&dimension[0]);
	if (fc->frame.dimension[1].value <= 0)
		LimitDimension(&dimension[1]);
	stack->allocation +=
		dimension[path].value +  child->trailing + child->leading;
	child->head = stack;
	child->next = stack->top;
	stack->top = child;
	child->dimension[0] = dimension[0].value;
	child->dimension[1] = dimension[1].value;
	child->align = fc->frame.alignment;
	stack->min_separation = fc->frame.separation[FRAME_LEADING];
	/*
	** Keep track of maximum width (excluding frame borders)
	*/
	separation = child->dimension[1-path] + child->left + child->right;
	if (stack->max_dimension < separation)
		stack->max_dimension = separation;
	/*
	** If the dimension is to be fixed from the first child, and
	** has not yet been done, then do it now. (Child borders are
	** are included).
	*/
	if (stack->info->dimension[1-path].value == XeFrameDimension_RULE_A)
	    {
		stack->info->dimension[1-path].value =
			child->dimension[1-path] + padding[1-path];
		LimitDimension(&stack->info->dimension[1-path]);
	    }
	child->sized = True;
    }
/*
** FrameLayout
**	Recompute positions and sizes of all children (but do not
**	configure them).
**
**	mode = 0,	query layout only (with possible geometry
**			query to the parent)
**
**	mode > 0	configure with possible resize geometry
**			request to the parent.
**
**	size != NULL	only if called from QueryGeometry (mode == 0) or
**			from Resize (mode > 0).
*/
#define LAYOUT_QUERY_ONLY 0
#define LAYOUT_AND_COMMIT 1	/* Call XtConfigureWidget */

static void FrameLayout(fw, one, mode, size)
XeFrameWidget fw;
LayoutChild *one;
int mode;
FrameSize *size;
    {
	/*
	** path
	**	is a "flip-flop" variable that is being used in indexing
	**	dimension and position arrays. It has always either 0 or
	**	1 value and
	**
	**	path	gives dimension/position in layout path direction,
	**
	**	1-path	gives dimension/position orthogonal to the layout
	**		path direction.
	*/
	int i, glue, path, invert;
	LayoutChild *child;
	XeFrameConstraints fc;
	XeFrameDimension dimension[2];
	LayoutInfo *info;

	if (fw->frame.layout_info)
	    {
		/*
		** Has been called recursively. Probaby because number of
		** managed children has changed. Cannot do much here, just
		** flag a request for the original layout function to redo
		** everything when it gets to it. (Just hope it doesn't go
		** into loop... --msa)
		** [ ...well, it seems to get into loop, redo setting
		**   commented out for now... -- msa ]
		*/
		/* fw->frame.layout_info->redo = True; */
		return;
	    }
	/*
	** Start with the dimensions specified by the resources or size.
	** Redo's may later occur with changed values.
	*/
	if (size != NULL)
	    {
		/*
		** If 'size' parameter is present, it will override any
		** resource specifiers for the size.
		*/
		dimension[0] = size->width;
		dimension[1] = size->height;
	    }
	else
	    {
		dimension[0] = fw->basic.dimension[0];
		dimension[1] = fw->basic.dimension[1];
	    }
    redo_layout:
	info = fw->frame.layout_info = (LayoutInfo *)XtCalloc
		(1, XtOffsetOf(LayoutInfo, list[0]) +
		 fw->composite.num_children * sizeof(info->list[0]));
	info->num_children = fw->composite.num_children;
	info->normal.top = info->reverse.top = info->fixed.top = NULL;
	info->normal.info = info->reverse.info = info->fixed.info = info;
	info->dimension[0] = dimension[0];
	info->dimension[1] = dimension[1];

	if (fw->frame.layout_path == 0 || fw->frame.layout_path == 180)
	    {
		path = FRAME_HORIZONTAL;
		if (mode && info->dimension[1].value == XeFrameDimension_MAX)
			info->dimension[1].value = fw->core.height;
	    }
	else
	    {
		path = FRAME_VERTICAL;
		if (mode && info->dimension[0].value == XeFrameDimension_MAX)
			info->dimension[0].value = fw->core.width;
	    }
	invert = fw->frame.layout_path == 90 || fw->frame.layout_path == 180;
	info->b_left = fw->frame.border[FRAME_LEFT].value.space
		+ fw->frame.border[FRAME_LEFT].value.width;
	info->b_right = fw->frame.border[FRAME_RIGHT].value.space
		+ fw->frame.border[FRAME_RIGHT].value.width;
	info->b_trailing = fw->frame.border[FRAME_TRAILING].value.space
		+ fw->frame.border[FRAME_TRAILING].value.width;
	info->b_leading = fw->frame.border[FRAME_LEADING].value.space
		+ fw->frame.border[FRAME_LEADING].value.width;
	/*
	** Phase 1: Compute the required size without limits
	*/
	for (i = 0; i < info->num_children; ++i)
	    {
		child = &info->list[i];
		child->child = fw->composite.children[i];
		if (one && child->child == one->child)
			*child = *one;
		else
		    {
			child->position[0] = child->child->core.x;
			child->position[1] = child->child->core.y;
			child->dimension[0] = child->child->core.width;
			child->dimension[1] = child->child->core.height;
			child->b = child->child->core.border_width;
		    }
		if (!XtIsManaged(child->child))
			continue;  /* Unmanaged don't take part in layout */
		fc = (XeFrameConstraints)child->child->core.constraints;
		if (fc->frame.position.type == XeFramePositionType_FIXED)
		    {
			/* Fixed positioning */
			int dimension, allocation;

			allocation = info->fixed.allocation;
			info->fixed.allocation = 0;
			AddToStack(child, fc, &info->fixed, path, False);
			child->position[0] = fc->frame.position.x;
			child->position[1] = fc->frame.position.y;
			child->located = True;
			/*
			** Adjust max dimension to cover the child in the
			** specified fixed position (dimension orthogonal
			** to the path).
			*/
			dimension = child->dimension[1-path]
				+ child->position[1-path]
				+ child->left + child->right;
			if (dimension > info->fixed.max_dimension)
				info->fixed.max_dimension = dimension;
			/*
			** Adjust allocation to cover new child (this is
			** a quick hack.. may be erroneus --msa)
			*/
			info->fixed.allocation += child->position[path];
			if (allocation > info->fixed.allocation)
				info->fixed.allocation = allocation;
		    }
		else if (fc->frame.position.type == XeFramePositionType_NORMAL)
			AddToStack(child, fc, &info->normal, path, False);
		else
			AddToStack(child, fc, &info->reverse, path, True);
	    }
	/*
	** If the dimension orthogonal to layout direction has not yet been
	** fixed, do it now (make it large enough to contain all children).
	*/
	if (info->dimension[1-path].value <= 0)
	    {
		info->dimension[1-path].value =
			MAX(info->fixed.max_dimension,
			    MAX(info->reverse.max_dimension,
				info->normal.max_dimension));
		info->dimension[1-path].value += info->b_left + info->b_right;
		LimitDimension(&info->dimension[1-path]);
	    }
	/*
	** Compute the dimension in layout path direction. If both stacks
	** are present, then use maximun center separation. If only normal
	** stack is present, then use the leading offset, and if only reverse
	** stack, then use the trailing offset. (NOTE, this is done only
	** if the dimension in path direction is not yet fixed.)
	*/
	if (info->normal.top && info->reverse.top)
		glue = MAX(info->normal.separation, info->reverse.separation);
	else
		glue = MAX(info->normal.end, info->reverse.end);
	glue += info->normal.start + info->reverse.start;
	if (info->dimension[path].value <= 0)
	    {
		info->dimension[path].value =
			info->normal.allocation + info->reverse.allocation
				+ info->b_trailing + glue + info->b_leading;
		info->dimension[path].value =
			MAX(info->dimension[path].value,
			    info->fixed.allocation);
	    }
	/*
	** Phase 2: Try to resize the frame itself
	*/
	if (mode == LAYOUT_QUERY_ONLY && size != NULL)
		/*
		** This branch occurs only if the call to FrameLayout came
		** from the QueryGeometry method.
		**
		** This means that the parent is asking about the preferred
		** geometry. There is no need to do any further processing
		** (like actually finding final positions for children etc)
		*/
		goto done_layout;
	else if ((info->dimension[0].value != fw->core.width ||
		  info->dimension[1].value != fw->core.height))
	    {
		XtGeometryResult result;
		XtWidgetGeometry request, reply;
		int dim[2], changed;
		
		request.request_mode = CWWidth | CWHeight;
		if (!mode)
			request.request_mode |= XtCWQueryOnly;
		dim[0] = request.width = info->dimension[0].value;
		dim[1] = request.height = info->dimension[1].value;
		result = XtMakeGeometryRequest((Widget)fw, &request, &reply);
		if (result == XtGeometryNo)
		    {
			dim[0] = fw->core.width;
		        dim[1] = fw->core.height;
		    }
		else if (result == XtGeometryAlmost)
		    {
			if (mode)
				XtMakeGeometryRequest
					((Widget)fw, &reply,
					 (XtWidgetGeometry *)NULL);
			dim[0] = reply.width;
			dim[1] = reply.height;
		    }
		/*
		** The parent refused to grant new size, redo everything
		** the size parent sees acceptable.
		*/
		changed = 0;
		if (info->dimension[0].value != dim[0])
		    {
			dimension[0].value = dim[0];
			dimension[0].min = 0;
			dimension[0].max = 0;
			changed = 1;
		    }
		if (info->dimension[1].value != dim[1])
		    {
			dimension[1].value = dim[1];
			dimension[1].min = 0;
			dimension[1].max = 0;
			changed = 1;
		    }
		if (changed)
		    {
			XFree((char *)fw->frame.layout_info);
			fw->frame.layout_info = NULL;
			/*
			** ..this should never cause a loop, unless there is
			** some fault in the layout computations... --msa
			*/
			goto redo_layout;
		    }
	    }
	if (info->maximize)
		MaximizeChildren(info, info->maximize,
				 info->dimension[path].value
				 - info->b_leading - glue - info->b_trailing
				 - info->normal.allocation
				 - info->reverse.allocation, path);
	/*
	** Phase 3: Configure floating widgets
	*/
	for (i = 0; i < info->num_children; ++i)
	    {
		int tmp;

		child = &info->list[i];
		if (!XtIsManaged(child->child))
			continue;
		fc = (XeFrameConstraints)child->child->core.constraints;
		tmp = info->dimension[1-path].value
			- child->left - child->right
			- info->b_left - info->b_right;
		if (fc->frame.dimension[1-path].value == XeFrameDimension_MAX)
		    {
			if (fc->frame.dimension[1-path].min > 0 &&
			    fc->frame.dimension[1-path].min > tmp)
				tmp = fc->frame.dimension[1-path].min;
			if (fc->frame.dimension[1-path].max > 0 &&
			    fc->frame.dimension[1-path].max < tmp)
				tmp = fc->frame.dimension[1-path].max;
			if (tmp >= child->dimension[1-path])
				child->dimension[1-path] = tmp;
			else
				/* THIS IS AN ERROR CONDITION! FIX! --msa */
				XeWidgetWarningMsg
					((Widget)fw, "layoutComputeError",
					 "Inconsitent computed width.",
					 (String *)NULL, 0);
		    }
	    }
	ComputeStack(info, info->normal.top,
		     invert ?
		     info->dimension[path].value - info->normal.allocation 
		     - info->b_trailing - info->normal.start:
		     info->b_trailing + info->normal.allocation
		     + info->normal.start,  path, invert);
	ComputeStack(info, info->reverse.top,
		     invert ?
		     info->b_leading + info->reverse.start :
		     info->dimension[path].value
		     - info->b_leading - info->reverse.start,
		     path, invert);
	/*
	** Phase 4: if this is not a query, do overflow check and callbacks
	** for the children that have requested it.
	*/
	if (mode)
	    {
		for (i = 0; i < info->num_children; ++i)
		    {
			XeFrameOverflowCallbackData data;

			child = &info->list[i];
			if (!child->child)
			    {
				info->redo = 1;
				continue;
			    }
			if (!XtIsManaged(child->child))
				continue;
			if (fw->frame.overflow_callbacks == NULL)
				continue;
			if (child->child->core.being_destroyed)
				continue;
			data.widget = child->child;
			data.child.x = child->position[0];
			data.child.y = child->position[1];
			data.child.width = child->dimension[0];
			data.child.height = child->dimension[1];
			data.frame.x = 0;
			data.frame.y = 0;
			data.frame.width = info->dimension[0].value;
			data.frame.height = info->dimension[1].value;
			if (data.child.x >= 0 &&
			    data.child.y >= 0 &&
			    data.frame.x + data.frame.width >=
			    data.child.x + data.child.width &&
			    data.frame.y + data.frame.height >= 
			    data.child.y + data.child.height)
				continue; /* child enclosed, okay */
			data.redo = 0;
			XtCallCallbackList
				((Widget)fw, fw->frame.overflow_callbacks,
				 (XtPointer)&data);
			/*
			** It is actually possible that the child got
			** destroyed in above call (because we were already
			** in Phase2 of destroy). If this happens, there is
			** not much else to do except try redoing the layout.
			*/
			if (data.redo || !child->child ||
			    !XtIsManaged(child->child))
				info->redo = 1;
		    }
	    }
	if (info->redo)
	    {
		/*
		** Do it the hard way, totally. Just in case the
		** number of children or something strange managed
		** to change...? --msa
		*/
		XFree((char *)fw->frame.layout_info);
		fw->frame.layout_info = NULL;
		goto redo_layout;
	    }
	/*
	** Phase 5: commit layout
	** (Forces min width and height as '1', should probably give an
	** error message too!)
	*/
	if (mode)
		for (i = 0; i < info->num_children; ++i)
		    {
			int width, height;
			Widget cw;
			
			child = &info->list[i];
			cw = child->child;

			if (!XtIsManaged(cw))
				continue;
			width = MAX(child->dimension[0], 1);
			height = MAX(child->dimension[1], 1);
			if (fw->frame.printing)
			    {
				/*
				** If printing is set, assume we are really
				** working with the cloned tree. As currently
				** each cloned widget really has the same
				** window, doing reconfigure to it is not
				** the best thing... (especially, when
				** there might not be any real window!!)
				*/
				cw->core.x = child->position[0];
				cw->core.y = child->position[1];
				cw->core.border_width = child->b;
				if (cw->core.width != width ||
				    cw->core.height != height)
				    {
					cw->core.width = width;
					cw->core.height = height;
					/*
					** Call resize method directly if size
					** changed (and if the widget is from
					** Xew, cannot call unknown Resize
					** safely for printing...)
					*/
#ifndef USING_MOTIF_122
					/*
					** MOTIF has some 'ResizeWrapper'
					** around the resize method, and this
					** seems require 'per display
					** information', which 'printer
					** display' does not have! Cannot
					** do anything else, but skip the
					** resize totally for MOTIF subclassed
					** Xew...
					*/
					if (cw->core.widget_class->core_class.resize &&
					    XtIsSubclass(cw, xeBasicWidgetClass))
						(*cw->core.widget_class->
						 core_class.resize)
							((Widget)cw);
#endif
				    }
			    }
			else
				XtConfigureWidget
					(cw,
					 child->position[0],
					 child->position[1],
					 width,
					 height,
					 child->b);
		    }
	/*
	** Phase 6: if this was called from GeometryManager for a specific
	**	child, then return computed geometry.
	*/
	if (one)
		for (i = 0; i < info->num_children; ++i)
			if (info->list[i].child == one->child)
			    {
				*one = info->list[i];
				break;
			    }
    done_layout:
	/*
	** Return the size to the caller, if requested
	*/
	if (size)
	    {
		size->width.value = info->dimension[0].value;
		size->height.value = info->dimension[1].value;
	    }
	XFree((char *)fw->frame.layout_info);
	fw->frame.layout_info = NULL;
    }

/*
** GeometryManager method
**	As all good Geometry Managers should, we will return No if the
**	request will have no effect; i.e. when the requestor is already
**	of the desired geometry.
*/
static XtGeometryResult GeometryManager(w, request, reply)
Widget w;
XtWidgetGeometry *request, *reply;
    {
	static LayoutChild init_LayoutChild;

	XeFrameWidget fw = (XeFrameWidget)XtParent(w);
	LayoutChild child;
	XtWidgetGeometry rq;

	/*
	** Because request and reply may actually be the same, we have
	** to make a local copy of the request.
	*/
	rq = *request;
	
	child = init_LayoutChild;	/* Initialize all fields */
	child.child = w;
	child.position[0] = (rq.request_mode & CWX) ? rq.x : w->core.x;
	child.position[1] = (rq.request_mode & CWY) ? rq.y : w->core.y;
	child.dimension[0]= 
		(rq.request_mode & CWWidth) ? rq.width : w->core.width;
	child.dimension[1]=
		(rq.request_mode & CWHeight) ? rq.height : w->core.height;
	child.b = (rq.request_mode & CWBorderWidth) ?
		rq.border_width : w->core.border_width;
	child.no_query = True;

	if (fw->frame.layout_info != 0) /* Recursive calls */
		return XtGeometryNo;
	if (child.dimension[0] == w->core.width &&
	    child.dimension[1] == w->core.height &&
	    child.position[0] == w->core.x &&
	    child.position[1] == w->core.y &&
	    child.b == w->core.border_width)
		return XtGeometryNo;
	/*
	** One needs to study the request more closely...
	*/
	FrameLayout(fw, &child, LAYOUT_QUERY_ONLY, (FrameSize *)NULL);
	if (reply)
	    {
		reply->request_mode = CWWidth|CWHeight|CWX|CWY|CWBorderWidth;
		reply->x = child.position[0];
		reply->y = child.position[1];
		reply->width = child.dimension[0];
		reply->height = child.dimension[1];
		reply->border_width = child.b;
	    }
	if ((!(rq.request_mode&CWWidth) || rq.width == child.dimension[0]) &&
	    (!(rq.request_mode&CWHeight) || rq.height == child.dimension[1]) &&
	    (!(rq.request_mode&CWBorderWidth) || rq.border_width == child.b) &&
	    (!(rq.request_mode&CWX) || rq.x == child.position[0]) &&
	    (!(rq.request_mode&CWY) || rq.y == child.position[1]))
	    {
		if (rq.request_mode & XtCWQueryOnly)
			return XtGeometryYes;
		FrameLayout(fw, &child, LAYOUT_AND_COMMIT, (FrameSize *)NULL);
		if (XtIsRealized((Widget)fw))
			/* This is brutal! Fine tune later! --msa */
			XClearArea(XtDisplay(fw),XtWindow(fw),0,0,0,0,True);
		return XtGeometryDone;
	    }
        return reply ? XtGeometryAlmost : XtGeometryNo;
    }


static void Initialize(request, init)
Widget request, init;
    {
	XeFrameWidget fw = (XeFrameWidget)init;

	fw->frame.layout_info = NULL;
	/*
	** Initialize Border drawing GC and border colors
	*/
	fw->frame.gc = _XeFrameCreateGC
		(init, &fw->frame.line,
		 fw->frame.border, XtNumber(fw->frame.border),
		 fw->frame.shadow_contrast);
	fw->frame.printing = False;
    }

static void Realize(w, valueMask, attributes)
Widget w;
Mask *valueMask;
XSetWindowAttributes *attributes;
    {
	XeFrameWidget fw = (XeFrameWidget) w;

	FrameLayout(fw,(LayoutChild *)NULL,LAYOUT_AND_COMMIT,(FrameSize*)NULL);
	/*
	** Ad hoc safety kludge: enforce minimum size 1 for both dimensions!
	*/
	if (!fw->core.width)
		fw->core.width = 1;
	if (!fw->core.height)
		fw->core.height = 1;
	(*SuperClass->core_class.realize) (w, valueMask, attributes);
    }

static void Destroy(w)
Widget w;
    {
	XeFrameWidget fw = (XeFrameWidget)w;

	_XeFrameFreeGC(w, fw->frame.gc,
		       fw->frame.border, XtNumber(fw->frame.border));
    } 

static void InsertChild(w)
register Widget w;
    {
	/*
	** insert the child widget in the composite children list with the
	** superclass insert_child routine.
	*/
	(*SuperClass->composite_class.insert_child)(w);
    }

static void DeleteChild(w)
Widget w;
    {
	XeFrameWidget fw = (XeFrameWidget)XtParent(w);
	if (fw->frame.layout_info)
	    {
		/*
		** The parent is currently doing layout and this child is
		** just being ripped off under it. Remove the reference
		** to this child from the layout info (This happens, not
		** sure how.. but it does.. --msa)
		*/
		int i = fw->frame.layout_info->num_children;

		while (--i >= 0)
			if (fw->frame.layout_info->list[i].child == w)
			    {
				fw->frame.layout_info->list[i].child = 0;
				break;
			    }
	    }
	/*
	** delete the child widget in the composite children list with the
	** superclass delete_child routine.
	*/
	(*SuperClass->composite_class.delete_child) (w);
    }

static void ChangeManaged(w)
Widget w;
    {
	XeFrameWidget fw = (XeFrameWidget)w;

	FrameLayout(fw,(LayoutChild *)NULL,LAYOUT_AND_COMMIT,(FrameSize*)NULL);

#ifdef USING_MOTIF_122
        /* update the traversal */
        _XmNavigChangeManaged(w);
#endif
        
    }

static void Resize(w)
Widget w;
    {
	XeFrameWidget fw = (XeFrameWidget)w;
	FrameSize size;

	/*
	** If the layout_info structure is already allocated just assume
	** layout will end up okay and do nothing. Otherwise, do layout
	** to exact requested size.
	*/
	if (!fw->frame.layout_info)
	    {
		size.width.value = fw->core.width;
		size.width.max = size.width.min = 0;
		size.height.value = fw->core.height;
		size.height.max = size.height.min = 0;
		size.border_width = fw->core.border_width;
		FrameLayout(fw, (LayoutChild *)NULL, LAYOUT_AND_COMMIT, &size);
	    }
    }

static void Redisplay(w, event, region)
Widget w;
XExposeEvent *event;
Region region;
    {
	XeFrameWidget fw = (XeFrameWidget)w;
	int i, left, right, top, bottom;

	if (fw->frame.layout_path == 0)
	    {
		left = FRAME_TRAILING;
		right = FRAME_LEADING;
		top = FRAME_LEFT;
		bottom = FRAME_RIGHT;
	    }
	else if (fw->frame.layout_path == 180)
	    {
		left = FRAME_LEADING;
		right = FRAME_TRAILING;
		top = FRAME_RIGHT;
		bottom = FRAME_LEFT;
	    }
	else if (fw->frame.layout_path == 90)
	    {
		left = FRAME_LEFT;
		right = FRAME_RIGHT;
		top = FRAME_LEADING;
		bottom = FRAME_TRAILING;
	    }
	else
	    {
		left = FRAME_RIGHT;
		right = FRAME_LEFT;
		top = FRAME_TRAILING;
		bottom = FRAME_LEADING;
	    }
	_XeDrawOwnBorder
		(w, fw->frame.gc, &fw->frame.line,
		 &fw->frame.border[left], &fw->frame.border[right],
		 &fw->frame.border[top], &fw->frame.border[bottom],
		 fw->frame.invert_shadow != 0);
	for (i = 0; i < fw->composite.num_children; ++i)
	    {
		XeFrameConstraints fc;
		Widget child = fw->composite.children[i];

		if (!XtIsManaged(child))
			continue;
		if (XtIsSubclass(child, fw->core.widget_class))
			continue; /* Frames draw borders themselves */
		fc = (XeFrameConstraints)child->core.constraints;
		_XeDrawChildBorder(w, fw->frame.gc, &fw->frame.line, child,
				   &fc->frame.border[left],
				   &fc->frame.border[right],
				   &fc->frame.border[top],
				   &fc->frame.border[bottom],
				   fc->frame.invert_shadow != 0);
	    }
	if (fw->basic.expose_callbacks &&
	    fw->core.widget_class == xeFrameWidgetClass)
	    {
		XeExposeCallbackData data;
		data.reason = XeCR_EXPOSE;
		data.event = event;
		data.region = region;
		XtCallCallbackList
			(w, fw->basic.expose_callbacks, (XtPointer)&data);
	    }
    }

/*
** QueryGeometry
*/
static XtGeometryResult QueryGeometry(w, request, preferred)
Widget w;
XtWidgetGeometry *request;
XtWidgetGeometry *preferred;
    {
	XeFrameWidget fw = (XeFrameWidget)w;
	XtWidgetGeometry rq;
	FrameSize size;

	rq = *request; /* make a local copy, just in case... */

	/*
	** NOTE: border_width is always accepted as is, if requested,
	** Thus whether it is requested or not does not affect the return
	** directly (it may indirectly affect width/height and make 'Yes'
	** into 'Almost' or 'No').
	*/
	if (rq.request_mode & CWWidth)
	    {
		size.width.value = rq.width;
		size.width.min = size.width.max = 0;
	    }
	else
		size.width = fw->basic.dimension[0];
	if (rq.request_mode & CWHeight)
	    {
		size.height.value = rq.height;
		size.height.min = size.height.max = 0;
	    }
	else
		size.height = fw->basic.dimension[1];
	size.border_width = (rq.request_mode & CWBorderWidth)
		? rq.border_width : fw->core.border_width;
	FrameLayout(fw, (LayoutChild *)NULL, LAYOUT_QUERY_ONLY, &size);
	preferred->request_mode = CWWidth | CWHeight | CWBorderWidth;
	preferred->width = size.width.value;
	preferred->height = size.height.value;
	preferred->border_width = size.border_width;
	if (((rq.request_mode & (CWWidth|CWHeight)) == (CWWidth|CWHeight)) &&
	    rq.width == preferred->width && rq.height == preferred->height)
		return XtGeometryYes;
	else if (preferred->width == w->core.width &&
		 preferred->height == w->core.height)
		return XtGeometryNo;
	else
		return XtGeometryAlmost;
    }
/*
**	****************
**	Printing support
**	****************
**	As frames are often just large empty space, it makes some sense
**	to have printing methods for them. The default print method with
**	the heavy XGetImage/XPutImage processing can be avoided.
*/
static void PrintInitialize(w, c, pp)
Widget w, c;
XePrintProcess *pp;
    {
	XeFrameWidget fw = (XeFrameWidget)w;
	XeFrameWidget clone = (XeFrameWidget)c;
	FrameSize size;

	Initialize(w, c);
	clone->frame.printing = fw->frame.printing = True;
	size.width.value = clone->core.width;
	size.width.max = size.width.min = 0;
	size.height.value = clone->core.height;
	size.height.max = size.height.min = 0;
	FrameLayout(clone, (LayoutChild *)NULL, LAYOUT_AND_COMMIT, &size);
	if (!XtParent(clone))
	    {
		clone->core.width = size.width.value;
		clone->core.height = size.height.value;
	    }
    }

static XtPointer Print(w, c, x, y, pp)
Widget w, c;
int x, y;
XePrintProcess pp;
    {
	XeFrameWidget clone = (XeFrameWidget)c;

	clone->basic.x_translation = x;
	clone->basic.y_translation = y;
	Redisplay(c, (XEvent *)NULL, (Region)0);
	return NULL;
    }

static void PrintCancel(w, clone, pp, client_data)
Widget w, clone;
XePrintProcess pp;
XtPointer client_data;
    {
    }


static void PrintDestroy(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	XeFrameWidget fw = (XeFrameWidget)w;

	fw->frame.printing = False;
	Destroy(c);
    }

static void PrintConstraintInitialize(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	_XeFrameInitialize(w, c, (ArgList)0, (Cardinal)0);
    }

static void PrintConstraintDestroy(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	_XeFrameDestroy(c);
    }

/*
**	**************
*/

static Boolean CheckBorder(old, set)
XeFrameBorderP *old, *set;
    {
	int i;

	for (i = 0; i < 4; ++i)
		if (old[i].value.width != set[i].value.width ||
		    old[i].value.space != set[i].value.space)
			return True;
	return False;
    }

static Boolean CheckDimension(old, set)
XeFrameDimension *old, *set;
    {
	return	old[0].value != set[0].value ||
		old[0].min != set[0].min ||
		old[0].max != set[0].max ||
		old[1].value != set[1].value ||
		old[1].min != set[1].min ||
		old[1].max != set[1].max;
    }


static Boolean  SetValues(old, request, set)
Widget old, request, set;
    {
	XeFrameWidget cw = (XeFrameWidget)old;
	XeFrameWidget nw = (XeFrameWidget)set;
	Boolean changed, layout;
	int i;

	/*
	** Force layout_path into one of the legal values
	*/
	if (nw->frame.layout_path != 0 &&
	    nw->frame.layout_path != 90 &&
	    nw->frame.layout_path != 180 &&
	    nw->frame.layout_path != 270)
		nw->frame.layout_path = 270;
	for (i = 0, changed = False; i < XtNumber(nw->frame.border); ++i)
		changed |= _XeChangeBorderColor
			(set, cw->frame.border, nw->frame.border, i,
			 cw->frame.shadow_contrast,
			 nw->frame.shadow_contrast);
	changed |= cw->frame.invert_shadow != nw->frame.invert_shadow;
	layout= CheckBorder(cw->frame.border, nw->frame.border) ||
		CheckDimension(cw->basic.dimension, nw->basic.dimension) ||
		cw->frame.layout_path != nw->frame.layout_path;
	if (layout)
		FrameLayout(nw,(LayoutChild *)NULL,LAYOUT_AND_COMMIT,
			    (FrameSize*)NULL);
	return True;
    }

static Boolean FrameSetValues(old, request, set, args, num_args)
Widget old, request, set;
ArgList args;
Cardinal *num_args;
    {
	XeFrameWidget fw = (XeFrameWidget)XtParent(set);
	int result = _XeFrameSetValues(old, request, set, args, num_args);

	if (result & XeFrameSetValues_GEOMETRY)
	    {
		FrameLayout(fw,(LayoutChild *)NULL,LAYOUT_AND_COMMIT,
			    (FrameSize*)NULL);
		if (XtIsRealized((Widget)fw))
			/* This is brutal! Fine tune later! --msa */
			XClearArea(XtDisplay(fw),XtWindow(fw),0,0,0,0,True);
	    }
	else if ((result & XeFrameSetValues_FRAMING) &&
		 XtIsRealized((Widget)fw))
	    {
		/*
		** Only framing attributes changed. Generate expose only to
		** the area child and framing.
		*/
		XeFrameExtents FE;

		_XeFrameExtents(set, set->core.width, set->core.height, &FE);
		XClearArea(XtDisplay(fw), XtWindow((Widget)fw),
			   set->core.x - FE.window.x,
			   set->core.y - FE.window.y,
			   FE.width, FE.height, True);
	    }
	return True;
    }

static void Notify(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
    {
	XeNotifyCallbackData call_data;
	call_data.reason = XeCR_NOTIFY;
	call_data.event = event;
	call_data.params = params;
	call_data.num_params = *num_params;
	XtCallCallbackList(w, ((XeFrameWidget)w)->basic.notify_callbacks,
			   (XtPointer)&call_data);
    }

#ifdef USING_MOTIF_122
/************************************************************************
 *
 *  WidgetNavigable - routine to handle traversal to children of this
 *  widget, if there are any, or else the widget itself, if no children. 
 *
 ************************************************************************/
static XmNavigability
WidgetNavigable( wid)
        Widget wid ;
{   
    if(wid->core.sensitive
       &&  wid->core.ancestor_sensitive
       &&  ((XmManagerWidget) wid)->manager.traversal_on) {
        XmNavigationType nav_type = ((XmManagerWidget) wid)
            ->manager.navigation_type ;
        Widget *children = ((XmManagerWidget) wid)->composite.children ;
        unsigned idx = 0 ;

        while( idx < ((XmManagerWidget) wid)->composite.num_children) {
            if(_XmGetNavigability( children[idx])) {
                if( (nav_type == XmSTICKY_TAB_GROUP)
                    ||  (nav_type == XmEXCLUSIVE_TAB_GROUP)
                    ||  (    (nav_type == XmTAB_GROUP)
                             &&  !_XmShellIsExclusive( wid))) {
                    return XmDESCENDANTS_TAB_NAVIGABLE ;
                }
                return XmDESCENDANTS_NAVIGABLE ;
            }
            ++idx ;
        }
            
        if((nav_type == XmSTICKY_TAB_GROUP)
           ||  (nav_type == XmEXCLUSIVE_TAB_GROUP)
           ||  ((nav_type == XmTAB_GROUP)
                &&  !_XmShellIsExclusive( wid))) {
            return XmTAB_NAVIGABLE ;
        }
        return XmCONTROL_NAVIGABLE ;
    }
    return XmNOT_NAVIGABLE ;
}

#endif
